/**
 * JottNote Chrome Extension - Main Popup Controller
 * Orchestrates all modules and handles UI interactions
 */

import {
    getNotes,
    getNotesByCreation,
    getNote,
    createNote,
    saveNote,
    deleteNote,
    getCurrentNoteId,
    setCurrentNoteId,
    getSettings,
    saveSettings,
    getTemplates,
    getTemplate,
    saveTemplate,
    deleteTemplate,
    getPlaceholders,
    savePlaceholders
} from './js/storage.js';

import {
    getBookmarks,
    getBookmark,
    saveBookmark,
    deleteBookmark,
    searchBookmarks,
    getAllTags,
    importFromChrome,
    updateAccessTime,
    extractTitleFromUrl,
    exportBookmarksToJSON,
    importBookmarksFromJSON
} from './js/bookmarks.js';

import {
    getSessions,
    getSession,
    saveSession,
    deleteSession,
    updateSession,
    saveCurrentSession,
    openSession,
    searchSessions,
    getAllSessionTags,
    exportSessionsToJSON,
    importSessionsFromJSON
} from './js/sessions.js';

import {
    goToPrevious,
    goToNext,
    jumpToFront,
    promoteNote,
    deleteCurrentNote,
    getNavigationInfo
} from './js/navigation.js';

import {
    initEditor,
    handleContentChange,
    forceSave,
    getCurrentContent,
    getCurrentNote,
    setCurrentNote,
    smartCopy,
    smartPaste,
    positionCursorAtEnd,
    positionCursorAtStart
} from './js/editor.js';

import {
    parseKeyword,
    shouldShowSlashCommand,
    getSlashKeywords,
    insertKeyword,
    parseTemplateCommand,
    updateKeywordsConfig
} from './js/keywords.js';

import { parseGameCommand } from './js/keywords.js';
import { parseBookmarksCommand } from './js/keywords.js';
import { parseSessionCommand } from './js/keywords.js';
import { parsePlaceholdersCommand } from './js/keywords.js';


import {
    processMathContent,
    renderMathHTML,
    resetVariables
} from './js/math-mode.js';

import {
    processDateContent,
    getFormattedToday
} from './js/date-mode.js';

import { THEMES, validateCustomTheme, CUSTOM_THEME_KEY } from './js/themes.js';

import { CURRENCY_LIST, CRYPTO_LIST } from './js/currency-data.js';
import { getStatsResult, countStats } from './js/stats-mode.js';
import { parseSortCommand, applySortToContent, shouldShowSortHelp } from './js/sort-mode.js';

import { handleLinkClick, autoShortenUrls, toggleLinkAtCursor, detectUrls, getFullUrlFromShortened, rebuildUrlMap, clearUrlMap } from './js/links.js';

import {
    parseTimerCommand,
    startTimer,
    stopTimer,
    togglePause,
    restartTimer,
    isTimerRunning,
    isTimerPaused
} from './js/timer.js';

import {
    parseCodeKeyword,
    highlightCode,
    isCodeMode,
    getSupportedLanguages
} from './js/code-mode.js';

import {
    initSpellCheck,
    setSpellCheckEnabled,
    isSpellCheckEnabled,
    findMisspelledWords,
    getSpellingSuggestions
} from './js/spell-check.js';

import {
    initLicense,
    isPro,
    openUpgrade,
    isThemeFree,
    isProMode,
    PRO_KEYWORDS
} from './js/license.js';

import { processUuidCommand } from './js/uuid-mode.js';
import { processLoremCommand } from './js/lorem-mode.js';
import { fetchWeather, fetchWeatherTomorrow, fetchWeatherDaily, parseWeatherCommand } from './js/weather-mode.js';
import {
    startStopwatch,
    stopStopwatch,
    toggleStopwatchPause,
    lapStopwatch,
    resetStopwatch,
    isStopwatchRunning
} from './js/stopwatch-mode.js';

import {
    initTicTacToe
} from './js/game-tictactoe.js';

import {
    initHangman
} from './js/game-hangman.js';

import {
    initQuiz
} from './js/game-quiz.js';

import {
    fetchWordData
} from './js/dictionary-mode.js';
import { parseDictionaryCommand } from './js/keywords.js';

import {
    initTypingTest
} from './js/game-type.js';

import {
    initReactionTime
} from './js/game-react.js';

import {
    initMinesweeper
} from './js/game-mines.js';

import {
    initMemory
} from './js/game-memory.js';

import {
    initWordle
} from './js/game-wordle.js';

// Syntax highlighting theme palettes
const SYNTAX_THEMES = {
    // ═══════════ Dark Themes (10) ═══════════
    'one-dark': {
        label: 'One Dark',
        mode: 'dark',
        colors: {
            '--syntax-base': '#abb2bf', '--syntax-tag': '#e06c75', '--syntax-keyword': '#c678dd',
            '--syntax-string': '#98c379', '--syntax-number': '#d19a66', '--syntax-function': '#61afef',
            '--syntax-property': '#56b6c2', '--syntax-comment': '#5c6370', '--syntax-punct': '#abb2bf',
            '--syntax-attr': '#e5c07b', '--syntax-selector': '#c678dd', '--syntax-value': '#e5c07b'
        }
    },
    'monokai': {
        label: 'Monokai',
        mode: 'dark',
        colors: {
            '--syntax-base': '#f8f8f2', '--syntax-tag': '#f92672', '--syntax-keyword': '#f92672',
            '--syntax-string': '#e6db74', '--syntax-number': '#ae81ff', '--syntax-function': '#a6e22e',
            '--syntax-property': '#66d9ef', '--syntax-comment': '#75715e', '--syntax-punct': '#f8f8f2',
            '--syntax-attr': '#a6e22e', '--syntax-selector': '#f92672', '--syntax-value': '#ae81ff'
        }
    },
    'dracula': {
        label: 'Dracula',
        mode: 'dark',
        colors: {
            '--syntax-base': '#f8f8f2', '--syntax-tag': '#ff79c6', '--syntax-keyword': '#ff79c6',
            '--syntax-string': '#f1fa8c', '--syntax-number': '#bd93f9', '--syntax-function': '#50fa7b',
            '--syntax-property': '#8be9fd', '--syntax-comment': '#6272a4', '--syntax-punct': '#f8f8f2',
            '--syntax-attr': '#50fa7b', '--syntax-selector': '#ff79c6', '--syntax-value': '#bd93f9'
        }
    },
    'nord': {
        label: 'Nord',
        mode: 'dark',
        colors: {
            '--syntax-base': '#d8dee9', '--syntax-tag': '#bf616a', '--syntax-keyword': '#b48ead',
            '--syntax-string': '#a3be8c', '--syntax-number': '#b48ead', '--syntax-function': '#88c0d0',
            '--syntax-property': '#8fbcbb', '--syntax-comment': '#616e88', '--syntax-punct': '#d8dee9',
            '--syntax-attr': '#d08770', '--syntax-selector': '#b48ead', '--syntax-value': '#ebcb8b'
        }
    },
    'solarized-dark': {
        label: 'Solarized Dark',
        mode: 'dark',
        colors: {
            '--syntax-base': '#839496', '--syntax-tag': '#cb4b16', '--syntax-keyword': '#859900',
            '--syntax-string': '#2aa198', '--syntax-number': '#d33682', '--syntax-function': '#268bd2',
            '--syntax-property': '#2aa198', '--syntax-comment': '#586e75', '--syntax-punct': '#839496',
            '--syntax-attr': '#b58900', '--syntax-selector': '#859900', '--syntax-value': '#cb4b16'
        }
    },
    'paraiso-dark': {
        label: 'Paraiso Dark',
        mode: 'dark',
        colors: {
            '--syntax-base': '#a39e9b', '--syntax-tag': '#ef6155', '--syntax-keyword': '#815ba4',
            '--syntax-string': '#48b685', '--syntax-number': '#f99b15', '--syntax-function': '#06b6ef',
            '--syntax-property': '#5bc4bf', '--syntax-comment': '#776e71', '--syntax-punct': '#a39e9b',
            '--syntax-attr': '#fec418', '--syntax-selector': '#815ba4', '--syntax-value': '#ef6155'
        }
    },
    'tokyo-night': {
        label: 'Tokyo Night',
        mode: 'dark',
        colors: {
            '--syntax-base': '#a9b1d6', '--syntax-tag': '#f7768e', '--syntax-keyword': '#bb9af7',
            '--syntax-string': '#9ece6a', '--syntax-number': '#ff9e64', '--syntax-function': '#7aa2f7',
            '--syntax-property': '#73daca', '--syntax-comment': '#565f89', '--syntax-punct': '#a9b1d6',
            '--syntax-attr': '#e0af68', '--syntax-selector': '#bb9af7', '--syntax-value': '#ff9e64'
        }
    },
    'gruvbox-dark': {
        label: 'Gruvbox Dark',
        mode: 'dark',
        colors: {
            '--syntax-base': '#ebdbb2', '--syntax-tag': '#fb4934', '--syntax-keyword': '#fb4934',
            '--syntax-string': '#b8bb26', '--syntax-number': '#d3869b', '--syntax-function': '#fabd2f',
            '--syntax-property': '#83a598', '--syntax-comment': '#928374', '--syntax-punct': '#ebdbb2',
            '--syntax-attr': '#8ec07c', '--syntax-selector': '#fb4934', '--syntax-value': '#fe8019'
        }
    },
    'ayu-dark': {
        label: 'Ayu Dark',
        mode: 'dark',
        colors: {
            '--syntax-base': '#bfbdb6', '--syntax-tag': '#f07178', '--syntax-keyword': '#ff8f40',
            '--syntax-string': '#aad94c', '--syntax-number': '#e6b450', '--syntax-function': '#ffb454',
            '--syntax-property': '#39bae6', '--syntax-comment': '#636a72', '--syntax-punct': '#bfbdb6',
            '--syntax-attr': '#59c2ff', '--syntax-selector': '#ff8f40', '--syntax-value': '#e6b450'
        }
    },
    'catppuccin-mocha': {
        label: 'Catppuccin Mocha',
        mode: 'dark',
        colors: {
            '--syntax-base': '#cdd6f4', '--syntax-tag': '#f38ba8', '--syntax-keyword': '#cba6f7',
            '--syntax-string': '#a6e3a1', '--syntax-number': '#fab387', '--syntax-function': '#89b4fa',
            '--syntax-property': '#94e2d5', '--syntax-comment': '#6c7086', '--syntax-punct': '#cdd6f4',
            '--syntax-attr': '#f9e2af', '--syntax-selector': '#cba6f7', '--syntax-value': '#f38ba8'
        }
    },

    // ═══════════ Light Themes (10) ═══════════
    'github-light': {
        label: 'GitHub Light',
        mode: 'light',
        colors: {
            '--syntax-base': '#24292e', '--syntax-tag': '#22863a', '--syntax-keyword': '#d73a49',
            '--syntax-string': '#032f62', '--syntax-number': '#005cc5', '--syntax-function': '#6f42c1',
            '--syntax-property': '#005cc5', '--syntax-comment': '#6a737d', '--syntax-punct': '#24292e',
            '--syntax-attr': '#6f42c1', '--syntax-selector': '#22863a', '--syntax-value': '#005cc5'
        }
    },
    'solarized-light': {
        label: 'Solarized Light',
        mode: 'light',
        colors: {
            '--syntax-base': '#657b83', '--syntax-tag': '#cb4b16', '--syntax-keyword': '#859900',
            '--syntax-string': '#2aa198', '--syntax-number': '#d33682', '--syntax-function': '#268bd2',
            '--syntax-property': '#2aa198', '--syntax-comment': '#93a1a1', '--syntax-punct': '#657b83',
            '--syntax-attr': '#b58900', '--syntax-selector': '#859900', '--syntax-value': '#cb4b16'
        }
    },
    'one-light': {
        label: 'One Light',
        mode: 'light',
        colors: {
            '--syntax-base': '#383a42', '--syntax-tag': '#e45649', '--syntax-keyword': '#a626a4',
            '--syntax-string': '#50a14f', '--syntax-number': '#986801', '--syntax-function': '#4078f2',
            '--syntax-property': '#0184bc', '--syntax-comment': '#a0a1a7', '--syntax-punct': '#383a42',
            '--syntax-attr': '#c18401', '--syntax-selector': '#a626a4', '--syntax-value': '#e45649'
        }
    },
    'paraiso-light': {
        label: 'Paraiso Light',
        mode: 'light',
        colors: {
            '--syntax-base': '#4f424c', '--syntax-tag': '#ef6155', '--syntax-keyword': '#815ba4',
            '--syntax-string': '#48b685', '--syntax-number': '#f99b15', '--syntax-function': '#06b6ef',
            '--syntax-property': '#5bc4bf', '--syntax-comment': '#8d8687', '--syntax-punct': '#4f424c',
            '--syntax-attr': '#fec418', '--syntax-selector': '#815ba4', '--syntax-value': '#ef6155'
        }
    },
    'ayu-light': {
        label: 'Ayu Light',
        mode: 'light',
        colors: {
            '--syntax-base': '#5c6166', '--syntax-tag': '#f07178', '--syntax-keyword': '#fa8d3e',
            '--syntax-string': '#86b300', '--syntax-number': '#a37acc', '--syntax-function': '#f2ae49',
            '--syntax-property': '#399ee6', '--syntax-comment': '#abb0b6', '--syntax-punct': '#5c6166',
            '--syntax-attr': '#399ee6', '--syntax-selector': '#fa8d3e', '--syntax-value': '#f07178'
        }
    },
    'gruvbox-light': {
        label: 'Gruvbox Light',
        mode: 'light',
        colors: {
            '--syntax-base': '#3c3836', '--syntax-tag': '#9d0006', '--syntax-keyword': '#9d0006',
            '--syntax-string': '#79740e', '--syntax-number': '#8f3f71', '--syntax-function': '#b57614',
            '--syntax-property': '#427b58', '--syntax-comment': '#928374', '--syntax-punct': '#3c3836',
            '--syntax-attr': '#076678', '--syntax-selector': '#9d0006', '--syntax-value': '#af3a03'
        }
    },
    'catppuccin-latte': {
        label: 'Catppuccin Latte',
        mode: 'light',
        colors: {
            '--syntax-base': '#4c4f69', '--syntax-tag': '#d20f39', '--syntax-keyword': '#8839ef',
            '--syntax-string': '#40a02b', '--syntax-number': '#fe640b', '--syntax-function': '#1e66f5',
            '--syntax-property': '#179299', '--syntax-comment': '#9ca0b0', '--syntax-punct': '#4c4f69',
            '--syntax-attr': '#df8e1d', '--syntax-selector': '#8839ef', '--syntax-value': '#d20f39'
        }
    },
    'vs-light': {
        label: 'VS Light',
        mode: 'light',
        colors: {
            '--syntax-base': '#000000', '--syntax-tag': '#800000', '--syntax-keyword': '#0000ff',
            '--syntax-string': '#a31515', '--syntax-number': '#098658', '--syntax-function': '#795e26',
            '--syntax-property': '#001080', '--syntax-comment': '#008000', '--syntax-punct': '#000000',
            '--syntax-attr': '#e50000', '--syntax-selector': '#800000', '--syntax-value': '#0000ff'
        }
    },
    'xcode-light': {
        label: 'Xcode Light',
        mode: 'light',
        colors: {
            '--syntax-base': '#000000', '--syntax-tag': '#643820', '--syntax-keyword': '#ad3da4',
            '--syntax-string': '#d12f1b', '--syntax-number': '#272ad8', '--syntax-function': '#4b21b0',
            '--syntax-property': '#0b4f79', '--syntax-comment': '#707f8c', '--syntax-punct': '#000000',
            '--syntax-attr': '#947100', '--syntax-selector': '#ad3da4', '--syntax-value': '#272ad8'
        }
    },
    'tokyo-night-light': {
        label: 'Tokyo Night Light',
        mode: 'light',
        colors: {
            '--syntax-base': '#343b59', '--syntax-tag': '#8c4351', '--syntax-keyword': '#7847bd',
            '--syntax-string': '#485e30', '--syntax-number': '#965027', '--syntax-function': '#34548a',
            '--syntax-property': '#33635c', '--syntax-comment': '#9699a3', '--syntax-punct': '#343b59',
            '--syntax-attr': '#8f5e15', '--syntax-selector': '#7847bd', '--syntax-value': '#8c4351'
        }
    }
};

// DOM Elements
let editor;
let headerToolbar;
let footerToolbar;
let keywordIndicator;
let slashPopup;
let settingsPanel;
let searchPanel;
let noteDate;
let timerDisplay;
let timerMinutes;
let timerSeconds;
let timerProgress;
let timerRounds;
let timerCompleteModal;
let timerChime;
let codeHelp;
let codeLanguagesHelp;
let timerHelp;
let codeEditorContainer;
let codeEditable;
let statsDisplay;
let statsLabel;
let statsValue;
let statsDetail;
let sumHelp;
let avgHelp;
let countHelp;
let listHelp;
let listDisplay;
let listBreakIndex = Infinity; // Line index where list section ends (for cursor padding)
let mathHelp;
let mathDisplay;
let countDisplay;
let editorHighlighter;
let currencyHelp;
let gameDisplay;
let gameHelp;
let dateHelp;
let dateDisplay;
let currencyListContent;
let findReplacePanel;
let findInput;
let findModeSelect;
let findCount;
let findExpandBtn;
let replaceRow;
let replaceInput;
let timerStartPopup;
let timerHoverPopup;
let linkHoverPopup;
let linkHoverUrl;
let templateHelp;
let templateList;
let placeholdersHelp;
let placeholdersList;
let sortHelp;
let spellCheckContextMenu;
let uuidHelp;
let uuidDisplay;
let loremHelp;
let weatherHelp;
let weatherDisplay;
let timerMsEl;
let timerLapsEl;
let bookmarksDisplay;
let bookmarksHelp;
let bookmarksTagsList;
let sessionsDisplay;
let sessionsHelp;
let sessionsTagsList;
let dictionaryHelp;
let dictionaryDisplay;
let sessionEditPanel;
let sessionEditTitle;
let sessionEditTags;
let sessionEditUrlsList;
let sessionDeleteSelectedBtn;
let currentEditingSessionId = null;

// State
let currentMode = null;
let settings = {};
let isCreatingTemplate = false;
let isCreatingPlaceholders = false;
let placeholdersCreateStartLine = -1;
let templateCreateStartLine = -1;
let isLoadingTemplate = false;
let isSortingMode = false;
let isAddingBookmark = false;
let currentBookmarkEntry = { url: '', tags: '', description: '' };
let bookmarkEntryStep = 0; // 0=url, 1=tags, 2=description
let bookmarksToSave = [];
let isEditingBookmark = false;
let editingBookmarkId = null;
let importDialogOpen = false;

// Input generation counter — incremented on each editor input event.
// Async mode-update chains check this to abandon stale work when a newer
// input has fired (prevents race conditions where old async completions
// re-show overlays after content was cleared).
let editorInputGeneration = 0;

// Find/Replace state
let findMatches = [];
let currentMatchIndex = -1;

// Fun facts for timer completion
const FUN_FACTS = [
    // WILDLIFE
    { category: 'WILDLIFE FACT-FILE', fact: 'Chimpanzees share over 98% of their DNA with humans.' },
    { category: 'WILDLIFE FACT-FILE', fact: 'Octopuses have three hearts and blue blood.' },
    { category: 'WILDLIFE FACT-FILE', fact: 'Elephants are the only animals that can\'t jump.' },
    { category: 'WILDLIFE FACT-FILE', fact: 'A group of flamingos is called a "flamboyance".' },
    { category: 'WILDLIFE FACT-FILE', fact: 'Dolphins sleep with one eye open.' },
    { category: 'WILDLIFE FACT-FILE', fact: 'A shrimp\'s heart is located in its head.' },
    { category: 'WILDLIFE FACT-FILE', fact: 'Sloths can hold their breath longer than dolphins — up to 40 minutes.' },
    { category: 'WILDLIFE FACT-FILE', fact: 'Crows can recognize and remember human faces for years.' },
    { category: 'WILDLIFE FACT-FILE', fact: 'Sea otters hold hands while sleeping to keep from drifting apart.' },
    { category: 'WILDLIFE FACT-FILE', fact: 'A group of owls is called a "parliament".' },
    { category: 'WILDLIFE FACT-FILE', fact: 'Butterflies taste with their feet.' },
    { category: 'WILDLIFE FACT-FILE', fact: 'Koalas sleep up to 22 hours per day.' },
    { category: 'WILDLIFE FACT-FILE', fact: 'Hummingbirds are the only birds that can fly backwards.' },
    { category: 'WILDLIFE FACT-FILE', fact: 'A snail can sleep for three years straight.' },
    { category: 'WILDLIFE FACT-FILE', fact: 'The mantis shrimp can punch at the speed of a .22 caliber bullet.' },
    { category: 'WILDLIFE FACT-FILE', fact: 'Wombat droppings are cube-shaped.' },
    { category: 'WILDLIFE FACT-FILE', fact: 'Cats have over 20 different vocalizations, including the purr.' },
    { category: 'WILDLIFE FACT-FILE', fact: 'Axolotls can regenerate their brain, heart, and limbs.' },
    { category: 'WILDLIFE FACT-FILE', fact: 'Honey bees can recognize human faces.' },
    { category: 'WILDLIFE FACT-FILE', fact: 'An octopus has nine brains — one central and one in each arm.' },
    // SPACE
    { category: 'SPACE FACT-FILE', fact: 'A day on Venus is longer than a year on Venus.' },
    { category: 'SPACE FACT-FILE', fact: 'There are more stars in the universe than grains of sand on Earth.' },
    { category: 'SPACE FACT-FILE', fact: 'The footprints on the Moon will be there for 100 million years.' },
    { category: 'SPACE FACT-FILE', fact: 'Neutron stars are so dense a teaspoon weighs about 6 billion tons.' },
    { category: 'SPACE FACT-FILE', fact: 'Space is completely silent — there\'s no medium for sound to travel.' },
    { category: 'SPACE FACT-FILE', fact: 'The Sun makes up 99.86% of the mass of the solar system.' },
    { category: 'SPACE FACT-FILE', fact: 'There\'s a giant cloud of alcohol in space — enough to fill 400 trillion pints of beer.' },
    { category: 'SPACE FACT-FILE', fact: 'Saturn would float in water because it\'s less dense than water.' },
    { category: 'SPACE FACT-FILE', fact: 'A year on Mercury is just 88 Earth days.' },
    { category: 'SPACE FACT-FILE', fact: 'The largest known star, UY Scuti, could fit 5 billion of our Suns.' },
    { category: 'SPACE FACT-FILE', fact: 'Olympus Mons on Mars is nearly 3 times the height of Mount Everest.' },
    { category: 'SPACE FACT-FILE', fact: 'The Milky Way is on a collision course with the Andromeda galaxy.' },
    { category: 'SPACE FACT-FILE', fact: 'It takes sunlight about 8 minutes and 20 seconds to reach Earth.' },
    { category: 'SPACE FACT-FILE', fact: 'Jupiter\'s Great Red Spot is a storm that has lasted over 350 years.' },
    { category: 'SPACE FACT-FILE', fact: 'There are more trees on Earth than stars in the Milky Way.' },
    { category: 'SPACE FACT-FILE', fact: 'The International Space Station orbits Earth every 90 minutes.' },
    { category: 'SPACE FACT-FILE', fact: 'One million Earths could fit inside the Sun.' },
    { category: 'SPACE FACT-FILE', fact: 'Astronauts grow up to 2 inches taller in space.' },
    { category: 'SPACE FACT-FILE', fact: 'The observable universe is about 93 billion light-years in diameter.' },
    { category: 'SPACE FACT-FILE', fact: 'Venus rotates in the opposite direction to most other planets.' },
    // SCIENCE
    { category: 'SCIENCE FACT-FILE', fact: 'Honey never spoils. Archaeologists found 3000-year-old edible honey.' },
    { category: 'SCIENCE FACT-FILE', fact: 'Water can boil and freeze at the same time (triple point).' },
    { category: 'SCIENCE FACT-FILE', fact: 'Bananas are radioactive (they contain potassium-40).' },
    { category: 'SCIENCE FACT-FILE', fact: 'Hot water freezes faster than cold water — it\'s called the Mpemba effect.' },
    { category: 'SCIENCE FACT-FILE', fact: 'Glass is technically neither a solid nor a liquid — it\'s an amorphous solid.' },
    { category: 'SCIENCE FACT-FILE', fact: 'Your body contains about 0.2 mg of gold.' },
    { category: 'SCIENCE FACT-FILE', fact: 'A single bolt of lightning contains enough energy to toast 100,000 slices of bread.' },
    { category: 'SCIENCE FACT-FILE', fact: 'Sound travels about 4.3 times faster in water than in air.' },
    { category: 'SCIENCE FACT-FILE', fact: 'You can\'t hum while holding your nose closed.' },
    { category: 'SCIENCE FACT-FILE', fact: 'The human brain uses about 20% of the body\'s total energy.' },
    { category: 'SCIENCE FACT-FILE', fact: 'There are more bacteria in your mouth than people on Earth.' },
    { category: 'SCIENCE FACT-FILE', fact: 'A teaspoon of soil contains more organisms than there are people alive.' },
    { category: 'SCIENCE FACT-FILE', fact: 'DNA in all your cells, stretched out, would reach Pluto and back.' },
    { category: 'SCIENCE FACT-FILE', fact: 'The average cloud weighs about 1.1 million pounds.' },
    { category: 'SCIENCE FACT-FILE', fact: 'A photon takes 40,000 years to travel from the Sun\'s core to its surface.' },
    { category: 'SCIENCE FACT-FILE', fact: 'Stomach acid is strong enough to dissolve metal.' },
    { category: 'SCIENCE FACT-FILE', fact: 'An average person walks the equivalent of 5 times around Earth in a lifetime.' },
    { category: 'SCIENCE FACT-FILE', fact: 'Your nose can detect over 1 trillion different scents.' },
    { category: 'SCIENCE FACT-FILE', fact: 'The Earth\'s core is as hot as the surface of the Sun.' },
    { category: 'SCIENCE FACT-FILE', fact: 'Humans share about 60% of their DNA with bananas.' },
    // HISTORY
    { category: 'HISTORY FACT-FILE', fact: 'Cleopatra lived closer in time to the Moon landing than the pyramids.' },
    { category: 'HISTORY FACT-FILE', fact: 'Oxford University is older than the Aztec Empire.' },
    { category: 'HISTORY FACT-FILE', fact: 'Ancient Romans used crushed mouse brains as toothpaste.' },
    { category: 'HISTORY FACT-FILE', fact: 'Nintendo was founded in 1889 — as a playing card company.' },
    { category: 'HISTORY FACT-FILE', fact: 'The shortest war in history lasted 38 minutes (Britain vs. Zanzibar, 1896).' },
    { category: 'HISTORY FACT-FILE', fact: 'The Great Wall of China is not visible from space with the naked eye.' },
    { category: 'HISTORY FACT-FILE', fact: 'Vikings used to give kittens to brides as essential household gifts.' },
    { category: 'HISTORY FACT-FILE', fact: 'Abraham Lincoln was a champion wrestler before becoming president.' },
    { category: 'HISTORY FACT-FILE', fact: 'In 1932, Australia lost a war against emus — the Great Emu War.' },
    { category: 'HISTORY FACT-FILE', fact: 'The first alarm clock could only ring at 4 AM.' },
    { category: 'HISTORY FACT-FILE', fact: 'Ketchup was sold as medicine in the 1830s.' },
    { category: 'HISTORY FACT-FILE', fact: 'Albert Einstein was offered the presidency of Israel in 1952. He declined.' },
    { category: 'HISTORY FACT-FILE', fact: 'Napoleon was actually above average height for his time.' },
    { category: 'HISTORY FACT-FILE', fact: 'The Eiffel Tower was supposed to be temporary — built for the 1889 World Fair.' },
    { category: 'HISTORY FACT-FILE', fact: 'The first computer programmer was Ada Lovelace in the 1840s.' },
    { category: 'HISTORY FACT-FILE', fact: 'Coca-Cola was originally green and contained cocaine.' },
    { category: 'HISTORY FACT-FILE', fact: 'Shakespeare invented over 1,700 words in the English language.' },
    { category: 'HISTORY FACT-FILE', fact: 'The Leaning Tower of Pisa took 200 years to build and started tilting after 5.' },
    { category: 'HISTORY FACT-FILE', fact: 'Ancient Egyptians used slabs of stone as pillows.' },
    { category: 'HISTORY FACT-FILE', fact: 'The first email was sent in 1971 by Ray Tomlinson to himself.' },
    // PRODUCTIVITY
    { category: 'PRODUCTIVITY FACT-FILE', fact: 'The average person spends 2 weeks of their life waiting for traffic lights.' },
    { category: 'PRODUCTIVITY FACT-FILE', fact: 'Taking short breaks improves focus by up to 50%.' },
    { category: 'PRODUCTIVITY FACT-FILE', fact: 'Writing things down by hand improves memory retention by 29%.' },
    { category: 'PRODUCTIVITY FACT-FILE', fact: 'The Pomodoro Technique was invented by a college student using a tomato-shaped timer.' },
    { category: 'PRODUCTIVITY FACT-FILE', fact: 'Multitasking can reduce productivity by up to 40%.' },
    { category: 'PRODUCTIVITY FACT-FILE', fact: 'Your brain makes better decisions in the morning than at night.' },
    { category: 'PRODUCTIVITY FACT-FILE', fact: 'The two-minute rule: if it takes less than 2 minutes, do it now.' },
    { category: 'PRODUCTIVITY FACT-FILE', fact: 'Walking meetings boost creative output by an average of 60%.' },
    { category: 'PRODUCTIVITY FACT-FILE', fact: 'People who plan their day the night before are significantly more productive.' },
    { category: 'PRODUCTIVITY FACT-FILE', fact: 'Listening to music without lyrics can improve concentration.' },
    { category: 'PRODUCTIVITY FACT-FILE', fact: 'The average office worker is interrupted every 11 minutes.' },
    { category: 'PRODUCTIVITY FACT-FILE', fact: 'It takes about 23 minutes to refocus after an interruption.' },
    { category: 'PRODUCTIVITY FACT-FILE', fact: 'Blue light from screens suppresses melatonin and disrupts sleep.' },
    { category: 'PRODUCTIVITY FACT-FILE', fact: 'Drinking water can boost your brain\'s reaction time by 14%.' },
    { category: 'PRODUCTIVITY FACT-FILE', fact: 'Working in 90-minute cycles aligns with your body\'s natural energy rhythm.' },
    // TECH
    { category: 'TECH FACT-FILE', fact: 'The first 1GB hard drive (1980) weighed 550 pounds and cost $40,000.' },
    { category: 'TECH FACT-FILE', fact: 'More people in the world have mobile phones than toilets.' },
    { category: 'TECH FACT-FILE', fact: 'The first website ever made is still online at info.cern.ch.' },
    { category: 'TECH FACT-FILE', fact: 'About 90% of the world\'s currency exists only as digital numbers.' },
    { category: 'TECH FACT-FILE', fact: 'The QWERTY keyboard was designed to slow typists down to prevent jams.' },
    { category: 'TECH FACT-FILE', fact: 'Google\'s first storage was made from LEGO bricks.' },
    { category: 'TECH FACT-FILE', fact: 'The average smartphone has more computing power than NASA had in 1969.' },
    { category: 'TECH FACT-FILE', fact: 'The first computer bug was an actual moth found in a Harvard computer.' },
    { category: 'TECH FACT-FILE', fact: 'Over 6,000 new computer viruses are created every month.' },
    { category: 'TECH FACT-FILE', fact: 'Amazon started as an online bookstore in Jeff Bezos\' garage.' },
    { category: 'TECH FACT-FILE', fact: 'The Firefox logo isn\'t actually a fox — it\'s a red panda.' },
    { category: 'TECH FACT-FILE', fact: 'Email existed before the World Wide Web.' },
    { category: 'TECH FACT-FILE', fact: 'The word "robot" comes from the Czech word "robota," meaning forced labor.' },
    { category: 'TECH FACT-FILE', fact: 'The first YouTube video was uploaded on April 23, 2005.' },
    { category: 'TECH FACT-FILE', fact: 'Over 500 hours of video are uploaded to YouTube every minute.' },
    // FOOD
    { category: 'FOOD FACT-FILE', fact: 'Strawberries aren\'t actually berries, but bananas are.' },
    { category: 'FOOD FACT-FILE', fact: 'Peanuts aren\'t nuts — they\'re legumes.' },
    { category: 'FOOD FACT-FILE', fact: 'A single spaghetti noodle is called a "spaghetto".' },
    { category: 'FOOD FACT-FILE', fact: 'Apples float in water because they\'re 25% air.' },
    { category: 'FOOD FACT-FILE', fact: 'Chocolate was once used as currency by the Aztecs.' },
    { category: 'FOOD FACT-FILE', fact: 'Crackers have holes to prevent air bubbles during baking.' },
    { category: 'FOOD FACT-FILE', fact: 'The most expensive pizza in the world costs $12,000.' },
    { category: 'FOOD FACT-FILE', fact: 'Carrots were originally purple, not orange.' },
    { category: 'FOOD FACT-FILE', fact: 'Avocados are a fruit, not a vegetable.' },
    { category: 'FOOD FACT-FILE', fact: 'It takes about 50 licks to finish a single scoop of ice cream.' }
];


/**
 * Detect if we're in side panel mode and apply the CSS class
 */
async function detectSidePanelMode() {
    try {
        // Robust check: Chrome Popups are strictly limited in height (usually ~600px).
        // Side Panels fill the vertical screen height.
        // We also check for the specific 'sidepanel' query param if we implement that later.
        const isTall = window.innerHeight > 620;
        const isExplicitSidePanel = window.location.search.includes('mode=sidepanel');

        if (isTall || isExplicitSidePanel) {
            document.documentElement.classList.add('side-panel-mode');
            document.body.classList.add('side-panel-mode');

        }
    } catch (error) {
        console.error('Side panel detection error:', error);
    }
}

// Run detection immediately and on resize
detectSidePanelMode();
window.addEventListener('resize', detectSidePanelMode);

/**

 * Initialize the extension
 */
async function init() {
    // If opened as a folder-picker helper window, show minimal UI and skip full init
    if (window.location.search.includes('pickFolder=true')) {
        document.body.innerHTML = '';
        document.body.style.cssText = 'margin:0; padding:40px; font-family:-apple-system,BlinkMacSystemFont,sans-serif; background:#1e1e1e; color:#e0e0e0; display:flex; flex-direction:column; align-items:center; justify-content:center; height:100vh; box-sizing:border-box;';
        document.body.innerHTML = `
            <p style="font-size:15px; margin:0 0 20px; text-align:center;">Select a folder for your JottNote backups</p>
            <button id="pick-folder-btn" style="padding:10px 28px; font-size:14px; cursor:pointer; border:1px solid #555; border-radius:6px; background:#2a2a2a; color:#e0e0e0;">Select Folder</button>
            <p id="pick-status" style="margin-top:16px; font-size:13px; color:#999; text-align:center;"></p>
        `;
        document.getElementById('pick-folder-btn').addEventListener('click', async () => {
            const statusEl = document.getElementById('pick-status');
            try {
                const handle = await window.showDirectoryPicker({ mode: 'readwrite' });
                const result = await chrome.storage.local.get('jottnote_settings');
                const s = result.jottnote_settings || {};
                s.backupFolderName = handle.name;
                await chrome.storage.local.set({ jottnote_settings: s });
                statusEl.style.color = '#6c6';
                statusEl.textContent = 'Folder set: ' + handle.name;
                setTimeout(() => window.close(), 1500);
            } catch (err) {
                if (err.name === 'AbortError') {
                    statusEl.textContent = 'Selection cancelled.';
                } else {
                    statusEl.style.color = '#c66';
                    statusEl.textContent = 'Error: ' + err.message;
                }
            }
        });
        return; // Skip full extension init
    }

    // If opened as an export helper window, show export UI and skip full init
    if (window.location.search.includes('exportNow=true')) {
        document.body.innerHTML = '';
        document.body.style.cssText = 'margin:0; padding:30px; font-family:-apple-system,BlinkMacSystemFont,sans-serif; background:#1e1e1e; color:#e0e0e0; display:flex; flex-direction:column; align-items:center; justify-content:center; height:100vh; box-sizing:border-box;';

        // Fetch notes
        const notesResult = await chrome.storage.local.get('jottnote_notes');
        const allNotes = notesResult.jottnote_notes || [];

        if (allNotes.length === 0) {
            document.body.innerHTML = `
                <p style="font-size:15px; color:#999;">No notes to export.</p>
            `;
            setTimeout(() => window.close(), 2000);
            return;
        }

        const timestamp = new Date().toISOString().replace(/[:.]/g, '-');
        document.body.innerHTML = `
            <p style="font-size:15px; margin:0 0 6px; text-align:center;">Export ${allNotes.length} note${allNotes.length === 1 ? '' : 's'} as a backup</p>
            <p style="font-size:12px; margin:0 0 20px; color:#999; text-align:center;">Choose where to save the backup file</p>
            <button id="export-btn" style="padding:10px 28px; font-size:14px; cursor:pointer; border:1px solid #555; border-radius:6px; background:#2a2a2a; color:#e0e0e0;">Save Backup</button>
            <p id="export-status" style="margin-top:16px; font-size:13px; color:#999; text-align:center;"></p>
        `;

        document.getElementById('export-btn').addEventListener('click', async () => {
            const statusEl = document.getElementById('export-status');
            try {
                const fileHandle = await window.showSaveFilePicker({
                    suggestedName: `jottnote-backup-${timestamp}.json`,
                    types: [{ description: 'JSON File', accept: { 'application/json': ['.json'] } }]
                });
                statusEl.textContent = 'Saving...';
                const writable = await fileHandle.createWritable();
                await writable.write(JSON.stringify({
                    exportedAt: new Date().toISOString(),
                    noteCount: allNotes.length,
                    notes: allNotes
                }, null, 2));
                await writable.close();
                statusEl.style.color = '#6c6';
                statusEl.textContent = 'Backup saved: ' + allNotes.length + ' notes';
                setTimeout(() => window.close(), 1500);
            } catch (err) {
                if (err.name === 'AbortError') {
                    statusEl.textContent = 'Save cancelled.';
                } else {
                    statusEl.style.color = '#c66';
                    statusEl.textContent = 'Error: ' + err.message;
                }
            }
        });
        return; // Skip full extension init
    }

    // Get DOM elements
    editor = document.getElementById('note-editor');
    headerToolbar = document.querySelector('.header-toolbar');
    footerToolbar = document.querySelector('.footer-toolbar');
    keywordIndicator = document.getElementById('keyword-indicator');
    slashPopup = document.getElementById('slash-popup');
    settingsPanel = document.getElementById('settings-panel');
    searchPanel = document.getElementById('search-panel');
    noteDate = document.getElementById('note-date');
    timerDisplay = document.getElementById('timer-display');
    timerMinutes = document.getElementById('timer-minutes');
    timerSeconds = document.getElementById('timer-seconds');
    timerProgress = document.getElementById('timer-progress');
    timerRounds = document.getElementById('timer-rounds');
    timerCompleteModal = document.getElementById('timer-complete-modal');
    timerChime = document.getElementById('timer-chime');
    codeHelp = document.getElementById('code-help');
    codeLanguagesHelp = document.getElementById('code-languages-help');
    timerHelp = document.getElementById('timer-help');
    codeEditorContainer = document.getElementById('code-editor-container');
    codeEditable = document.getElementById('code-editable');
    statsDisplay = document.getElementById('stats-display');
    statsLabel = document.getElementById('stats-label');
    statsValue = document.getElementById('stats-value');
    statsDetail = document.getElementById('stats-detail');
    sumHelp = document.getElementById('sum-help');
    avgHelp = document.getElementById('avg-help');
    countHelp = document.getElementById('count-help');
    listHelp = document.getElementById('list-help');
    listDisplay = document.getElementById('list-display');
    mathHelp = document.getElementById('math-help');
    mathDisplay = document.getElementById('math-display');
    countDisplay = document.getElementById('count-display');
    editorHighlighter = document.getElementById('editor-highlighter');
    currencyHelp = document.getElementById('currency-help');
    currencyListContent = document.getElementById('currency-list-content');
    gameDisplay = document.getElementById('game-display');
    gameHelp = document.getElementById('game-help');
    dateHelp = document.getElementById('date-help');
    dateDisplay = document.getElementById('date-display');

    // Bookmarks elements
    bookmarksDisplay = document.getElementById('bookmarks-display');
    bookmarksHelp = document.getElementById('bookmarks-help');
    bookmarksTagsList = document.getElementById('bookmarks-tags-list');

    // Sessions elements
    sessionsDisplay = document.getElementById('sessions-display');
    sessionsHelp = document.getElementById('sessions-help');
    sessionsTagsList = document.getElementById('sessions-tags-list');

    // Session edit panel
    sessionEditPanel = document.getElementById('session-edit-panel');
    sessionEditTitle = document.getElementById('session-edit-title');
    sessionEditTags = document.getElementById('session-edit-tags');
    sessionEditUrlsList = document.getElementById('session-edit-urls-list');
    sessionDeleteSelectedBtn = document.getElementById('session-delete-selected-btn');

    // Find/Replace elements
    findReplacePanel = document.getElementById('find-replace-panel');
    findInput = document.getElementById('find-input');
    findModeSelect = document.getElementById('find-mode-select');
    findCount = document.getElementById('find-count');
    findExpandBtn = document.getElementById('find-expand-btn');
    replaceRow = document.getElementById('replace-row');
    replaceInput = document.getElementById('replace-input');

    // Notification popups
    timerStartPopup = document.getElementById('timer-start-popup');
    timerHoverPopup = document.getElementById('timer-hover-popup');

    // Link hover popup
    linkHoverPopup = document.getElementById('link-hover-popup');
    linkHoverUrl = document.getElementById('link-hover-url');
    const linkHoverShortcuts = document.getElementById('link-hover-shortcuts');

    // Update shortcuts text based on platform
    const isMac = navigator.platform.toUpperCase().indexOf('MAC') >= 0;
    if (!isMac) {
        linkHoverShortcuts.textContent = 'Ctrl+Click to open | Ctrl⇧+Click to expand.';
    }

    // Template elements
    templateHelp = document.getElementById('template-help');
    placeholdersHelp = document.getElementById('placeholders-help');
    placeholdersList = document.getElementById('placeholders-list');
    templateList = document.getElementById('template-list');

    // Sort help element
    sortHelp = document.getElementById('sort-help');

    // Utility keyword elements
    uuidHelp = document.getElementById('uuid-help');
    uuidDisplay = document.getElementById('uuid-display');
    loremHelp = document.getElementById('lorem-help');
    weatherHelp = document.getElementById('weather-help');
    weatherDisplay = document.getElementById('weather-display');
    dictionaryHelp = document.getElementById('dictionary-help');
    dictionaryDisplay = document.getElementById('dictionary-display');
    timerMsEl = document.getElementById('timer-ms');
    timerLapsEl = document.getElementById('timer-laps');

    // Spell check context menu
    spellCheckContextMenu = document.getElementById('spell-check-context-menu');

    // Populate currency help list (Fiat + Crypto)
    buildCurrencyList();

    // License check - pass callback for status updates
    await initLicense((isPro) => {
        if (isPro) {
            onProActivated(false);
        } else {
            updateProSettingsTab();
        }
    });

    settings = await getSettings();
    applySettings();

    // Setup Pro UI elements
    setupProUI();

    // Initialize spell check (non-blocking, with proper error handling)
    if (isPro() && settings.spellCheckEnabled) {
        try {
            initSpellCheck(true).catch(err => {
                console.warn('Spell check initialization failed:', err);
            });
        } catch (err) {
            console.warn('Spell check init error:', err);
        }
    }

    // Auto-delete unmodified notes on startup
    await autoDeleteOldNotes(settings);

    // Load or create current note (respects newNoteOnLaunch and autoCreatePeriod)
    await loadCurrentNote();

    // Track popup close time for autoCreatePeriod
    window.addEventListener('beforeunload', async () => {
        settings.lastPopupCloseTime = Date.now();
        await saveSettings(settings);
    });

    // Setup delegated event listeners (once, instead of per-render)
    setupDelegatedListeners();

    // Setup event listeners
    setupToolbarHover();
    setupEditorEvents();
    setupNavigationButtons();
    setupSettingsPanel();
    setupSearchPanel();
    setupFindReplace();
    setupKeyboardShortcuts();
    setupSlashCommands();
    setupTimer();
    setupCodeEditor();
    setupSpellCheck();
    setupProUI();

    // Render themes
    renderThemeSelector();

    // Apply Pro gates to slash popup items
    applySlashProGates();

    // Focus active editor (handles code mode vs normal editor)
    focusActiveEditor();
}

/**
 * Load a note into the editor and update all UI states
 * Ensures code mode buffer is cleared to prevent ghost content
 * @param {Object} note
 */
async function loadNote(note) {
    // 1. Clear code buffer (CRITICAL FIX)
    // If we don't clear this, updateCodeMode will overwrite the new note
    // with the old code content if code mode was active
    if (typeof codeEditable !== 'undefined' && codeEditable) {
        codeEditable.textContent = '';
    }

    // 1.5. Pre-clear highlighter to prevent ghost text during mode transitions
    if (editorHighlighter) {
        editorHighlighter.innerHTML = '';
        editorHighlighter.classList.add('hidden');
        editorHighlighter.style.display = 'none';
    }

    // 1.6. Detect code mode EARLY — hide textarea/highlighter BEFORE async gaps
    // This prevents Chrome from painting the textarea content during await pauses
    const noteContent = note.content || '';
    const willBeCodeMode = isCodeMode(noteContent);
    if (willBeCodeMode && editor) {
        editor.style.display = 'none';
    }

    // 2. Load content
    initEditor(note, editor);

    // 2.5. Rebuild URL map for shortened links
    rebuildUrlMap(noteContent);

    // 3. Update UI
    await updateNoteDate(note);
    updateKeywordIndicator();
    await updateNavigationState();

    // 4. Update Modes — runs synchronously after awaits complete
    try {
        updateCodeMode(); // Handles code mode state (entering/exiting)
        if (typeof updateStatsMode === 'function') updateStatsMode();
        if (typeof updateListMode === 'function') updateListMode();
        if (typeof updateEditorHighlighter === 'function') updateEditorHighlighter();
        // Skip math mode in code mode — CSS values get misinterpreted as math
        if (typeof updateMathMode === 'function' && !willBeCodeMode) {
            updateMathMode(editor.value);
        }
        if (typeof updateDateMode === 'function' && !willBeCodeMode) {
            const { mode: loadMode } = parseKeyword(editor.value);
            if (loadMode === 'date') updateDateMode(editor.value);
        }
    } catch (e) {
        console.error('Error updating modes:', e);
    }

    // 4.5. If the loaded note is empty, ensure a clean slate so placeholder shows
    if (!noteContent.trim()) {
        resetAllModes();
        updateEditorHighlighter();
    }

    // 5. Restore inline styles if NOT in code mode (code mode keeps them hidden)
    // Also restore if we pre-hid editors (willBeCodeMode) but updateCodeMode didn't
    // actually enter code-mode (e.g. "code:" without a language returns null from
    // parseCodeKeyword while isCodeMode still returns true)
    const editorAreaRef = document.querySelector('.editor-area');
    if (!willBeCodeMode || (willBeCodeMode && !editorAreaRef.classList.contains('code-mode'))) {
        restoreNonCodeEditors();
    }
}

/**
 * Auto-delete unmodified notes based on autoDeletePeriod setting
 * Runs once on startup. Slider values: 0=Today, 1=1week, 2=1month, 3=1year, 4=Never
 */
async function autoDeleteOldNotes(settings) {
    const period = settings.autoDeletePeriod ?? 4; // default: Never
    if (period >= 4) return; // Never — skip

    const now = Date.now();
    const thresholds = [
        0,                       // 0 = Today (delete everything older than start of today)
        7 * 24 * 3600000,        // 1 = 1 week
        30 * 24 * 3600000,       // 2 = 1 month
        365 * 24 * 3600000       // 3 = 1 year
    ];

    let cutoff;
    if (period === 0) {
        // "Today" means delete notes not modified today
        const today = new Date();
        today.setHours(0, 0, 0, 0);
        cutoff = today.getTime();
    } else {
        cutoff = now - thresholds[period];
    }

    const allNotes = await getNotes();
    let deletedCount = 0;

    for (const note of allNotes) {
        const modifiedTime = new Date(note.modifiedAt || note.createdAt).getTime();
        // Only delete if not modified since cutoff AND note has content (skip empty new notes)
        if (modifiedTime < cutoff && note.content && note.content.trim()) {
            await deleteNote(note.id);
            deletedCount++;
        }
    }

    if (deletedCount > 0) {

    }
}

/**
 * Load the current note or create a new one
 * Respects newNoteOnLaunch and autoCreatePeriod settings
 */
async function loadCurrentNote() {
    let noteId = await getCurrentNoteId();
    let note = null;

    // Check if we should create a new note on launch
    if (settings.newNoteOnLaunch) {
        note = await createNote();
        await setCurrentNoteId(note.id);
        await loadNote(note);
        return;
    }

    // Check auto-create timer (create new note if enough time has passed since last close)
    const autoCreatePeriod = settings.autoCreatePeriod ?? 5; // default: Never
    if (autoCreatePeriod < 5) { // 5 = Never
        const lastCloseTime = settings.lastPopupCloseTime || 0;
        if (lastCloseTime > 0) {
            const elapsed = Date.now() - lastCloseTime;
            const thresholds = [0, 3 * 60000, 30 * 60000, 60 * 60000, 24 * 3600000]; // Always, 3min, 30min, 1hr, 1day
            const threshold = thresholds[autoCreatePeriod];

            if (elapsed >= threshold) {
                note = await createNote();
                await setCurrentNoteId(note.id);
                await loadNote(note);
                return;
            }
        }
    }

    if (noteId) {
        note = await getNote(noteId);
    }

    if (!note) {
        // No current note, create new one
        note = await createNote();

        // Check if this is the very first launch (onboarding)
        const obResult = await chrome.storage.local.get('jottnote_onboarding_done');
        if (!obResult.jottnote_onboarding_done) {
            note.content = `Welcome to JottNote!\n\nType "/" on a new line to explore all modes.\n\nKeywords go on the first line of a note:\n  math — calculator & conversions\n  list — checkboxes & tasks\n  sum — add up numbers\n\nNavigate notes with the arrows below.\nDelete this note anytime.`;
            await saveNote(note);
        }
    }

    await setCurrentNoteId(note.id);
    await loadNote(note);

    // Show onboarding after loading is complete
    const obCheck = await chrome.storage.local.get('jottnote_onboarding_done');
    if (!obCheck.jottnote_onboarding_done) {
        showOnboarding();
    }
}

/**
 * Setup delegated event listeners (once during init, not per-render)
 * Prevents accumulating duplicate listeners on re-render.
 */
function setupDelegatedListeners() {
    // Math click-to-copy
    if (mathDisplay) {
        mathDisplay.addEventListener('click', (e) => {
            const span = e.target.closest('.math-result[data-copy]');
            if (span) {
                const value = span.getAttribute('data-copy');
                if (value) {
                    copyToClipboard(value);
                    showCopyFeedback(span);
                }
            }
        });
    }

    // Date click-to-copy
    if (dateDisplay) {
        dateDisplay.addEventListener('click', (e) => {
            const span = e.target.closest('.date-result[data-copy]');
            if (span) {
                const value = span.getAttribute('data-copy');
                if (value) {
                    copyToClipboard(value);
                    showCopyFeedback(span);
                }
            }
        });
    }

    // List checkboxes
    if (listDisplay) {
        listDisplay.addEventListener('click', (e) => {
            const checkbox = e.target.closest('.list-checkbox');
            if (checkbox) {
                handleCheckboxClick(e);
            }
        });
    }

    // Spell check suggestions
    if (spellCheckContextMenu) {
        spellCheckContextMenu.addEventListener('click', (e) => {
            const item = e.target.closest('.spell-suggestion');
            if (item) {
                const originalWord = item.getAttribute('data-word');
                const replacement = item.getAttribute('data-suggestion');
                replaceWord(originalWord, replacement);
                hideSpellCheckMenu();
            }
        });
    }

    // Tag cloud pills
    const tagCloudEl = document.getElementById('search-tag-cloud');
    if (tagCloudEl) {
        tagCloudEl.addEventListener('click', (e) => {
            const pill = e.target.closest('.search-tag-pill');
            if (pill) {
                const searchInput = document.getElementById('search-input');
                searchInput.value = pill.dataset.tag;
                searchInput.dispatchEvent(new Event('input'));
            }
        });
    }
}

/**
 * Setup toolbar hover behavior
 */
function setupToolbarHover() {
    const container = document.querySelector('.jottnote-container');

    let headerTimeout, footerTimeout;

    container.addEventListener('mousemove', (e) => {
        // Don't process toolbar hover during onboarding
        const obOverlay = document.getElementById('onboarding-overlay');
        if (obOverlay && obOverlay.classList.contains('visible')) return;

        const rect = container.getBoundingClientRect();
        const y = e.clientY - rect.top;
        const height = rect.height;

        // Header zone (top 50px)
        // Don't show header if settings panel or search panel is open
        if (y < 50 && !settingsPanel.classList.contains('visible') && !searchPanel.classList.contains('visible')) {
            clearTimeout(headerTimeout);
            headerToolbar.classList.add('visible');
        } else {
            headerTimeout = setTimeout(() => {
                headerToolbar.classList.remove('visible');
            }, 200);
        }

        // Footer zone (bottom 50px)
        if (y > height - 50) {
            clearTimeout(footerTimeout);
            footerToolbar.classList.add('visible');
        } else {
            footerTimeout = setTimeout(() => {
                footerToolbar.classList.remove('visible');
            }, 200);
        }
    });

    container.addEventListener('mouseleave', () => {
        headerTimeout = setTimeout(() => {
            headerToolbar.classList.remove('visible');
        }, 200);
        footerTimeout = setTimeout(() => {
            footerToolbar.classList.remove('visible');
        }, 200);
    });
}

/**
 * Setup editor events
 */
function setupEditorEvents() {
    // Content change with auto-save
    editor.addEventListener('input', async (e) => {
        // Bump generation counter — async chains check this to detect stale work
        const gen = ++editorInputGeneration;

        // Check if we just pressed Enter on a template load command
        // Pattern: first line is "template:name" and second line is empty (cursor just moved there)
        const content = editor.value;
        const lines = content.split('\n');

        if (lines.length >= 2 && !isLoadingTemplate) {
            const firstLine = lines[0].trim().toLowerCase();
            const secondLine = lines[1];

            // Check if first line is a template load command and second line is empty/just created
            if (firstLine.startsWith('template:') &&
                firstLine !== 'template:create' &&
                firstLine !== 'template:end' &&
                firstLine !== 'template' &&
                secondLine.trim() === '') {

                const templateName = lines[0].trim().substring(9).trim();
                if (templateName) {
                    // Load template - use try-finally to ensure flag is reset
                    isLoadingTemplate = true;

                    // Safety timeout to force-reset flag after 2 seconds
                    const safetyTimeout = setTimeout(() => {
                        console.warn('[Template] Safety timeout triggered - resetting flag');
                        isLoadingTemplate = false;
                    }, 2000);

                    try {
                        await loadTemplate(templateName);
                    } catch (error) {
                        console.error('[Template] Load error:', error);
                    } finally {
                        clearTimeout(safetyTimeout);
                        isLoadingTemplate = false;
                    }
                    return; // Skip other processing
                }
            }
        }

        handleContentChange(editor, async (savedNote) => {
            await updateNoteDate(savedNote);
        });

        // If content is empty, force-reset all mode states for a clean slate
        if (!content.trim()) {
            resetAllModes();
            updateEditorHighlighter();
            return;
        }

        updateKeywordIndicator();
        checkSlashCommand();
        updateCodeMode();
        updateTimerHelp();
        updateCurrencyHelp();
        await updateTemplateHelp();
        if (gen !== editorInputGeneration) return;
        await handleTemplateCommands();
        if (gen !== editorInputGeneration) return;
        await updatePlaceholdersHelp();
        if (gen !== editorInputGeneration) return;
        await handlePlaceholdersCommands();
        if (gen !== editorInputGeneration) return;
        await updateBookmarksMode();
        if (gen !== editorInputGeneration) return;
        handleBookmarksAdd();
        await updateSessionsMode();
        if (gen !== editorInputGeneration) return;
        handleSessionsAdd();
        updateSortHelp();
        await handleSortCommand();
        if (gen !== editorInputGeneration) return;
        updateStatsMode();
        updateListMode();
        updateEditorHighlighter();
    });

    // Sync scroll for highlighter and list display
    editor.addEventListener('scroll', () => {
        if (editorHighlighter) {
            editorHighlighter.scrollTop = editor.scrollTop;
            editorHighlighter.scrollLeft = editor.scrollLeft;
        }
        if (listDisplay) {
            listDisplay.scrollTop = editor.scrollTop;
            listDisplay.scrollLeft = editor.scrollLeft;
        }
    });

    // Handle paste - skip smart paste in code mode
    editor.addEventListener('paste', (e) => {
        const content = editor.value;
        const inCodeMode = isCodeMode(content);

        if (!e.shiftKey && !inCodeMode) {
            e.preventDefault();
            const text = e.clipboardData.getData('text/plain');
            smartPaste(editor, text, {
                stripLeadingSpaces: settings.editPasteRemoveSpaces !== undefined ? settings.editPasteRemoveSpaces : true,
                stripListNumbers: settings.editPasteRemoveNumbers !== undefined ? settings.editPasteRemoveNumbers : true,
                stripBullets: settings.editPasteRemoveBullets !== undefined ? settings.editPasteRemoveBullets : true,
                stripMarkdown: settings.editPasteRemoveMarkdown || false,
                stripEmptyLines: settings.editPasteRemoveEmpty || false
            });
        }
        // In code mode, allow normal paste (preserves formatting)
    });

    // Handle copy
    editor.addEventListener('copy', async (e) => {
        if (editor.selectionStart === editor.selectionEnd) {
            e.preventDefault();
            await smartCopy(editor, {
                skipKeywords: settings.editSkipKeywordCopy !== undefined ? settings.editSkipKeywordCopy : true,
                skipCheckTriggers: settings.editSkipChecklistCopy || false,
                checkTrigger: settings.checkTrigger || '/x'
            });
        }
    });

    // Auto-copy selected text
    editor.addEventListener('mouseup', async () => {
        if (settings.editAutoCopy && editor.selectionStart !== editor.selectionEnd) {
            const selectedText = editor.value.substring(editor.selectionStart, editor.selectionEnd);
            try {
                await navigator.clipboard.writeText(selectedText);
            } catch (err) {
                console.warn('Auto-copy failed:', err);
            }
        }
    });

    // Keep indentation on Enter
    editor.addEventListener('keydown', (e) => {
        if (e.key === 'Enter' && (settings.editKeepIndent !== undefined ? settings.editKeepIndent : true)) {
            const content = editor.value;
            if (isCodeMode(content)) return; // Skip in code mode
            const cursorPos = editor.selectionStart;
            const lineStart = content.lastIndexOf('\n', cursorPos - 1) + 1;
            const currentLine = content.substring(lineStart, cursorPos);
            const indent = currentLine.match(/^(\s*)/)[1];
            if (indent) {
                // In list mode, don't carry indent from empty/whitespace-only lines
                // This resets the cursor to the standard position after a list break
                const { mode } = parseKeyword(content);
                if (mode === 'list' && currentLine.trim() === '') {
                    return; // Let default Enter behavior insert a plain newline
                }
                e.preventDefault();
                const before = content.substring(0, cursorPos);
                const after = content.substring(editor.selectionEnd);
                editor.value = before + '\n' + indent + after;
                editor.selectionStart = editor.selectionEnd = cursorPos + 1 + indent.length;
                editor.dispatchEvent(new Event('input', { bubbles: true }));
            }
        }
    });

    // List mode: Tab to indent, Shift+Tab to unindent, Backspace at line start to unindent
    editor.addEventListener('keydown', (e) => {
        const content = editor.value;
        const { mode } = parseKeyword(content);
        if (mode !== 'list') return;

        const cursorPos = editor.selectionStart;
        const lines = content.split('\n');

        // Find which line cursor is on and the offset within that line
        let charCount = 0;
        let lineIdx = 0;
        for (let i = 0; i < lines.length; i++) {
            if (charCount + lines[i].length >= cursorPos || i === lines.length - 1) {
                lineIdx = i;
                break;
            }
            charCount += lines[i].length + 1; // +1 for \n
        }
        const lineStart = charCount;
        const posInLine = cursorPos - lineStart;
        const currentLine = lines[lineIdx];

        if (e.key === 'Tab') {
            e.preventDefault();
            if (lineIdx === 0) return; // Don't indent keyword line

            if (e.shiftKey) {
                // Shift+Tab: unindent — remove up to 2 leading spaces
                const leadingMatch = currentLine.match(/^( {1,2})/);
                if (leadingMatch) {
                    const spacesToRemove = leadingMatch[1].length;
                    lines[lineIdx] = currentLine.substring(spacesToRemove);
                    editor.value = lines.join('\n');
                    editor.selectionStart = editor.selectionEnd = Math.max(lineStart, cursorPos - spacesToRemove);
                    editor.dispatchEvent(new Event('input', { bubbles: true }));
                }
            } else {
                // Tab: indent — add 2 spaces at line start
                lines[lineIdx] = '  ' + currentLine;
                editor.value = lines.join('\n');
                editor.selectionStart = editor.selectionEnd = cursorPos + 2;
                editor.dispatchEvent(new Event('input', { bubbles: true }));
            }
        }

        if (e.key === 'Backspace' && lineIdx > 0) {
            const leadingSpaces = currentLine.match(/^(\s*)/)[1].length;
            // If cursor is at start of actual text content (within leading whitespace) and there are leading spaces
            if (leadingSpaces > 0 && posInLine <= leadingSpaces) {
                e.preventDefault();
                const spacesToRemove = Math.min(2, leadingSpaces);
                lines[lineIdx] = currentLine.substring(spacesToRemove);
                editor.value = lines.join('\n');
                editor.selectionStart = editor.selectionEnd = Math.max(lineStart, cursorPos - spacesToRemove);
                editor.dispatchEvent(new Event('input', { bubbles: true }));
            }
        }
    });

    // List mode: update textarea padding when cursor moves between list and normal-text areas
    editor.addEventListener('click', () => {
        const editorArea = document.querySelector('.editor-area');
        if (editorArea && editorArea.classList.contains('list-mode')) {
            updateListCursorPadding();
        }
    });
    editor.addEventListener('keyup', (e) => {
        if (['ArrowUp', 'ArrowDown', 'ArrowLeft', 'ArrowRight', 'Home', 'End', 'Enter', 'Backspace', 'Delete'].includes(e.key)) {
            const editorArea = document.querySelector('.editor-area');
            if (editorArea && editorArea.classList.contains('list-mode')) {
                updateListCursorPadding();
            }
        }
    });

    // Auto-shorten URLs when space or enter is pressed
    editor.addEventListener('keydown', (e) => {
        if (e.key === ' ' || e.key === 'Enter') {
            // Delay slightly to let the character be inserted first (only if not a template command)
            setTimeout(() => {
                autoShortenUrls(editor, !settings.disableLinkShortening);
            }, 10);
        }
    });

    // Check and update link popup based on cursor position
    function updateLinkPopup() {
        const cursorPos = editor.selectionStart;
        const content = editor.value;

        // Find if cursor is within a link (text between + markers)
        const linkPattern = /\+([^\+]+)\+/g;
        let match;
        let hoveredLink = null;

        while ((match = linkPattern.exec(content)) !== null) {
            const start = match.index;
            const end = start + match[0].length;

            // Check if cursor is within this link
            if (cursorPos >= start && cursorPos <= end) {
                const linkText = match[1]; // Text without + markers

                // Try to get full URL from shortened version
                const fullUrl = getFullUrlFromShortened(linkText) || getFullUrlFromShortened(match[0]);

                if (fullUrl) {
                    hoveredLink = fullUrl;
                } else {
                    // Fallback: if it looks like a URL, use it as is
                    if (linkText.match(/https?:\/\//)) {
                        hoveredLink = linkText;
                    }
                }
                break;
            }
        }

        if (hoveredLink) {
            linkHoverUrl.textContent = hoveredLink;
            linkHoverPopup.classList.add('visible');
        } else {
            linkHoverPopup.classList.remove('visible');
        }
    }

    // Update link popup when cursor position changes
    editor.addEventListener('click', (e) => {
        // Don't show popup if CMD/CTRL is pressed (we're doing an action)
        if (!e.metaKey && !e.ctrlKey) {
            setTimeout(updateLinkPopup, 10);
        }
    });
    editor.addEventListener('keyup', (e) => {
        // Update on arrow keys, home, end, etc
        setTimeout(updateLinkPopup, 10);
    });
    editor.addEventListener('focus', updateLinkPopup);

    // Update periodically while in editor (for arrow key navigation)
    let popupInterval = null;
    editor.addEventListener('focus', () => {
        if (popupInterval) clearInterval(popupInterval);
        popupInterval = setInterval(updateLinkPopup, 100);
    });
    editor.addEventListener('blur', () => {
        if (popupInterval) {
            clearInterval(popupInterval);
            popupInterval = null;
        }
        linkHoverPopup.classList.remove('visible');
    });

    // Handle link clicks (CMD+Click to open, CMD+SHIFT+Click to expand/shorten)
    editor.addEventListener('click', (e) => {
        // Only handle if CMD/CTRL is pressed
        if (!e.metaKey && !e.ctrlKey) return;

        const cursorPos = editor.selectionStart;
        const content = editor.value;

        // Find if click is within a link (text between + markers)
        const linkPattern = /\+([^\+]+)\+/g;
        let match;
        let clickedLinkText = null;

        while ((match = linkPattern.exec(content)) !== null) {
            const start = match.index;
            const end = start + match[0].length;

            if (cursorPos >= start && cursorPos <= end) {
                clickedLinkText = match[1]; // Text without + markers
                break;
            }
        }

        if (!clickedLinkText) return;

        // CMD/CTRL + SHIFT + Click: Toggle expansion
        if (e.shiftKey) {
            e.preventDefault();
            toggleLinkAtCursor(editor);
            // Update popup after toggle
            setTimeout(updateLinkPopup, 50);
            return;
        }

        // CMD/CTRL + Click: Open link
        e.preventDefault();

        // Try to get full URL from shortened version
        let urlToOpen = getFullUrlFromShortened(clickedLinkText);

        // If not found in map, check if it's already a full URL
        if (!urlToOpen && clickedLinkText.match(/^https?:\/\//)) {
            urlToOpen = clickedLinkText;
        }

        if (urlToOpen && isSafeUrl(urlToOpen)) {
            window.open(urlToOpen, '_blank');
        }
    });

    // Sync scroll between editor and code display overlay
    editor.addEventListener('scroll', () => {
        if (codeDisplay && codeDisplay.classList.contains('visible')) {
            codeDisplay.scrollTop = editor.scrollTop;
            codeDisplay.scrollLeft = editor.scrollLeft;
        }
    });

    // Session edit panel event listeners
    document.getElementById('session-edit-close').addEventListener('click', closeSessionEditPanel);
    document.getElementById('session-edit-cancel-btn').addEventListener('click', closeSessionEditPanel);
    document.getElementById('session-edit-save-btn').addEventListener('click', saveSessionEdit);
    sessionDeleteSelectedBtn.addEventListener('click', () => {
        const checkedBoxes = sessionEditUrlsList.querySelectorAll('.session-edit-url-checkbox:checked');
        if (checkedBoxes.length > 0 && confirm(`Delete ${checkedBoxes.length} URL(s)?`)) {
            // Just uncheck them - they'll be filtered out on save
            showTemplateNotification(`${checkedBoxes.length} URL(s) will be deleted on save`, 2000);
        }
    });

    // Aggressive Persistence Handlers
    // Ensure data is saved immediately when the extension loses focus or closes
    editor.addEventListener('blur', () => forceSave(editor));
    window.addEventListener('blur', () => forceSave(editor));
    window.addEventListener('beforeunload', () => forceSave(editor));
}

/**
 * Setup navigation buttons
 */
/**
 * Navigate with slide animation
 * @param {string} direction - 'next' (slide left) or 'prev' (slide right)
 * @param {Function} action - Async function to load the note
 */
async function navigateWithAnimation(direction, action) {
    const editorArea = document.querySelector('.editor-area');

    // Determine animation classes
    // Next: Slide content Left (Out Left, In Right)
    // Prev: Slide content Right (Out Right, In Left)
    const exitClass = direction === 'next' ? 'anim-slide-out-left' : 'anim-slide-out-right';
    const enterClass = direction === 'next' ? 'anim-slide-in-right' : 'anim-slide-in-left';

    // Start Exit Animation
    editorArea.classList.add(exitClass);

    // Wait for animation (200ms)
    await new Promise(resolve => setTimeout(resolve, 200));

    // Perform Navigation (Content Update)
    const note = await action();

    // Prepare Enter Animation
    // Remove exit class (would be opacity 0) and add enter class (starts opacity 0)
    editorArea.classList.remove(exitClass);

    if (note) {
        editorArea.classList.add(enterClass);

        // Wait for enter animation
        await new Promise(resolve => setTimeout(resolve, 200));

        // Cleanup
        editorArea.classList.remove(enterClass);
    } else {
        // If no note found (e.g. at end), just reset
        // Usually action() returns null if no move
        // But for 'slide' effect, we want to slide back?
        // Or if we stayed on same note (Next -> Empty), we just fade back in?
        // Force fade in
        editorArea.style.opacity = '0';
        requestAnimationFrame(() => {
            editorArea.style.transition = 'opacity 0.2s';
            editorArea.style.opacity = '1';
            setTimeout(() => {
                editorArea.style.transition = '';
                editorArea.style.opacity = '';
            }, 200);
        });
    }
}

/**
 * Setup navigation buttons
 */
function setupNavigationButtons() {
    document.getElementById('prev-btn').addEventListener('click', async () => {
        await forceSave(editor);

        await navigateWithAnimation('prev', async () => {
            const note = await goToPrevious();
            if (note) {
                await loadNote(note);
                return note;
            }
            return null;
        });
    });

    document.getElementById('next-btn').addEventListener('click', async () => {
        await forceSave(editor);

        await navigateWithAnimation('next', async () => {
            const content = getCurrentContent();
            const note = await goToNext(content);
            if (note) {
                await loadNote(note);
                return note;
            }
            return null;
        });
    });

    document.getElementById('jump-btn').addEventListener('click', async () => {
        await forceSave(editor);
        const note = await jumpToFront();
        if (note) {
            await loadNote(note);
        }
    });

    // Pin to top button
    document.getElementById('pin-btn').addEventListener('click', async () => {
        const currentNote = getCurrentNote();
        if (currentNote) {
            await forceSave(editor);
            const note = await promoteNote(currentNote.id);
            if (note) {
                await loadNote(note);
            }
        }
    });

    // New note button
    document.getElementById('new-note-btn').addEventListener('click', async () => {
        await forceSave(editor);

        // Animate as 'Next' (New content)
        await navigateWithAnimation('next', async () => {
            const newNote = await createNote();
            await setCurrentNoteId(newNote.id);
            await loadNote(newNote);
            return newNote;
        });
    });

    // Delete button
    document.getElementById('delete-btn').addEventListener('click', async () => {
        const currentNote = getCurrentNote();
        if (currentNote) {
            // Delete with fade out?
            // Simple fade out
            const editorArea = document.querySelector('.editor-area');
            editorArea.style.transition = 'opacity 0.2s';
            editorArea.style.opacity = '0';

            await new Promise(r => setTimeout(r, 200));

            const note = await deleteCurrentNote(currentNote.id);
            if (note) {
                await loadNote(note);
            }

            editorArea.style.opacity = '1';
            setTimeout(() => { editorArea.style.transition = ''; }, 200);
        }
    });

    // Copy formatted button
    document.getElementById('copy-formatted-btn').addEventListener('click', async () => {
        const currentNote = getCurrentNote();
        if (!currentNote || !currentNote.content.trim()) return;

        try {
            const content = currentNote.content;
            const { mode } = parseKeyword(content);
            let html = noteContentToHtml(content);
            let plainText = stripKeywordLine(content);

            // Append computed results for math/sum/avg/count modes
            if (mode === 'math') {
                const results = processMathContent(content, { significantDigits: 2 });
                const mathLines = buildMathCopyText(content, results);
                plainText = mathLines.plain;
                html = mathLines.html;
            } else if (mode === 'sum' || mode === 'avg') {
                const result = getStatsResult(mode, content);
                if (result) {
                    plainText += '\n\n' + result.label + ': ' + result.value;
                    html += `<p style="margin-top:12px;"><strong>${escapeHtml(result.label)}:</strong> ${escapeHtml(result.value)}</p>`;
                }
            } else if (mode === 'count') {
                const stats = countStats(content);
                const countText = `Items: ${stats.items}\nWords: ${stats.words}\nCharacters: ${stats.chars}\nSentences: ${stats.sentences}`;
                plainText += '\n\n' + countText;
                html += `<p style="margin-top:12px;">${escapeHtml(countText).replace(/\n/g, '<br>')}</p>`;
            }

            const blob = new Blob([html], { type: 'text/html' });
            const textBlob = new Blob([plainText], { type: 'text/plain' });
            await navigator.clipboard.write([
                new ClipboardItem({
                    'text/html': blob,
                    'text/plain': textBlob
                })
            ]);
            showTemplateNotification('Copied as formatted text', 1500);
        } catch (err) {
            console.error('Formatted copy failed:', err);
            showTemplateNotification('Copy failed', 1500);
        }
    });

    // Export button (footer only)
    document.getElementById('export-btn').addEventListener('click', exportNote);

    // Click-to-copy on sum/avg stats value
    if (statsValue) {
        statsValue.style.cursor = 'pointer';
        statsValue.title = 'Click to copy';
        statsValue.addEventListener('click', () => {
            const value = statsValue.textContent;
            if (value) {
                copyToClipboard(value);
                showCopyFeedback(statsValue);
            }
        });
    }
}

/**
 * Setup settings panel
 */
function setupSettingsPanel() {
    // Open settings - resize popup to 550x375
    document.getElementById('settings-btn').addEventListener('click', () => {
        document.body.style.width = '550px';
        document.body.style.height = '375px';
        settingsPanel.classList.add('visible');
    });

    // Close settings - resize popup back to 400x500
    document.getElementById('close-settings-btn').addEventListener('click', () => {
        settingsPanel.classList.remove('visible');
        document.body.style.width = '400px';
        document.body.style.height = '500px';
    });

    // Tab switching
    const tabs = document.querySelectorAll('.settings-tab');
    const tabContents = document.querySelectorAll('.settings-tab-content');

    tabs.forEach(tab => {
        tab.addEventListener('click', () => {
            const tabName = tab.dataset.tab;

            // Remove active class from all tabs and contents
            tabs.forEach(t => t.classList.remove('active'));
            tabContents.forEach(c => c.classList.remove('active'));

            // Add active class to clicked tab and corresponding content
            tab.classList.add('active');
            const content = document.querySelector(`.settings-tab-content[data-tab="${tabName}"]`);
            if (content) {
                content.classList.add('active');
            }

            // Initialize shortcuts if switching to shortcuts tab
            if (tabName === 'shortcuts') {
                if (isPro()) {
                    initShortcuts();
                } else {
                    const shortcutsContent = document.querySelector('.settings-tab-content[data-tab="shortcuts"]');
                    if (shortcutsContent) {
                        shortcutsContent.innerHTML = `
                            <div style="display:flex;flex-direction:column;align-items:center;justify-content:center;height:100%;gap:12px;padding:20px;text-align:center;">
                                <div style="font-size:14px;color:var(--text-secondary);">Keyboard Shortcuts customization is a Pro feature</div>
                                <button class="pro-hint-btn" onclick="document.getElementById('pro-upgrade-btn')?.click()">Upgrade — $4.99</button>
                            </div>
                        `;
                    }
                }
            }
        });
    });

    // Text size select
    document.getElementById('text-size-select').value = settings.textSize || 'm';
    document.getElementById('text-size-select').addEventListener('change', async (e) => {
        settings.textSize = e.target.value;
        await saveSettings(settings);
        applySettings();
    });

    // Paper Type
    document.getElementById('paper-type-select').value = settings.paperType || 'blank';
    document.getElementById('paper-type-select').addEventListener('change', async (e) => {
        settings.paperType = e.target.value;
        await saveSettings(settings);
        applySettings();
    });

    // Paper Opacity
    document.getElementById('paper-opacity-select').value = settings.paperOpacity || '0.1';
    document.getElementById('paper-opacity-select').addEventListener('change', async (e) => {
        settings.paperOpacity = e.target.value;
        await saveSettings(settings);
        applySettings();
    });
    // ── Text Editing Settings Tab ──

    // Copy section
    document.getElementById('edit-auto-copy').checked = settings.editAutoCopy || false;
    document.getElementById('edit-auto-copy').addEventListener('change', async (e) => {
        settings.editAutoCopy = e.target.checked;
        await saveSettings(settings);
    });

    document.getElementById('edit-skip-keyword-copy').checked = settings.editSkipKeywordCopy !== undefined ? settings.editSkipKeywordCopy : true;
    document.getElementById('edit-skip-keyword-copy').addEventListener('change', async (e) => {
        settings.editSkipKeywordCopy = e.target.checked;
        await saveSettings(settings);
    });

    document.getElementById('edit-skip-checklist-copy').checked = settings.editSkipChecklistCopy || false;
    document.getElementById('edit-skip-checklist-copy').addEventListener('change', async (e) => {
        settings.editSkipChecklistCopy = e.target.checked;
        await saveSettings(settings);
    });

    // Paste section
    document.getElementById('edit-paste-remove-spaces').checked = settings.editPasteRemoveSpaces !== undefined ? settings.editPasteRemoveSpaces : true;
    document.getElementById('edit-paste-remove-spaces').addEventListener('change', async (e) => {
        settings.editPasteRemoveSpaces = e.target.checked;
        await saveSettings(settings);
    });

    document.getElementById('edit-paste-remove-numbers').checked = settings.editPasteRemoveNumbers !== undefined ? settings.editPasteRemoveNumbers : true;
    document.getElementById('edit-paste-remove-numbers').addEventListener('change', async (e) => {
        settings.editPasteRemoveNumbers = e.target.checked;
        await saveSettings(settings);
    });

    document.getElementById('edit-paste-remove-bullets').checked = settings.editPasteRemoveBullets !== undefined ? settings.editPasteRemoveBullets : true;
    document.getElementById('edit-paste-remove-bullets').addEventListener('change', async (e) => {
        settings.editPasteRemoveBullets = e.target.checked;
        await saveSettings(settings);
    });

    document.getElementById('edit-paste-remove-markdown').checked = settings.editPasteRemoveMarkdown || false;
    document.getElementById('edit-paste-remove-markdown').addEventListener('change', async (e) => {
        settings.editPasteRemoveMarkdown = e.target.checked;
        await saveSettings(settings);
    });

    document.getElementById('edit-paste-remove-empty').checked = settings.editPasteRemoveEmpty || false;
    document.getElementById('edit-paste-remove-empty').addEventListener('change', async (e) => {
        settings.editPasteRemoveEmpty = e.target.checked;
        await saveSettings(settings);
    });

    // Typing section
    document.getElementById('edit-keep-indent').checked = settings.editKeepIndent !== undefined ? settings.editKeepIndent : true;
    document.getElementById('edit-keep-indent').addEventListener('change', async (e) => {
        settings.editKeepIndent = e.target.checked;
        await saveSettings(settings);
    });

    document.getElementById('edit-macos-replacements').checked = settings.editMacosReplacements || false;
    document.getElementById('edit-macos-replacements').addEventListener('change', async (e) => {
        settings.editMacosReplacements = e.target.checked;
        await saveSettings(settings);
    });

    // Hyperlinks section
    const editDisableLinkShortening = document.getElementById('edit-disable-link-shortening');
    const editDisableAllLinks = document.getElementById('edit-disable-all-links');

    editDisableLinkShortening.checked = settings.disableLinkShortening || false;
    editDisableLinkShortening.addEventListener('change', async (e) => {
        settings.disableLinkShortening = e.target.checked;
        await saveSettings(settings);
    });

    editDisableAllLinks.checked = settings.disableAllLinks || false;
    editDisableAllLinks.addEventListener('change', async (e) => {
        settings.disableAllLinks = e.target.checked;
        await saveSettings(settings);
    });

    // Export settings
    document.getElementById('export-omit-keyword').checked = settings.exportOmitKeyword || false;
    document.getElementById('export-omit-keyword').addEventListener('change', async (e) => {
        settings.exportOmitKeyword = e.target.checked;
        await saveSettings(settings);
    });

    document.getElementById('export-use-first-line').checked = settings.exportUseFirstLine || false;
    document.getElementById('export-use-first-line').addEventListener('change', async (e) => {
        settings.exportUseFirstLine = e.target.checked;
        await saveSettings(settings);
    });

    // Export all notes button
    document.getElementById('export-all-notes-btn').addEventListener('click', exportAllNotes);

    // Export sessions button
    const exportSessionsBtn = document.getElementById('export-sessions-btn');
    if (exportSessionsBtn) {
        exportSessionsBtn.addEventListener('click', exportSessions);
    }

    // Import sessions button
    const importSessionsZipBtn = document.getElementById('import-sessions-zip-btn');
    if (importSessionsZipBtn) {
        importSessionsZipBtn.addEventListener('click', () => {
            const input = document.createElement('input');
            input.type = 'file';
            input.accept = '.zip';
            input.addEventListener('change', async (e) => {
                const file = e.target.files[0];
                if (file) {
                    await importSessionsFromZip(file);
                }
            });
            input.click();
        });
    }

    // Import bookmarks from JSON via settings button
    const importBookmarksZipBtn = document.getElementById('import-bookmarks-zip-btn');
    if (importBookmarksZipBtn) {
        importBookmarksZipBtn.addEventListener('click', () => {
            const input = document.createElement('input');
            input.type = 'file';
            input.accept = '.json';
            input.addEventListener('change', async (e) => {
                const file = e.target.files[0];
                if (file) {
                    try {
                        const text = await file.text();
                        const result = await importBookmarksFromJSON(text);
                        showTemplateNotification(`Imported ${result.imported} bookmark(s), ${result.skipped} skipped`, 3000);
                    } catch (err) {
                        showTemplateNotification(`Import failed: ${err.message}`, 3000);
                    }
                }
            });
            input.click();
        });
    }

    // Notes tab settings
    // Spell check checkbox
    const spellCheckCheckbox = document.getElementById('spell-check-checkbox');
    if (spellCheckCheckbox) {
        spellCheckCheckbox.checked = settings.spellCheckEnabled || false;
        spellCheckCheckbox.addEventListener('change', async (e) => {
            if (!isPro()) {
                e.target.checked = false;
                showTemplateNotification('Spell Check is a Pro feature', 2000);
                return;
            }
            settings.spellCheckEnabled = e.target.checked;
            await saveSettings(settings);

            try {
                await setSpellCheckEnabled(e.target.checked);
                forceUpdateUI();
            } catch (err) {
                console.error('Spell check toggle failed:', err);
                showTemplateNotification('Spell check toggle failed', 2000);
            }
        });
    }


    // New note on launch checkbox
    document.getElementById('new-note-on-launch-checkbox').checked = settings.newNoteOnLaunch || false;
    document.getElementById('new-note-on-launch-checkbox').addEventListener('change', async (e) => {
        settings.newNoteOnLaunch = e.target.checked;
        await saveSettings(settings);
    });

    // Auto-delete slider
    document.getElementById('auto-delete-slider').value = settings.autoDeletePeriod ?? 4;
    document.getElementById('auto-delete-slider').addEventListener('input', async (e) => {
        if (!isPro()) {
            e.target.value = 4; // Reset to 'Never'
            showTemplateNotification('Auto-Delete is a Pro feature', 2000);
            return;
        }
        settings.autoDeletePeriod = parseInt(e.target.value);
        await saveSettings(settings);
    });

    // Delete date button
    document.getElementById('delete-date-btn').addEventListener('click', async () => {
        const dateInput = document.getElementById('delete-since-date');
        const selectedDate = dateInput.value;

        if (!selectedDate) {
            alert('Please select a date first.');
            return;
        }

        const confirmDelete = confirm(`Delete all notes not modified since ${selectedDate}?`);
        if (!confirmDelete) return;

        const cutoffDate = new Date(selectedDate);
        const allNotes = await getNotes();
        let deletedCount = 0;

        for (const note of allNotes) {
            const modifiedDate = new Date(note.modifiedAt || note.createdAt);
            if (modifiedDate < cutoffDate) {
                await deleteNote(note.id);
                deletedCount++;
            }
        }

        alert(`Deleted ${deletedCount} note(s).`);

        // Reload current note if it was deleted
        const currentId = await getCurrentNoteId();
        const currentNote = await getNote(currentId);
        if (!currentNote) {
            await loadCurrentNote();
        }
    });

    // Auto-create new note slider
    document.getElementById('auto-create-slider').value = settings.autoCreatePeriod ?? 5;
    document.getElementById('auto-create-slider').addEventListener('input', async (e) => {
        settings.autoCreatePeriod = parseInt(e.target.value);
        await saveSettings(settings);
    });

    // Show note count checkbox
    document.getElementById('show-note-count-checkbox').checked = settings.showNoteCount || false;
    document.getElementById('show-note-count-checkbox').addEventListener('change', async (e) => {
        settings.showNoteCount = e.target.checked;
        await saveSettings(settings);
        updateNoteCount();
    });

    // Show tag cloud in search checkbox
    document.getElementById('show-tag-cloud-checkbox').checked = settings.showTagCloud !== false;
    document.getElementById('show-tag-cloud-checkbox').addEventListener('change', async (e) => {
        settings.showTagCloud = e.target.checked;
        await saveSettings(settings);
    });

    // Backup count input
    document.getElementById('backup-count-input').value = settings.backupCount ?? 12;
    document.getElementById('backup-count-input').addEventListener('change', async (e) => {
        if (!isPro()) {
            showTemplateNotification('Auto-Backups is a Pro feature', 2000);
            return;
        }
        settings.backupCount = parseInt(e.target.value) || 12;
        await saveSettings(settings);
    });

    // Backup interval slider
    document.getElementById('backup-interval-slider').value = settings.backupInterval ?? 8;
    document.getElementById('backup-interval-slider').addEventListener('input', async (e) => {
        if (!isPro()) {
            showTemplateNotification('Auto-Backups is a Pro feature', 2000);
            return;
        }
        settings.backupInterval = parseInt(e.target.value);
        await saveSettings(settings);
    });

    // ── Notes & Backups Folder ──

    // Backup folder UI elements
    const folderPathEl = document.getElementById('backup-folder-path');
    const exportNowBtn = document.getElementById('export-now-btn');
    const chooseFolderBtn = document.getElementById('choose-folder-btn');

    function updateFolderUI(folderName) {
        if (folderName) {
            folderPathEl.textContent = folderName;
            exportNowBtn.style.display = '';
            chooseFolderBtn.textContent = 'Change Folder';
        } else {
            folderPathEl.textContent = 'No folder selected';
            exportNowBtn.style.display = 'none';
            chooseFolderBtn.textContent = 'Choose Folder';
        }
    }

    // Initialize display from saved settings
    updateFolderUI(settings.backupFolderName || null);

    // Listen for storage changes so the UI updates after the helper window saves
    chrome.storage.onChanged.addListener((changes) => {
        if (changes.jottnote_settings?.newValue?.backupFolderName !== undefined) {
            const newName = changes.jottnote_settings.newValue.backupFolderName;
            settings.backupFolderName = newName;
            updateFolderUI(newName || null);
        }
    });

    // Choose folder — opens a small helper window where the native picker works
    // (extension popups close when native OS dialogs take focus, but browser windows don't)
    chooseFolderBtn.addEventListener('click', () => {
        chrome.windows.create({
            url: chrome.runtime.getURL('popup.html?pickFolder=true'),
            type: 'popup',
            width: 420,
            height: 200,
            focused: true
        });
    });

    // Export now — opens a helper window with a save-file picker
    // (same approach as folder selection — browser windows survive native dialogs)
    exportNowBtn.addEventListener('click', () => {
        chrome.windows.create({
            url: chrome.runtime.getURL('popup.html?exportNow=true'),
            type: 'popup',
            width: 420,
            height: 220,
            focused: true
        });
    });

    // Keywords tab settings

    // Helper: populate a main keyword <select> with the built-in default + custom keywords
    function populateMainKeywordSelect(selectId, defaultKeyword, customKeywordsStr, currentValue) {
        const select = document.getElementById(selectId);
        if (!select) return;
        const customs = (customKeywordsStr || '')
            .split(',').map(k => k.trim().toLowerCase()).filter(k => k && k !== defaultKeyword);
        select.innerHTML = '';
        // Default keyword is always first
        const defOpt = document.createElement('option');
        defOpt.value = defaultKeyword;
        defOpt.textContent = defaultKeyword;
        select.appendChild(defOpt);
        // Add custom keywords as options
        customs.forEach(kw => {
            const opt = document.createElement('option');
            opt.value = kw;
            opt.textContent = kw;
            select.appendChild(opt);
        });
        select.value = currentValue || defaultKeyword;
    }

    // Helper: populate the main checker <select> for list check keywords
    function populateMainCheckerSelect(selectId, defaultChecker, customCheckersStr, currentValue) {
        const select = document.getElementById(selectId);
        if (!select) return;
        const customs = (customCheckersStr || '')
            .split(',').map(k => k.trim().toLowerCase()).filter(k => k && k !== defaultChecker);
        select.innerHTML = '';
        const defOpt = document.createElement('option');
        defOpt.value = defaultChecker;
        defOpt.textContent = defaultChecker;
        select.appendChild(defOpt);
        customs.forEach(kw => {
            const opt = document.createElement('option');
            opt.value = kw;
            opt.textContent = kw;
            select.appendChild(opt);
        });
        select.value = currentValue || defaultChecker;
    }

    // List section
    document.getElementById('list-keywords-input').value = settings.listKeywords ?? '';
    document.getElementById('list-keywords-input').addEventListener('change', async (e) => {
        settings.listKeywords = e.target.value;
        await saveSettings(settings);
        populateMainKeywordSelect('list-main-keyword-select', 'list', e.target.value, settings.listMainKeyword);
        applySettings();
    });

    populateMainKeywordSelect('list-main-keyword-select', 'list', settings.listKeywords, settings.listMainKeyword);
    document.getElementById('list-main-keyword-select').addEventListener('change', async (e) => {
        settings.listMainKeyword = e.target.value;
        await saveSettings(settings);
        applySettings();
    });

    document.getElementById('list-check-keywords-input').value = settings.listCheckKeywords ?? '/x, done';
    document.getElementById('list-check-keywords-input').addEventListener('change', async (e) => {
        settings.listCheckKeywords = e.target.value;
        await saveSettings(settings);
        populateMainCheckerSelect('list-main-checker-select', '/x', e.target.value, settings.listMainChecker);
    });

    populateMainCheckerSelect('list-main-checker-select', '/x', settings.listCheckKeywords, settings.listMainChecker);
    document.getElementById('list-main-checker-select').addEventListener('change', async (e) => {
        settings.listMainChecker = e.target.value;
        await saveSettings(settings);
    });

    // Auto-complete children setting
    document.getElementById('list-auto-complete-children').checked = settings.listAutoCompleteChildren || false;
    document.getElementById('list-auto-complete-children').addEventListener('change', async (e) => {
        settings.listAutoCompleteChildren = e.target.checked;
        await saveSettings(settings);
    });

    // Math section
    document.getElementById('math-keywords-input').value = settings.mathKeywords ?? '';
    document.getElementById('math-keywords-input').addEventListener('change', async (e) => {
        settings.mathKeywords = e.target.value;
        await saveSettings(settings);
        populateMainKeywordSelect('math-main-keyword-select', 'math', e.target.value, settings.mathMainKeyword);
        applySettings();
    });

    populateMainKeywordSelect('math-main-keyword-select', 'math', settings.mathKeywords, settings.mathMainKeyword);
    document.getElementById('math-main-keyword-select').addEventListener('change', async (e) => {
        settings.mathMainKeyword = e.target.value;
        await saveSettings(settings);
        applySettings();
    });

    document.getElementById('math-sig-digits-slider').value = settings.mathSigDigits ?? 2;
    document.getElementById('math-sig-digits-slider').addEventListener('input', async (e) => {
        settings.mathSigDigits = parseInt(e.target.value);
        await saveSettings(settings);
    });

    document.getElementById('math-thousands-comma-checkbox').checked = settings.mathThousandsComma ?? true;
    document.getElementById('math-thousands-comma-checkbox').addEventListener('change', async (e) => {
        settings.mathThousandsComma = e.target.checked;
        await saveSettings(settings);
    });

    document.getElementById('math-update-currencies-checkbox').checked = settings.mathUpdateCurrencies ?? true;
    document.getElementById('math-update-currencies-checkbox').addEventListener('change', async (e) => {
        settings.mathUpdateCurrencies = e.target.checked;
        await saveSettings(settings);
    });

    document.getElementById('math-custom-rates-textarea').value = settings.mathCustomRates || '';
    document.getElementById('math-custom-rates-textarea').addEventListener('change', async (e) => {
        settings.mathCustomRates = e.target.value;
        await saveSettings(settings);
    });

    document.getElementById('math-primary-symbol-select').value = settings.mathPrimarySymbol || '$';
    document.getElementById('math-primary-symbol-select').addEventListener('change', async (e) => {
        settings.mathPrimarySymbol = e.target.value;
        await saveSettings(settings);
    });

    document.getElementById('math-primary-currency-select').value = settings.mathPrimaryCurrency || 'USD';
    document.getElementById('math-primary-currency-select').addEventListener('change', async (e) => {
        settings.mathPrimaryCurrency = e.target.value;
        await saveSettings(settings);
    });

    document.getElementById('math-secondary-currency-select').value = settings.mathSecondaryCurrency || 'CAD';
    document.getElementById('math-secondary-currency-select').addEventListener('change', async (e) => {
        settings.mathSecondaryCurrency = e.target.value;
        await saveSettings(settings);
    });

    // Sum section
    document.getElementById('sum-keywords-input').value = settings.sumKeywords ?? '';
    document.getElementById('sum-keywords-input').addEventListener('change', async (e) => {
        settings.sumKeywords = e.target.value;
        await saveSettings(settings);
        populateMainKeywordSelect('sum-main-keyword-select', 'sum', e.target.value, settings.sumMainKeyword);
        applySettings();
    });

    populateMainKeywordSelect('sum-main-keyword-select', 'sum', settings.sumKeywords, settings.sumMainKeyword);
    document.getElementById('sum-main-keyword-select').addEventListener('change', async (e) => {
        settings.sumMainKeyword = e.target.value;
        await saveSettings(settings);
        applySettings();
    });

    // Avg section
    document.getElementById('avg-keywords-input').value = settings.avgKeywords ?? '';
    document.getElementById('avg-keywords-input').addEventListener('change', async (e) => {
        settings.avgKeywords = e.target.value;
        await saveSettings(settings);
        populateMainKeywordSelect('avg-main-keyword-select', 'avg', e.target.value, settings.avgMainKeyword);
        applySettings();
    });

    populateMainKeywordSelect('avg-main-keyword-select', 'avg', settings.avgKeywords, settings.avgMainKeyword);
    document.getElementById('avg-main-keyword-select').addEventListener('change', async (e) => {
        settings.avgMainKeyword = e.target.value;
        await saveSettings(settings);
        applySettings();
    });

    // Count section
    document.getElementById('count-keywords-input').value = settings.countKeywords ?? '';
    document.getElementById('count-keywords-input').addEventListener('change', async (e) => {
        settings.countKeywords = e.target.value;
        await saveSettings(settings);
        populateMainKeywordSelect('count-main-keyword-select', 'count', e.target.value, settings.countMainKeyword);
        applySettings();
    });

    populateMainKeywordSelect('count-main-keyword-select', 'count', settings.countKeywords, settings.countMainKeyword);
    document.getElementById('count-main-keyword-select').addEventListener('change', async (e) => {
        settings.countMainKeyword = e.target.value;
        await saveSettings(settings);
        applySettings();
    });

    // Code section
    document.getElementById('code-keywords-input').value = settings.codeKeywords ?? '';
    document.getElementById('code-keywords-input').addEventListener('change', async (e) => {
        settings.codeKeywords = e.target.value;
        await saveSettings(settings);
        populateMainKeywordSelect('code-main-keyword-select', 'code', e.target.value, settings.codeMainKeyword);
        applySettings();
    });

    populateMainKeywordSelect('code-main-keyword-select', 'code', settings.codeKeywords, settings.codeMainKeyword);
    document.getElementById('code-main-keyword-select').addEventListener('change', async (e) => {
        settings.codeMainKeyword = e.target.value;
        await saveSettings(settings);
        applySettings();
    });

    // Timer section
    document.getElementById('timer-keywords-input').value = settings.timerKeywords ?? '';
    document.getElementById('timer-keywords-input').addEventListener('change', async (e) => {
        settings.timerKeywords = e.target.value;
        await saveSettings(settings);
        populateMainKeywordSelect('timer-main-keyword-select', 'timer', e.target.value, settings.timerMainKeyword);
        applySettings();
    });

    populateMainKeywordSelect('timer-main-keyword-select', 'timer', settings.timerKeywords, settings.timerMainKeyword);
    document.getElementById('timer-main-keyword-select').addEventListener('change', async (e) => {
        settings.timerMainKeyword = e.target.value;
        await saveSettings(settings);
        applySettings();
    });

    document.getElementById('timer-pause-on-close-checkbox').checked = settings.timerPauseOnClose || false;
    document.getElementById('timer-pause-on-close-checkbox').addEventListener('change', async (e) => {
        settings.timerPauseOnClose = e.target.checked;
        await saveSettings(settings);
    });

    // Date section
    document.getElementById('date-keywords-input').value = settings.dateKeywords ?? '';
    document.getElementById('date-keywords-input').addEventListener('change', async (e) => {
        settings.dateKeywords = e.target.value;
        await saveSettings(settings);
        populateMainKeywordSelect('date-main-keyword-select', 'date', e.target.value, settings.dateMainKeyword);
        applySettings();
    });

    populateMainKeywordSelect('date-main-keyword-select', 'date', settings.dateKeywords, settings.dateMainKeyword);
    document.getElementById('date-main-keyword-select').addEventListener('change', async (e) => {
        settings.dateMainKeyword = e.target.value;
        await saveSettings(settings);
        applySettings();
    });

    // Date format select (Notes tab)
    document.getElementById('date-format-select').value = settings.dateFormat || 'MMM D, YYYY';
    document.getElementById('date-format-select').addEventListener('change', async (e) => {
        settings.dateFormat = e.target.value;
        await saveSettings(settings);
        // Re-render date mode if active
        const content = editor.value;
        const { mode } = parseKeyword(content);
        if (mode === 'date') {
            updateDateMode(content);
        }
    });

    // ── Timer Settings Tab ──

    // General
    document.getElementById('timer-pause-on-quit').checked = settings.timerPauseOnQuit || false;
    document.getElementById('timer-pause-on-quit').addEventListener('change', async (e) => {
        settings.timerPauseOnQuit = e.target.checked;
        await saveSettings(settings);
    });

    document.getElementById('timer-show-in-menubar').checked = settings.timerShowInMenubar !== undefined ? settings.timerShowInMenubar : true;
    document.getElementById('timer-show-in-menubar').addEventListener('change', async (e) => {
        settings.timerShowInMenubar = e.target.checked;
        await saveSettings(settings);
    });

    // Countdown Timer
    document.getElementById('countdown-notify').checked = settings.countdownNotify || false;
    document.getElementById('countdown-notify').addEventListener('change', async (e) => {
        settings.countdownNotify = e.target.checked;
        await saveSettings(settings);
    });

    document.getElementById('countdown-fullscreen-alert').checked = settings.countdownFullscreenAlert !== undefined ? settings.countdownFullscreenAlert : true;
    document.getElementById('countdown-fullscreen-alert').addEventListener('change', async (e) => {
        settings.countdownFullscreenAlert = e.target.checked;
        await saveSettings(settings);
    });

    document.getElementById('countdown-play-sound').checked = settings.countdownPlaySound !== undefined ? settings.countdownPlaySound : true;
    document.getElementById('countdown-play-sound').addEventListener('change', async (e) => {
        settings.countdownPlaySound = e.target.checked;
        await saveSettings(settings);
    });

    // Pomodoro Timer
    document.getElementById('pomodoro-notify').checked = settings.pomodoroNotify || false;
    document.getElementById('pomodoro-notify').addEventListener('change', async (e) => {
        settings.pomodoroNotify = e.target.checked;
        await saveSettings(settings);
    });

    document.getElementById('pomodoro-fullscreen-alert').checked = settings.pomodoroFullscreenAlert !== undefined ? settings.pomodoroFullscreenAlert : true;
    document.getElementById('pomodoro-fullscreen-alert').addEventListener('change', async (e) => {
        settings.pomodoroFullscreenAlert = e.target.checked;
        await saveSettings(settings);
    });

    document.getElementById('pomodoro-play-sound').checked = settings.pomodoroPlaySound !== undefined ? settings.pomodoroPlaySound : true;
    document.getElementById('pomodoro-play-sound').addEventListener('change', async (e) => {
        settings.pomodoroPlaySound = e.target.checked;
        await saveSettings(settings);
    });

    // Sound Volume
    const volumeSlider = document.getElementById('timer-volume-slider');
    const volumeLabel = document.getElementById('timer-volume-label');
    volumeSlider.value = settings.timerVolume !== undefined ? settings.timerVolume : 40;
    volumeLabel.textContent = volumeSlider.value + '%';

    // Update the slider fill color to reflect the current value
    function updateVolumeSliderFill() {
        const val = volumeSlider.value;
        const pct = val + '%';
        volumeSlider.style.background = `linear-gradient(to right, #5BA3F5 0%, #5BA3F5 ${pct}, rgba(255,255,255,0.15) ${pct}, rgba(255,255,255,0.15) 100%)`;
    }
    updateVolumeSliderFill();

    volumeSlider.addEventListener('input', (e) => {
        volumeLabel.textContent = e.target.value + '%';
        updateVolumeSliderFill();
    });

    volumeSlider.addEventListener('change', async (e) => {
        settings.timerVolume = parseInt(e.target.value);
        await saveSettings(settings);
    });

    document.getElementById('timer-test-sound-btn').addEventListener('click', () => {
        const chime = document.getElementById('timer-chime');
        if (chime) {
            const vol = (settings.timerVolume !== undefined ? settings.timerVolume : 40) / 100;
            chime.volume = Math.min(Math.max(vol, 0), 1);
            chime.currentTime = 0;
            chime.play().catch(err => console.warn('Could not play test sound:', err));
        }
    });

    // General section
    document.getElementById('disable-all-keywords-checkbox').checked = settings.disableAllKeywords || false;
    document.getElementById('disable-all-keywords-checkbox').addEventListener('change', async (e) => {
        settings.disableAllKeywords = e.target.checked;
        await saveSettings(settings);
    });



    // Side panel mode checkbox (Misc tab)
    const sidePanelCheckbox = document.getElementById('side-panel-mode-checkbox');
    if (sidePanelCheckbox) {
        sidePanelCheckbox.checked = settings.sidePanelMode || false;
        sidePanelCheckbox.addEventListener('change', async (e) => {
            if (!isPro()) {
                e.target.checked = false;
                showTemplateNotification('Side Panel is a Pro feature', 2000);
                return;
            }
            settings.sidePanelMode = e.target.checked;
            await saveSettings(settings);

            // Explicitly notify background script to update behavior immediately
            try {
                await chrome.runtime.sendMessage({
                    type: 'UPDATE_SIDE_PANEL_MODE',
                    enabled: settings.sidePanelMode
                });

            } catch (err) {
                console.warn('Failed to send side panel update:', err);
            }
        });
    }

    // Bookmarks import section

    const importBookmarksBtn = document.getElementById('import-bookmarks-btn');
    const importBookmarksStatus = document.getElementById('import-bookmarks-status');

    if (importBookmarksBtn && importBookmarksStatus) {
        importBookmarksBtn.addEventListener('click', async () => {
            importBookmarksStatus.textContent = 'Importing...';
            importBookmarksStatus.style.color = 'var(--text-secondary)';

            try {
                const count = await importFromChrome();
                importBookmarksStatus.textContent = `✓ Imported ${count} bookmarks`;
                importBookmarksStatus.style.color = 'var(--accent-3-main)';
                setTimeout(() => {
                    importBookmarksStatus.textContent = '';
                }, 3000);
            } catch (err) {
                importBookmarksStatus.textContent = `✗ Error: ${err.message}`;
                importBookmarksStatus.style.color = 'var(--accent-5-main)';
                setTimeout(() => {
                    importBookmarksStatus.textContent = '';
                }, 5000);
            }
        });
    }

    // Save session button
    const saveSessionBtn = document.getElementById('save-session-btn');
    const saveSessionStatus = document.getElementById('save-session-status');

    if (saveSessionBtn && saveSessionStatus) {
        saveSessionBtn.addEventListener('click', async () => {
            saveSessionStatus.textContent = 'Saving...';
            saveSessionStatus.style.color = 'var(--text-secondary)';

            try {
                await saveCurrentSession('', []);
                saveSessionStatus.textContent = '✓ Session saved';
                saveSessionStatus.style.color = 'var(--accent-2-main)';
                setTimeout(() => {
                    saveSessionStatus.textContent = '';
                }, 3000);
            } catch (err) {
                saveSessionStatus.textContent = `✗ Error: ${err.message}`;
                saveSessionStatus.style.color = 'var(--accent-5-main)';
                setTimeout(() => {
                    saveSessionStatus.textContent = '';
                }, 5000);
            }
        });
    }

    // ── Syntax Highlighting Settings ──

    // Populate default language dropdown
    const defaultLangSelect = document.getElementById('default-code-language-select');
    if (defaultLangSelect) {
        const languages = getSupportedLanguages();
        defaultLangSelect.innerHTML = '';
        languages.forEach(lang => {
            const opt = document.createElement('option');
            opt.value = lang.aliases[0]; // first alias (e.g. 'js', 'py')
            opt.textContent = lang.name.charAt(0).toUpperCase() + lang.name.slice(1);
            defaultLangSelect.appendChild(opt);
        });
        defaultLangSelect.value = settings.defaultCodeLanguage || 'js';
        defaultLangSelect.addEventListener('change', async (e) => {
            settings.defaultCodeLanguage = e.target.value;
            await saveSettings(settings);
        });
    }



}

/**
 * Apply syntax theme CSS custom properties by mapping the active theme's accent colors.
 * Code mode shares the same palette as the rest of the UI, ensuring visual consistency.
 * Uses accent mains AND secondaries for maximum color variety in code highlighting.
 */
function applySyntaxTheme() {
    const root = document.documentElement;
    const cs = getComputedStyle(root);

    // Read current accent and derived values from the active theme
    const textPri = cs.getPropertyValue('--text-primary').trim();
    const textSec = cs.getPropertyValue('--text-secondary').trim();
    const textLight = cs.getPropertyValue('--text-light').trim() || textSec;
    const a1 = cs.getPropertyValue('--accent-1-main').trim();
    const a1sec = cs.getPropertyValue('--accent-1-secondary').trim() || a1;
    const a2 = cs.getPropertyValue('--accent-2-main').trim();
    const a3 = cs.getPropertyValue('--accent-3-main').trim();
    const a3sec = cs.getPropertyValue('--accent-3-secondary').trim() || a3;
    const a4 = cs.getPropertyValue('--accent-4-main').trim();
    const a4sec = cs.getPropertyValue('--accent-4-secondary').trim() || a4;
    const a5 = cs.getPropertyValue('--accent-5-main').trim();
    const a5sec = cs.getPropertyValue('--accent-5-secondary').trim() || a5;
    const a6 = cs.getPropertyValue('--accent-6-main').trim();
    const a6sec = cs.getPropertyValue('--accent-6-secondary').trim() || a6;

    // Map syntax vars to theme accents — each token type gets a distinct color
    root.style.setProperty('--syntax-base', textPri);         // Default code text
    root.style.setProperty('--syntax-punct', textLight);      // Brackets, parens, commas
    root.style.setProperty('--syntax-comment', textSec);      // Comments (italic)
    root.style.setProperty('--syntax-keyword', a1);           // Keywords (if, for, return)
    root.style.setProperty('--syntax-selector', a1sec);       // CSS selectors
    root.style.setProperty('--syntax-function', a2);          // Function calls
    root.style.setProperty('--syntax-string', a3);            // String literals
    root.style.setProperty('--syntax-operator', a3sec);       // Operators (=, +, &&)
    root.style.setProperty('--syntax-tag', a4);               // Type names, HTML tags
    root.style.setProperty('--syntax-builtin', a4sec);        // Built-in functions/objects
    root.style.setProperty('--syntax-number', a5);            // Numeric literals
    root.style.setProperty('--syntax-value', a5sec);          // Value keywords (true, null)
    root.style.setProperty('--syntax-property', a6);          // Object properties
    root.style.setProperty('--syntax-attr', a6sec);           // Decorators, attributes
}

/**
 * Hex ↔ HSL helpers for color derivation
 */
function hexToHSL(hex) {
    hex = hex.replace('#', '');
    if (hex.length === 3) hex = hex.split('').map(c => c + c).join('');
    const r = parseInt(hex.slice(0, 2), 16) / 255;
    const g = parseInt(hex.slice(2, 4), 16) / 255;
    const b = parseInt(hex.slice(4, 6), 16) / 255;
    const max = Math.max(r, g, b), min = Math.min(r, g, b);
    let h = 0, s = 0, l = (max + min) / 2;
    if (max !== min) {
        const d = max - min;
        s = l > 0.5 ? d / (2 - max - min) : d / (max + min);
        if (max === r) h = ((g - b) / d + (g < b ? 6 : 0)) / 6;
        else if (max === g) h = ((b - r) / d + 2) / 6;
        else h = ((r - g) / d + 4) / 6;
    }
    return { h: h * 360, s: s * 100, l: l * 100 };
}

function hslToHex(h, s, l) {
    h = ((h % 360) + 360) % 360;
    s = Math.max(0, Math.min(100, s)) / 100;
    l = Math.max(0, Math.min(100, l)) / 100;
    const c = (1 - Math.abs(2 * l - 1)) * s;
    const x = c * (1 - Math.abs(((h / 60) % 2) - 1));
    const m = l - c / 2;
    let r = 0, g = 0, b = 0;
    if (h < 60) { r = c; g = x; }
    else if (h < 120) { r = x; g = c; }
    else if (h < 180) { g = c; b = x; }
    else if (h < 240) { g = x; b = c; }
    else if (h < 300) { r = x; b = c; }
    else { r = c; b = x; }
    return '#' + [r + m, g + m, b + m].map(v =>
        Math.round(v * 255).toString(16).padStart(2, '0')
    ).join('').toUpperCase();
}

/**
 * Derive a lighter/darker variant by shifting lightness toward a target.
 * @param {string} hex - base color
 * @param {number} amount - lightness shift (positive = lighter, negative = darker)
 * @param {number} [desat=0] - desaturation amount (0–100)
 */
function shiftColor(hex, amount, desat = 0) {
    const hsl = hexToHSL(hex);
    hsl.l = Math.max(0, Math.min(100, hsl.l + amount));
    hsl.s = Math.max(0, Math.min(100, hsl.s - desat));
    return hslToHex(hsl.h, hsl.s, hsl.l);
}

/**
 * Mix two hex colors at a given ratio (0 = all color1, 1 = all color2).
 */
function mixColors(hex1, hex2, ratio) {
    const h1 = hex1.replace('#', ''), h2 = hex2.replace('#', '');
    const r = Math.round(parseInt(h1.slice(0, 2), 16) * (1 - ratio) + parseInt(h2.slice(0, 2), 16) * ratio);
    const g = Math.round(parseInt(h1.slice(2, 4), 16) * (1 - ratio) + parseInt(h2.slice(2, 4), 16) * ratio);
    const b = Math.round(parseInt(h1.slice(4, 6), 16) * (1 - ratio) + parseInt(h2.slice(4, 6), 16) * ratio);
    return '#' + [r, g, b].map(v => Math.max(0, Math.min(255, v)).toString(16).padStart(2, '0')).join('').toUpperCase();
}

/**
 * Compute and apply derived color variations from the 10 main theme colors.
 * Custom themes can override any derived value by including it in their colors object.
 * The function checks if a CSS var is already set by the theme before overriding.
 */
function computeDerivedColors() {
    const root = document.documentElement;
    const cs = getComputedStyle(root);

    // Read main colors
    const bg = cs.getPropertyValue('--bg-primary').trim();
    const text = cs.getPropertyValue('--text-primary').trim();
    const textSec = cs.getPropertyValue('--text-secondary').trim();
    const a1 = cs.getPropertyValue('--accent-1-main').trim();
    const a2 = cs.getPropertyValue('--accent-2-main').trim();
    const a3 = cs.getPropertyValue('--accent-3-main').trim();
    const a4 = cs.getPropertyValue('--accent-4-main').trim();
    const a5 = cs.getPropertyValue('--accent-5-main').trim();
    const a6 = cs.getPropertyValue('--accent-6-main').trim();
    const border = cs.getPropertyValue('--border').trim();

    // Determine if theme is dark or light
    const bgHSL = hexToHSL(bg);
    const isDark = bgHSL.l < 50;

    // Lightness shift direction: dark themes lighten, light themes darken
    const dir = isDark ? 1 : -1;

    // ─── Derived vars with defaults (only set if not already overridden) ───
    const derived = {
        // Background
        '--bg-fade': shiftColor(bg, dir * 6),

        // Text variations (progressively lighter toward bg)
        '--text-subtle-plus': shiftColor(textSec, dir * 10),
        '--text-highlight': mixColors(text, bg, 0.65),
        '--text-light': mixColors(text, bg, 0.55),
        '--text-superlight': mixColors(text, bg, 0.75),
        '--text-hyperlight': mixColors(text, bg, 0.85),
        '--text-reverse': isDark ? '#FFFFFF' : '#000000',

        // Accent secondary/tertiary (muted variations)
        '--accent-1-secondary': shiftColor(a1, dir * 12, 8),
        '--accent-1-tertiary': shiftColor(a1, dir * 24, 15),
        '--accent-2-secondary': shiftColor(a2, dir * 12, 8),
        '--accent-3-secondary': shiftColor(a3, dir * 12, 8),
        '--accent-4-secondary': shiftColor(a4, dir * 12, 8),
        '--accent-5-secondary': shiftColor(a5, dir * 12, 8),
        '--accent-6-secondary': shiftColor(a6, dir * 12, 8),

        // Grid colors (subtle background variations)
        '--grid-superlight': mixColors(bg, text, 0.04),
        '--grid-clear': mixColors(bg, text, 0.08),
        '--grid-bold': mixColors(bg, text, 0.15)
    };

    // Apply each derived var ONLY if not already set by the theme
    // (Custom themes can include these keys in their colors object to override)
    for (const [prop, value] of Object.entries(derived)) {
        // Check if the theme explicitly set this property
        // Inline styles from theme application take priority
        if (!root.style.getPropertyValue(prop)) {
            root.style.setProperty(prop, value);
        }
    }
}

/**
 * Clear all derived color overrides so they can be recomputed for a new theme.
 * Called before applying a new theme's colors.
 */
function clearDerivedColors() {
    const derivedKeys = [
        '--bg-fade',
        '--text-subtle-plus', '--text-highlight', '--text-light',
        '--text-superlight', '--text-hyperlight', '--text-reverse',
        '--accent-1-secondary', '--accent-1-tertiary',
        '--accent-2-secondary', '--accent-3-secondary',
        '--accent-4-secondary', '--accent-5-secondary', '--accent-6-secondary',
        '--grid-superlight', '--grid-clear', '--grid-bold'
    ];
    const root = document.documentElement;
    derivedKeys.forEach(key => root.style.removeProperty(key));
}

/**
 * Setup search panel
 */
function setupSearchPanel() {
    const searchInput = document.getElementById('search-input');
    const searchResults = document.getElementById('search-results');
    const searchCancel = document.getElementById('search-cancel');

    // Open search
    document.getElementById('search-btn').addEventListener('click', () => {
        searchPanel.classList.add('visible');
        searchInput.value = '';
        searchInput.focus();
        performSearch('');
    });

    // Cancel button closes search
    searchCancel.addEventListener('click', () => {
        searchPanel.classList.remove('visible');
        focusActiveEditor();
    });

    // Close search on Escape
    searchInput.addEventListener('keydown', async (e) => {
        if (e.key === 'Escape') {
            searchPanel.classList.remove('visible');
            focusActiveEditor();
        } else if (e.key === 'Enter') {
            const selected = searchResults.querySelector('.search-result-item.selected');
            if (selected) {
                await selectSearchResult(selected.dataset.id);
            }
        } else if (e.key === 'ArrowDown' || e.key === 'ArrowUp') {
            e.preventDefault();
            navigateSearchResults(e.key === 'ArrowDown' ? 1 : -1);
        }
    });

    // Search on input
    searchInput.addEventListener('input', (e) => {
        performSearch(e.target.value);
    });
}

/**
 * Setup Find and Replace panel
 */
function setupFindReplace() {
    // Expand/collapse replace row
    findExpandBtn.addEventListener('click', () => {
        findReplacePanel.classList.toggle('expanded');
        if (findReplacePanel.classList.contains('expanded')) {
            replaceInput.focus();
        }
    });

    // Mode change
    findModeSelect.addEventListener('change', () => {
        performFindInNote();
    });

    // Find input events - stop propagation to prevent editor interference
    findInput.addEventListener('input', (e) => {
        e.stopPropagation();
        performFindInNote();
    });

    findInput.addEventListener('keydown', (e) => {
        e.stopPropagation(); // Prevent editor from receiving keystrokes
        if (e.key === 'Escape') {
            closeFindReplace();
        } else if (e.key === 'Enter') {
            e.preventDefault();
            if (e.shiftKey) {
                navigateFindMatch(-1);
            } else {
                navigateFindMatch(1);
            }
        } else if (e.key === 'Tab' && !e.shiftKey) {
            e.preventDefault();
            findReplacePanel.classList.add('expanded');
            replaceInput.focus();
        }
    });

    // Prevent focus loss on click within panel
    findInput.addEventListener('focus', (e) => {
        e.stopPropagation();
    });

    // Replace input events - stop propagation to prevent editor interference
    replaceInput.addEventListener('keydown', (e) => {
        e.stopPropagation();
        if (e.key === 'Escape') {
            closeFindReplace();
        } else if (e.key === 'Enter') {
            e.preventDefault();
            if (e.shiftKey) {
                replaceAllMatches();
            } else {
                replaceCurrentMatch();
            }
        }
    });

    replaceInput.addEventListener('input', (e) => {
        e.stopPropagation();
    });

    replaceInput.addEventListener('focus', (e) => {
        e.stopPropagation();
    });

    // Navigation buttons
    document.getElementById('find-prev-btn').addEventListener('click', () => {
        navigateFindMatch(-1);
    });

    document.getElementById('find-next-btn').addEventListener('click', () => {
        navigateFindMatch(1);
    });

    // Replace buttons
    document.getElementById('replace-btn').addEventListener('click', () => {
        replaceCurrentMatch();
    });

    document.getElementById('replace-all-btn').addEventListener('click', () => {
        replaceAllMatches();
    });

    // Done button
    document.getElementById('find-done-btn').addEventListener('click', () => {
        closeFindReplace();
    });
}

/**
 * Open Find and Replace panel
 */
function openFindReplace() {
    findReplacePanel.classList.add('visible');
    findInput.focus();

    // If text is selected, use it as search term
    const selectedText = editor.value.substring(editor.selectionStart, editor.selectionEnd);
    if (selectedText && selectedText.length < 100 && !selectedText.includes('\n')) {
        findInput.value = selectedText;
    }

    performFindInNote();
}

/**
 * Close Find and Replace panel
 */
function closeFindReplace() {
    findReplacePanel.classList.remove('visible');
    findReplacePanel.classList.remove('expanded');
    findMatches = [];
    currentMatchIndex = -1;
    findInput.value = '';
    replaceInput.value = '';

    // Clear find highlighting and restore normal editor display
    if (editorHighlighter) {
        editorHighlighter.innerHTML = '';
    }
    // Trigger normal editor update
    updateEditorHighlighter();

    focusActiveEditor();
}

/**
 * Clear find highlighting (called when editor content changes)
 */
function clearFindHighlighting() {
    if (findReplacePanel && findReplacePanel.classList.contains('visible')) {
        // Re-run search when content changes
        performFindInNote();
    }
}

/**
 * Perform find in current note
 */
function performFindInNote() {
    const searchTerm = findInput.value;
    const mode = findModeSelect.value;
    const content = editor.value;

    findMatches = [];
    currentMatchIndex = -1;

    if (!searchTerm) {
        updateFindCount();
        return;
    }

    try {
        let regex;
        const flags = 'gi'; // Always case-insensitive

        switch (mode) {
            case 'contains':
                regex = new RegExp(escapeRegex(searchTerm), flags);
                break;
            case 'word':
                regex = new RegExp(`\\b${escapeRegex(searchTerm)}\\b`, flags);
                break;
            case 'starts':
                regex = new RegExp(`^${escapeRegex(searchTerm)}`, flags + 'm');
                break;
            case 'ends':
                regex = new RegExp(`${escapeRegex(searchTerm)}$`, flags + 'm');
                break;
            case 'regex':
                regex = new RegExp(searchTerm, flags);
                break;
            default:
                regex = new RegExp(escapeRegex(searchTerm), flags);
        }

        let match;
        while ((match = regex.exec(content)) !== null) {
            findMatches.push({
                start: match.index,
                end: match.index + match[0].length,
                text: match[0]
            });
            // Prevent infinite loop for zero-length matches
            if (match[0].length === 0) regex.lastIndex++;
        }

        // Set current match to the one closest to cursor
        if (findMatches.length > 0) {
            const cursorPos = editor.selectionStart;
            currentMatchIndex = 0;
            for (let i = 0; i < findMatches.length; i++) {
                if (findMatches[i].start >= cursorPos) {
                    currentMatchIndex = i;
                    break;
                }
            }
            highlightCurrentMatch();
        }
    } catch (e) {
        // Invalid regex
        findMatches = [];
    }

    updateFindCount();
    updateFindHighlighting();
}

/**
 * Escape special regex characters
 */
function escapeRegex(str) {
    return str.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
}

/**
 * Update find count display
 */
function updateFindCount() {
    if (findMatches.length === 0) {
        findCount.textContent = '0 hits';
        findReplacePanel.classList.remove('has-hits');
    } else {
        findCount.textContent = `${currentMatchIndex + 1}/${findMatches.length}`;
        findReplacePanel.classList.add('has-hits');
    }
}

/**
 * Navigate to next/previous match
 */
function navigateFindMatch(direction) {
    if (findMatches.length === 0) return;

    currentMatchIndex += direction;
    if (currentMatchIndex >= findMatches.length) {
        currentMatchIndex = 0;
    } else if (currentMatchIndex < 0) {
        currentMatchIndex = findMatches.length - 1;
    }

    highlightCurrentMatch();
    updateFindCount();
}

/**
 * Highlight and scroll to current match
 */
function highlightCurrentMatch() {
    if (currentMatchIndex < 0 || currentMatchIndex >= findMatches.length) return;

    const match = findMatches[currentMatchIndex];

    // Scroll editor to show match (without stealing focus)
    const lineHeight = parseInt(getComputedStyle(editor).lineHeight) || 20;
    const lines = editor.value.substring(0, match.start).split('\n').length;
    const scrollTop = (lines - 3) * lineHeight;
    editor.scrollTop = Math.max(0, scrollTop);

    // Update visual highlighting in editor
    updateFindHighlighting();
}

/**
 * Update visual highlighting of matches in the editor-highlighter overlay
 */
function updateFindHighlighting() {
    if (!editorHighlighter) return;

    // Never populate highlighter during code mode
    const editorArea = document.querySelector('.editor-area');
    if (editorArea && editorArea.classList.contains('code-mode')) return;

    const content = editor.value;
    if (findMatches.length === 0 || !findReplacePanel.classList.contains('visible')) {
        // No matches or panel closed - show normal content
        return;
    }

    // Build highlighted HTML
    let html = '';
    let lastEnd = 0;

    for (let i = 0; i < findMatches.length; i++) {
        const match = findMatches[i];
        const isCurrent = i === currentMatchIndex;

        // Add text before this match
        if (match.start > lastEnd) {
            html += escapeHtmlForHighlight(content.substring(lastEnd, match.start));
        }

        // Add highlighted match
        const className = isCurrent ? 'find-highlight current' : 'find-highlight';
        html += `<span class="${className}">${escapeHtmlForHighlight(match.text)}</span>`;

        lastEnd = match.end;
    }

    // Add remaining text
    if (lastEnd < content.length) {
        html += escapeHtmlForHighlight(content.substring(lastEnd));
    }

    editorHighlighter.innerHTML = html;
    editorHighlighter.classList.remove('hidden');
}

/**
 * Replace current match
 */
function replaceCurrentMatch() {
    if (currentMatchIndex < 0 || currentMatchIndex >= findMatches.length) return;

    const match = findMatches[currentMatchIndex];
    const replaceText = replaceInput.value;

    // Temporarily focus editor for replacement
    editor.focus();
    editor.setSelectionRange(match.start, match.end);

    // Use execCommand for undo support
    document.execCommand('insertText', false, replaceText);

    // Close find panel after replacing (same as Done button behavior)
    closeFindReplace();
}

/**
 * Replace all matches
 */
function replaceAllMatches() {
    if (findMatches.length === 0) return;

    const replaceText = replaceInput.value;
    const content = editor.value;

    // Replace from end to start to preserve indices
    let newContent = content;
    for (let i = findMatches.length - 1; i >= 0; i--) {
        const match = findMatches[i];
        newContent = newContent.substring(0, match.start) + replaceText + newContent.substring(match.end);
    }

    // Set content and trigger save
    editor.value = newContent;
    editor.dispatchEvent(new Event('input', { bubbles: true }));

    // Close find panel after replacing all (same as Done button behavior)
    closeFindReplace();
}

/**
 * Extract keyword from note content
 */
function extractKeyword(content) {
    const firstLine = content.split('\n')[0].trim().toLowerCase();
    const keywords = ['list', 'math', 'sum', 'avg', 'count', 'code', 'timer'];

    for (const kw of keywords) {
        if (firstLine === kw || firstLine.startsWith(kw + ':') || firstLine.startsWith(kw + ' ')) {
            return kw;
        }
    }
    return null;
}

/**
 * Format note preview with pipe separators
 */
function formatPreview(content, keyword) {
    // Remove keyword line if present
    let lines = content.split('\n');
    if (keyword && lines.length > 0) {
        lines = lines.slice(1);
    }

    // Filter empty lines and join with pipes
    const nonEmpty = lines
        .map(l => l.trim())
        .filter(l => l.length > 0)
        .slice(0, 5);

    // Truncate each line and join with |
    const preview = nonEmpty
        .map(l => l.length > 40 ? l.substring(0, 40) + '...' : l)
        .join('  |  ');

    return preview || '[Empty Note]';
}

/**
 * Format relative time (e.g., "Edited 3 hours ago")
 */
function formatRelativeTime(dateString) {
    const date = new Date(dateString);
    const now = new Date();
    const diffMs = now - date;
    const diffMins = Math.floor(diffMs / 60000);
    const diffHours = Math.floor(diffMs / 3600000);
    const diffDays = Math.floor(diffMs / 86400000);

    if (diffMins < 1) return 'Edited just now';
    if (diffMins < 60) return `Edited ${diffMins} minute${diffMins === 1 ? '' : 's'} ago`;
    if (diffHours < 24) return `Edited ${diffHours} hour${diffHours === 1 ? '' : 's'} ago`;
    if (diffDays < 7) return `Edited ${diffDays} day${diffDays === 1 ? '' : 's'} ago`;
    return null; // Don't show relative time for older notes
}

/**
 * Format date as "Month Day" or "Month Day, Year"
 */
function formatNoteDate(dateString) {
    const date = new Date(dateString);
    const now = new Date();
    const thisYear = now.getFullYear() === date.getFullYear();

    if (thisYear) {
        return date.toLocaleDateString('en-US', {
            month: 'long',
            day: 'numeric'
        });
    }
    return date.toLocaleDateString('en-US', {
        year: 'numeric',
        month: 'long',
        day: 'numeric'
    });
}

/**
 * Collect all tags across all notes
 * Returns Map<tag, count>
 */
function collectAllTags(notes) {
    const tagCounts = new Map();
    notes.forEach(note => {
        const tags = extractTags(note.content);
        tags.forEach(tag => {
            tagCounts.set(tag, (tagCounts.get(tag) || 0) + 1);
        });
    });
    return tagCounts;
}

/**
 * Render tag cloud pills into container
 */
function renderTagCloud(container, tagCounts) {
    if (tagCounts.size === 0) {
        container.style.display = 'none';
        return;
    }

    // Sort by count descending, then alphabetically
    const sorted = [...tagCounts.entries()].sort((a, b) => b[1] - a[1] || a[0].localeCompare(b[0]));

    container.innerHTML = sorted.map(([tag, count]) => {
        const accIndex = getTagAccentIndex(tag);
        return `<span class="search-tag-pill search-tag-pill-acc${accIndex}" data-tag="${escapeHtml(tag)}">${escapeHtml(tag)}<span class="search-tag-count">${count}</span></span>`;
    }).join('');

    container.style.display = 'flex';
}

/**
 * Highlight tags in search result preview text
 */
function highlightTagsInPreview(escapedPreview, tagQuery) {
    const tag = tagQuery.toLowerCase();
    const accIndex = getTagAccentIndex(tag);
    // Escape the tag for regex use
    const escapedTag = tag.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
    return escapedPreview.replace(
        new RegExp(`(${escapedTag})`, 'gi'),
        `<span class="search-tag-match-acc${accIndex}">$1</span>`
    );
}

/**
 * Perform search
 */
async function performSearch(query) {
    const notes = await getNotes();
    const searchResults = document.getElementById('search-results');
    const tagCloud = document.getElementById('search-tag-cloud');

    const trimmed = query.trim();
    const isTagQuery = trimmed.startsWith('#');

    // Show tag cloud when query is empty or starts with # (if enabled)
    if (settings.showTagCloud !== false && (!trimmed || isTagQuery)) {
        const tagCounts = collectAllTags(notes);
        renderTagCloud(tagCloud, tagCounts);
    } else {
        tagCloud.style.display = 'none';
    }

    let filtered;
    if (!trimmed) {
        filtered = notes;
    } else if (isTagQuery && trimmed === '#') {
        // Show all notes that have any tag
        filtered = notes.filter(n => extractTags(n.content).length > 0);
    } else if (isTagQuery) {
        // Filter notes containing the specific tag
        const searchTag = trimmed.toLowerCase();
        filtered = notes.filter(n => extractTags(n.content).includes(searchTag));
    } else {
        // Normal text search
        filtered = notes.filter(n => n.content.toLowerCase().includes(trimmed.toLowerCase()));
    }

    searchResults.innerHTML = filtered.map((note, index) => {
        const { keyword, mode, title, hasKeyword } = parseKeyword(note.content);
        const preview = formatPreview(note.content, hasKeyword ? keyword : null);
        const relativeTime = formatRelativeTime(note.modifiedAt);
        const date = formatNoteDate(note.modifiedAt);

        let previewHtml = escapeHtml(preview);
        // Highlight matching text in preview
        if (isTagQuery && trimmed.length > 1) {
            previewHtml = highlightTagsInPreview(previewHtml, trimmed);
        } else if (!isTagQuery && trimmed.length > 0) {
            const escapedQuery = trimmed.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
            previewHtml = previewHtml.replace(
                new RegExp(`(${escapedQuery})`, 'gi'),
                '<span class="search-match-highlight">$1</span>'
            );
        }

        // Build keyword header line with accent color
        let keywordHtml = '';
        if (hasKeyword && keyword) {
            const accentClass = getKeywordAccentClass(mode);
            const firstLine = note.content.split('\n')[0];
            const restOfLine = firstLine.substring(keyword.length);
            keywordHtml = `<div class="search-result-keyword"><span class="${accentClass}">${escapeHtml(keyword)}</span>${restOfLine ? escapeHtml(restOfLine) : ''}</div>`;
        }

        return `
      <div class="search-result-item ${index === 0 ? 'selected' : ''}" data-id="${note.id}">
        ${keywordHtml}
        <div class="search-result-preview">${previewHtml}</div>
        ${relativeTime ? `<div class="search-result-relative">${relativeTime}</div>` : ''}
        <div class="search-result-date">${date}</div>
      </div>
    `;
    }).join('');

    // Add click handlers
    searchResults.querySelectorAll('.search-result-item').forEach(item => {
        item.addEventListener('click', () => selectSearchResult(item.dataset.id));
    });
}

/**
 * Navigate search results
 */
function navigateSearchResults(direction) {
    const searchResults = document.getElementById('search-results');
    const items = searchResults.querySelectorAll('.search-result-item');
    const currentIndex = Array.from(items).findIndex(i => i.classList.contains('selected'));

    items[currentIndex]?.classList.remove('selected');

    let newIndex = currentIndex + direction;
    if (newIndex < 0) newIndex = items.length - 1;
    if (newIndex >= items.length) newIndex = 0;

    items[newIndex]?.classList.add('selected');
}

/**
 * Select search result and promote to front
 */
async function selectSearchResult(noteId) {
    const note = await promoteNote(noteId);
    if (note) {
        await setCurrentNoteId(note.id);
        await loadNote(note);
    }

    document.getElementById('search-panel').classList.remove('visible');
    focusActiveEditor();
}

/**
 * Setup keyboard shortcuts
 */
function setupKeyboardShortcuts() {
    document.addEventListener('keydown', async (e) => {
        // Previous note
        if (matchesShortcut(e, 'previousNote')) {
            e.preventDefault();
            await forceSave(editor);
            const note = await goToPrevious();
            if (note) {
                await loadNote(note);
            }
        }

        // Next note
        if (matchesShortcut(e, 'nextNote')) {
            e.preventDefault();
            await forceSave(editor);
            const content = getCurrentContent();
            const note = await goToNext(content);
            if (note) {
                await loadNote(note);
            }
        }

        // Move note to front
        if (matchesShortcut(e, 'moveToFront')) {
            e.preventDefault();
            const currentNote = getCurrentNote();
            if (currentNote) {
                await forceSave(editor);
                const note = await promoteNote(currentNote.id);
                if (note) {
                    await loadNote(note);
                }
            }
        }

        // Delete note
        if (matchesShortcut(e, 'deleteNote')) {
            e.preventDefault();
            const currentNote = getCurrentNote();
            if (currentNote) {
                const note = await deleteCurrentNote(currentNote.id);
                if (note) {
                    await loadNote(note);
                }
            }
        }

        // Search
        if (matchesShortcut(e, 'search')) {
            e.preventDefault();
            searchPanel.classList.add('visible');
            document.getElementById('search-input').focus();
            performSearch('');
        }

        // Find and Replace
        if (matchesShortcut(e, 'findReplace')) {
            e.preventDefault();
            openFindReplace();
        }

        // New note
        if (matchesShortcut(e, 'newNote')) {
            e.preventDefault();
            await forceSave(editor);
            const note = await createNewNote();
            if (note) {
                await loadNote(note);
            }
        }

        // Pin note to front (internal, not in settings UI)
        if (matchesShortcut(e, 'pinNote')) {
            e.preventDefault();
            const currentNote = getCurrentNote();
            if (currentNote) {
                await forceSave(editor);
                const note = await promoteNote(currentNote.id);
                if (note) {
                    await loadNote(note);
                }
            }
        }

        // Jump to front note
        if (matchesShortcut(e, 'jumpToFront')) {
            e.preventDefault();
            await forceSave(editor);
            const note = await jumpToFront();
            if (note) {
                await loadNote(note);
            }
        }

        // Hide/Show main window
        if (matchesShortcut(e, 'hideShow')) {
            e.preventDefault();
            window.close();
        }

        // Quick export
        if (matchesShortcut(e, 'quickExport')) {
            e.preventDefault();
            exportNote();
        }

        // Insert date at cursor
        if (matchesShortcut(e, 'insertDate')) {
            e.preventDefault();
            const dateStr = getFormattedToday(settings.dateFormat || 'MMM D, YYYY');
            const editorArea = document.querySelector('.editor-area');
            if (editorArea && editorArea.classList.contains('code-mode') && codeEditable) {
                // Insert into contenteditable code editor
                const sel = window.getSelection();
                if (sel.rangeCount) {
                    const range = sel.getRangeAt(0);
                    range.deleteContents();
                    range.insertNode(document.createTextNode(dateStr));
                    range.collapse(false);
                    codeEditable.dispatchEvent(new Event('input', { bubbles: true }));
                }
            } else if (editor) {
                // Insert into textarea
                const start = editor.selectionStart;
                const end = editor.selectionEnd;
                editor.value = editor.value.substring(0, start) + dateStr + editor.value.substring(end);
                editor.selectionStart = editor.selectionEnd = start + dateStr.length;
                editor.dispatchEvent(new Event('input', { bubbles: true }));
            }
        }

        // Increase text size
        if (matchesShortcut(e, 'increaseText')) {
            e.preventDefault();
            const currentSize = parseInt(settings.textSize) || 14;
            if (currentSize < 24) {
                settings.textSize = currentSize + 1;
                await saveSettings(settings);
                applySettings();
            }
        }

        // Decrease text size
        if (matchesShortcut(e, 'decreaseText')) {
            e.preventDefault();
            const currentSize = parseInt(settings.textSize) || 14;
            if (currentSize > 10) {
                settings.textSize = currentSize - 1;
                await saveSettings(settings);
                applySettings();
            }
        }

        // Escape - Close panels
        if (e.key === 'Escape') {
            // Resize popup back if settings was open
            if (settingsPanel.classList.contains('visible')) {
                document.body.style.width = '400px';
                document.body.style.height = '500px';
            }
            settingsPanel.classList.remove('visible');
            searchPanel.classList.remove('visible');
            slashPopup.classList.remove('visible');
            // Close find/replace if open (closeFindReplace handles its own focus)
            if (findReplacePanel.classList.contains('visible')) {
                closeFindReplace();
                return; // closeFindReplace already focuses active editor
            }
            focusActiveEditor();
        }

        // Number keys for slash command selection (1-9, 0 for 10th item)
        if (slashPopup.classList.contains('visible') && (e.key >= '0' && e.key <= '9')) {
            e.preventDefault();
            const keywords = getSlashKeywords();
            // 1-9 map to indices 0-8, 0 maps to index 9
            const index = e.key === '0' ? 9 : parseInt(e.key) - 1;
            if (keywords[index]) {
                insertKeyword(editor, keywords[index].key);
                slashPopup.classList.remove('visible');
            }
        }

        // Letter keys for slash command selection (a=11th, b=12th, etc.)
        if (slashPopup.classList.contains('visible') && e.key.length === 1 && e.key >= 'a' && e.key <= 'z' && !e.metaKey && !e.ctrlKey && !e.altKey) {
            const letterIndex = 10 + (e.key.charCodeAt(0) - 'a'.charCodeAt(0)); // a=10, b=11, ...
            const keywords = getSlashKeywords();
            if (letterIndex < keywords.length) {
                e.preventDefault();
                insertKeyword(editor, keywords[letterIndex].key);
                slashPopup.classList.remove('visible');
            }
        }

        // Any other key while slash popup is visible (except modifiers) — hide it
        if (slashPopup.classList.contains('visible') &&
            e.key !== 'Escape' && e.key !== 'Shift' && e.key !== 'Control' &&
            e.key !== 'Alt' && e.key !== 'Meta' &&
            !(e.key >= '0' && e.key <= '9') &&
            !(e.key >= 'a' && e.key <= 'z')) {
            slashPopup.classList.remove('visible');
        }
    });

    // Also hide slash popup when cursor moves away
    editor.addEventListener('mousedown', () => {
        if (slashPopup.classList.contains('visible')) {
            slashPopup.classList.remove('visible');
        }
    });
}

/**
 * Setup slash commands
 */
function setupSlashCommands() {
    slashPopup.querySelectorAll('.slash-item').forEach(item => {
        item.addEventListener('click', () => {
            insertKeyword(editor, item.dataset.keyword);
            slashPopup.classList.remove('visible');
            editor.focus();
        });
    });
}

/**
 * Check if slash command should show
 */
function checkSlashCommand() {
    const content = editor.value;
    const cursorPos = editor.selectionStart;

    if (shouldShowSlashCommand(content, cursorPos)) {
        slashPopup.classList.add('visible');
    } else {
        slashPopup.classList.remove('visible');
    }
}

/**
 * Update the keyword indicator
 */
function updateKeywordIndicator() {
    const content = editor.value;
    const { mode, hasKeyword } = parseKeyword(content);

    // Clear Pro upgrade hint when mode changes
    hideProUpgradeHint();

    // Check if in template creation mode
    if (isCreatingTemplate) {
        keywordIndicator.textContent = 'TEMPLATE';
        keywordIndicator.classList.add('visible', 'template-create');
        currentMode = 'template';
        return;
    }

    // Check if in placeholders creation mode
    if (isCreatingPlaceholders) {
        keywordIndicator.textContent = 'PLACEHOLDER';
        keywordIndicator.classList.add('visible', 'template-create');
        currentMode = 'placeholder';
        return;
    }

    // Check if in sort mode
    if (isSortingMode) {
        keywordIndicator.textContent = 'SORT';
        keywordIndicator.classList.add('visible', 'template-create');
        currentMode = 'sort';
        return;
    }

    // Check if in bookmarks add or edit mode
    if (isAddingBookmark) {
        keywordIndicator.textContent = 'BOOKMARKS (⌘S to save)';
        keywordIndicator.classList.add('visible', 'template-create');
        currentMode = 'bookmarks';
        return;
    }

    if (isEditingBookmark) {
        keywordIndicator.textContent = 'EDIT BOOKMARK (⌘S to save)';
        keywordIndicator.classList.add('visible', 'template-create');
        currentMode = 'bookmarks';
        return;
    }

    if (hasKeyword && mode) {
        keywordIndicator.textContent = mode.toUpperCase();
        keywordIndicator.classList.add('visible');
        keywordIndicator.classList.remove('template-create');
        currentMode = mode;
    } else {
        keywordIndicator.classList.remove('visible', 'template-create');
        currentMode = null;
    }
}

/**
 * Update the note date display
 */
async function updateNoteDate(note) {
    let dateText = '';

    if (note && note.modifiedAt) {
        const date = new Date(note.modifiedAt);
        dateText = date.toLocaleDateString('en-US', {
            month: 'short',
            day: 'numeric',
            year: 'numeric'
        });
    } else {
        dateText = 'New Note';
    }

    // Add note count if enabled
    if (settings.showNoteCount) {
        const allNotes = await getNotesByCreation();
        const currentId = await getCurrentNoteId();
        const currentIndex = allNotes.findIndex(n => n.id === currentId);

        if (currentIndex >= 0 && allNotes.length > 0) {
            dateText += ` (${currentIndex + 1} of ${allNotes.length})`;
        }
    }

    noteDate.textContent = dateText;
}

/**
 * Update note count display (called when setting changes)
 */
async function updateNoteCount() {
    const currentNote = getCurrentNote();
    await updateNoteDate(currentNote);
}

/**
 * Update navigation button states
 */
async function updateNavigationState() {
    const info = await getNavigationInfo();

    document.getElementById('prev-btn').disabled = info.isLast;
    document.getElementById('jump-btn').disabled = info.isFirst;
}

/**
 * Strip the keyword line from note content
 */
/**
 * Build copy text for math mode — pairs each expression with its result
 */
function buildMathCopyText(content, results) {
    const lines = content.split('\n');
    let plainLines = [];
    let htmlLines = [];

    // Skip first line (keyword)
    for (const item of results) {
        if (item.type === 'empty') {
            plainLines.push('');
            htmlLines.push('<br>');
        } else if (item.type === 'calculation') {
            // "5 + 3 = 8"
            const expr = item.expression.endsWith('=') ? item.expression.slice(0, -1).trim() : item.expression;
            plainLines.push(`${expr} = ${item.formatted}`);
            htmlLines.push(`<p style="margin:0 0 4px 0;">${escapeHtml(expr)} = <strong>${escapeHtml(item.formatted)}</strong></p>`);
        } else if (item.type === 'variable') {
            plainLines.push(`${item.name} : ${item.formatted}`);
            htmlLines.push(`<p style="margin:0 0 4px 0;">${escapeHtml(item.name)} : <strong>${escapeHtml(item.formatted)}</strong></p>`);
        } else if (item.type === 'comment') {
            plainLines.push('// ' + item.text);
            htmlLines.push(`<p style="margin:0 0 4px 0;color:#888;font-style:italic;">${escapeHtml('// ' + item.text)}</p>`);
        } else if (item.type === 'text') {
            plainLines.push(item.text);
            htmlLines.push(`<p style="margin:0 0 4px 0;">${escapeHtml(item.text)}</p>`);
        }
    }

    return {
        plain: plainLines.join('\n'),
        html: htmlLines.join('')
    };
}

function stripKeywordLine(content) {
    const { hasKeyword } = parseKeyword(content);
    if (!hasKeyword) return content;
    const lines = content.split('\n');
    lines.shift();
    return lines.join('\n');
}

/**
 * Convert note content to formatted HTML for clipboard
 * Strips keyword line, converts markdown to HTML
 */
function noteContentToHtml(content) {
    // Strip keyword line
    const text = stripKeywordLine(content);
    const lines = text.split('\n');
    let html = '';
    let inList = false;
    let inSubList = false;

    // Check if the note is in list mode (for checkbox conversion)
    const { mode } = parseKeyword(content);
    const isListMode = mode === 'list';
    const checkTrigger = (settings.checkTrigger || '/x').toLowerCase();

    for (let i = 0; i < lines.length; i++) {
        let line = lines[i];

        // Skip comment lines
        if (line.trim().startsWith('//')) continue;

        // Handle headings
        if (line.trim().startsWith('### ')) {
            if (inList) { html += '</ul>'; inList = false; }
            html += `<h3>${escapeHtml(line.trim().substring(4))}</h3>`;
            continue;
        }
        if (line.trim().startsWith('## ')) {
            if (inList) { html += '</ul>'; inList = false; }
            html += `<h2>${escapeHtml(line.trim().substring(3))}</h2>`;
            continue;
        }
        if (line.trim().startsWith('# ')) {
            if (inList) { html += '</ul>'; inList = false; }
            html += `<h1>${escapeHtml(line.trim().substring(2))}</h1>`;
            continue;
        }

        // Handle list mode lines as checkboxes
        if (isListMode && line.trim().length > 0) {
            const leadingSpaces = line.match(/^(\s*)/)[1].length;
            const isSub = leadingSpaces >= 2;
            const trimmed = line.trim();

            // Check for [x] marker first (from checkbox click), then check trigger
            const xMatch = trimmed.match(/^\[x\]\s*(.*)/i);
            const isChecked = xMatch || trimmed.toLowerCase().endsWith(checkTrigger);
            let itemText;
            if (xMatch) {
                itemText = xMatch[1];
            } else if (isChecked) {
                itemText = trimmed.substring(0, trimmed.length - checkTrigger.length).trim();
            } else {
                itemText = trimmed;
            }

            if (isSub) {
                // Sub-item: open nested list if not already in one
                if (!inSubList) {
                    html += '<ul style="list-style:none;padding-left:20px;">';
                    inSubList = true;
                }
            } else {
                // Parent item: close any open sub-list first
                if (inSubList) { html += '</ul>'; inSubList = false; }
                if (!inList) { html += '<ul style="list-style:none;padding-left:0;">'; inList = true; }
            }

            const checkbox = isChecked ? '&#9745; ' : '&#9744; ';
            const style = isChecked ? ' style="text-decoration:line-through;opacity:0.6;"' : '';
            html += `<li${style}>${checkbox}${applyInlineFormatting(escapeHtml(itemText))}</li>`;
            continue;
        }

        // Empty line closes list
        if (line.trim() === '') {
            if (inSubList) { html += '</ul>'; inSubList = false; }
            if (inList) { html += '</ul>'; inList = false; }
            html += '<br>';
            continue;
        }

        // Regular paragraph line
        if (inSubList) { html += '</ul>'; inSubList = false; }
        if (inList) { html += '</ul>'; inList = false; }
        html += `<p style="margin:0 0 4px 0;">${applyInlineFormatting(escapeHtml(line))}</p>`;
    }

    if (inSubList) html += '</ul>';
    if (inList) html += '</ul>';
    return html;
}

/**
 * Apply inline markdown formatting (bold, italic, underline, strikethrough)
 */
function applyInlineFormatting(escapedHtml) {
    return escapedHtml
        .replace(/\*\*(.+?)\*\*/g, '<strong>$1</strong>')
        .replace(/__(.+?)__/g, '<u>$1</u>')
        .replace(/_(.+?)_/g, '<em>$1</em>')
        .replace(/~(.+?)~/g, '<s>$1</s>');
}

/**
 * Export note as .txt file
 */
async function exportNote() {
    const currentNote = getCurrentNote();
    if (!currentNote) return;

    // Get content and process according to settings
    let content = currentNote.content || '';

    // Omit keyword if setting is enabled
    if (settings.exportOmitKeyword) {
        const lines = content.split('\n');
        const { hasKeyword } = parseKeyword(content);
        if (hasKeyword && lines.length > 0) {
            lines.shift(); // Remove first line (keyword line)
            content = lines.join('\n');
        }
    }

    // Generate filename
    const filename = generateExportFilename(currentNote);

    // Download as .txt file
    downloadTextFile(content, filename);
}

/**
 * Generate filename for exported note
 * @param {Object} note - Note object
 * @returns {string} Filename
 */
function generateExportFilename(note) {
    if (!note) return 'JottNote.txt';

    const content = note.content || '';

    // If "Use first line as title" setting is enabled
    if (settings.exportUseFirstLine && content.trim()) {
        const lines = content.split('\n');
        let firstLine = lines[0] || '';

        // Remove keyword if present
        const { hasKeyword } = parseKeyword(content);
        if (hasKeyword) {
            firstLine = lines[1] || firstLine;
        }

        firstLine = firstLine.trim();

        if (firstLine) {
            // Get first 2 words, or first word if only one word
            const words = firstLine.split(/\s+/);
            let filename = words.slice(0, 2).join(' ');

            // If only got 1 word and it's short, that's fine
            if (!filename) {
                filename = words[0] || 'JottNote';
            }

            // Sanitize filename (remove invalid characters)
            filename = filename.replace(/[<>:"/\\|?*]/g, '-');
            return `${filename}.txt`;
        }
    }

    // Default: JottNote - [Last Modified]
    const date = new Date(note.modifiedAt || note.createdAt);
    const dateStr = date.toLocaleDateString('en-US', {
        month: 'short',
        day: 'numeric',
        year: 'numeric'
    }).replace(/,/g, '');

    return `JottNote - ${dateStr}.txt`;
}

/**
 * Download text as file
 * @param {string} content - File content
 * @param {string} filename - Filename
 */
function downloadTextFile(content, filename) {
    const blob = new Blob([content], { type: 'text/plain' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = filename;
    a.click();
    URL.revokeObjectURL(url);
}

/**
 * Export all notes as .zip file
 */
async function exportAllNotes() {
    // Check if JSZip is available
    if (typeof JSZip === 'undefined') {
        alert('Export library not loaded. Please reload the extension.');
        return;
    }

    const allNotes = await getNotes();

    if (allNotes.length === 0) {
        alert('No notes to export.');
        return;
    }

    const zip = new JSZip();
    const usedFilenames = new Set();

    // Add each note to the zip
    for (const note of allNotes) {
        let content = note.content || '';

        // Omit keyword if setting is enabled
        if (settings.exportOmitKeyword) {
            const lines = content.split('\n');
            const { hasKeyword } = parseKeyword(content);
            if (hasKeyword && lines.length > 0) {
                lines.shift(); // Remove first line (keyword line)
                content = lines.join('\n');
            }
        }

        // Generate filename
        let filename = generateExportFilename(note);

        // Handle duplicate filenames
        let baseName = filename.replace(/\.txt$/, '');
        let extension = '.txt';
        let counter = 1;
        while (usedFilenames.has(filename)) {
            filename = `${baseName} (${counter})${extension}`;
            counter++;
        }
        usedFilenames.add(filename);

        // Add file to zip
        zip.file(filename, content);
    }

    // Generate zip file
    try {
        const zipBlob = await zip.generateAsync({ type: 'blob' });

        // Download zip
        const url = URL.createObjectURL(zipBlob);
        const a = document.createElement('a');
        a.href = url;

        // Generate zip filename with current date
        const date = new Date();
        const dateStr = date.toLocaleDateString('en-US', {
            month: 'short',
            day: 'numeric',
            year: 'numeric'
        }).replace(/,/g, '');
        a.download = `JottNote Export - ${dateStr}.zip`;

        a.click();
        URL.revokeObjectURL(url);
    } catch (error) {
        console.error('Export failed:', error);
        alert('Failed to export notes. Please try again.');
    }
}

/**
 * Apply settings to the UI
 */
function applySettings() {
    // Theme Logic
    // Default theme: 'sanrio' for free, 'gundam' for Pro (both light & dark)
    const defaultTheme = isPro() ? 'gundam' : 'sanrio';
    const lightThemeKey = settings.lightTheme || defaultTheme;
    const darkThemeKey = settings.darkTheme || defaultTheme;

    // Determine active mode based on system preference (for now, could be time-based if requested)
    // User asked for "dependant on time of day". System preference usually proxies this well.
    // Let's check system preference:
    const isDarkMode = window.matchMedia && window.matchMedia('(prefers-color-scheme: dark)').matches;

    // Select theme data
    const themeMode = isDarkMode ? 'dark' : 'light';
    const activeThemeKey = isDarkMode ? darkThemeKey : lightThemeKey;

    // Load theme - check for custom theme key
    const applyThemeColors = (td) => {
        if (td && td.colors) {
            const root = document.documentElement;
            clearDerivedColors(); // Reset derived vars before applying new theme
            Object.entries(td.colors).forEach(([property, value]) => {
                root.style.setProperty(property, value);
            });
        }
    };

    if (activeThemeKey === 'custom') {
        // Load custom theme from storage
        chrome.storage.local.get(CUSTOM_THEME_KEY, (result) => {
            const customTheme = result[CUSTOM_THEME_KEY];
            applyThemeColors(customTheme || THEMES['sanrio']);
        });
    } else {
        const themeData = THEMES[activeThemeKey] || THEMES['sanrio'];
        applyThemeColors(themeData);
    }

    // Set dataset for other CSS hooks if needed
    document.body.dataset.theme = themeMode;

    // Text size
    document.body.classList.remove('text-xs', 'text-s', 'text-m', 'text-l', 'text-xl');
    document.body.classList.add(`text-${settings.textSize || 'm'}`);

    // Paper Settings
    const editorArea = document.querySelector('.editor-area');
    if (editorArea) {
        // Remove existing paper classes
        editorArea.classList.remove('paper-blank', 'paper-lines', 'paper-dots', 'paper-grid-sm', 'paper-grid-lg');

        // Add selected paper class (default: blank)
        const paperType = settings.paperType || 'blank';
        editorArea.classList.add(`paper-${paperType}`);

        const opacity = settings.paperOpacity || '0.1';
        editorArea.style.setProperty('--paper-opacity', opacity);
    }

    // Update keywords config
    updateKeywordsConfig(settings);

    // Compute derived color variations (secondaries, text variants, grid)
    computeDerivedColors();

    // Apply syntax highlighting theme (must come AFTER derived colors)
    applySyntaxTheme();
}

/**
 * Update code mode display
 * Uses Prism.js for syntax highlighting with overlapping textarea
 */

/**
 * Restore textarea and highlighter visibility after leaving code mode
 * Removes inline display:none that was set when entering code mode
 */
function restoreNonCodeEditors() {
    if (editor) editor.style.display = '';
    if (editorHighlighter) editorHighlighter.style.display = '';
}

function updateCodeMode() {
    const content = editor.value;
    const firstLine = content.split('\n')[0];
    let codeResult = parseCodeKeyword(firstLine);
    const editorArea = document.querySelector('.editor-area');

    // Pro gate: if code mode detected but not Pro, show upgrade hint
    if (codeResult && !isPro()) {
        showProUpgradeHint('code');
        return;
    }

    if (!codeResult) {
        // Not in code mode - exit code mode if we were in it
        codeHelp.classList.remove('visible');
        if (codeLanguagesHelp) codeLanguagesHelp.classList.remove('visible');
        if (editorArea.classList.contains('code-mode')) {
            const cursorPos = getCodeCaretOffset();
            editor.value = getCodeText();
            editorArea.classList.remove('code-mode');
            restoreNonCodeEditors();
            codeEditable.textContent = '';

            editor.focus();
            editor.selectionStart = editor.selectionEnd = cursorPos;
            if (typeof updateEditorHighlighter === 'function') updateEditorHighlighter();
        }
        return;
    }

    if (codeResult.showLanguages) {
        codeHelp.classList.remove('visible');
        if (codeLanguagesHelp) {
            populateCodeLanguages();
            codeLanguagesHelp.classList.add('visible');
        }

        if (editorArea.classList.contains('code-mode')) {
            editor.value = getCodeText();
            editorArea.classList.remove('code-mode');
            restoreNonCodeEditors();
            codeEditable.textContent = '';
            editor.focus();
            if (typeof updateEditorHighlighter === 'function') updateEditorHighlighter();
        }
        return;
    }

    if (codeLanguagesHelp) codeLanguagesHelp.classList.remove('visible');

    if (codeResult.showHelp) {
        const hasMoreContent = content.split('\n').length > 1;

        if (hasMoreContent && settings.defaultCodeLanguage) {
            // Has content beyond keyword line — use default language
            codeHelp.classList.remove('visible');
            codeResult = { showHelp: false, language: settings.defaultCodeLanguage, displayName: settings.defaultCodeLanguage };
        } else {
            if (!hasMoreContent) {
                codeHelp.classList.add('visible');
            } else {
                codeHelp.classList.remove('visible');
            }

            if (editorArea.classList.contains('code-mode')) {
                editor.value = getCodeText();
                editorArea.classList.remove('code-mode');
                restoreNonCodeEditors();
                codeEditable.textContent = '';
                editor.focus();
                if (typeof updateEditorHighlighter === 'function') updateEditorHighlighter();
            }
        }
    }

    if (codeResult.language) {
        // Enter code mode with syntax highlighting
        codeHelp.classList.remove('visible');

        if (!editorArea.classList.contains('code-mode')) {
            // Entering code mode
            editorArea.classList.add('code-mode');

            // Force-hide textarea and highlighter with inline styles (nuclear option)
            // CSS class-based hiding can be overridden; inline styles cannot
            if (editorHighlighter) {
                editorHighlighter.innerHTML = '';
                editorHighlighter.style.display = 'none';
            }
            if (editor) {
                editor.style.display = 'none';
            }

            // Set content and apply highlighting
            highlightCodeContent(codeResult.language, content);

            codeEditable.scrollTop = 0;
            // Defer focus — contenteditable needs a paint frame after display:none → block
            requestAnimationFrame(() => {
                codeEditable.focus();
                setCodeCaretToEnd();
            });
        } else {
            // Already in code mode, update highlighting preserving cursor
            const saved = saveCodeCaret();
            highlightCodeContent(codeResult.language, getCodeText());
            restoreCodeCaret(saved);
        }
    }
}

/**
 * Populate the code languages reference list
 */
function populateCodeLanguages() {
    const container = document.getElementById('code-languages-list');
    if (!container || container.children.length > 0) return; // Already populated

    const languages = getSupportedLanguages();
    languages.forEach(({ name, aliases }) => {
        const item = document.createElement('div');
        item.className = 'code-lang-item';
        item.innerHTML = `<span class="code-lang-name">${name}</span><span class="code-lang-aliases">${aliases.join(', ')}</span>`;
        container.appendChild(item);
    });
}

/**
 * Update timer help display
 * Shows help when FIRST line is exactly "timer" and no other content
 */
function updateTimerHelp() {
    const content = editor.value;
    const hasMoreContent = content.split('\n').length > 1;
    const { mode } = parseKeyword(content);

    // Show help only if in timer mode and no other content
    if (mode === 'timer' && !hasMoreContent) {
        if (!isPro()) { showProUpgradeHint('timer'); return; }
        timerHelp.classList.add('visible');
    } else {
        timerHelp.classList.remove('visible');
    }
}



/**
 * Update currency help display
 * Shows help when FIRST line is exactly "currency" and no other content
 */
function updateCurrencyHelp() {
    if (!currencyHelp) return;

    const content = editor.value;
    const hasMoreContent = content.split('\n').length > 1;
    const { mode } = parseKeyword(content);

    // Currency is mapped to math mode with "currency" keyword
    if (mode === 'currency' && !hasMoreContent) {
        currencyHelp.classList.add('visible');
    } else {
        currencyHelp.classList.remove('visible');
    }
}

/**
 * Update template help display
 * Shows help when FIRST line is exactly "template" and no other content
 */
async function updateTemplateHelp() {
    if (!templateHelp) return;

    const content = editor.value;
    const lines = content.split('\n');
    const firstLineTrimmed = lines[0].trim().toLowerCase();
    const hasMoreContent = lines.length > 1 && lines.slice(1).some(line => line.trim() !== '');

    // Only show help when first line is exactly "template" with no colon/suffix and no other content
    if (firstLineTrimmed === 'template' && !hasMoreContent) {
        if (!isPro()) { showProUpgradeHint('template'); return; }
        // Populate template list
        await populateTemplateList();
        templateHelp.classList.add('visible');
    } else {
        templateHelp.classList.remove('visible');
    }
}

/**
 * Populate template list in help guide
 */
async function populateTemplateList() {
    if (!templateList) return;

    const templates = await getTemplates();

    if (templates.length === 0) {
        templateList.innerHTML = '<div class="template-list-empty">No templates saved yet.</div>';
    } else {
        const html = templates.map(t =>
            `<div class="template-list-item">template:${t.name}</div>`
        ).join('');
        templateList.innerHTML = html;
    }
}

/**
 * Update sort help display
 * Shows help when FIRST line is exactly "sort" and no other content
 */
function updateSortHelp() {
    if (!sortHelp) return;

    const content = editor.value;
    const showHelp = shouldShowSortHelp(content);

    if (showHelp) {
        if (!isPro()) { showProUpgradeHint('sort'); return; }
        sortHelp.classList.add('visible');
    } else {
        sortHelp.classList.remove('visible');
    }
}

/**
 * Handle sort commands and apply sorting
 */
async function handleSortCommand() {
    if (!isPro()) return;
    const content = editor.value;
    const { type, hasSortCommand, isActive, shouldExecute } = parseSortCommand(content);

    if (!hasSortCommand || type === 'help') {
        // Reset sort mode if no command
        if (isSortingMode) {
            isSortingMode = false;
            updateKeywordIndicator();
        }
        return;
    }

    // Set sort mode when active
    if (isActive && type && type !== 'help') {
        if (!isSortingMode) {
            isSortingMode = true;
            updateKeywordIndicator();
        }
    }

    // Only apply the sort when sort:end is typed
    if (shouldExecute && type && type !== 'help') {
        const sortedContent = applySortToContent(content, type);
        editor.value = sortedContent;

        // Reset sort mode
        isSortingMode = false;

        // Trigger auto-save
        handleContentChange(editor, async (savedNote) => {
            await updateNoteDate(savedNote);
        });

        // Update UI
        updateKeywordIndicator();
        updateEditorHighlighter();
    }
}

/**
 * Update stats display for sum/avg/count modes
 */
function updateStatsMode() {
    const content = editor.value;
    const hasMoreContent = content.split('\n').length > 1 || content.includes(':');
    const { mode } = parseKeyword(content);

    // Hide all help guides first
    sumHelp.classList.remove('visible');
    avgHelp.classList.remove('visible');
    countHelp.classList.remove('visible');
    listHelp.classList.remove('visible');
    mathHelp.classList.remove('visible');
    if (dateHelp) dateHelp.classList.remove('visible');
    if (uuidHelp) uuidHelp.classList.remove('visible');
    if (loremHelp) loremHelp.classList.remove('visible');
    if (weatherHelp) weatherHelp.classList.remove('visible');
    if (dictionaryHelp) dictionaryHelp.classList.remove('visible');

    // Show help guide when typing just the keyword (no other content)
    // Check by mode instead of hardcoded keywords
    if (mode === 'sum' && !hasMoreContent) {
        sumHelp.classList.add('visible');
    } else if (mode === 'avg' && !hasMoreContent) {
        avgHelp.classList.add('visible');
    } else if (mode === 'count' && !hasMoreContent) {
        countHelp.classList.add('visible');
    } else if (mode === 'math' && !hasMoreContent) {
        mathHelp.classList.add('visible');
    } else if (mode === 'date' && !hasMoreContent) {
        if (dateHelp) dateHelp.classList.add('visible');
    } else if (mode === 'uuid' && !hasMoreContent) {
        if (!isPro()) { showProUpgradeHint('uuid'); }
        else if (uuidHelp && !(uuidDisplay && uuidDisplay.classList.contains('visible'))) uuidHelp.classList.add('visible');
    } else if (mode === 'lorem' && !hasMoreContent) {
        if (!isPro()) { showProUpgradeHint('lorem'); }
        else if (loremHelp) loremHelp.classList.add('visible');
    } else if (mode === 'weather' && !hasMoreContent) {
        if (!isPro()) { showProUpgradeHint('weather'); }
        else if (weatherHelp && !(weatherDisplay && weatherDisplay.classList.contains('visible'))) weatherHelp.classList.add('visible');
    } else if ((mode === 'define' || mode === 'related') && !hasMoreContent) {
        if (!isPro()) { showProUpgradeHint('define'); }
        else if (dictionaryHelp && !(dictionaryDisplay && dictionaryDisplay.classList.contains('visible'))) dictionaryHelp.classList.add('visible');
    }

    // Check if we're in a stats mode and have content
    if (mode === 'count' && hasMoreContent) {
        // Count mode uses its own display overlay
        updateCountMode(content);
        statsDisplay.classList.remove('visible');
    } else if (mode === 'sum' || mode === 'avg') {
        const result = getStatsResult(mode, content);
        if (result) {
            // Format as "Total: 107" or "Average: 25"
            statsLabel.textContent = result.label + ': ';
            statsValue.textContent = result.value;

            // Apply specific accent class
            statsValue.classList.remove('stats-sum', 'stats-avg', 'stats-count');
            if (mode === 'sum') statsValue.classList.add('stats-sum');
            else if (mode === 'avg') statsValue.classList.add('stats-avg');

            statsDisplay.classList.add('visible');
        }
        countDisplay.classList.remove('visible');
    } else {
        statsDisplay.classList.remove('visible');
        countDisplay.classList.remove('visible');
    }

    // Update math display if in math mode
    if (mode === 'math') {
        updateMathMode(content);
    } else if (mode === 'game') {
        updateGameMode(content);
    } else {
        mathDisplay.classList.remove('visible');
        gameDisplay.classList.remove('active');
        if (gameHelp) gameHelp.classList.remove('visible');
    }

    // Update date display if in date mode
    if (mode === 'date' && hasMoreContent) {
        updateDateMode(content);
    } else {
        if (dateDisplay) {
            dateDisplay.classList.remove('visible');
            dateDisplay.innerHTML = '';
        }
    }

    // Hide uuid display when not in uuid mode or when showing help (no content yet)
    if (mode !== 'uuid' || !hasMoreContent) {
        if (uuidDisplay) uuidDisplay.classList.remove('visible');
    }
    if (mode !== 'weather') {
        if (weatherDisplay) weatherDisplay.classList.remove('visible');
        hideWeatherBg();
    }
    if (mode !== 'define' && mode !== 'related') {
        if (dictionaryDisplay) dictionaryDisplay.classList.remove('visible');
    }
}

/**
 * Update list mode display
 * Shows help guide when "list" is typed alone, checkboxes when content exists
 */
function updateListMode() {
    const content = editor.value;
    const hasMoreContent = content.split('\n').length > 1 || content.includes(':');
    const { mode } = parseKeyword(content);

    // Show/hide list help guide - check mode instead of hardcoded keyword
    if (mode === 'list' && !hasMoreContent) {
        listHelp.classList.add('visible');
    } else {
        listHelp.classList.remove('visible');
    }

    // Update list display if in list mode
    const editorArea = document.querySelector('.editor-area');
    if (mode === 'list') {
        renderListDisplay(content);
        editorHighlighter.classList.add('hidden');
        editorArea.classList.add('list-mode');
        updateListCursorPadding();
    } else {
        listDisplay.classList.remove('visible');
        // Only show editor highlighter if NOT in code mode
        if (!editorArea.classList.contains('code-mode')) {
            editorHighlighter.classList.remove('hidden');
        }
        editorArea.classList.remove('list-mode');
        editor.style.paddingLeft = '';
        listBreakIndex = Infinity;
    }
}

/**
 * Toggle textarea padding based on whether cursor is in list vs normal-text area.
 * List items need 24px padding for checkbox alignment; normal text after break needs 0.
 */
function updateListCursorPadding() {
    if (listBreakIndex === Infinity || listBreakIndex >= editor.value.split('\n').length) {
        editor.style.paddingLeft = '24px';
        return;
    }
    const cursorPos = editor.selectionStart;
    const textBefore = editor.value.substring(0, cursorPos);
    const cursorLine = textBefore.split('\n').length - 1; // 0-indexed line number

    if (cursorLine >= listBreakIndex) {
        editor.style.paddingLeft = '';
    } else {
        editor.style.paddingLeft = '24px';
    }
}

/**
 * Render the list display with checkboxes
 */
function renderListDisplay(content) {
    const lines = content.split('\n');
    const firstLine = lines[0] || '';

    // Build HTML - first line shows keyword
    const { keyword, mode, hasKeyword } = parseKeyword(content);
    let html = '';

    if (hasKeyword && mode === 'list') {
        // Find keyword in first line (supports custom keywords)
        const keywordLower = keyword.toLowerCase();
        const firstLineLower = firstLine.toLowerCase();
        const keywordIndex = firstLineLower.indexOf(keywordLower);

        if (keywordIndex !== -1) {
            const leadingSpace = firstLine.substring(0, keywordIndex);
            const keywordText = firstLine.substring(keywordIndex, keywordIndex + keyword.length);
            const rest = firstLine.substring(keywordIndex + keyword.length);
            // Accent 2 Main for list keyword
            html = `<div class="list-keyword-line">${escapeHtml(leadingSpace)}<span class="keyword-highlight keyword-highlight-acc2">${escapeHtml(keywordText)}</span>${escapeHtml(rest)}</div>`;
        } else {
            html = '<div class="list-keyword-line">&nbsp;</div>';
        }
    } else {
        html = '<div class="list-keyword-line">&nbsp;</div>';
    }

    // First pass: find where list section ends (empty line followed by content)
    let listEndIndex = lines.length; // Default: entire content is list
    for (let i = 1; i < lines.length - 1; i++) {
        if (lines[i].trim() === '' && lines[i + 1].trim() !== '') {
            listEndIndex = i; // The empty line is where list ends
            break;
        }
    }
    listBreakIndex = listEndIndex; // Store for cursor padding tracking

    // Process remaining lines
    for (let i = 1; i < lines.length; i++) {
        const line = lines[i];
        const isEmpty = line.trim() === '';
        const isAfterList = i >= listEndIndex;

        if (isAfterList) {
            // After list section - render as normal text (no checkbox, no indent)
            if (isEmpty) {
                html += `<div class="list-line normal-text empty">${escapeHtml(line) || '&nbsp;'}</div>`;
            } else {
                html += `<div class="list-line normal-text"><span class="list-text">${escapeHtml(line)}</span></div>`;
            }
        } else if (isEmpty) {
            // Empty line within list section
            html += `<div class="list-line empty">${escapeHtml(line) || '&nbsp;'}</div>`;
        } else {
            // Detect indentation level: 2+ leading spaces = sub-item
            const leadingSpaces = line.match(/^(\s*)/)[1].length;
            const isSub = leadingSpaces >= 2;
            const subClass = isSub ? ' list-line-sub' : '';
            const checkboxSubClass = isSub ? ' list-checkbox-sub' : '';

            // Check if line has [x] marker (account for leading whitespace)
            const trimmed = line.trimStart();
            const checkedMatch = trimmed.match(/^(\[x\])(.*)$/i);

            const isChecked = !!checkedMatch;
            const checkedClass = isChecked ? 'checked' : '';

            html += `<div class="list-line${subClass}">`;
            html += `<div class="list-checkbox${checkboxSubClass} ${checkedClass}" data-line="${i}"></div>`;

            // Make marker transparent but keep it for cursor alignment
            if (checkedMatch) {
                // [x] marker - make invisible, show rest of text
                // Preserve leading whitespace for cursor alignment
                const leadingWs = line.substring(0, leadingSpaces);
                html += `<span class="list-text ${checkedClass}"><span class="list-marker">${escapeHtml(leadingWs)}${escapeHtml(checkedMatch[1])}</span>${escapeHtml(checkedMatch[2])}</span>`;
            } else {
                // No marker - show full line
                html += `<span class="list-text ${checkedClass}">${escapeHtml(line)}</span>`;
            }
            html += `</div>`;
        }
    }

    listDisplay.innerHTML = html;
    listDisplay.classList.add('visible');
}

/**
 * Handle checkbox click - toggle checked state
 */
function handleCheckboxClick(e) {
    const checkbox = e.target;
    const lineIndex = parseInt(checkbox.dataset.line, 10);
    const lines = editor.value.split('\n');

    if (lineIndex >= 0 && lineIndex < lines.length) {
        const line = lines[lineIndex];
        const trimmed = line.trim();

        // Get leading whitespace
        const leadingSpace = line.match(/^(\s*)/)[1] || '';

        // Check current state and toggle
        const checkedMatch = trimmed.match(/^\[x\]\s*(.*)/i);

        let newLine;
        let isNowChecked;
        if (checkedMatch) {
            // Currently checked, uncheck it - remove marker entirely
            newLine = leadingSpace + checkedMatch[1];
            isNowChecked = false;
        } else {
            // Not checked - check it (add [x] marker)
            newLine = leadingSpace + '[x] ' + trimmed;
            isNowChecked = true;
        }

        lines[lineIndex] = newLine;

        // Auto-complete children if setting is enabled and this is a parent (non-indented) item
        if (settings.listAutoCompleteChildren && leadingSpace.length < 2) {
            for (let j = lineIndex + 1; j < lines.length; j++) {
                const childLine = lines[j];
                const childTrimmed = childLine.trim();
                if (childTrimmed === '') continue; // skip empty lines within children
                const childLeading = childLine.match(/^(\s*)/)[1] || '';
                if (childLeading.length < 2) break; // hit next parent, stop

                const childCheckedMatch = childTrimmed.match(/^\[x\]\s*(.*)/i);
                if (isNowChecked && !childCheckedMatch) {
                    // Check the child
                    lines[j] = childLeading + '[x] ' + childTrimmed;
                } else if (!isNowChecked && childCheckedMatch) {
                    // Uncheck the child
                    lines[j] = childLeading + childCheckedMatch[1];
                }
            }
        }

        editor.value = lines.join('\n');

        // Trigger save and update
        handleContentChange(editor, () => { });
        updateListMode();
        updateEditorHighlighter();
    }
}

/**
 * Update math mode display with inline calculation results
 */
function updateMathMode(content) {
    // Skip math mode when code mode is active — CSS values (800px, #333, etc.)
    // get misinterpreted as math expressions, creating ghost text overlay
    const editorArea = document.querySelector('.editor-area');
    if (editorArea && editorArea.classList.contains('code-mode')) {
        if (mathDisplay) {
            mathDisplay.classList.remove('visible');
            mathDisplay.innerHTML = '';
        }
        return;
    }

    const results = processMathContent(content, { significantDigits: 2 });

    // Build the HTML for math display - show keyword on first line
    const firstLine = content.split('\n')[0] || '';
    const { keyword, mode, hasKeyword } = parseKeyword(content);
    let html = '';

    if (hasKeyword && mode === 'math') {
        // Find keyword in first line (supports custom keywords)
        const keywordLower = keyword.toLowerCase();
        const firstLineLower = firstLine.toLowerCase();
        const keywordIndex = firstLineLower.indexOf(keywordLower);

        if (keywordIndex !== -1) {
            const leadingSpace = firstLine.substring(0, keywordIndex);
            const keywordText = firstLine.substring(keywordIndex, keywordIndex + keyword.length);
            const rest = firstLine.substring(keywordIndex + keyword.length);
            // Accent 1 Main for math/currency keyword
            html = `<div class="math-line">${escapeHtml(leadingSpace)}<span class="keyword-highlight keyword-highlight-acc1">${escapeHtml(keywordText)}</span>${escapeHtml(rest)}</div>`;
        } else {
            html = '<div class="math-line">&nbsp;</div>';
        }
    } else {
        html = '<div class="math-line">&nbsp;</div>';
    }

    for (const item of results) {
        if (item.type === 'empty') {
            html += '<div class="math-line">&nbsp;</div>';
        } else if (item.type === 'calculation') {
            // Show just the result in orange
            html += `<div class="math-line"><span class="math-expression">${escapeHtml(item.expression)}</span> <span class="math-result" data-copy="${escapeHtml(item.formatted)}" title="Click to copy">${escapeHtml(item.formatted)}</span></div>`;
        } else if (item.type === 'variable') {
            // Show variable value: Variable (Acc1Sec) : (Acc2Main) Value
            html += `<div class="math-line"><span class="math-expression"><span class="math-variable-assignment">${escapeHtml(item.name)}</span><span class="math-colon"> : </span></span><span class="math-result" data-copy="${escapeHtml(item.formatted)}" title="Click to copy">${escapeHtml(item.formatted)}</span></div>`;
        } else {
            html += '<div class="math-line">&nbsp;</div>';
        }
    }

    mathDisplay.innerHTML = html;
    mathDisplay.classList.add('visible');
}

/**
 * Update date mode display
 */
function updateDateMode(content) {
    if (!isPro()) { showProUpgradeHint('date'); return; }

    // Skip date mode when code mode is active
    const editorArea = document.querySelector('.editor-area');
    if (editorArea && editorArea.classList.contains('code-mode')) {
        if (dateDisplay) {
            dateDisplay.classList.remove('visible');
            if (weatherDisplay && weatherDisplay.classList.contains('visible')) {
                hideWeatherBg();
                weatherDisplay.classList.remove('visible');
            }
            if (dictionaryDisplay && dictionaryDisplay.classList.contains('visible')) {
                dictionaryDisplay.classList.remove('visible');
            }
            dateDisplay.innerHTML = '';
        }
        return;
    }

    const dateFormat = settings.dateFormat || 'MMM D, YYYY';
    const results = processDateContent(content, dateFormat);

    const firstLine = content.split('\n')[0] || '';
    const { keyword, mode, hasKeyword } = parseKeyword(content);
    let html = '';

    if (hasKeyword && mode === 'date') {
        const keywordLower = keyword.toLowerCase();
        const firstLineLower = firstLine.toLowerCase();
        const keywordIndex = firstLineLower.indexOf(keywordLower);

        if (keywordIndex !== -1) {
            const leadingSpace = firstLine.substring(0, keywordIndex);
            const keywordText = firstLine.substring(keywordIndex, keywordIndex + keyword.length);
            const rest = firstLine.substring(keywordIndex + keyword.length);
            html = `<div class="date-line">${escapeHtml(leadingSpace)}<span class="keyword-highlight keyword-highlight-acc6">${escapeHtml(keywordText)}</span>${escapeHtml(rest)}</div>`;
        } else {
            html = '<div class="date-line">&nbsp;</div>';
        }
    } else {
        html = '<div class="date-line">&nbsp;</div>';
    }

    for (const item of results) {
        if (item.type === 'empty') {
            html += '<div class="date-line">&nbsp;</div>';
        } else if (item.type === 'date-result') {
            html += `<div class="date-line"><span class="date-expression">${escapeHtml(item.expression)}</span> <span class="date-result" data-copy="${escapeHtml(item.formatted)}" title="Click to copy">${escapeHtml(item.formatted)}</span></div>`;
        } else if (item.type === 'comment') {
            html += `<div class="date-line"><span class="date-expression">//${escapeHtml(item.text)}</span></div>`;
        } else {
            html += '<div class="date-line">&nbsp;</div>';
        }
    }

    dateDisplay.innerHTML = html;
    dateDisplay.classList.add('visible');
}


/**
 * Update UUID mode display - shows generated values
 */
function updateUuidMode(content) {
    if (!uuidDisplay) return;
    if (!isPro()) { showProUpgradeHint('uuid'); return; }

    const result = processUuidCommand(content);

    if (result.type === 'help' || result.results.length === 0) {
        uuidDisplay.classList.remove('visible');
        return;
    }

    let html = `<div class="uuid-display-header">${escapeHtml(result.description)}</div>`;
    result.results.forEach(val => {
        html += `<div class="uuid-display-row" data-copy="${escapeHtml(val)}" title="Click to copy">${escapeHtml(val)}</div>`;
    });
    html += '<div class="uuid-display-copied" id="uuid-copied-msg">Copied!</div>';

    uuidDisplay.innerHTML = html;
    uuidDisplay.classList.add('visible');

    // Hide help guide when showing results
    if (uuidHelp) uuidHelp.classList.remove('visible');

    // Click-to-copy handlers
    uuidDisplay.querySelectorAll('.uuid-display-row').forEach(row => {
        row.addEventListener('click', () => {
            const val = row.getAttribute('data-copy');
            if (val) {
                navigator.clipboard.writeText(val).catch(() => { });
                const msg = uuidDisplay.querySelector('#uuid-copied-msg');
                if (msg) {
                    msg.classList.add('visible');
                    setTimeout(() => msg.classList.remove('visible'), 1500);
                }
            }
        });
    });
}

/**
 * Handle lorem insert - generates text and inserts into editor
 */
function handleLoremInsert(content) {
    if (!isPro()) { showProUpgradeHint('lorem'); return; }

    const result = processLoremCommand(content);
    if (result.help || !result.text) return;

    const firstLine = content.split('\n')[0];
    editor.value = firstLine + '\n' + result.text;
    editor.selectionStart = editor.selectionEnd = editor.value.length;
    editor.dispatchEvent(new Event('input', { bubbles: true }));
}

/**
 * Handle Dictionary & Thesaurus lookups
 */
async function handleDictionaryQuery(word, mode) {
    if (!isPro()) { showProUpgradeHint('define'); return; }

    if (dictionaryHelp) dictionaryHelp.classList.remove('visible');

    dictionaryDisplay.innerHTML = '<div class="dictionary-loading">Looking up word...</div>';
    dictionaryDisplay.classList.add('visible');

    try {
        const data = await fetchWordData(word);

        let html = `<div class="dictionary-header">
                        <div class="dictionary-word">${escapeHtml(data.word)}</div>
                        ${data.phonetic ? `<div class="dictionary-phonetic">${escapeHtml(data.phonetic)}</div>` : ''}
                    </div>`;

        if (mode === 'related') {
            // Thesaurus Mode
            let hasSynonyms = data.synonyms && data.synonyms.length > 0;
            let hasAntonyms = data.antonyms && data.antonyms.length > 0;

            if (!hasSynonyms && !hasAntonyms) {
                html += `<div class="dictionary-empty">No synonyms or antonyms found for "${escapeHtml(word)}".</div>`;
            } else {
                html += `<div class="dictionary-thesaurus-container">`;

                if (hasSynonyms) {
                    html += `<div class="dictionary-synonyms-section">
                                <div class="dictionary-section-title">Synonyms</div>
                                <div class="dictionary-tags">
                                    ${data.synonyms.slice(0, 15).map(s => `<span class="dictionary-tag synonym">${escapeHtml(s)}</span>`).join('')}
                                </div>
                             </div>`;
                }

                if (hasAntonyms) {
                    html += `<div class="dictionary-antonyms-section">
                                <div class="dictionary-section-title">Antonyms</div>
                                <div class="dictionary-tags">
                                    ${data.antonyms.slice(0, 10).map(a => `<span class="dictionary-tag antonym">${escapeHtml(a)}</span>`).join('')}
                                </div>
                             </div>`;
                }
                html += `</div>`;
            }
        } else {
            // Define Mode
            if (!data.meanings || data.meanings.length === 0) {
                html += `<div class="dictionary-empty">No definitions found for "${escapeHtml(word)}".</div>`;
            } else {
                html += `<div class="dictionary-meanings">`;
                data.meanings.forEach(meaning => {
                    html += `<div class="dictionary-meaning">
                                <div class="dictionary-pos">${escapeHtml(meaning.partOfSpeech)}</div>
                                <ul class="dictionary-defs">`;
                    meaning.definitions.forEach(def => {
                        html += `<li class="dictionary-def-item">
                                    <div class="dictionary-def-text">${escapeHtml(def.definition)}</div>
                                    ${def.example ? `<div class="dictionary-def-example">"${escapeHtml(def.example)}"</div>` : ''}
                                 </li>`;
                    });
                    html += `   </ul>
                             </div>`;
                });
                html += `</div>`;
            }
        }

        dictionaryDisplay.innerHTML = html;

    } catch (err) {
        dictionaryDisplay.innerHTML = `<div class="dictionary-error">${escapeHtml(err.message || 'Lookup failed')}</div>`;
    }
}

/**
 * Show weather gradient on editor-area background
 */
function showWeatherBg(gradient) {
    const area = document.querySelector('.editor-area');
    if (area) {
        area.style.setProperty('--weather-gradient', gradient);
        area.classList.add('has-weather');
    }
}

/**
 * Remove weather gradient from editor-area
 */
function hideWeatherBg() {
    const area = document.querySelector('.editor-area');
    if (area) {
        area.classList.remove('has-weather');
        area.style.removeProperty('--weather-gradient');
    }
}

/**
 * Update weather mode display — full-area immersive layout
 */
async function updateWeatherMode(content) {
    if (!weatherDisplay) return;
    if (!isPro()) { showProUpgradeHint('weather'); return; }

    const { mode, city } = parseWeatherCommand(content);

    // Hide help guide and show loading state
    if (weatherHelp) weatherHelp.classList.remove('visible');
    showWeatherBg('linear-gradient(160deg, #56CCF2, #2F80ED)');
    weatherDisplay.innerHTML = '<div class="weather-loading">Fetching weather...</div>';
    weatherDisplay.classList.add('visible');

    try {
        let data;
        if (mode === 'tomorrow') {
            data = await fetchWeatherTomorrow(city);
        } else if (mode === 'daily') {
            data = await fetchWeatherDaily(city);
        } else {
            data = await fetchWeather(city);
        }

        showWeatherBg(data.gradient);

        if (data.type === 'current') {
            const hourlyHtml = data.hourly.map(h =>
                `<div class="weather-hour-item"><span class="weather-hour-time">${h.hour}</span><span class="weather-hour-icon">${h.icon}</span><span class="weather-hour-temp">${h.temp}°</span></div>`
            ).join('');

            weatherDisplay.innerHTML = `
                <div class="weather-location">${escapeHtml(data.location)} <span style="float: right; font-size: 0.85em; opacity: 0.8; font-weight: normal;">${data.localTime}</span></div>
                <div class="weather-main">
                    <div class="weather-main-left">
                        <span class="weather-icon">${data.icon}</span>
                        <span class="weather-temp">${data.temperature}°</span>
                    </div>
                    <div class="weather-main-right">
                        <div class="weather-feels">Feels ${data.feelsLike}°</div>
                        <div class="weather-desc">${escapeHtml(data.description)}</div>
                    </div>
                </div>
                <div class="weather-hourly">${hourlyHtml}</div>`;
        } else if (data.type === 'tomorrow') {
            const hourlyHtml = data.hourly.map(h =>
                `<div class="weather-hour-item"><span class="weather-hour-time">${h.hour}</span><span class="weather-hour-icon">${h.icon}</span><span class="weather-hour-temp">${h.temp}°</span></div>`
            ).join('');

            weatherDisplay.innerHTML = `
                <div class="weather-location">${escapeHtml(data.location)} — Tomorrow <span style="float: right; font-size: 0.85em; opacity: 0.8; font-weight: normal;">${data.localTime}</span></div>
                <div class="weather-main">
                    <div class="weather-main-left">
                        <span class="weather-icon">${data.icon}</span>
                        <span class="weather-temp">${data.tempHigh}°</span>
                    </div>
                    <div class="weather-main-right">
                        <div class="weather-feels">Low ${data.tempLow}°</div>
                        <div class="weather-desc">${escapeHtml(data.description)}</div>
                    </div>
                </div>
                <div class="weather-hourly">${hourlyHtml}</div>`;
        } else if (data.type === 'daily') {
            const daysHtml = data.days.map(d =>
                `<div class="weather-day-row"><span class="weather-day-name">${d.day}</span><span class="weather-day-icon">${d.icon}</span><div class="weather-day-temps"><span class="weather-day-high">${d.high}°</span><span class="weather-day-low">${d.low}°</span></div></div>`
            ).join('');

            weatherDisplay.innerHTML = `
                <div class="weather-location">${escapeHtml(data.location)} <span style="float: right; font-size: 0.85em; opacity: 0.8; font-weight: normal;">${data.localTime}</span></div>
                <div class="weather-main">
                    <div class="weather-main-left">
                        <span class="weather-icon">${data.icon}</span>
                        <span class="weather-temp">${data.temperature}°</span>
                    </div>
                    <div class="weather-main-right">
                        <div class="weather-feels">Feels ${data.feelsLike}°</div>
                        <div class="weather-desc">${escapeHtml(data.description)}</div>
                    </div>
                </div>
                <div class="weather-daily">${daysHtml}</div>`;
        }
    } catch (err) {
        weatherDisplay.innerHTML = `<div class="weather-error">${escapeHtml(err.message || 'Failed to fetch weather')}</div>`;
    }
}

/**
 * Lapwatch tick callback — reuses timer display elements
 */
function onLapwatchTick(data) {
    if (!data) return;

    const [mins, secs] = data.time.split(':');
    timerMinutes.textContent = mins;
    timerSeconds.textContent = secs;
    if (timerMsEl) timerMsEl.textContent = data.ms;

    // Progress bar fills in 5-second intervals
    const cycleProgress = (data.elapsed % 5000) / 5000;
    timerProgress.style.setProperty('--timer-progress', cycleProgress);

    // Render laps
    if (timerLapsEl && data.laps.length > 0) {
        timerLapsEl.innerHTML = data.laps.map(lap =>
            `<div class="timer-lap-item"><span class="lap-num">${lap.index}</span> ${lap.deltaStr.display}</div>`
        ).reverse().join('');
    }
}

/**
 * Update count mode display with statistics
 */
function updateCountMode(content) {
    const stats = countStats(content);
    const lines = content.split('\n');
    const { keyword, mode, hasKeyword } = parseKeyword(content);

    let html = '';

    // Render all content lines
    for (let i = 0; i < lines.length; i++) {
        const line = lines[i];

        if (i === 0) {
            // First line - check for keyword
            if (hasKeyword && mode === 'count') {
                // Find keyword in first line (supports custom keywords)
                const keywordLower = keyword.toLowerCase();
                const lineLower = line.toLowerCase();
                const keywordIndex = lineLower.indexOf(keywordLower);

                if (keywordIndex !== -1) {
                    const leadingSpace = line.substring(0, keywordIndex);
                    const keywordText = line.substring(keywordIndex, keywordIndex + keyword.length);
                    const rest = line.substring(keywordIndex + keyword.length);
                    html += `<div class="count-line">${escapeHtml(leadingSpace)}<span class="keyword-highlight keyword-highlight-acc3">${escapeHtml(keywordText)}</span>${escapeHtml(rest)}</div>`;
                } else {
                    html += `<div class="count-line">${escapeHtml(line)}</div>`;
                }
            } else {
                html += `<div class="count-line">${escapeHtml(line)}</div>`;
            }
        } else {
            // All other lines - make content transparent so it doesn't overlap
            if (line.trim().length > 0) {
                html += `<div class="count-line"><span style="color: transparent;">${escapeHtml(line)}</span></div>`;
            } else {
                html += '<div class="count-line">&nbsp;</div>';
            }
        }
    }

    // Add 2 empty lines between content and statistics
    html += '<div class="count-line">&nbsp;</div>';
    html += '<div class="count-line">&nbsp;</div>';

    // Add statistics - clickable to copy
    html += `<div class="count-line clickable" data-copy="${stats.items}" title="Click to copy">Items: ${stats.items}</div>`;
    html += `<div class="count-line clickable" data-copy="${stats.words}" title="Click to copy">Words: ${stats.words}</div>`;
    html += `<div class="count-line clickable" data-copy="${stats.chars}" title="Click to copy">Characters: ${stats.chars}</div>`;
    html += `<div class="count-line clickable" data-copy="${stats.sentences}" title="Click to copy">Sentences: ${stats.sentences}</div>`;
    html += `<div class="count-line clickable" data-copy="${stats.fleschReadingEase}" title="Click to copy">Flesch Reading Ease Score: ${stats.fleschReadingEase}</div>`;
    html += `<div class="count-line clickable" data-copy="${stats.fleschKincaidGrade}" title="Click to copy">Flesch-Kincaid Grade Level: ${stats.fleschKincaidGrade}</div>`;

    countDisplay.innerHTML = html;
    countDisplay.classList.add('visible');

    // Add click handlers for copying
    const clickableLines = countDisplay.querySelectorAll('.count-line.clickable');
    clickableLines.forEach(line => {
        line.addEventListener('click', (e) => {
            const value = e.target.getAttribute('data-copy');
            if (value) {
                copyToClipboard(value);
                showCopyFeedback(e.target);
            }
        });
    });
}

/**
 * Update game mode - show help or render game
 */
function updateGameMode(content) {
    if (!isPro()) { showProUpgradeHint('game'); return; }

    const { type, gameName } = parseGameCommand(content);

    if (!type) {
        gameDisplay.classList.remove('active');
        if (gameHelp) gameHelp.classList.remove('visible');
        return;
    }

    if (type === 'help') {
        // Show game help guide
        gameDisplay.classList.remove('active');
        if (gameHelp) {
            gameHelp.classList.add('visible');
        }
    } else if (type === 'tic') {
        // Show Tic-Tac-Toe game
        if (gameHelp) gameHelp.classList.remove('visible');
        gameDisplay.classList.add('active');
        initTicTacToe(gameDisplay);
    } else if (type === 'hangman') {
        // Show Hangman game
        if (gameHelp) gameHelp.classList.remove('visible');
        gameDisplay.classList.add('active');
        initHangman(gameDisplay);
    } else if (type === 'quiz') {
        // Show Quiz game
        if (gameHelp) gameHelp.classList.remove('visible');
        gameDisplay.classList.add('active');
        initQuiz(gameDisplay);
    } else if (type === 'type') {
        // Show Typing Speed Test
        if (gameHelp) gameHelp.classList.remove('visible');
        gameDisplay.classList.add('active');
        initTypingTest(gameDisplay);
    } else if (type === 'react') {
        // Show Reaction Time
        if (gameHelp) gameHelp.classList.remove('visible');
        gameDisplay.classList.add('active');
        initReactionTime(gameDisplay);
    } else if (type === 'mines') {
        // Show Minesweeper
        if (gameHelp) gameHelp.classList.remove('visible');
        gameDisplay.classList.add('active');
        initMinesweeper(gameDisplay);
    } else if (type === 'memory') {
        // Show Memory Match
        if (gameHelp) gameHelp.classList.remove('visible');
        gameDisplay.classList.add('active');
        initMemory(gameDisplay);
    } else if (type === 'wordle') {
        // Show Wordle
        if (gameHelp) gameHelp.classList.remove('visible');
        gameDisplay.classList.add('active');
        initWordle(gameDisplay);
    } else {
        // Future games
        gameDisplay.classList.remove('active');
        if (gameHelp) gameHelp.classList.remove('visible');
    }
}

/**
 * Copy text to clipboard
 * @param {string} text - Text to copy
 */
function copyToClipboard(text) {
    navigator.clipboard.writeText(text).catch(err => {
        console.error('Failed to copy:', err);
    });
}

/**
 * Show visual feedback that text was copied
 * @param {HTMLElement} element - Element that was clicked
 */
function showCopyFeedback(element) {
    const originalText = element.textContent;
    element.textContent = 'Copied!';

    setTimeout(() => {
        element.textContent = originalText;
    }, 1000);
}

/**
 * Handle template commands
 */
async function handleTemplateCommands() {
    if (!isPro()) return;

    const content = editor.value;
    const { type, templateName } = parseTemplateCommand(content);

    if (!type) {
        // Reset creation mode if no template command
        if (isCreatingTemplate) {
            isCreatingTemplate = false;
            templateCreateStartLine = -1;
            updateKeywordIndicator();
        }
        return;
    }

    if (type === 'create') {
        // Start template creation mode
        if (!isCreatingTemplate) {
            isCreatingTemplate = true;
            const lines = content.split('\n');
            templateCreateStartLine = lines.indexOf(lines.find(l => l.trim().toLowerCase() === 'template:create'));
            updateKeywordIndicator();
        }
    } else if (type === 'end' && isCreatingTemplate) {
        // End template creation and save
        await saveTemplateFromContent();
    }
    // Note: template loading (type === 'load') is handled in keydown Enter handler
}

/**
 * Save template from content between template:create and template:end
 */
async function saveTemplateFromContent() {
    const content = editor.value;
    const lines = content.split('\n');

    // Find create and end lines
    const createLineIndex = lines.findIndex(l => l.trim().toLowerCase() === 'template:create');
    const endLineIndex = lines.findIndex(l => l.trim().toLowerCase() === 'template:end');

    if (createLineIndex === -1 || endLineIndex === -1) {
        showTemplateNotification('Error: Missing template:create or template:end', 2000);
        return;
    }

    if (endLineIndex <= createLineIndex + 1) {
        showTemplateNotification('Error: Template must have a name and content', 2000);
        return;
    }

    // Extract template name (first line after template:create)
    const templateName = lines[createLineIndex + 1].trim();

    if (!templateName) {
        showTemplateNotification('Error: Template name cannot be empty', 2000);
        return;
    }

    if (templateName.includes(':')) {
        showTemplateNotification('Error: Template name cannot contain ":"', 2000);
        return;
    }

    // Extract template content (lines between name and template:end)
    const templateContent = lines.slice(createLineIndex + 2, endLineIndex).join('\n');

    if (!templateContent.trim()) {
        showTemplateNotification('Error: Template content cannot be empty', 2000);
        return;
    }

    // Save template
    try {
        await saveTemplate(templateName, templateContent);
    } catch (error) {
        showTemplateNotification('Error: Failed to save template', 2000);
        console.error('Template save error:', error);
        return;
    }

    // Show success notification
    showTemplateNotification(`Template "${templateName}" saved`, 2000);

    // Reset creation mode first
    isCreatingTemplate = false;
    templateCreateStartLine = -1;

    // Clear the note
    editor.value = '';
    editor.scrollTop = 0;

    // Force update immediately AND on next tick to guarantee UI refresh
    forceUpdateUI();
    setTimeout(forceUpdateUI, 10);
}

/**
 * Update placeholders help display
 * Shows help when first line is exactly "placeholders" with no other content
 */
async function updatePlaceholdersHelp() {
    if (!placeholdersHelp) return;
    if (!isPro()) return;

    const content = editor.value;
    const lines = content.split('\n');
    const firstLineTrimmed = lines[0].trim().toLowerCase();
    const hasMoreContent = lines.length > 1 && lines.slice(1).some(line => line.trim() !== '');

    if (firstLineTrimmed === 'placeholder' && !hasMoreContent) {
        await populatePlaceholdersList();
        placeholdersHelp.classList.add('visible');
    } else {
        placeholdersHelp.classList.remove('visible');
    }
}

/**
 * Populate the saved placeholders list in the help guide
 */
async function populatePlaceholdersList() {
    if (!placeholdersList) return;

    const placeholders = await getPlaceholders();
    const keys = Object.keys(placeholders);

    if (keys.length === 0) {
        placeholdersList.innerHTML = '<div class="placeholders-list-empty">No placeholders saved yet.</div>';
    } else {
        placeholdersList.innerHTML = keys.map(key =>
            `<div class="placeholders-list-item"><span class="placeholder-key">{{${key}}}</span> <span class="placeholder-value">→ ${placeholders[key]}</span></div>`
        ).join('');
    }
}

/**
 * Handle placeholders commands (create/end)
 */
async function handlePlaceholdersCommands() {
    if (!isPro()) return;

    const content = editor.value;
    const { type } = parsePlaceholdersCommand(content);

    if (!type) {
        if (isCreatingPlaceholders) {
            isCreatingPlaceholders = false;
            placeholdersCreateStartLine = -1;
            updateKeywordIndicator();
        }
        return;
    }

    if (type === 'create') {
        if (!isCreatingPlaceholders) {
            isCreatingPlaceholders = true;
            const lines = content.split('\n');
            placeholdersCreateStartLine = lines.indexOf(lines.find(l => l.trim().toLowerCase() === 'placeholder:create'));
            updateKeywordIndicator();
        }
    } else if (type === 'end' && isCreatingPlaceholders) {
        await savePlaceholdersFromContent();
    }
}

/**
 * Save placeholders from content between placeholders:create and placeholders:end
 * Format: one per line, "key value" (first word is key, rest is value)
 */
async function savePlaceholdersFromContent() {
    const content = editor.value;
    const lines = content.split('\n');

    const createLineIndex = lines.findIndex(l => l.trim().toLowerCase() === 'placeholder:create');
    const endLineIndex = lines.findIndex(l => l.trim().toLowerCase() === 'placeholder:end');

    if (createLineIndex === -1 || endLineIndex === -1) {
        showTemplateNotification('Error: Missing placeholder:create or placeholder:end', 2000);
        return;
    }

    if (endLineIndex <= createLineIndex + 1) {
        showTemplateNotification('Error: No placeholders defined', 2000);
        return;
    }

    // Parse lines between create and end
    const map = {};
    let count = 0;
    for (let i = createLineIndex + 1; i < endLineIndex; i++) {
        const line = lines[i].trim();
        if (!line) continue;

        // First word is key, rest is value
        const spaceIndex = line.indexOf(' ');
        if (spaceIndex > 0) {
            const key = line.substring(0, spaceIndex).trim();
            const value = line.substring(spaceIndex + 1).trim();
            if (key && value) {
                map[key] = value;
                count++;
            }
        }
    }

    if (count === 0) {
        showTemplateNotification('Error: No valid placeholders found (use: key value)', 2000);
        return;
    }

    try {
        await savePlaceholders(map);
    } catch (error) {
        showTemplateNotification('Error: Failed to save placeholders', 2000);
        console.error('Placeholder save error:', error);
        return;
    }

    showTemplateNotification(`${count} placeholder(s) saved`, 2000);

    isCreatingPlaceholders = false;
    placeholdersCreateStartLine = -1;

    editor.value = '';
    editor.scrollTop = 0;

    forceUpdateUI();
    setTimeout(forceUpdateUI, 10);
}

/**
 * Load and apply a template by name
 */
async function loadTemplate(templateName) {
    let template;

    try {
        template = await getTemplate(templateName);
    } catch (error) {
        showTemplateNotification('Error: Failed to load template', 2000);
        console.error('[loadTemplate] Error:', error);
        return;
    }

    if (!template) {
        showTemplateNotification(`Template "${templateName}" not found`, 2000);
        return;
    }

    let content = template.content;

    // Resolve placeholders
    const { resolved, customPlaceholders } = await resolveTemplatePlaceholders(content);
    content = resolved;

    if (customPlaceholders.length > 0) {
        // Show prompt modal for custom placeholders
        showPlaceholderPrompt(customPlaceholders, (values) => {
            // Replace custom placeholders with user-provided values
            for (const name of customPlaceholders) {
                const regex = new RegExp(`\\{\\{${name.replace(/[.*+?^${}()|[\]\\]/g, '\\$&')}\\}\\}`, 'g');
                content = content.replace(regex, values[name] || name);
            }
            applyTemplateContent(content, templateName);
        });
    } else {
        applyTemplateContent(content, templateName);
    }
}

/**
 * Final step: set editor content from a resolved template
 */
function applyTemplateContent(content, templateName) {
    editor.value = content;
    editor.scrollTop = 0;
    positionCursorAtStart(editor);

    showTemplateNotification(`Template "${templateName}" loaded`, 1500);

    forceUpdateUI();
    setTimeout(forceUpdateUI, 10);
}

/**
 * Resolve built-in placeholders and identify custom ones
 * @returns {{ resolved: string, customPlaceholders: string[] }}
 */
async function resolveTemplatePlaceholders(content) {
    // Find all {{...}} placeholders
    const placeholderRegex = /\{\{([^}]+)\}\}/g;
    const allMatches = [...content.matchAll(placeholderRegex)];

    if (allMatches.length === 0) {
        return { resolved: content, customPlaceholders: [] };
    }

    const builtins = new Set(['date', 'time', 'datetime', 'clipboard']);
    const customSet = new Set();
    let resolved = content;

    // Build built-in values
    const dateStr = getFormattedToday(settings.dateFormat || 'MMM D, YYYY');
    const now = new Date();
    const timeStr = now.toLocaleTimeString('en-US', { hour: 'numeric', minute: '2-digit', hour12: true });
    const datetimeStr = `${dateStr} ${timeStr}`;

    let clipboardText = '';
    try {
        clipboardText = await navigator.clipboard.readText();
    } catch (e) {
        clipboardText = '';
    }

    const builtinValues = {
        date: dateStr,
        time: timeStr,
        datetime: datetimeStr,
        clipboard: clipboardText
    };

    // Load saved custom placeholders
    const savedPlaceholders = await getPlaceholders();

    // Replace built-ins, saved placeholders, and collect unknown custom ones
    for (const match of allMatches) {
        const name = match[1].trim();
        const lower = name.toLowerCase();
        if (builtins.has(lower)) {
            const regex = new RegExp(`\\{\\{\\s*${name.replace(/[.*+?^${}()|[\]\\]/g, '\\$&')}\\s*\\}\\}`, 'g');
            resolved = resolved.replace(regex, builtinValues[lower]);
        } else if (savedPlaceholders[name] !== undefined) {
            // Auto-fill from saved placeholders
            const regex = new RegExp(`\\{\\{\\s*${name.replace(/[.*+?^${}()|[\]\\]/g, '\\$&')}\\s*\\}\\}`, 'g');
            resolved = resolved.replace(regex, savedPlaceholders[name]);
        } else {
            customSet.add(name);
        }
    }

    return { resolved, customPlaceholders: [...customSet] };
}

/**
 * Show a modal prompting user for custom placeholder values
 * @param {string[]} placeholders - list of custom placeholder names
 * @param {Function} callback - called with { name: value } map when applied
 */
function showPlaceholderPrompt(placeholders, callback) {
    const modal = document.getElementById('placeholder-modal');
    const contentEl = document.getElementById('placeholder-modal-content');
    const applyBtn = document.getElementById('placeholder-modal-apply');
    const cancelBtn = document.getElementById('placeholder-modal-cancel');
    const closeBtn = document.getElementById('placeholder-modal-close');

    // Build input fields
    contentEl.innerHTML = '';
    for (const name of placeholders) {
        const field = document.createElement('div');
        field.className = 'placeholder-field';
        field.innerHTML = `
            <label>${name}</label>
            <input type="text" data-placeholder="${name}" placeholder="Enter ${name}..." autocomplete="off" spellcheck="false">
        `;
        contentEl.appendChild(field);
    }

    modal.classList.add('visible');

    // Focus the first input
    const firstInput = contentEl.querySelector('input');
    if (firstInput) setTimeout(() => firstInput.focus(), 50);

    // Collect values and apply
    function apply() {
        const values = {};
        contentEl.querySelectorAll('input').forEach(input => {
            values[input.dataset.placeholder] = input.value;
        });
        cleanup();
        callback(values);
    }

    // Cancel: use placeholder names as-is
    function cancel() {
        const values = {};
        placeholders.forEach(name => { values[name] = name; });
        cleanup();
        callback(values);
    }

    function cleanup() {
        modal.classList.remove('visible');
        applyBtn.removeEventListener('click', apply);
        cancelBtn.removeEventListener('click', cancel);
        closeBtn.removeEventListener('click', cancel);
        document.removeEventListener('keydown', handleKeydown);
    }

    function handleKeydown(e) {
        if (e.key === 'Escape') {
            e.preventDefault();
            cancel();
        } else if (e.key === 'Enter') {
            e.preventDefault();
            apply();
        }
    }

    applyBtn.addEventListener('click', apply);
    cancelBtn.addEventListener('click', cancel);
    closeBtn.addEventListener('click', cancel);
    document.addEventListener('keydown', handleKeydown);
}

/**
 * Update bookmarks mode and display
 */
async function updateBookmarksMode() {
    const content = editor.value;
    const editorArea = document.querySelector('.editor-area');
    const { type, tag, query, hasBookmarksCommand } = parseBookmarksCommand(content);

    if (!hasBookmarksCommand) {
        // Not in bookmarks mode - hide all bookmarks UI
        if (bookmarksDisplay) bookmarksDisplay.classList.remove('visible');
        if (bookmarksHelp) bookmarksHelp.classList.remove('visible');
        return;
    }

    if (!isPro()) { showProUpgradeHint('bookmarks'); return; }

    // Ensure search panel is hidden when in bookmarks mode
    if (searchPanel) {
        searchPanel.classList.remove('visible');
    }

    // Handle different command types
    if (type === 'help') {
        // Show help guide with tags
        await populateTagsList();
        bookmarksHelp.classList.add('visible');
        bookmarksDisplay.classList.remove('visible');
        // Hide editor highlighter when showing help
        if (editorHighlighter) {
            editorHighlighter.classList.add('hidden');
            editorHighlighter.innerHTML = '';
        }
    } else if (type === 'add') {
        // Add mode - hide help and display, show editor normally
        bookmarksHelp.classList.remove('visible');
        bookmarksDisplay.classList.remove('visible');
        // Keep editor highlighter visible so user can see their typing (unless in code mode)
        if (editorHighlighter && !editorArea.classList.contains('code-mode')) {
            editorHighlighter.classList.remove('hidden');
        }
        // Guided entry handled by handleBookmarksAdd()
    } else if (type === 'end') {
        // End command detected - save bookmarks
        await saveAllBookmarks();
    } else if (type === 'filter') {
        // Show filtered bookmarks
        const bookmarks = await getBookmarks(tag);
        await renderBookmarksDisplay(bookmarks, tag);
        bookmarksHelp.classList.remove('visible');
        // Hide editor highlighter when showing bookmarks display
        if (editorHighlighter) {
            editorHighlighter.classList.add('hidden');
            editorHighlighter.innerHTML = '';
        }
    } else if (type === 'search') {
        // Show search results
        const bookmarks = await searchBookmarks(query);
        await renderBookmarksDisplay(bookmarks, null, `Search: "${query}"`);
        bookmarksHelp.classList.remove('visible');
        // Hide editor highlighter when showing bookmarks display
        if (editorHighlighter) {
            editorHighlighter.classList.add('hidden');
            editorHighlighter.innerHTML = '';
        }
    } else if (type === 'import') {
        bookmarksHelp.classList.remove('visible');
        bookmarksDisplay.classList.remove('visible');
        triggerBookmarksImport();
    } else if (type === 'edit') {
        // Edit mode - hide help and display, show editor normally
        bookmarksHelp.classList.remove('visible');
        bookmarksDisplay.classList.remove('visible');
        // Keep editor highlighter visible so user can see their typing (unless in code mode)
        if (editorHighlighter && !editorArea.classList.contains('code-mode')) {
            editorHighlighter.classList.remove('hidden');
        }
        // Editing handled by same guided entry system
    }
}

/**
 * Trigger bookmarks import from JSON file
 */
async function triggerBookmarksImport() {
    try {
        showTemplateNotification('Importing Chrome bookmarks...', 2000);
        const count = await importFromChrome();
        showTemplateNotification(`Imported ${count} bookmark(s)`, 3000);
    } catch (err) {
        showTemplateNotification(`Import failed: ${err.message}`, 3000);
    }
    // Refresh to bookmarks view
    editor.value = 'bookmarks';
    editor.dispatchEvent(new Event('input', { bubbles: true }));
}

/**
 * Render bookmarks display
 */
async function renderBookmarksDisplay(bookmarks, filterTag = null, title = null) {
    if (!bookmarksDisplay) return;

    const content = editor.value;
    const firstLine = content.split('\n')[0] || '';

    let html = '';

    // Show the full first line (the search command)
    if (firstLine) {
        // Find "bookmarks" in the first line and highlight it
        const lowerFirstLine = firstLine.toLowerCase();
        const bookmarksIndex = lowerFirstLine.indexOf('bookmarks');

        if (bookmarksIndex !== -1) {
            const before = firstLine.substring(0, bookmarksIndex);
            const keyword = firstLine.substring(bookmarksIndex, bookmarksIndex + 9); // "bookmarks"
            const after = firstLine.substring(bookmarksIndex + 9);

            html += `<div class="bookmarks-keyword-line">`;
            html += escapeHtml(before);
            html += `<span class="keyword-highlight keyword-highlight-acc5">${escapeHtml(keyword)}</span>`;
            html += escapeHtml(after);
            html += `</div>`;
        } else {
            html += `<div class="bookmarks-keyword-line">${escapeHtml(firstLine)}</div>`;
        }
    }

    // Show custom title if provided
    if (title) {
        html += `<div style="color: var(--text-secondary); margin-bottom: 10px; font-size: 12px;">${escapeHtml(title)}</div>`;
    }

    // Show bookmarks
    if (bookmarks.length === 0) {
        html += `<div class="bookmarks-empty">No bookmarks found</div>`;
    } else {
        for (const bookmark of bookmarks) {
            html += `<div class="bookmarks-item" data-bookmark-id="${bookmark.id}">`;
            html += `<div class="bookmarks-item-content">`;
            html += `<span class="bookmarks-item-title">${escapeHtml(bookmark.title)}</span>`;
            html += `<span class="bookmarks-item-url">${escapeHtml(bookmark.url)}</span>`;

            if (bookmark.tags && bookmark.tags.length > 0) {
                html += `<div class="bookmarks-item-tags">`;
                for (const tag of bookmark.tags) {
                    html += `<span class="bookmarks-tag">${escapeHtml(tag)}</span>`;
                }
                html += `</div>`;
            }

            if (bookmark.description) {
                html += `<span class="bookmarks-item-description">${escapeHtml(bookmark.description)}</span>`;
            }
            html += `</div>`;

            html += `<div class="bookmarks-item-actions">`;
            html += `<button class="bookmarks-btn bookmarks-btn-edit" data-bookmark-id="${bookmark.id}" title="Edit bookmark">✎</button>`;
            html += `<button class="bookmarks-btn bookmarks-btn-delete" data-bookmark-id="${bookmark.id}" title="Delete bookmark">×</button>`;
            html += `</div>`;

            html += `</div>`;
        }
    }

    bookmarksDisplay.innerHTML = html;
    bookmarksDisplay.classList.add('visible');

    // Wire up click handlers for bookmark content (open URL)
    const bookmarkContents = bookmarksDisplay.querySelectorAll('.bookmarks-item-content');
    bookmarkContents.forEach(content => {
        content.addEventListener('click', (e) => {
            const item = content.closest('.bookmarks-item');
            const bookmarkId = item.getAttribute('data-bookmark-id');
            handleBookmarkClick(bookmarkId);
        });
    });

    // Wire up edit button handlers
    const editButtons = bookmarksDisplay.querySelectorAll('.bookmarks-btn-edit');
    editButtons.forEach(btn => {
        btn.addEventListener('click', (e) => {
            e.stopPropagation();
            const bookmarkId = btn.getAttribute('data-bookmark-id');
            handleBookmarkEdit(bookmarkId);
        });
    });

    // Wire up delete button handlers
    const deleteButtons = bookmarksDisplay.querySelectorAll('.bookmarks-btn-delete');
    deleteButtons.forEach(btn => {
        btn.addEventListener('click', (e) => {
            e.stopPropagation();
            const bookmarkId = btn.getAttribute('data-bookmark-id');
            handleBookmarkDelete(bookmarkId);
        });
    });
}

/**
 * Populate tags list in help guide
 */
async function populateTagsList() {
    if (!bookmarksTagsList) return;

    const tags = await getAllTags();

    if (tags.length === 0) {
        bookmarksTagsList.innerHTML = '<span style="color: var(--text-subtle); font-size: 11px;">No tags yet</span>';
    } else {
        let html = '';
        for (const { tag, count } of tags) {
            html += `<div class="bookmarks-tag-filter" data-tag="${escapeHtml(tag)}">`;
            html += `${escapeHtml(tag)}<span class="bookmarks-tag-count">(${count})</span>`;
            html += `</div>`;
        }
        bookmarksTagsList.innerHTML = html;

        // Wire up tag filter clicks
        const tagFilters = bookmarksTagsList.querySelectorAll('.bookmarks-tag-filter');
        tagFilters.forEach(filter => {
            filter.addEventListener('click', () => {
                const tag = filter.getAttribute('data-tag');
                editor.value = `bookmarks:${tag}`;
                editor.dispatchEvent(new Event('input', { bubbles: true }));
            });
        });
    }
}

/**
 * Handle bookmark click - open in new tab
 */
async function handleBookmarkClick(bookmarkId) {
    try {
        const bookmark = await getBookmark(bookmarkId);
        if (!bookmark) return;

        // Update access time
        await updateAccessTime(bookmarkId);

        // Open in new tab (validate protocol)
        if (!isSafeUrl(bookmark.url)) return;
        chrome.tabs.create({ url: bookmark.url });
    } catch (err) {
        console.error('Failed to open bookmark:', err);
        showTemplateNotification('Failed to open bookmark', 2000);
    }
}

/**
 * Handle bookmark edit - load bookmark into editor
 */
async function handleBookmarkEdit(bookmarkId) {
    const bookmark = await getBookmark(bookmarkId);
    if (!bookmark) return;

    // Load bookmark into editor in edit format
    const tagsStr = bookmark.tags && bookmark.tags.length > 0 ? bookmark.tags.join(', ') : '';
    const descriptionStr = bookmark.description || '';

    editor.value = `bookmarks:edit:${bookmarkId}\nURL: ${bookmark.url}\nTags: ${tagsStr}\nDescription: ${descriptionStr}`;

    // Set editing state
    isEditingBookmark = true;
    editingBookmarkId = bookmarkId;

    // Trigger update
    editor.dispatchEvent(new Event('input', { bubbles: true }));
    editor.focus();
}

/**
 * Handle bookmark delete - confirm and delete
 */
async function handleBookmarkDelete(bookmarkId) {
    const bookmark = await getBookmark(bookmarkId);
    if (!bookmark) return;

    // Confirm deletion
    const confirmed = confirm(`Delete bookmark?\n\n${bookmark.title}\n${bookmark.url}`);
    if (!confirmed) return;

    // Delete the bookmark
    const success = await deleteBookmark(bookmarkId);

    if (success) {
        // Refresh the current view
        const content = editor.value;
        const { type, tag, query } = parseBookmarksCommand(content);

        if (type === 'search') {
            // Refresh search results
            const bookmarks = await searchBookmarks(query);
            await renderBookmarksDisplay(bookmarks, null, `Search: "${query}"`);
        } else if (type === 'filter') {
            // Refresh filtered results
            const bookmarks = await getBookmarks(tag);
            await renderBookmarksDisplay(bookmarks, tag);
        } else {
            // Go back to help/all bookmarks view
            editor.value = 'bookmarks';
            editor.dispatchEvent(new Event('input', { bubbles: true }));
        }
    }
}

/**
 * Handle bookmarks add mode with guided entry
 */
function handleBookmarksAdd() {
    const content = editor.value;
    const lines = content.split('\n');
    const { type } = parseBookmarksCommand(content);

    if (type !== 'add') {
        isAddingBookmark = false;
        bookmarksToSave = [];
        bookmarkEntryStep = 0;
        return;
    }

    // If just typed "bookmarks:add", initialize
    if (lines.length === 1 || !isAddingBookmark) {
        isAddingBookmark = true;
        bookmarkEntryStep = 0;
        currentBookmarkEntry = { url: '', tags: '', description: '' };
        return;
    }

    // Parse all lines to build bookmark entries
    parseBookmarkFields(lines);
}

/**
 * Parse bookmark fields from content lines
 */
function parseBookmarkFields(lines) {
    // Reset bookmarks to save
    bookmarksToSave = [];
    let currentEntry = { url: '', tags: '', description: '' };

    for (let i = 1; i < lines.length; i++) {
        const line = lines[i];

        if (line.startsWith('URL: ')) {
            currentEntry.url = line.substring(5).trim();
        } else if (line.startsWith('Tags: ')) {
            currentEntry.tags = line.substring(6).trim();
        } else if (line.startsWith('Description: ')) {
            currentEntry.description = line.substring(13).trim();

            // If we have a URL, save this entry
            if (currentEntry.url) {
                bookmarksToSave.push({ ...currentEntry });
                currentEntry = { url: '', tags: '', description: '' };
            }
        }
    }

    // Store current entry for next field
    currentBookmarkEntry = currentEntry;
}

/**
 * Insert bookmark template for guided entry
 */
function insertBookmarkTemplate() {
    const cursorPos = editor.selectionStart;
    const content = editor.value;

    let template = '';
    if (bookmarkEntryStep === 0) {
        template = 'URL: ';
        bookmarkEntryStep = 1;
    } else if (bookmarkEntryStep === 1) {
        template = 'Tags: ';
        bookmarkEntryStep = 2;
    } else if (bookmarkEntryStep === 2) {
        template = 'Description: ';
        bookmarkEntryStep = 0; // Reset for next bookmark
    }

    // Insert template at cursor
    const before = content.substring(0, cursorPos);
    const after = content.substring(cursorPos);
    editor.value = before + template + after;
    editor.selectionStart = editor.selectionEnd = cursorPos + template.length;

    // Trigger update
    editor.dispatchEvent(new Event('input', { bubbles: true }));
}

/**
 * Save all bookmarks from add mode or edit mode
 */
async function saveAllBookmarks() {
    const content = editor.value;
    const lines = content.split('\n');
    const { type, bookmarkId: parsedBookmarkId } = parseBookmarksCommand(content);

    // Handle edit mode - check both flag and parsed command
    const editId = isEditingBookmark && editingBookmarkId ? editingBookmarkId : (type === 'edit' ? parsedBookmarkId : null);

    if (editId) {
        const bookmark = await getBookmark(editId);
        if (!bookmark) {
            console.error('Bookmark not found for editing:', editId);
            isEditingBookmark = false;
            editingBookmarkId = null;
            return;
        }

        // Parse edited fields
        let url = bookmark.url;
        let tags = bookmark.tags || [];
        let description = bookmark.description || '';

        for (let i = 1; i < lines.length; i++) {
            const line = lines[i];
            if (line.startsWith('URL: ')) {
                url = line.substring(5).trim();
            } else if (line.startsWith('Tags: ')) {
                const tagsStr = line.substring(6).trim();
                tags = tagsStr ? tagsStr.split(',').map(t => t.trim()).filter(t => t) : [];
            } else if (line.startsWith('Description: ')) {
                description = line.substring(13).trim();
            }
        }

        // Update bookmark - keep the original ID!
        bookmark.url = url;
        bookmark.title = extractTitleFromUrl(url);
        bookmark.tags = tags;
        bookmark.description = description;
        bookmark.modifiedAt = new Date().toISOString();

        try {
            await saveBookmark(bookmark);
        } catch (err) {
            console.error('Failed to save bookmark:', err);
            showTemplateNotification('Failed to save bookmark', 2000);
            return;
        }

        // Clear editor and reset state
        editor.value = '';
        isEditingBookmark = false;
        editingBookmarkId = null;
        editor.dispatchEvent(new Event('input', { bubbles: true }));

        showTemplateNotification('Bookmark updated', 2000);
        return;
    }

    // Handle add mode - create new bookmarks
    const bookmarksToSaveNow = [];
    let currentEntry = { url: '', tags: '', description: '' };

    for (let i = 1; i < lines.length; i++) {
        const line = lines[i];
        const trimmedLine = line.trim().toLowerCase();

        // Stop at bookmarks:end or bookmarks:save
        if (trimmedLine === 'bookmarks:end' || trimmedLine === 'bookmarks:save') {
            // Save current entry if it has a URL
            if (currentEntry.url) {
                bookmarksToSaveNow.push({ ...currentEntry });
            }
            break;
        }

        if (line.startsWith('URL: ')) {
            // Save previous entry if exists
            if (currentEntry.url) {
                bookmarksToSaveNow.push({ ...currentEntry });
            }
            // Start new entry
            currentEntry = {
                url: line.substring(5).trim(),
                tags: '',
                description: ''
            };
        } else if (line.startsWith('Tags: ')) {
            currentEntry.tags = line.substring(6).trim();
        } else if (line.startsWith('Description: ')) {
            currentEntry.description = line.substring(13).trim();
        }
    }

    // Save the last entry if it has a URL
    if (currentEntry.url) {
        bookmarksToSaveNow.push({ ...currentEntry });
    }

    if (bookmarksToSaveNow.length === 0) {
        // No bookmarks to save - just clear
        editor.value = '';
        isAddingBookmark = false;
        bookmarkEntryStep = 0;
        bookmarksToSave = [];
        editor.dispatchEvent(new Event('input', { bubbles: true }));
        return;
    }

    let savedCount = 0;
    for (const entry of bookmarksToSaveNow) {
        if (!entry.url) continue;

        const bookmark = {
            url: entry.url,
            title: extractTitleFromUrl(entry.url),
            tags: entry.tags ? entry.tags.split(',').map(t => t.trim()).filter(t => t) : [],
            description: entry.description || '',
            createdAt: new Date().toISOString(),
            modifiedAt: new Date().toISOString(),
            accessedAt: new Date().toISOString(),
            chromeBookmarkId: null
        };

        try {
            await saveBookmark(bookmark);
            savedCount++;
        } catch (err) {
            console.error('Failed to save bookmark:', err);
        }
    }

    // Clear editor and reset state
    editor.value = '';
    isAddingBookmark = false;
    bookmarksToSave = [];
    bookmarkEntryStep = 0;

    // Show notification
    showTemplateNotification(`${savedCount} bookmark(s) saved`, 2000);

    // Trigger update
    editor.dispatchEvent(new Event('input', { bubbles: true }));
}

// ============================================================================
// SESSION MANAGEMENT
// ============================================================================

/**
 * Update sessions mode display based on parsed command
 */
async function updateSessionsMode() {
    const content = editor.value;
    const { type, tag, query, sessionId, hasSessionCommand } = parseSessionCommand(content);

    // Close edit panel if user starts typing something else
    if (sessionEditPanel && sessionEditPanel.classList.contains('visible') &&
        (!hasSessionCommand || (type !== 'edit' || sessionId !== currentEditingSessionId))) {
        closeSessionEditPanel();
    }

    if (!hasSessionCommand) {
        sessionsDisplay.classList.remove('visible');
        sessionsHelp.classList.remove('visible');
        return;
    }

    if (!isPro()) { showProUpgradeHint('session'); return; }

    // Hide sessions display and help initially
    sessionsDisplay.classList.remove('visible');
    sessionsHelp.classList.remove('visible');

    // Handle different command types
    if (type === 'help') {
        // Show help
        sessionsHelp.classList.add('visible');
        return;
    }

    if (type === 'add') {
        // Show add form
        handleSessionsAdd();
        return;
    }

    if (type === 'end') {
        // Save session from editor
        await saveSessionFromEditor();
        return;
    }

    if (type === 'import') {
        triggerSessionsImport();
        return;
    }

    if (type === 'edit') {
        // Edit existing session
        await editSession(sessionId);
        return;
    }

    // For filter, search, or default (show all)
    let sessions = await getSessions();

    // Apply filtering
    if (type === 'filter' && tag) {
        sessions = sessions.filter(s =>
            s.tags && s.tags.some(t => t.toLowerCase() === tag.toLowerCase())
        );
    } else if (type === 'search' && query) {
        sessions = await searchSessions(query);
    }

    // Render sessions
    renderSessionsDisplay(sessions);
}

/**
 * Render sessions display
 */
function renderSessionsDisplay(sessions) {
    if (!sessions || sessions.length === 0) {
        sessionsDisplay.innerHTML = '<div class="sessions-empty">No sessions found.</div>';
        sessionsDisplay.classList.add('visible');
        return;
    }

    let html = '';

    sessions.forEach(session => {
        const tagsHtml = session.tags && session.tags.length > 0
            ? session.tags.map(tag => `<span class="sessions-tag">${escapeHtml(tag)}</span>`).join('')
            : '';

        const dateStr = session.dateCreated
            ? new Date(session.dateCreated).toLocaleDateString()
            : '';

        const urlsHtml = session.urls && session.urls.length > 0
            ? session.urls.map(item => {
                // Handle both old format (string) and new format (object with url and title)
                const urlString = typeof item === 'string' ? item : item.url;
                const urlTitle = typeof item === 'object' && item.title ? item.title : null;

                let displayUrl = urlTitle || urlString;
                try {
                    const urlObj = new URL(urlString);
                    if (!urlTitle) {
                        displayUrl = urlObj.hostname + urlObj.pathname;
                    }
                    if (displayUrl.length > 50) {
                        displayUrl = displayUrl.substring(0, 47) + '...';
                    }
                } catch (e) {
                    // If URL parsing fails, truncate as-is
                    if (displayUrl.length > 50) {
                        displayUrl = displayUrl.substring(0, 47) + '...';
                    }
                }
                return `<div class="sessions-url-item" data-url="${escapeHtml(urlString)}" title="${escapeHtml(urlString)}">${escapeHtml(displayUrl)}</div>`;
            }).join('')
            : '<div class="sessions-url-item sessions-no-urls">No URLs in this session</div>';

        html += `
            <div class="sessions-item" data-session-id="${session.id}">
                <div class="sessions-item-header">
                    <div class="sessions-item-title">${escapeHtml(session.title || 'Untitled Session')}</div>
                    <div class="sessions-item-actions">
                        <button class="sessions-btn sessions-btn-open" title="Open all tabs">Open All</button>
                        <button class="sessions-btn sessions-btn-edit" title="Edit session">Edit</button>
                        <button class="sessions-btn sessions-btn-delete" title="Delete session">Delete</button>
                    </div>
                </div>
                <div class="sessions-item-meta">
                    <span class="sessions-date">${dateStr}</span>
                    ${tagsHtml}
                </div>
                <div class="sessions-urls">
                    ${urlsHtml}
                </div>
            </div>
        `;
    });

    sessionsDisplay.innerHTML = html;
    sessionsDisplay.classList.add('visible');

    // Add event listeners
    sessionsDisplay.querySelectorAll('.sessions-btn-open').forEach(btn => {
        btn.addEventListener('click', async (e) => {
            e.stopPropagation();
            const sessionId = btn.closest('.sessions-item').dataset.sessionId;
            try {
                await openSession(sessionId);
                showTemplateNotification('Session tabs opened', 2000);
            } catch (err) {
                console.error('Failed to open session:', err);
                showTemplateNotification('Failed to open session', 2000);
            }
        });
    });

    sessionsDisplay.querySelectorAll('.sessions-btn-edit').forEach(btn => {
        btn.addEventListener('click', async (e) => {
            e.stopPropagation();
            const sessionId = btn.closest('.sessions-item').dataset.sessionId;
            try {
                await editSession(sessionId);
            } catch (err) {
                console.error('Failed to edit session:', err);
                showTemplateNotification('Failed to load session', 2000);
            }
        });
    });

    sessionsDisplay.querySelectorAll('.sessions-btn-delete').forEach(btn => {
        btn.addEventListener('click', async (e) => {
            e.stopPropagation();
            const sessionId = btn.closest('.sessions-item').dataset.sessionId;
            const sessions = await getSessions();
            const session = sessions.find(s => s.id === sessionId);

            if (confirm(`Delete session "${session?.title || 'Untitled'}"?`)) {
                try {
                    await deleteSession(sessionId);
                    showTemplateNotification('Session deleted', 2000);
                    await updateSessionsMode();
                } catch (err) {
                    console.error('Failed to delete session:', err);
                    showTemplateNotification('Failed to delete session', 2000);
                }
            }
        });
    });

    // Add click handlers for individual URLs
    sessionsDisplay.querySelectorAll('.sessions-url-item[data-url]').forEach(urlItem => {
        urlItem.addEventListener('click', (e) => {
            e.stopPropagation();
            const url = urlItem.dataset.url;
            if (url && isSafeUrl(url)) {
                chrome.tabs.create({ url });
            }
        });
    });
}

/**
 * Handle adding new session (session:add command)
 */
function handleSessionsAdd() {
    const content = editor.value;
    const lines = content.split('\n');
    const { type } = parseSessionCommand(content);

    if (type !== 'add') {
        return;
    }

    // Only expand template if user pressed Enter (content has newline after "session:add")
    // AND the template hasn't been expanded yet (check if "Title:" exists)
    const firstLine = lines[0] || '';
    const hasTemplate = content.includes('Title:') && content.includes('Tags:');

    // Only expand if: first line is "session:add", there are multiple lines,
    // second line is empty (user just pressed Enter), and template not yet expanded
    const secondLine = lines[1] || '';
    const justPressedEnter = lines.length === 2 && secondLine.trim() === '';

    if (justPressedEnter && !hasTemplate && firstLine.trim().toLowerCase() === 'session:add') {
        const template = `session:add

Title:
Tags:
(tabs will be captured when you type "session:end")`;

        editor.value = template;
        editor.focus();

        // Position cursor right after "Title:" (before the newline)
        const titleLineStart = template.indexOf('Title:');
        if (titleLineStart !== -1) {
            // +6 for "Title:" length puts cursor right after the colon
            editor.setSelectionRange(titleLineStart + 6, titleLineStart + 6);
        }
    }
}

/**
 * Save session from editor content (session:end command)
 */
async function saveSessionFromEditor() {
    const content = editor.value;
    const lines = content.split('\n');
    const firstLine = lines[0] || '';
    const trimmedFirst = firstLine.trim().toLowerCase();

    let title = '';
    let tags = [];
    let urls = [];
    let sessionId = null;

    // Check if we're editing an existing session
    const isEditing = trimmedFirst.startsWith('session:edit:');
    if (isEditing) {
        sessionId = trimmedFirst.substring(13).trim();
    }

    // Parse fields from content
    for (const line of lines) {
        const lower = line.toLowerCase().trim();
        if (lower.startsWith('title:')) {
            title = line.substring(line.indexOf(':') + 1).trim();
        } else if (lower.startsWith('tags:')) {
            const tagsStr = line.substring(line.indexOf(':') + 1).trim();
            tags = tagsStr.split(',')
                .map(t => t.trim())
                .filter(t => t.length > 0);
        } else if (lower.startsWith('url:')) {
            const url = line.substring(line.indexOf(':') + 1).trim();
            if (url) {
                urls.push({ url, title: '' });
            }
        }
    }

    try {
        if (isEditing && sessionId) {
            // Update existing session
            const sessions = await getSessions();
            const existingSession = sessions.find(s => s.id === sessionId);

            if (!existingSession) {
                showTemplateNotification('Session not found', 2000);
                return;
            }

            // Update session with new values
            const updatedSession = {
                ...existingSession,
                title: title || existingSession.title,
                tags: tags.length > 0 ? tags : existingSession.tags,
                urls: urls.length > 0 ? urls : existingSession.urls,
                dateModified: new Date().toISOString()
            };

            await saveSession(updatedSession);
            showTemplateNotification('Session updated', 2000);
        } else {
            // Create new session from current browser tabs
            await saveCurrentSession(title, tags);
            showTemplateNotification('Session saved', 2000);
        }

        // Clear editor
        editor.value = '';

        // Trigger update
        editor.dispatchEvent(new Event('input', { bubbles: true }));
    } catch (error) {
        console.error('Failed to save session:', error);
        showTemplateNotification('Failed to save session', 3000);
    }
}

/**
 * Parse session fields from editor content
 */
function parseSessionFields(content) {
    const lines = content.split('\n');
    let title = '';
    let tags = [];

    for (const line of lines) {
        const lower = line.toLowerCase().trim();
        if (lower.startsWith('title:')) {
            title = line.substring(line.indexOf(':') + 1).trim();
        } else if (lower.startsWith('tags:')) {
            const tagsStr = line.substring(line.indexOf(':') + 1).trim();
            tags = tagsStr.split(',')
                .map(t => t.trim())
                .filter(t => t.length > 0);
        }
    }

    return { title, tags };
}

/**
 * Edit existing session - opens edit panel
 */
async function editSession(sessionId) {
    const sessions = await getSessions();
    const session = sessions.find(s => s.id === sessionId);

    if (!session) {
        showTemplateNotification('Session not found', 2000);
        return;
    }

    // Store current editing session ID
    currentEditingSessionId = sessionId;

    // Populate form fields
    sessionEditTitle.value = session.title || '';
    sessionEditTags.value = session.tags ? session.tags.join(', ') : '';

    // Populate URLs list with checkboxes
    renderSessionEditUrls(session.urls || []);

    // Show the panel
    sessionEditPanel.classList.add('visible');
    sessionEditTitle.focus();
}

/**
 * Render URLs list in edit panel with checkboxes
 */
function renderSessionEditUrls(urls) {
    if (!urls || urls.length === 0) {
        sessionEditUrlsList.innerHTML = '<div style="padding: 20px; text-align: center; color: var(--text-subtle);">No URLs in this session</div>';
        return;
    }

    let html = '';
    urls.forEach((item, index) => {
        const urlString = typeof item === 'string' ? item : item.url;
        const urlTitle = typeof item === 'object' && item.title ? item.title : null;

        html += `
            <div class="session-edit-url-item" data-index="${index}">
                <input type="checkbox" class="session-edit-url-checkbox" data-index="${index}">
                <div class="session-edit-url-text">
                    ${urlTitle ? `<span class="session-edit-url-title">${escapeHtml(urlTitle)}</span>` : ''}
                    <span class="session-edit-url-link">${escapeHtml(urlString)}</span>
                </div>
            </div>
        `;
    });

    sessionEditUrlsList.innerHTML = html;

    // Add checkbox change listeners
    sessionEditUrlsList.querySelectorAll('.session-edit-url-checkbox').forEach(checkbox => {
        checkbox.addEventListener('change', updateDeleteButtonVisibility);
    });
}

/**
 * Update delete button visibility based on checkbox selection
 */
function updateDeleteButtonVisibility() {
    const checkedBoxes = sessionEditUrlsList.querySelectorAll('.session-edit-url-checkbox:checked');
    if (checkedBoxes.length > 0) {
        sessionDeleteSelectedBtn.style.display = 'flex';
        sessionDeleteSelectedBtn.textContent = `🗑️ Delete Selected (${checkedBoxes.length})`;
    } else {
        sessionDeleteSelectedBtn.style.display = 'none';
    }
}

/**
 * Close session edit panel
 */
function closeSessionEditPanel() {
    sessionEditPanel.classList.remove('visible');
    currentEditingSessionId = null;
    sessionEditTitle.value = '';
    sessionEditTags.value = '';
    sessionEditUrlsList.innerHTML = '';
    sessionDeleteSelectedBtn.style.display = 'none';
}

/**
 * Save session edit changes
 */
async function saveSessionEdit() {
    if (!currentEditingSessionId) return;

    const sessions = await getSessions();
    const session = sessions.find(s => s.id === currentEditingSessionId);

    if (!session) {
        showTemplateNotification('Session not found', 2000);
        closeSessionEditPanel();
        return;
    }

    // Get updated values
    const title = sessionEditTitle.value.trim();
    const tagsStr = sessionEditTags.value.trim();
    const tags = tagsStr ? tagsStr.split(',').map(t => t.trim()).filter(t => t) : [];

    // Get URLs (excluding deleted ones)
    const checkboxes = sessionEditUrlsList.querySelectorAll('.session-edit-url-checkbox');
    const urls = [];
    checkboxes.forEach((checkbox, index) => {
        if (!checkbox.checked) { // Keep unchecked URLs
            if (session.urls && session.urls[index]) {
                urls.push(session.urls[index]);
            }
        }
    });

    // Update session
    const updatedSession = {
        ...session,
        title: title || session.title,
        tags: tags,
        urls: urls,
        dateModified: new Date().toISOString()
    };

    try {
        await saveSession(updatedSession);
        showTemplateNotification('Session updated', 2000);
    } catch (err) {
        console.error('Failed to update session:', err);
        showTemplateNotification('Failed to update session', 2000);
    }

    // Close panel and refresh display
    closeSessionEditPanel();
    await updateSessionsMode();
}

/**
 * Populate sessions tags list in settings
 */
async function populateSessionsTagsList() {
    const sessions = await getSessions();
    const tagsSet = new Set();

    sessions.forEach(session => {
        if (session.tags) {
            session.tags.forEach(tag => tagsSet.add(tag));
        }
    });

    return Array.from(tagsSet).sort();
}

/**
 * Export sessions to JSON file
 */
async function exportSessions() {
    try {
        const sessions = await getSessions();
        const jsonData = exportSessionsToJSON(sessions);
        const blob = new Blob([jsonData], { type: 'application/json' });
        const url = URL.createObjectURL(blob);

        const a = document.createElement('a');
        a.href = url;
        a.download = `jottnote-sessions-${Date.now()}.json`;
        a.click();

        URL.revokeObjectURL(url);
        showTemplateNotification('Sessions exported', 2000);
    } catch (error) {
        console.error('Export failed:', error);
        showTemplateNotification('Export failed', 2000);
    }
}

/**
 * Trigger sessions import from JSON file
 */
async function triggerSessionsImport() {
    try {
        showTemplateNotification('Saving current session...', 2000);
        await saveCurrentSession('', []);
        showTemplateNotification('Session saved', 3000);
    } catch (err) {
        showTemplateNotification(`Save failed: ${err.message}`, 3000);
    }
    // Refresh to session view
    editor.value = 'session';
    editor.dispatchEvent(new Event('input', { bubbles: true }));
}

/**
 * Import sessions from ZIP file
 */
async function importSessionsFromZip(file) {
    try {
        const zip = await JSZip.loadAsync(file);
        const sessionsFile = zip.file('sessions.json');

        if (!sessionsFile) {
            showTemplateNotification('No sessions.json found in ZIP', 3000);
            return;
        }

        const jsonData = await sessionsFile.async('string');
        const result = await importSessionsFromJSON(jsonData);

        showTemplateNotification(`Imported ${result.imported} session(s)`, 2000);
    } catch (error) {
        console.error('Import failed:', error);
        showTemplateNotification('Import failed', 2000);
    }
}

/**
 * Reset all keyword modes and overlays to a clean state
 * Called when editor content is fully cleared
 */
function resetAllModes() {
    const editorArea = document.querySelector('.editor-area');

    // Remove mode classes from editor area
    if (editorArea) {
        editorArea.classList.remove('code-mode', 'list-mode');
    }
    editor.style.paddingLeft = '';
    listBreakIndex = Infinity;
    restoreNonCodeEditors();

    // Restore editor highlighter
    if (editorHighlighter) {
        editorHighlighter.classList.remove('hidden');
        editorHighlighter.innerHTML = '';
    }

    // Reset all mode state flags
    isCreatingTemplate = false;
    isCreatingPlaceholders = false;
    isLoadingTemplate = false;
    isSortingMode = false;
    isAddingBookmark = false;
    isEditingBookmark = false;
    editingBookmarkId = null;
    currentMode = null;

    // Hide slash command popup
    if (slashPopup) slashPopup.classList.remove('visible');

    // Hide all help guides
    if (listHelp) listHelp.classList.remove('visible');
    if (sumHelp) sumHelp.classList.remove('visible');
    if (avgHelp) avgHelp.classList.remove('visible');
    if (countHelp) countHelp.classList.remove('visible');
    if (mathHelp) mathHelp.classList.remove('visible');
    if (codeHelp) codeHelp.classList.remove('visible');
    if (codeLanguagesHelp) codeLanguagesHelp.classList.remove('visible');
    if (timerHelp) timerHelp.classList.remove('visible');
    if (currencyHelp) currencyHelp.classList.remove('visible');
    if (templateHelp) templateHelp.classList.remove('visible');
    if (placeholdersHelp) placeholdersHelp.classList.remove('visible');
    if (sortHelp) sortHelp.classList.remove('visible');
    if (bookmarksHelp) bookmarksHelp.classList.remove('visible');
    if (sessionsHelp) sessionsHelp.classList.remove('visible');
    if (gameHelp) gameHelp.classList.remove('visible');
    if (dateHelp) dateHelp.classList.remove('visible');
    if (uuidHelp) uuidHelp.classList.remove('visible');
    if (loremHelp) loremHelp.classList.remove('visible');
    if (weatherHelp) weatherHelp.classList.remove('visible');

    // Hide all display overlays
    if (listDisplay) listDisplay.classList.remove('visible');
    if (statsDisplay) statsDisplay.classList.remove('visible');
    if (countDisplay) countDisplay.classList.remove('visible');
    if (mathDisplay) mathDisplay.classList.remove('visible');
    if (gameDisplay) gameDisplay.classList.remove('active');
    if (dateDisplay) { dateDisplay.classList.remove('visible'); dateDisplay.innerHTML = ''; }
    if (bookmarksDisplay) bookmarksDisplay.classList.remove('visible');
    if (sessionsDisplay) sessionsDisplay.classList.remove('visible');
    if (uuidDisplay) uuidDisplay.classList.remove('visible');
    if (weatherDisplay) weatherDisplay.classList.remove('visible');
    hideWeatherBg();

    // Clear code mode state
    if (codeEditable) codeEditable.textContent = '';

    // Ensure textarea value is truly empty so placeholder shows
    // (whitespace-only content still hides the HTML placeholder attribute)
    if (editor && !editor.value.trim()) {
        editor.value = '';
    }

    // Reset keyword indicator
    updateKeywordIndicator();
}

/**
 * Force update all UI modes and helpers
 * Clears blocking states and ensures overlays match content
 */
function forceUpdateUI() {
    isLoadingTemplate = false;

    // 1. Force cleanup of previous states
    const editorArea = document.querySelector('.editor-area');
    if (editorArea) {
        editorArea.classList.remove('code-mode', 'list-mode');
    }
    editor.style.paddingLeft = '';
    listBreakIndex = Infinity;
    restoreNonCodeEditors();
    // Clear highlighter content and keep hidden until modes are re-evaluated
    if (editorHighlighter) {
        editorHighlighter.innerHTML = '';
    }

    // 2. Ensure cursor is at start
    if (editor) {
        editor.scrollTop = 0;
        // Only set cursor if it's not already at start (avoids fighting user)
        // usage in loadTemplate handles initial positioning
    }

    // 3. Update all modes
    try {
        updateKeywordIndicator();
        updateCodeMode();
        updateListMode();
        updateStatsMode();
        updateMathMode(editor.value);
        updateTimerHelp();
        updateCurrencyHelp();
        updateTemplateHelp();
        updateSortHelp();
        updateEditorHighlighter();

        // 4. Save content
        if (editor) {
            handleContentChange(editor, () => { });
        }
    } catch (e) {
        console.error('Error in forceUpdateUI:', e);
    }
}

/**
 * Show template notification (error or success)
 */
function showTemplateNotification(message, duration = 2000) {
    // Create notification element if it doesn't exist
    let notification = document.getElementById('template-notification');

    if (!notification) {
        notification = document.createElement('div');
        notification.id = 'template-notification';
        notification.className = 'notification-popup';
        document.querySelector('.jottnote-container').appendChild(notification);
    }

    notification.textContent = message;
    notification.classList.add('visible');

    // Hide after duration
    setTimeout(() => {
        notification.classList.remove('visible');
    }, duration);
}

/**
 * Update editor highlighter
 * Colors keywords in red using overlay method
 */
function updateEditorHighlighter() {
    if (!editorHighlighter) return;

    // Skip if find panel is visible - find highlighting takes priority
    if (findReplacePanel && findReplacePanel.classList.contains('visible')) {
        return;
    }

    // Skip if code mode is active - code editor handles its own highlighting
    // Actively clear highlighter to prevent ghost text overlay on scroll
    const editorArea = document.querySelector('.editor-area');
    if (editorArea && editorArea.classList.contains('code-mode')) {
        editorHighlighter.innerHTML = '';
        editorHighlighter.classList.add('hidden');
        editorHighlighter.style.display = 'none';
        return;
    }

    // Skip if list mode is active - list-display handles its own rendering
    if (editorArea && editorArea.classList.contains('list-mode')) {
        editorHighlighter.innerHTML = '';
        editorHighlighter.classList.add('hidden');
        editorHighlighter.style.display = 'none';
        return;
    }

    // Skip if bookmarks display is visible - bookmarks handles its own display
    if (bookmarksDisplay && bookmarksDisplay.classList.contains('visible')) {
        return;
    }

    const content = editor.value;
    const { keyword, mode, hasKeyword } = parseKeyword(content);

    // Use escapeHtmlForHighlight to match code mode behavior and prevent misalignment
    let html = escapeHtmlForHighlight(content);

    // If we have a keyword on the first line, highlight it
    if (hasKeyword && keyword) {
        const firstLineIdx = content.indexOf('\n');
        const firstLine = firstLineIdx === -1 ? content : content.substring(0, firstLineIdx);

        // Find the keyword in the first line
        const escapedKey = keyword.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
        const match = firstLine.match(new RegExp(`\\b${escapedKey}\\b`, 'i'));

        if (match) {
            const start = match.index;
            const end = start + match[0].length;

            const before = escapeHtmlForHighlight(firstLine.substring(0, start));
            const keyText = escapeHtmlForHighlight(firstLine.substring(start, end));
            const after = escapeHtmlForHighlight(firstLine.substring(end));

            const accentClass = getKeywordAccentClass(mode);

            // Reconstruct first line with highlighted keyword using specific class
            // Note: We reconstruct the whole HTML to ensure partial replacement works
            const hlLine = `${before}<span class="keyword-highlight ${accentClass}">${keyText}</span>${after}`;

            const rest = firstLineIdx === -1 ? '' : escapeHtmlForHighlight(content.substring(firstLineIdx));
            html = hlLine + rest;
        }
    }

    // Apply tag highlighting (before spell check so tagged words aren't underlined)
    html = applyTagHighlighting(html);

    // Apply spell check if enabled (with safe integration and error handling)
    if (typeof isSpellCheckEnabled === 'function' && isSpellCheckEnabled()) {
        try {
            if (typeof applySpellCheckToHtml === 'function') {
                html = applySpellCheckToHtml(html, content, hasKeyword);
            }
        } catch (err) {
            console.error('Spell check highlighting failed:', err);
            // Continue without spell check - don't break the UI
        }
    }

    // Add a trailing break if the content ends with a newline
    if (content.endsWith('\n')) {
        html += '<br>';
    }

    editorHighlighter.innerHTML = html;
    editorHighlighter.classList.remove('hidden');
    editorHighlighter.style.display = '';

    // Sync scroll
    editorHighlighter.scrollTop = editor.scrollTop;
    editorHighlighter.scrollLeft = editor.scrollLeft;
}

/**
 * Apply spell check highlighting to HTML content
 * This wraps misspelled words with spell-error spans
 */
function applySpellCheckToHtml(html, plainContent, hasKeyword) {
    // Find misspelled words in the content
    const misspelled = findMisspelledWords(plainContent);
    if (misspelled.length === 0) return html;

    // Skip first line if we have a keyword
    const firstLineEnd = plainContent.indexOf('\n');
    const filteredMisspelled = hasKeyword && firstLineEnd !== -1
        ? misspelled.filter(m => m.start >= firstLineEnd)
        : misspelled;

    if (filteredMisspelled.length === 0) return html;

    // Build a map of words to highlight
    const wordsToHighlight = new Set(filteredMisspelled.map(m => m.word.toLowerCase()));

    // Apply spell check highlighting using regex
    // Match whole words that are in our misspelled set
    let result = html;
    wordsToHighlight.forEach(word => {
        // Escape special regex characters in the word
        const escapedWord = word.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
        // Match the word with proper boundaries, case-insensitive
        const regex = new RegExp(`\\b(${escapedWord})\\b(?![^<]*>)`, 'gi');
        result = result.replace(regex, '<span class="spell-error" data-word="$1">$1</span>');
    });

    return result;
}

/**
 * Setup spell check context menu
 */
function setupSpellCheck() {
    if (!editorHighlighter || !spellCheckContextMenu) {
        console.warn('Spell check elements not found, skipping setup');
        return;
    }

    // Click on spell error to show suggestions menu
    editorHighlighter.addEventListener('click', (e) => {
        // Only show menu if spell check is enabled
        if (!isSpellCheckEnabled()) return;

        // Check if the clicked element is a spell error
        const target = e.target;
        if (!target.classList || !target.classList.contains('spell-error')) {
            hideSpellCheckMenu();
            return;
        }

        e.preventDefault();
        e.stopPropagation();

        const word = target.getAttribute('data-word') || target.textContent;
        const rect = target.getBoundingClientRect();

        // Position menu below the clicked word
        showSpellCheckMenu(word, rect.left, rect.bottom + 4);
    });

    // Hide menu when clicking elsewhere
    document.addEventListener('click', (e) => {
        if (spellCheckContextMenu && !spellCheckContextMenu.contains(e.target)) {
            hideSpellCheckMenu();
        }
    });
}

/**
 * Show spell check context menu with suggestions
 */
function showSpellCheckMenu(misspelledWord, x, y) {
    if (!spellCheckContextMenu) return;

    const suggestions = getSpellingSuggestions(misspelledWord);

    // Build menu content
    let menuHtml = `<div class="spell-check-word">${escapeHtml(misspelledWord)}</div>`;

    if (suggestions.length > 0) {
        menuHtml += '<div class="spell-check-suggestions">';
        suggestions.forEach(suggestion => {
            menuHtml += `<div class="spell-suggestion" data-word="${escapeHtml(misspelledWord)}" data-suggestion="${escapeHtml(suggestion)}">${escapeHtml(suggestion)}</div>`;
        });
        menuHtml += '</div>';
    } else {
        menuHtml += '<div class="spell-check-no-suggestions">No suggestions</div>';
    }

    spellCheckContextMenu.innerHTML = menuHtml;
    spellCheckContextMenu.classList.add('visible');

    // Position menu
    positionContextMenu(spellCheckContextMenu, x, y);

}

/**
 * Hide spell check context menu
 */
function hideSpellCheckMenu() {
    if (spellCheckContextMenu) {
        spellCheckContextMenu.classList.remove('visible');
    }
}

/**
 * Position context menu near click, ensuring it stays in viewport
 */
function positionContextMenu(menu, x, y) {
    // Start with click position
    menu.style.left = `${x}px`;
    menu.style.top = `${y}px`;

    // Get menu dimensions after showing it
    const rect = menu.getBoundingClientRect();
    const viewportWidth = window.innerWidth;
    const viewportHeight = window.innerHeight;

    // Adjust horizontal position if menu goes off-screen
    if (rect.right > viewportWidth) {
        menu.style.left = `${viewportWidth - rect.width - 10}px`;
    }

    // Adjust vertical position if menu goes off-screen
    if (rect.bottom > viewportHeight) {
        menu.style.top = `${viewportHeight - rect.height - 10}px`;
    }
}

/**
 * Replace a misspelled word with a suggestion in the editor
 */
function replaceWord(originalWord, replacement) {
    if (!editor) return;

    const content = editor.value;

    // Use a regex to replace the first occurrence of the word (case-insensitive, whole word)
    const escapedWord = originalWord.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
    const regex = new RegExp(`\\b${escapedWord}\\b`, 'i');

    const newContent = content.replace(regex, replacement);

    if (newContent !== content) {
        editor.value = newContent;
        handleContentChange(editor, () => { });
        updateEditorHighlighter();
    }
}

/**
 * Focus the active editor (code textarea in code mode, main editor otherwise)
 */
function focusActiveEditor() {
    const editorArea = document.querySelector('.editor-area');
    if (editorArea && editorArea.classList.contains('code-mode')) {
        // Defer focus to next frame — Chrome can't focus a contenteditable
        // element the same frame it transitions from display:none
        requestAnimationFrame(() => {
            codeEditable.focus();
            // Ensure caret is visible
            if (!window.getSelection().rangeCount) {
                setCodeCaretToEnd();
            }
        });
    } else {
        editor.focus();
    }
}

/**
 * Escape HTML for display
 */
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

/**
 * Check if a URL uses a safe protocol (http or https)
 * @param {string} url
 * @returns {boolean}
 */
function isSafeUrl(url) {
    try {
        const parsed = new URL(url);
        return parsed.protocol === 'http:' || parsed.protocol === 'https:';
    } catch {
        return false;
    }
}

/**
 * Get plain text from the contenteditable code editor
 * Handles both text nodes (\n) and <br> elements
 */
function getCodeText() {
    if (!codeEditable) return '';
    // Walk DOM tree to extract text — avoids phantom newline issues with innerText
    let text = '';
    let firstBlock = true;
    function walk(node) {
        if (node.nodeType === Node.TEXT_NODE) {
            text += node.textContent;
        } else if (node.nodeName === 'BR') {
            if (node.classList && node.classList.contains('cursor-helper')) return;
            text += '\n';
        } else if ((node.nodeName === 'DIV' || node.nodeName === 'P') && node.parentNode === codeEditable) {
            // Block elements Chrome may create — add newline separator
            if (!firstBlock) text += '\n';
            firstBlock = false;
            for (const child of node.childNodes) {
                walk(child);
            }
        } else {
            for (const child of node.childNodes) {
                walk(child);
            }
        }
    }
    walk(codeEditable);
    return text;
}

/**
 * Get caret offset as character count from start of contenteditable
 */
function getCodeCaretOffset() {
    const sel = window.getSelection();
    if (!sel.rangeCount || !codeEditable) return 0;

    const range = sel.getRangeAt(0);
    const preRange = document.createRange();
    preRange.selectNodeContents(codeEditable);
    preRange.setEnd(range.startContainer, range.startOffset);

    // Walk cloned fragment to count characters
    const frag = preRange.cloneContents();
    let offset = 0;
    let firstBlock = true;
    function walk(node, isTopLevel) {
        if (node.nodeType === Node.TEXT_NODE) {
            offset += node.textContent.length;
        } else if (node.nodeName === 'BR') {
            if (node.classList && node.classList.contains('cursor-helper')) return;
            offset += 1;
        } else if ((node.nodeName === 'DIV' || node.nodeName === 'P') && isTopLevel) {
            if (!firstBlock) offset += 1;
            firstBlock = false;
            for (const child of node.childNodes) { walk(child, false); }
        } else {
            for (const child of node.childNodes) { walk(child, isTopLevel); }
        }
    }
    for (const child of frag.childNodes) { walk(child, true); }
    return offset;
}

/**
 * Save caret position in contenteditable
 */
function saveCodeCaret() {
    const sel = window.getSelection();
    if (!sel.rangeCount || !codeEditable) return null;
    const start = getCodeCaretOffset();
    return { start };
}

/**
 * Restore caret position in contenteditable from character offset
 */
function restoreCodeCaret(saved) {
    if (!saved || !codeEditable) return;
    setCodeCaretAtOffset(saved.start);
}

/**
 * Set caret at a specific character offset in contenteditable
 */
function setCodeCaretAtOffset(targetOffset) {
    if (!codeEditable) return;
    const sel = window.getSelection();
    const range = document.createRange();

    let currentOffset = 0;
    let found = false;

    function walk(node) {
        if (found) return;
        if (node.nodeType === Node.TEXT_NODE) {
            const len = node.textContent.length;
            if (currentOffset + len >= targetOffset) {
                range.setStart(node, targetOffset - currentOffset);
                range.collapse(true);
                found = true;
                return;
            }
            currentOffset += len;
        } else if (node.nodeName === 'BR') {
            if (node.classList && node.classList.contains('cursor-helper')) return;
            if (currentOffset + 1 >= targetOffset) {
                // Place cursor after the <br>
                const parent = node.parentNode;
                const idx = Array.from(parent.childNodes).indexOf(node) + 1;
                range.setStart(parent, idx);
                range.collapse(true);
                found = true;
                return;
            }
            currentOffset += 1;
        } else {
            for (const child of node.childNodes) {
                walk(child);
                if (found) return;
            }
        }
    }

    walk(codeEditable);

    if (!found) {
        // Place cursor at end
        range.selectNodeContents(codeEditable);
        range.collapse(false);
    }

    sel.removeAllRanges();
    sel.addRange(range);
}

/**
 * Set caret to end of contenteditable
 */
function setCodeCaretToEnd() {
    if (!codeEditable) return;
    const sel = window.getSelection();
    const range = document.createRange();
    range.selectNodeContents(codeEditable);
    range.collapse(false);
    sel.removeAllRanges();
    sel.addRange(range);
}

/**
 * Highlight code content using custom SyntaxHighlighter
 * Sets innerHTML on the contenteditable code editor
 * @param {string} language - Language identifier
 * @param {string} code - Plain text code content
 */
function highlightCodeContent(language, code) {
    if (!codeEditable) return;
    const lines = code.split('\n');
    const firstLine = lines[0];
    const codeContent = lines.slice(1).join('\n');

    // First line in red (code:lang command)
    const firstLineHtml = `<span class="code-first-line">${escapeHtmlForHighlight(firstLine)}</span>`;

    let html;
    if (lines.length > 1) {
        // Content exists after first line (preserve newline even if code is empty)
        if (typeof SyntaxHighlighter !== 'undefined' && codeContent) {
            html = firstLineHtml + '\n' + SyntaxHighlighter.highlight(codeContent, language);
        } else {
            html = firstLineHtml + '\n' + escapeHtmlForHighlight(codeContent);
        }
    } else {
        html = firstLineHtml;
    }

    // Trailing <br> ensures Chrome renders cursor on empty last line
    codeEditable.innerHTML = html + '<br class="cursor-helper">';
}

/**
 * Escape HTML for display
 * Only escape < > & - quotes don't need escaping and would cause cursor misalignment
 */
function escapeHtmlForHighlight(text) {
    return text
        .replace(/&/g, '&amp;')
        .replace(/</g, '&lt;')
        .replace(/>/g, '&gt;');
}

// --- Tag Support ---

const TAG_REGEX = /#[a-zA-Z][a-zA-Z0-9_-]*/g;

/**
 * Extract unique tags from note content
 */
function extractTags(content) {
    // Skip code mode notes — hex colors (#FFFFFF), CSS selectors (#header),
    // preprocessor directives (#include), and comments (#) conflict with tags
    if (isCodeMode(content)) return [];
    const matches = content.match(TAG_REGEX);
    if (!matches) return [];
    return [...new Set(matches.map(t => t.toLowerCase()))];
}

/**
 * Get accent color index (1-5) for a tag name using djb2 hash
 */
function getTagAccentIndex(tagName) {
    let hash = 5381;
    const name = tagName.toLowerCase();
    for (let i = 0; i < name.length; i++) {
        hash = ((hash << 5) + hash) + name.charCodeAt(i);
        hash = hash & hash; // Convert to 32-bit integer
    }
    return (Math.abs(hash) % 5) + 1;
}

/**
 * Get the accent CSS class for a keyword mode
 * Maps 19 keyword modes across 6 accent groups for full palette use
 */
function getKeywordAccentClass(mode) {
    switch (mode) {
        // Accent 1: Calculation-related
        case 'math':
        case 'currency':
            return 'keyword-highlight-acc1';

        // Accent 2: Organizational tools
        case 'list':
        case 'template':
        case 'placeholder':
            return 'keyword-highlight-acc2';

        // Accent 3: Number/data operations
        case 'sum':
        case 'avg':
        case 'count':
        case 'sort':
            return 'keyword-highlight-acc3';

        // Accent 4: Technical/utility
        case 'code':
        case 'timer':
            return 'keyword-highlight-acc4';

        // Accent 5: Storage/management
        case 'bookmarks':
        case 'session':
        case 'paste':
            return 'keyword-highlight-acc5';

        // Accent 6: Generators & reference
        case 'game':
        case 'date':
        case 'uuid':
        case 'lorem':
        case 'weather':
        case 'define':
        case 'related':
            return 'keyword-highlight-acc6';

        default:
            return 'keyword-highlight-acc1';
    }
}

/**
 * Apply tag highlighting to HTML-escaped content
 * Wraps #tag patterns with accent-colored spans
 */
function applyTagHighlighting(html) {
    // Match #tags but not inside existing HTML tags
    return html.replace(/(#[a-zA-Z][a-zA-Z0-9_-]*)(?![^<]*>)/g, (match) => {
        const accIndex = getTagAccentIndex(match);
        return `<span class="tag-highlight tag-highlight-acc${accIndex}">${match}</span>`;
    });
}

/**
 * Escape HTML for Prism (basic escaping)
 */
function escapeHtmlForPrism(text) {
    return text
        .replace(/&/g, '&amp;')
        .replace(/</g, '&lt;')
        .replace(/>/g, '&gt;');
}

/**
 * Setup code editor event listeners (contenteditable approach)
 */
function setupCodeEditor() {
    if (!codeEditable) return;

    // Scroll guard: ensure highlighter/textarea stay hidden during code mode scrolling
    codeEditable.addEventListener('scroll', () => {
        if (editorHighlighter && editorHighlighter.innerHTML) {
            editorHighlighter.innerHTML = '';
        }
        if (editorHighlighter) editorHighlighter.style.display = 'none';
        if (editor) editor.style.display = 'none';
    });

    // Keydown handler - Tab indent, Enter auto-indent
    codeEditable.addEventListener('keydown', (e) => {
        // Tab key: insert spaces
        if (e.key === 'Tab') {
            e.preventDefault();
            const text = getCodeText();
            const offset = getCodeCaretOffset();

            if (e.shiftKey) {
                // Shift+Tab: remove leading spaces from current line
                const beforeCursor = text.substring(0, offset);
                const lineStart = beforeCursor.lastIndexOf('\n') + 1;
                const line = text.substring(lineStart);
                if (line.startsWith('    ')) {
                    const newText = text.substring(0, lineStart) + line.substring(4);
                    editor.value = newText;
                    const firstLine = newText.split('\n')[0];
                    const codeResult = parseCodeKeyword(firstLine);
                    if (codeResult && codeResult.language) {
                        highlightCodeContent(codeResult.language, newText);
                        setCodeCaretAtOffset(Math.max(lineStart, offset - 4));
                    }
                } else if (line.match(/^( {1,3})/)) {
                    const spaces = line.match(/^( +)/)[1].length;
                    const newText = text.substring(0, lineStart) + line.substring(spaces);
                    editor.value = newText;
                    const firstLine = newText.split('\n')[0];
                    const codeResult = parseCodeKeyword(firstLine);
                    if (codeResult && codeResult.language) {
                        highlightCodeContent(codeResult.language, newText);
                        setCodeCaretAtOffset(Math.max(lineStart, offset - spaces));
                    }
                }
            } else {
                // Tab: insert 4 spaces
                const newText = text.substring(0, offset) + '    ' + text.substring(offset);
                editor.value = newText;
                const firstLine = newText.split('\n')[0];
                const codeResult = parseCodeKeyword(firstLine);
                if (codeResult && codeResult.language) {
                    highlightCodeContent(codeResult.language, newText);
                    setCodeCaretAtOffset(offset + 4);
                }
            }
            handleContentChange(editor, async (savedNote) => {
                await updateNoteDate(savedNote);
            });
        }

        // Enter key: insert newline with auto-indent
        if (e.key === 'Enter') {
            e.preventDefault();
            const text = getCodeText();
            const offset = getCodeCaretOffset();
            const beforeCursor = text.substring(0, offset);
            const lineStart = beforeCursor.lastIndexOf('\n') + 1;
            const currentLine = text.substring(lineStart, offset);
            const indent = currentLine.match(/^(\s*)/)[1];

            const insertion = '\n' + indent;
            const newText = beforeCursor + insertion + text.substring(offset);
            editor.value = newText;
            const firstLine = newText.split('\n')[0];
            const codeResult = parseCodeKeyword(firstLine);
            if (codeResult && codeResult.language) {
                highlightCodeContent(codeResult.language, newText);
                setCodeCaretAtOffset(offset + insertion.length);
            }
            handleContentChange(editor, async (savedNote) => {
                await updateNoteDate(savedNote);
            });
        }
    });

    // Input handler - fires on normal typing
    codeEditable.addEventListener('input', () => {
        const text = getCodeText();
        editor.value = text;
        updateCodeMode();
        handleContentChange(editor, async (savedNote) => {
            await updateNoteDate(savedNote);
        });

        // If content is now empty (user cleared everything in code mode),
        // ensure all modes are fully reset so the placeholder shows
        if (!editor.value.trim()) {
            resetAllModes();
            updateEditorHighlighter();
        }
    });

    // Paste handler - insert plain text only
    codeEditable.addEventListener('paste', (e) => {
        e.preventDefault();
        const plainText = e.clipboardData.getData('text/plain');
        // Remove trailing newlines
        const trimmed = plainText.replace(/\n+$/, '');
        document.execCommand('insertText', false, trimmed);
    });
}


/**
 * Setup timer functionality
 */
function setupTimer() {
    // Timer click to pause/resume (handles both timer and lapwatch)
    timerDisplay.addEventListener('click', () => {
        if (isStopwatchRunning()) {
            const paused = toggleStopwatchPause();
            timerDisplay.classList.toggle('paused', paused);
        } else if (isTimerRunning()) {
            const paused = togglePause();
            timerDisplay.classList.toggle('paused', paused);
        }
    });

    // Double-click to stop (handles both timer and lapwatch)
    timerDisplay.addEventListener('dblclick', () => {
        if (isStopwatchRunning()) {
            stopStopwatch();
            timerDisplay.classList.remove('lapwatch');
            hideTimer();
        } else if (isTimerRunning()) {
            stopTimer();
            hideTimer();
        }
    });

    // Timer hover to show instructions
    timerDisplay.addEventListener('mouseenter', () => {
        if (isTimerRunning() || isStopwatchRunning()) {
            timerHoverPopup.classList.add('visible');
        }
    });

    timerDisplay.addEventListener('mouseleave', () => {
        timerHoverPopup.classList.remove('visible');
    });

    // Check for timer commands on editor input
    editor.addEventListener('keydown', (e) => {
        if (e.key === 'Enter' && !e.shiftKey) {
            // Skip if event was already handled (by template loader or other handlers)
            if (e.defaultPrevented) {
                return;
            }

            const content = editor.value;
            const cursorPos = editor.selectionStart;

            // Find the start of the current line
            const beforeCursor = content.substring(0, cursorPos);
            const lineStart = beforeCursor.lastIndexOf('\n') + 1;
            const currentLine = content.substring(lineStart, cursorPos).trim();

            // Check if in bookmarks add or edit mode
            const { type: bookmarksType } = parseBookmarksCommand(content);
            if (bookmarksType === 'add' || bookmarksType === 'edit') {
                // Get current line content properly
                const fullCurrentLine = content.substring(lineStart, cursorPos);

                // If we just pressed Enter after "bookmarks:add", insert first template
                if (currentLine.toLowerCase().trim() === 'bookmarks:add') {
                    e.preventDefault();
                    e.stopImmediatePropagation();
                    // Insert newline and first template
                    const newContent = content.substring(0, cursorPos) + '\nURL: ' + content.substring(cursorPos);
                    editor.value = newContent;
                    editor.selectionStart = editor.selectionEnd = cursorPos + 6; // Position after "URL: "
                    editor.dispatchEvent(new Event('input', { bubbles: true }));
                    isAddingBookmark = true;
                    bookmarkEntryStep = 1; // Next is Tags
                    return;
                }

                // Insert next field template based on current line
                if (fullCurrentLine.startsWith('URL:')) {
                    e.preventDefault();
                    e.stopImmediatePropagation();
                    const newContent = content.substring(0, cursorPos) + '\nTags: ' + content.substring(cursorPos);
                    editor.value = newContent;
                    editor.selectionStart = editor.selectionEnd = cursorPos + 7; // Position after "Tags: "
                    editor.dispatchEvent(new Event('input', { bubbles: true }));
                    bookmarkEntryStep = 2; // Next is Description
                    return;
                } else if (fullCurrentLine.startsWith('Tags:')) {
                    e.preventDefault();
                    e.stopImmediatePropagation();
                    const newContent = content.substring(0, cursorPos) + '\nDescription: ' + content.substring(cursorPos);
                    editor.value = newContent;
                    editor.selectionStart = editor.selectionEnd = cursorPos + 14; // Position after "Description: "
                    editor.dispatchEvent(new Event('input', { bubbles: true }));
                    bookmarkEntryStep = 0; // Reset for next bookmark
                    return;
                } else if (fullCurrentLine.startsWith('Description:')) {
                    e.preventDefault();
                    e.stopImmediatePropagation();
                    // In edit mode, just move to next line (don't create new bookmark template)
                    if (bookmarksType === 'edit') {
                        const newContent = content.substring(0, cursorPos) + '\n' + content.substring(cursorPos);
                        editor.value = newContent;
                        editor.selectionStart = editor.selectionEnd = cursorPos + 1;
                        editor.dispatchEvent(new Event('input', { bubbles: true }));
                        return;
                    }
                    // In add mode, insert blank line then next bookmark template
                    const newContent = content.substring(0, cursorPos) + '\n\nURL: ' + content.substring(cursorPos);
                    editor.value = newContent;
                    editor.selectionStart = editor.selectionEnd = cursorPos + 7; // Position after "URL: "
                    editor.dispatchEvent(new Event('input', { bubbles: true }));
                    bookmarkEntryStep = 1; // Next is Tags
                    return;
                }
            }

            // Check for utility keyword commands on Enter (first line only)
            const { mode: currentKeywordMode } = parseKeyword(content);
            if (lineStart === 0) {
                if (currentKeywordMode === 'uuid') {
                    e.preventDefault();
                    e.stopImmediatePropagation();
                    // Force content to have a second line so processUuidCommand detects Enter
                    updateUuidMode(content + '\n');
                    return;
                }

                if (currentKeywordMode === 'lorem') {
                    e.preventDefault();
                    e.stopImmediatePropagation();
                    // Force content to have a second line so processLoremCommand detects Enter
                    handleLoremInsert(content + '\n');
                    return;
                }

                if (currentKeywordMode === 'weather') {
                    e.preventDefault();
                    e.stopImmediatePropagation();
                    updateWeatherMode(content);
                    return;
                }

                if (currentKeywordMode === 'define' || currentKeywordMode === 'related') {
                    e.preventDefault();
                    e.stopImmediatePropagation();
                    const { word, hasDefineCommand } = parseDictionaryCommand(content);
                    if (hasDefineCommand && word) {
                        handleDictionaryQuery(word, currentKeywordMode);
                    }
                    return;
                }
            }

            // Check if current line is a timer command
            const timerConfig = parseTimerCommand(currentLine);

            if (timerConfig) {
                e.preventDefault();
                e.stopImmediatePropagation();
                handleTimerCommand(timerConfig, currentLine, lineStart, cursorPos);
            }
        }

        // Cmd+S saves bookmarks when in add or edit mode
        if ((e.metaKey || e.ctrlKey) && e.key === 's' && (isAddingBookmark || isEditingBookmark)) {
            e.preventDefault();
            e.stopImmediatePropagation();
            saveAllBookmarks();
            return;
        }

        // Escape cancels bookmark add or edit mode
        if (e.key === 'Escape' && (isAddingBookmark || isEditingBookmark)) {
            e.preventDefault();
            editor.value = '';
            isAddingBookmark = false;
            isEditingBookmark = false;
            editingBookmarkId = null;
            bookmarksToSave = [];
            bookmarkEntryStep = 0;
            editor.dispatchEvent(new Event('input', { bubbles: true }));
            return;
        }

        // Escape stops timer
        if (e.key === 'Escape' && isTimerRunning()) {
            stopTimer();
            hideTimer();
        }

        // Escape stops stopwatch (lapwatch mode)
        if (e.key === 'Escape' && isStopwatchRunning()) {
            stopStopwatch();
            timerDisplay.classList.remove('lapwatch');
            hideTimer();
        }
    });
}

/**
 * Show timer start notification
 * @param {Object} config - Timer configuration
 */
function showTimerStartNotification(config) {
    let message = '';

    if (config.type === 'stopwatch' || config.type === 'lapwatch') {
        message = 'Stopwatch started';
    } else if (config.type === 'countdown') {
        const minutes = Math.floor(config.duration / 60);
        const seconds = config.duration % 60;

        if (minutes > 0 && seconds > 0) {
            message = `${minutes} min ${seconds} sec countdown started`;
        } else if (minutes > 0) {
            message = `${minutes} min countdown started`;
        } else {
            message = `${seconds} sec countdown started`;
        }
    } else if (config.type === 'pomodoro') {
        message = 'Pomodoro started';
    }

    timerStartPopup.textContent = message;
    timerStartPopup.classList.add('visible');

    // Hide after 2 seconds
    setTimeout(() => {
        timerStartPopup.classList.remove('visible');
    }, 2000);
}

/**
 * Handle timer command
 * @param {Object} config - Timer configuration
 * @param {string} commandText - The timer command text
 * @param {number} lineStart - Start position of the line in editor
 * @param {number} lineEnd - End position of the line in editor (cursor position)
 */
function handleTimerCommand(config, commandText, lineStart, lineEnd) {
    if (!isPro()) { showProUpgradeHint('timer'); return; }

    if (config.type === 'control') {
        // Handle control commands
        if (config.action === 'pause' && isTimerRunning()) {
            const paused = togglePause();
            timerDisplay.classList.toggle('paused', paused);
        } else if (config.action === 'restart' && isTimerRunning()) {
            restartTimer();
        } else if (config.action === 'stop') {
            stopTimer();
            hideTimer();
        }
        // Clear the command from editor
        clearTimerCommand(lineStart, lineEnd);
        return;
    }

    // Stopwatch with laps (timer:stopwatch)
    if (config.type === 'lapwatch') {
        if (config.action === 'start') {
            if (!isStopwatchRunning()) {
                startStopwatch(onLapwatchTick);
                timerDisplay.classList.add('lapwatch');
                showTimer();
                showTimerStartNotification({ type: 'lapwatch' });
            }
        } else if (config.action === 'pause') {
            if (isStopwatchRunning()) {
                const paused = toggleStopwatchPause();
                timerDisplay.classList.toggle('paused', paused);
            }
        } else if (config.action === 'lap') {
            lapStopwatch();
        } else if (config.action === 'stop') {
            stopStopwatch();
            timerDisplay.classList.remove('lapwatch');
            hideTimer();
        } else if (config.action === 'reset') {
            resetStopwatch();
            timerDisplay.classList.remove('lapwatch');
            hideTimer();
            if (timerLapsEl) timerLapsEl.innerHTML = '';
        }
        clearTimerCommand(lineStart, lineEnd);
        return;
    }

    // Start new timer
    startTimer(config, onTimerTick, onTimerComplete);
    showTimer();

    // Show timer start notification
    showTimerStartNotification(config);

    // Clear the timer command from the editor
    clearTimerCommand(lineStart, lineEnd);
}

/**
 * Clear timer command from editor at specified position
 * @param {number} lineStart - Start position of the line
 * @param {number} lineEnd - End position of the line (cursor position)
 */
function clearTimerCommand(lineStart, lineEnd) {
    const content = editor.value;

    // Check if there's a newline before the line
    const hasNewlineBefore = lineStart > 0 && content[lineStart - 1] === '\n';

    // Remove the timer command line
    const removeStart = hasNewlineBefore ? lineStart - 1 : lineStart;
    const removeEnd = lineEnd;

    const before = content.substring(0, removeStart);
    const after = content.substring(removeEnd);

    editor.value = before + after;
    editor.selectionStart = editor.selectionEnd = removeStart;
    editor.dispatchEvent(new Event('input', { bubbles: true }));
}

/**
 * Timer tick callback
 */
function onTimerTick(data) {
    if (!data) {
        hideTimer();
        return;
    }

    // Parse MM:SS format and update separate elements
    const [mins, secs] = data.time.split(':');
    timerMinutes.textContent = mins;
    timerSeconds.textContent = secs;

    // Update progress bar
    if (data.progress !== null) {
        timerProgress.style.setProperty('--timer-progress', data.progress);
    } else {
        // Stopwatch - show pulsing indicator
        timerProgress.style.setProperty('--timer-progress', 1);
    }

    // Update break state for pomodoro
    timerDisplay.classList.toggle('break', data.isBreak === true);

    // Update round counter for pomodoro
    if (data.type === 'pomodoro' && timerRounds) {
        timerRounds.style.display = 'block';
        timerRounds.textContent = `${data.completedRounds}x`;
    } else if (timerRounds) {
        timerRounds.style.display = 'none';
    }
}

/**
 * Timer complete callback
 */
function onTimerComplete(data) {
    // Only countdown timers show completion modal and play chime
    // Pomodoro timers cycle silently
    if (data.type === 'countdown') {
        // Play chime sound
        if (timerChime) {
            timerChime.currentTime = 0;
            timerChime.play().catch(() => { });
        }

        // Show completion modal
        hideTimer();
        showTimerCompleteModal();
    }
    // Pomodoro timers do nothing here - they cycle continuously without interruption
}

/**
 * Show timer complete modal with random fun fact
 */
function showTimerCompleteModal() {
    const randomFact = FUN_FACTS[Math.floor(Math.random() * FUN_FACTS.length)];

    document.getElementById('timer-complete-category').textContent = randomFact.category;
    document.getElementById('timer-complete-fact').textContent = randomFact.fact;

    timerCompleteModal.classList.add('visible');

    // Setup dismiss button
    const dismissBtn = document.getElementById('timer-complete-btn');
    dismissBtn.onclick = dismissTimerComplete;

    // Also dismiss on Enter or Escape
    const handleKey = (e) => {
        if (e.key === 'Enter' || e.key === 'Escape') {
            dismissTimerComplete();
            document.removeEventListener('keydown', handleKey);
        }
    };
    document.addEventListener('keydown', handleKey);
}

/**
 * Dismiss timer complete modal
 */
function dismissTimerComplete() {
    timerCompleteModal.classList.remove('visible');
    focusActiveEditor();
}

/**
 * Show timer display
 */
function showTimer() {
    timerDisplay.classList.add('visible');
    document.querySelector('.editor-area').classList.add('has-timer');
}

/**
 * Hide timer display
 */
function hideTimer() {
    timerDisplay.classList.remove('visible', 'paused', 'break', 'lapwatch');
    document.querySelector('.editor-area').classList.remove('has-timer');
    timerMinutes.textContent = '00';
    timerSeconds.textContent = '00';
    if (timerMsEl) timerMsEl.textContent = '';
    if (timerLapsEl) timerLapsEl.innerHTML = '';
    timerProgress.style.setProperty('--timer-progress', 0);
    // Hide hover popup if visible
    timerHoverPopup.classList.remove('visible');
}

/**
 * Update currency help display
 * Shows help when current line is exactly "currency"
 */




// ═══════════════════════════════════════════════════
// PRO / FREEMIUM GATE LOGIC
// ═══════════════════════════════════════════════════

/**
 * Show the inline Pro upgrade hint in the editor area.
 * @param {string} modeName - The mode name to show in the hint (e.g. 'Code', 'Timer')
 */
function showProUpgradeHint(modeName) {
    const hint = document.getElementById('pro-upgrade-hint');
    const hintText = document.getElementById('pro-hint-text');
    if (hint && hintText) {
        const displayName = modeName.charAt(0).toUpperCase() + modeName.slice(1);
        hintText.textContent = `${displayName} mode is a Pro feature`;
        hint.classList.add('visible');
    }
}

/**
 * Hide the inline Pro upgrade hint.
 */
function hideProUpgradeHint() {
    const hint = document.getElementById('pro-upgrade-hint');
    if (hint) {
        hint.classList.remove('visible');
    }
}

/**
 * Apply Pro lock states to slash popup items.
 * Free users see Pro keywords greyed out with a "PRO" badge.
 */
function applySlashProGates() {
    if (isPro()) return; // Pro users see everything

    const items = document.querySelectorAll('.slash-item');
    items.forEach(item => {
        const keyword = item.dataset.keyword;
        if (PRO_KEYWORDS.includes(keyword)) {
            item.classList.add('pro-locked');
            // Add PRO badge if not already there
            if (!item.querySelector('.slash-pro-badge')) {
                const badge = document.createElement('span');
                badge.className = 'slash-pro-badge';
                badge.textContent = 'PRO';
                item.appendChild(badge);
            }
        }
    });
}

/**
 * Setup Pro UI: upgrade buttons, Pro settings tab, etc.
 */
function setupProUI() {
    // Inline upgrade hint button
    const proHintBtn = document.getElementById('pro-hint-btn');
    if (proHintBtn) {
        proHintBtn.addEventListener('click', () => {
            openUpgrade(onProActivated);
        });
    }

    // Pro settings tab upgrade button
    const proUpgradeBtn = document.getElementById('pro-upgrade-btn');
    if (proUpgradeBtn) {
        proUpgradeBtn.addEventListener('click', () => {
            openUpgrade(onProActivated);
        });
    }

    // Refresh status link
    const refreshLink = document.getElementById('pro-refresh-status');
    if (refreshLink) {
        refreshLink.addEventListener('click', async (e) => {
            e.preventDefault();
            refreshLink.textContent = 'Checking...';
            const isNowPro = await refreshProStatus();
            if (isNowPro) {
                onProActivated();
            } else {
                refreshLink.textContent = 'Status: Free (Click to refresh)';
                setTimeout(() => {
                    refreshLink.textContent = 'Restore Purchase';
                }, 2000);
            }
        });
    }

    // Update Pro settings tab state
    updateProSettingsTab();
}

/**
 * Build/rebuild the currency help list (Fiat + Crypto).
 * Called at init and again when Pro is activated so crypto shows up.
 */
function buildCurrencyList() {
    if (!currencyListContent || !CURRENCY_LIST || !CRYPTO_LIST) return;

    const fiatHTML = CURRENCY_LIST.map(c => `
        <div class="currency-item">
            <span class="currency-code">${c.code}</span>
            <span class="currency-name">${c.name}</span>
        </div>
    `).join('');

    let cryptoSection = '';
    if (isPro()) {
        const cryptoHTML = CRYPTO_LIST.map(c => `
            <div class="currency-item crypto">
                <span class="currency-code">${c.code}</span>
                <span class="currency-name">${c.name}</span>
            </div>
        `).join('');
        cryptoSection = `
            <div class="currency-section">
                <div class="currency-section-title">Cryptocurrencies</div>
                ${cryptoHTML}
            </div>
        `;
    } else {
        cryptoSection = `
            <div class="currency-section pro-locked">
                <div class="currency-section-title">Cryptocurrencies <span class="slash-pro-badge">PRO</span></div>
            </div>
        `;
    }

    currencyListContent.innerHTML = `
        <div class="currency-section">
            <div class="currency-section-title">Fiat Currencies</div>
            ${fiatHTML}
        </div>
        ${cryptoSection}
    `;
}

/**
 * Called when Pro is successfully activated (payment confirmed).
 */
function onProActivated(showToast = true) {
    // Update Pro settings tab
    updateProSettingsTab();

    // Remove all Pro locks from slash popup
    document.querySelectorAll('.slash-item.pro-locked').forEach(item => {
        item.classList.remove('pro-locked');
        const badge = item.querySelector('.slash-pro-badge');
        if (badge) badge.remove();
    });

    // Remove theme locks
    renderThemeSelector();

    // Rebuild currency list to show crypto for Pro users
    buildCurrencyList();

    // Hide any visible upgrade hint
    hideProUpgradeHint();

    // Show success toast only on fresh activation, not every startup
    if (showToast) {
        showTemplateNotification('Pro activated! All features unlocked.', 3000);
    }

    // Re-apply settings so default Pro theme takes effect
    if (settings) {
        applySettings();
    }

    // Re-process current content to unlock any gated modes
    if (editor && editor.value) {
        updateKeywordIndicator();
    }
}

/**
 * Update the Pro settings tab to reflect current Pro status.
 */
function updateProSettingsTab() {
    const freeState = document.getElementById('pro-free-state');
    const activeState = document.getElementById('pro-active-state');
    const badge = document.getElementById('pro-badge-indicator');
    const proTab = document.querySelector('.settings-tab-content[data-tab="pro"]');

    if (isPro()) {
        if (freeState) freeState.style.display = 'none';
        if (activeState) activeState.style.display = '';
        if (badge) { badge.textContent = 'Pro'; badge.className = 'pro-badge active'; }
        if (proTab) proTab.classList.add('pro-active');
    } else {
        if (freeState) freeState.style.display = '';
        if (activeState) activeState.style.display = 'none';
        if (badge) { badge.textContent = 'Free'; badge.className = 'pro-badge free'; }
        if (proTab) proTab.classList.remove('pro-active');
    }
}

// Initialize on DOM load
document.addEventListener('DOMContentLoaded', init);


/**
 * Render Theme Selector Grid
 */
function renderThemeSelector() {
    const grid = document.getElementById('theme-grid');

    if (!grid) return;

    grid.innerHTML = '';

    const renderCard = (key, theme) => {
        const card = document.createElement('div');
        card.dataset.key = key;

        // Inline preview styles (background from theme)
        const c = theme.colors;
        card.style.background = c['--bg-primary'];

        // Is this theme active as Light or Dark?
        const defaultTheme = isPro() ? 'gundam' : 'sanrio';
        const isLightActive = settings.lightTheme === key || (!settings.lightTheme && key === defaultTheme);
        const isDarkActive = settings.darkTheme === key || (!settings.darkTheme && key === defaultTheme);

        // Build class name with selection state
        let cardClass = 'theme-card';
        if (isLightActive) cardClass += ' selected-light';
        if (isDarkActive) cardClass += ' selected-dark';
        // Lock non-free themes for free users
        const locked = !isPro() && !isThemeFree(key);
        if (locked) cardClass += ' pro-locked';
        card.className = cardClass;

        // Dots - show accent colors from the theme
        const dotsHtml = [
            c['--accent-1-main'], c['--accent-2-main'], c['--accent-3-main'],
            c['--accent-4-main'], c['--accent-5-main'], c['--accent-1-secondary'],
            c['--accent-3-secondary'], c['--accent-4-secondary']
        ].filter(Boolean).map(color => `<div class="theme-dot" style="background-color: ${color}"></div>`).join('');

        card.innerHTML = `
            <div class="theme-header">
                <span class="theme-name" style="color: ${c['--text-primary']}">${theme.name}</span>
                <div class="theme-toggles">
                    <button class="theme-toggle-btn ${isLightActive ? 'active' : ''}" title="Set as Light Theme" data-action="light">
                        <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><circle cx="12" cy="12" r="5"></circle><line x1="12" y1="1" x2="12" y2="3"></line><line x1="12" y1="21" x2="12" y2="23"></line><line x1="4.22" y1="4.22" x2="5.64" y2="5.64"></line><line x1="18.36" y1="18.36" x2="19.78" y2="19.78"></line><line x1="1" y1="12" x2="3" y2="12"></line><line x1="21" y1="12" x2="23" y2="12"></line><line x1="4.22" y1="19.78" x2="5.64" y2="18.36"></line><line x1="18.36" y1="5.64" x2="19.78" y2="4.22"></line></svg>
                    </button>
                    <button class="theme-toggle-btn ${isDarkActive ? 'active' : ''}" title="Set as Dark Theme" data-action="dark">
                        <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M21 12.79A9 9 0 1 1 11.21 3 7 7 0 0 0 21 12.79z"></path></svg>
                    </button>
                </div>
            </div>
            <div class="theme-dots">
                ${dotsHtml}
            </div>
        `;

        // Handle Clicks
        const btns = card.querySelectorAll('.theme-toggle-btn');
        btns.forEach(btn => {
            btn.addEventListener('click', async (e) => {
                e.stopPropagation(); // Prevent bubbling if card has click
                if (locked) {
                    showTemplateNotification('This theme requires Pro', 2000);
                    return;
                }
                const action = btn.dataset.action;

                if (action === 'light') {
                    settings.lightTheme = key;
                } else {
                    settings.darkTheme = key;
                }
                await saveSettings(settings);

                // Re-render UI to update active states
                renderThemeSelector();

                // Apply valid settings
                applySettings();
            });
        });

        return card;
    };

    // Render All Themes
    Object.entries(THEMES).forEach(([key, theme]) => {
        grid.appendChild(renderCard(key, theme));
    });

    // Render custom theme card if one exists
    chrome.storage.local.get(CUSTOM_THEME_KEY, (result) => {
        const customTheme = result[CUSTOM_THEME_KEY];
        if (!customTheme) return;

        const card = renderCard('custom', customTheme);

        // Add delete button to bottom-right corner of custom theme card
        card.style.position = 'relative';
        const deleteBtn = document.createElement('button');
        deleteBtn.className = 'theme-delete-btn';
        deleteBtn.title = 'Remove custom theme';
        deleteBtn.innerHTML = '&times;';
        deleteBtn.style.cssText = 'position:absolute;bottom:6px;right:6px;width:20px;height:20px;border-radius:50%;border:1.5px solid rgba(255,255,255,0.3);background:rgba(0,0,0,0.45);color:#fff;font-size:14px;line-height:1;cursor:pointer;display:flex;align-items:center;justify-content:center;padding:0;opacity:0.7;transition:opacity 0.15s;';
        deleteBtn.addEventListener('mouseenter', () => { deleteBtn.style.opacity = '1'; });
        deleteBtn.addEventListener('mouseleave', () => { deleteBtn.style.opacity = '0.7'; });
        deleteBtn.addEventListener('click', async (e) => {
            e.stopPropagation();
            chrome.storage.local.remove(CUSTOM_THEME_KEY, async () => {
                const defaultTheme = isPro() ? 'gundam' : 'sanrio';
                if (settings.lightTheme === 'custom') settings.lightTheme = defaultTheme;
                if (settings.darkTheme === 'custom') settings.darkTheme = defaultTheme;
                await saveSettings(settings);
                renderThemeSelector();
                applySettings();
                showTemplateNotification('Custom theme removed', 2000);
            });
        });
        card.appendChild(deleteBtn);

        grid.appendChild(card);
    });
}

/**
 * Import a custom theme from JSON data
 */
function importCustomTheme(jsonData) {
    let themeData;
    try {
        themeData = typeof jsonData === 'string' ? JSON.parse(jsonData) : jsonData;
    } catch (e) {
        showTemplateNotification('Invalid JSON file', 2500);
        return;
    }

    const { valid, errors } = validateCustomTheme(themeData);
    if (!valid) {
        showTemplateNotification('Invalid theme: ' + errors[0], 3000);
        return;
    }

    // Ensure derived values exist
    if (!themeData.colors['--bg-secondary']) {
        const bg = themeData.colors['--bg-primary'];
        const hex = bg.replace('#', '');
        const lum = (parseInt(hex.substring(0, 2), 16) * 299 + parseInt(hex.substring(2, 4), 16) * 587 + parseInt(hex.substring(4, 6), 16) * 114) / 1000;
        const shift = lum > 128 ? -12 : 14;
        const r = Math.max(0, Math.min(255, parseInt(hex.substring(0, 2), 16) + shift));
        const g = Math.max(0, Math.min(255, parseInt(hex.substring(2, 4), 16) + shift));
        const b = Math.max(0, Math.min(255, parseInt(hex.substring(4, 6), 16) + shift));
        const secondary = '#' + [r, g, b].map(c => c.toString(16).padStart(2, '0')).join('').toUpperCase();
        themeData.colors['--bg-secondary'] = secondary;
        themeData.colors['--bg-tertiary'] = secondary;
    }
    if (!themeData.colors['--text-muted']) themeData.colors['--text-muted'] = themeData.colors['--text-secondary'];
    if (!themeData.colors['--accent']) themeData.colors['--accent'] = themeData.colors['--accent-1-main'];
    if (!themeData.colors['--accent-light']) themeData.colors['--accent-light'] = themeData.colors['--accent-2-main'];

    chrome.storage.local.set({ [CUSTOM_THEME_KEY]: themeData }, () => {
        showTemplateNotification('Custom theme "' + themeData.name + '" imported!', 2500);
        renderThemeSelector();
    });
}

// Wire up import button
document.addEventListener('DOMContentLoaded', () => {
    const importBtn = document.getElementById('import-custom-theme-btn');
    if (importBtn) {
        importBtn.addEventListener('click', () => {
            const input = document.createElement('input');
            input.type = 'file';
            input.accept = '.json';
            input.addEventListener('change', (e) => {
                const file = e.target.files[0];
                if (!file) return;
                const reader = new FileReader();
                reader.onload = (ev) => importCustomTheme(ev.target.result);
                reader.readAsText(file);
            });
            input.click();
        });
    }
});

/**
 * SHORTCUTS MANAGEMENT
 */

// Default shortcuts configuration
// NOTE: Chrome extensions cannot intercept most ⌘/Ctrl shortcuts (⌘N, ⌘T, ⌘W, ⌘D, ⌘F, ⌘S, etc.)
// We use ⌥ (Alt/Option) as the primary modifier to avoid conflicts
const DEFAULT_SHORTCUTS = {
    newNote: { name: 'New Note', keys: ['Meta', 'Shift', 'T'], default: ['Meta', 'Shift', 'T'] },
    deleteNote: { name: 'Delete Note', keys: ['Meta', 'Shift', 'D'], default: ['Meta', 'Shift', 'D'] },
    previousNote: { name: 'Previous Note', keys: ['Meta', 'Shift', '['], default: ['Meta', 'Shift', '['] },
    nextNote: { name: 'Next Note', keys: ['Meta', 'Shift', ']'], default: ['Meta', 'Shift', ']'] },
    jumpToFront: { name: 'Jump to Front', keys: ['Meta', 'Shift', 'J'], default: ['Meta', 'Shift', 'J'] },
    moveToFront: { name: 'Move Note to Front', keys: ['Meta', 'Shift', 'M'], default: ['Meta', 'Shift', 'M'] },
    search: { name: 'Search Notes', keys: ['Meta', 'Shift', 'S'], default: ['Meta', 'Shift', 'S'] },
    findReplace: { name: 'Find and Replace', keys: ['Meta', 'Shift', 'F'], default: ['Meta', 'Shift', 'F'] },
    pinNote: { name: 'Pin Note', keys: ['Meta', 'Shift', 'P'], default: ['Meta', 'Shift', 'P'] },
    quickExport: { name: 'Quick Export', keys: ['Meta', 'Shift', 'E'], default: ['Meta', 'Shift', 'E'] },
    hideShow: { name: 'Hide/Show Window', keys: ['Meta', 'Shift', 'O'], default: ['Meta', 'Shift', 'O'] },
    increaseText: { name: 'Increase Text Size', keys: ['Meta', 'Shift', 'ArrowUp'], default: ['Meta', 'Shift', 'ArrowUp'] },
    decreaseText: { name: 'Decrease Text Size', keys: ['Meta', 'Shift', 'ArrowDown'], default: ['Meta', 'Shift', 'ArrowDown'] },
    insertDate: { name: 'Insert Date', keys: ['Meta', 'Shift', 'I'], default: ['Meta', 'Shift', 'I'] },
    globalHotkey: { name: 'Global Hotkey (Alt+A)', keys: ['Alt', 'A'], default: ['Alt', 'A'] }
};

let currentShortcuts = { ...DEFAULT_SHORTCUTS };
let selectedShortcutId = null;
let isRecordingShortcut = false;
let recordingShortcutId = null;
let recordedKeys = [];

/**
 * Initialize shortcuts UI
 */
async function initShortcuts() {
    // Load saved shortcuts from storage
    const savedShortcuts = await chrome.storage.local.get('shortcuts');
    if (savedShortcuts.shortcuts) {
        currentShortcuts = { ...DEFAULT_SHORTCUTS, ...savedShortcuts.shortcuts };
    }

    // Render shortcuts list
    renderShortcutsList();

    // Setup event listeners
    setupShortcutsEvents();
}

/**
 * Render the shortcuts list
 */
function renderShortcutsList() {
    const list = document.getElementById('shortcuts-list');
    if (!list) return;

    list.innerHTML = '';

    let firstShortcutId = null;

    Object.entries(currentShortcuts).forEach(([id, shortcut], index) => {
        if (index === 0) firstShortcutId = id;

        const item = document.createElement('div');
        item.className = 'shortcut-item';
        item.dataset.shortcutId = id;

        const name = document.createElement('div');
        name.className = 'shortcut-item-name';
        name.textContent = shortcut.name;

        const keys = document.createElement('div');
        keys.className = shortcut.keys.length > 0 ? 'shortcut-item-keys' : 'shortcut-item-keys empty';
        keys.textContent = shortcut.keys.length > 0 ? formatShortcutDisplay(shortcut.keys) : 'Not set';

        item.appendChild(name);
        item.appendChild(keys);

        item.addEventListener('click', () => selectShortcut(id));

        list.appendChild(item);
    });

    // Preserve current selection if one exists, otherwise select first
    const idToSelect = selectedShortcutId && currentShortcuts[selectedShortcutId]
        ? selectedShortcutId
        : firstShortcutId;

    if (idToSelect) {
        selectShortcut(idToSelect);
    }
}

/**
 * Format shortcut keys for display (e.g., "⌘N" or "⌘⇧F")
 */
function formatShortcutDisplay(keys) {
    return keys.map(key => {
        if (key === 'Meta') return '⌘';
        if (key === 'Control') return '⌃';
        if (key === 'Alt') return '⌥';
        if (key === 'Shift') return '⇧';
        if (key === 'ArrowUp') return '↑';
        if (key === 'ArrowDown') return '↓';
        if (key === 'ArrowLeft') return '←';
        if (key === 'ArrowRight') return '→';
        return key;
    }).join('');
}

/**
 * Select a shortcut to edit
 */
function selectShortcut(id) {
    selectedShortcutId = id;
    clearConflictMessage();

    // Update list UI
    document.querySelectorAll('.shortcut-item').forEach(item => {
        item.classList.toggle('selected', item.dataset.shortcutId === id);
    });

    // Show editor panel
    const editor = document.getElementById('shortcut-editor');
    const shortcut = currentShortcuts[id];

    editor.classList.add('visible');

    // Update keys display
    const keysDisplay = document.getElementById('shortcut-keys-display');
    keysDisplay.innerHTML = '';

    if (shortcut.keys.length > 0) {
        shortcut.keys.forEach(key => {
            const keyEl = document.createElement('div');
            keyEl.className = 'shortcut-key';
            if (['Meta', 'Control', 'Alt', 'Shift'].includes(key)) {
                keyEl.classList.add('modifier');
            }
            keyEl.textContent = formatKeyForDisplay(key);
            keysDisplay.appendChild(keyEl);
        });
    } else {
        const keyEl = document.createElement('div');
        keyEl.className = 'shortcut-key';
        keyEl.style.opacity = '0.5';
        keyEl.textContent = '?';
        keysDisplay.appendChild(keyEl);
    }

    // Add click handler to start recording
    keysDisplay.onclick = () => startRecordingShortcut(id);

    // Update action name
    document.getElementById('shortcut-action-name').textContent = shortcut.name;

    // Update reset button text
    const resetBtn = document.getElementById('shortcut-reset-btn');
    resetBtn.textContent = 'Reset shortcut to default (' + formatShortcutDisplay(shortcut.default) + ')';
}

/**
 * Start recording a new shortcut
 */
function startRecordingShortcut(id) {
    if (isRecordingShortcut) return;

    isRecordingShortcut = true;
    recordingShortcutId = id;
    recordedKeys = [];

    // Update UI to show recording state
    const keysDisplay = document.getElementById('shortcut-keys-display');
    const recordingHint = document.getElementById('shortcut-recording-hint');

    keysDisplay.classList.add('recording');
    keysDisplay.innerHTML = '<div class="shortcut-key" style="opacity: 0.5;">...</div>';
    recordingHint.style.display = 'block';

    // Add global keydown listener
    document.addEventListener('keydown', handleShortcutKeyDown);
}

/**
 * Stop recording shortcut
 */
function stopRecordingShortcut(save = false) {
    if (!isRecordingShortcut) return;

    isRecordingShortcut = false;

    // Remove global keydown listener
    document.removeEventListener('keydown', handleShortcutKeyDown);

    // Update UI
    const keysDisplay = document.getElementById('shortcut-keys-display');
    const recordingHint = document.getElementById('shortcut-recording-hint');

    keysDisplay.classList.remove('recording');
    recordingHint.style.display = 'none';

    if (save && recordedKeys.length > 0) {
        // Validate and save the shortcut
        const validation = validateShortcut(recordedKeys, recordingShortcutId);

        if (validation.valid) {
            saveCustomShortcut(recordingShortcutId, recordedKeys);
            clearConflictMessage();
        } else {
            // Show error inline and revert
            showConflictMessage(validation.error);
            selectShortcut(recordingShortcutId);
        }
    } else {
        // Cancelled - revert to current shortcut
        selectShortcut(recordingShortcutId);
    }

    recordingShortcutId = null;
    recordedKeys = [];
}

/**
 * Handle keydown during recording
 */
function handleShortcutKeyDown(event) {
    event.preventDefault();
    event.stopPropagation();

    // ESC to cancel
    if (event.key === 'Escape') {
        stopRecordingShortcut(false);
        return;
    }

    // Build key combination
    const keys = [];

    if (event.metaKey || event.key === 'Meta') keys.push('Meta');
    if (event.ctrlKey || event.key === 'Control') keys.push('Control');
    if (event.altKey || event.key === 'Alt') keys.push('Alt');
    if (event.shiftKey || event.key === 'Shift') keys.push('Shift');

    // Use event.code to get the physical key, avoiding macOS Alt+letter special chars
    // (e.g., Alt+N = "Dead", Alt+G = "©" — but event.code = "KeyN", "KeyG")
    const code = event.code;
    const isModifierCode = ['MetaLeft', 'MetaRight', 'ControlLeft', 'ControlRight',
        'AltLeft', 'AltRight', 'ShiftLeft', 'ShiftRight'].includes(code);

    if (!isModifierCode && code) {
        let mainKey;
        if (code.startsWith('Key')) {
            mainKey = code.slice(3); // "KeyN" -> "N"
        } else if (code.startsWith('Digit')) {
            mainKey = code.slice(5); // "Digit1" -> "1"
        } else if (code.startsWith('Numpad')) {
            mainKey = code.slice(6); // "Numpad1" -> "1"
        } else if (code.startsWith('Arrow')) {
            mainKey = code; // "ArrowUp", "ArrowDown", etc.
        } else {
            // Brackets, slashes, etc. — use event.key but fallback to code
            mainKey = event.key.length === 1 ? event.key : code;
        }
        keys.push(code.startsWith('Arrow') ? mainKey : mainKey.toUpperCase());
    }

    // Need at least one modifier + one key
    if (keys.length >= 2) {
        recordedKeys = keys;

        // Update display
        const keysDisplay = document.getElementById('shortcut-keys-display');
        keysDisplay.innerHTML = '';

        keys.forEach(key => {
            const keyEl = document.createElement('div');
            keyEl.className = 'shortcut-key';
            if (['Meta', 'Control', 'Alt', 'Shift'].includes(key)) {
                keyEl.classList.add('modifier');
            }
            keyEl.textContent = formatKeyForDisplay(key);
            keysDisplay.appendChild(keyEl);
        });

        // Auto-save after short delay
        setTimeout(() => {
            if (isRecordingShortcut) {
                stopRecordingShortcut(true);
            }
        }, 500);
    }
}

/**
 * Validate shortcut combination
 */
function validateShortcut(keys, currentId) {
    // Global Hotkey is the only shortcut that doesn't require Cmd/Ctrl+Shift
    if (currentId === 'globalHotkey') {
        const hasModifier = keys.some(k => ['Meta', 'Control', 'Alt', 'Shift'].includes(k));
        if (!hasModifier) {
            return { valid: false, error: 'Shortcut must include a modifier key (⌘, Ctrl, Alt, or Shift)' };
        }
    } else {
        // All other shortcuts must use Cmd/Ctrl + Shift + Key
        const hasCmdOrCtrl = keys.includes('Meta') || keys.includes('Control');
        const hasShift = keys.includes('Shift');
        if (!hasCmdOrCtrl || !hasShift) {
            return { valid: false, error: 'Shortcuts must use Cmd/Ctrl + Shift + Key' };
        }
    }

    // Normalize keys for comparison: treat Meta and Control as equivalent
    const normalizeForCompare = (k) => k.map(key => (key === 'Meta' || key === 'Control') ? 'CmdCtrl' : key);
    const normalizedKeys = JSON.stringify(normalizeForCompare(keys));

    // Check for conflicts with other shortcuts
    for (const [id, shortcut] of Object.entries(currentShortcuts)) {
        if (id === currentId) continue;
        if (shortcut.keys.length === 0) continue;

        if (JSON.stringify(normalizeForCompare(shortcut.keys)) === normalizedKeys) {
            return { valid: false, error: `Already used by "${shortcut.name}"` };
        }
    }

    // Reserved Chrome browser shortcuts that cannot be overridden in extensions
    // Note: Cmd/Ctrl+Shift combos are generally available inside extension popups,
    // so only single-modifier Cmd/Ctrl shortcuts are reserved here
    const reserved = [
        // Mac (Meta/Cmd) — single-modifier, intercepted by Chrome
        ['Meta', 'Q'], ['Meta', 'W'], ['Meta', 'T'], ['Meta', 'R'],
        ['Meta', 'L'], ['Meta', 'H'], ['Meta', 'M'], ['Meta', ','],
        ['Meta', 'J'], ['Meta', 'Y'],
        // Windows/Linux (Control) — single-modifier, intercepted by Chrome
        ['Control', 'Q'], ['Control', 'W'], ['Control', 'T'], ['Control', 'R'],
        ['Control', 'L'], ['Control', 'H'], ['Control', 'J'],
        // Clipboard (both platforms) — intercepted before extensions
        ['Meta', 'A'], ['Meta', 'C'], ['Meta', 'V'], ['Meta', 'X'], ['Meta', 'Z'],
        ['Control', 'A'], ['Control', 'C'], ['Control', 'V'], ['Control', 'X'], ['Control', 'Z']
    ];

    for (const combo of reserved) {
        if (JSON.stringify(normalizeForCompare(combo)) === normalizedKeys) {
            return { valid: false, error: 'Reserved browser shortcut' };
        }
    }

    return { valid: true };
}

/**
 * Save custom shortcut
 */
async function saveCustomShortcut(id, keys) {
    // Update current shortcuts
    currentShortcuts[id].keys = keys;

    // Save to storage
    await chrome.storage.local.set({ shortcuts: currentShortcuts });

    // Update UI
    renderShortcutsList();
    selectShortcut(id);

    showTemplateNotification('Shortcut updated successfully', 1500);
}

/**
 * Format individual key for large display
 */
function formatKeyForDisplay(key) {
    if (key === 'Meta') return '⌘';
    if (key === 'Control') return '⌃';
    if (key === 'Alt') return '⌥';
    if (key === 'Shift') return '⇧';
    if (key === 'ArrowUp') return '↑';
    if (key === 'ArrowDown') return '↓';
    if (key === 'ArrowLeft') return '←';
    if (key === 'ArrowRight') return '→';
    return key;
}

/**
 * Setup shortcut events
 */
function setupShortcutsEvents() {
    // Reset button
    document.getElementById('shortcut-reset-btn').addEventListener('click', () => {
        if (!selectedShortcutId) return;

        currentShortcuts[selectedShortcutId].keys = [...currentShortcuts[selectedShortcutId].default];
        saveShortcuts();
        renderShortcutsList();
        selectShortcut(selectedShortcutId);
        clearConflictMessage();
    });

    // Remove button
    document.getElementById('shortcut-remove-btn').addEventListener('click', () => {
        if (!selectedShortcutId) return;

        currentShortcuts[selectedShortcutId].keys = [];
        saveShortcuts();
        renderShortcutsList();
        selectShortcut(selectedShortcutId);
        clearConflictMessage();
    });

    // Click on keys display to start recording
    document.getElementById('shortcut-keys-display').addEventListener('click', () => {
        if (!selectedShortcutId) return;
        startRecordingShortcut(selectedShortcutId);
    });
}


/**
 * Save shortcuts to storage

 */
async function saveShortcuts() {
    await chrome.storage.local.set({ shortcuts: currentShortcuts });
}

/**
 * Get shortcut for action
 */
function getShortcut(actionId) {
    return currentShortcuts[actionId]?.keys || [];
}

/**
 * Check if event matches shortcut
 */
function matchesShortcut(event, actionId) {
    const shortcut = getShortcut(actionId);
    if (shortcut.length === 0) return false;

    const hasMeta = shortcut.includes('Meta');
    const hasCtrl = shortcut.includes('Control');
    const hasAlt = shortcut.includes('Alt');
    const hasShift = shortcut.includes('Shift');

    // Treat Meta (Cmd) and Control (Ctrl) as interchangeable for cross-platform support
    const wantsCmdOrCtrl = hasMeta || hasCtrl;
    const eventHasCmdOrCtrl = event.metaKey || event.ctrlKey;

    const mainKey = shortcut.find(k => !['Meta', 'Control', 'Alt', 'Shift'].includes(k));

    return (
        (wantsCmdOrCtrl === eventHasCmdOrCtrl) &&
        (hasAlt === event.altKey) &&
        (hasShift === event.shiftKey) &&
        (mainKey && event.key.toUpperCase() === mainKey.toUpperCase())
    );
}

/**
 * Show conflict message inline
 */
function showConflictMessage(msg) {
    const el = document.getElementById('shortcut-conflict-msg');
    if (el) el.textContent = msg;
}

/**
 * Clear conflict message
 */
function clearConflictMessage() {
    const el = document.getElementById('shortcut-conflict-msg');
    if (el) el.textContent = '';
}

/**
 * Show first-launch onboarding overlay with icon labels
 */
function showOnboarding() {
    const overlay = document.getElementById('onboarding-overlay');
    if (!overlay) return;

    const container = document.querySelector('.jottnote-container');
    const containerRect = container.getBoundingClientRect();

    // Force both toolbars visible
    headerToolbar.classList.add('visible');
    footerToolbar.classList.add('visible');

    // Clear any previous content
    overlay.innerHTML = '';

    // Header icon labels (appear below the icons, arrow points up)
    const headerIcons = [
        { id: 'settings-btn', text: 'Settings' },
        { id: 'search-btn', text: 'Search' },
        { id: 'new-note-btn', text: 'New Note' },
        { id: 'pin-btn', text: 'Pin Note' }
    ];

    // Footer icon labels (appear above the icons, arrow points down)
    const footerIcons = [
        { id: 'delete-btn', text: 'Delete' },
        { id: 'copy-formatted-btn', text: 'Copy' },
        { id: 'prev-btn', text: 'Previous' },
        { id: 'next-btn', text: 'Next' },
        { id: 'jump-btn', text: 'Jump to Front' },
        { id: 'export-btn', text: 'Export' }
    ];

    // Place header labels, staggering rows to avoid overlap
    let prevHeaderRight = -Infinity;
    const ROW1_OFFSET = 6;
    const ROW2_OFFSET = 24;

    headerIcons.forEach(({ id, text }) => {
        const btn = document.getElementById(id);
        if (!btn) return;
        const btnRect = btn.getBoundingClientRect();
        const label = document.createElement('div');
        label.className = 'onboarding-label arrow-up';
        label.textContent = text;

        // Measure label width to detect overlap
        label.style.visibility = 'hidden';
        label.style.position = 'absolute';
        overlay.appendChild(label);
        const labelW = label.offsetWidth;
        const centerX = btnRect.left - containerRect.left + btnRect.width / 2;
        const leftEdge = centerX - labelW / 2;

        // Stagger to second row if it would overlap the previous label
        let offset = ROW1_OFFSET;
        if (leftEdge < prevHeaderRight + 4) {
            offset = ROW2_OFFSET;
            label.classList.remove('arrow-up');
        } else {
            prevHeaderRight = centerX + labelW / 2;
        }

        label.style.visibility = '';
        label.style.top = (btnRect.bottom - containerRect.top + offset) + 'px';
        label.style.left = centerX + 'px';
        label.style.transform = 'translateX(-50%)';
    });

    // Place footer labels, staggering rows to avoid overlap
    let prevFooterRight = -Infinity;

    footerIcons.forEach(({ id, text }) => {
        const btn = document.getElementById(id);
        if (!btn) return;
        const btnRect = btn.getBoundingClientRect();
        const label = document.createElement('div');
        label.className = 'onboarding-label arrow-down';
        label.textContent = text;

        // Measure label width to detect overlap
        label.style.visibility = 'hidden';
        label.style.position = 'absolute';
        overlay.appendChild(label);
        const labelW = label.offsetWidth;
        const centerX = btnRect.left - containerRect.left + btnRect.width / 2;
        const leftEdge = centerX - labelW / 2;

        // Stagger to second row if it would overlap the previous label
        let offset = ROW1_OFFSET;
        if (leftEdge < prevFooterRight + 4) {
            offset = ROW2_OFFSET;
            label.classList.remove('arrow-down');
        } else {
            prevFooterRight = centerX + labelW / 2;
        }

        label.style.visibility = '';
        label.style.bottom = (containerRect.bottom - btnRect.top + offset) + 'px';
        label.style.left = centerX + 'px';
        label.style.transform = 'translateX(-50%)';
    });

    // Dismiss hint in center
    const hint = document.createElement('div');
    hint.className = 'onboarding-dismiss';
    hint.textContent = 'Click anywhere to start';
    overlay.appendChild(hint);

    // Show overlay
    overlay.classList.add('visible');

    // Dismiss handler (delayed so popup-open click doesn't immediately dismiss)
    setTimeout(() => {
        const dismiss = (e) => {
            dismissOnboarding();
            document.removeEventListener('click', dismiss);
            document.removeEventListener('keydown', dismiss);
        };
        document.addEventListener('click', dismiss);
        document.addEventListener('keydown', dismiss);
    }, 300);
}

/**
 * Dismiss onboarding overlay and persist the flag
 */
function dismissOnboarding() {
    const overlay = document.getElementById('onboarding-overlay');
    if (!overlay) return;

    overlay.classList.remove('visible');
    overlay.innerHTML = '';

    // Return toolbars to auto-hide
    headerToolbar.classList.remove('visible');
    footerToolbar.classList.remove('visible');

    // Persist so onboarding never shows again
    chrome.storage.local.set({ jottnote_onboarding_done: true });
}
