/**
 * JottNote Syntax Highlighter
 * Multi-language support with consistent color scheme:
 * - Red/coral (#e06c75): tags, type names, important tokens
 * - Purple (#c678dd): keywords, attributes
 * - Green (#98c379): strings
 * - Cyan (#56b6c2): properties, operators
 * - Orange (#d19a66): numbers, constants
 * - Blue (#61afef): functions, methods
 * - Gray (#5c6370): comments
 * - Gold (#e5c07b): value literals (true, false, null, self, this)
 * - Operators (#56b6c2): punctuation/operators
 */

const SyntaxHighlighter = {

    // Only escape characters that MUST be escaped for HTML safety
    // Do NOT escape quotes - they display as-is in text content
    // Escaping quotes causes cursor position mismatch (' is 6 chars but ' is 1)
    escapeHtml: function (text) {
        return text
            .replace(/&/g, '&amp;')
            .replace(/</g, '&lt;')
            .replace(/>/g, '&gt;');
    },

    // Helper: Extract strings into safe placeholders before any HTML spans are created.
    // This prevents string regexes from matching inside span attributes.
    _extractStrings: function (text, patterns) {
        const strings = [];
        let result = text;
        for (const pattern of patterns) {
            result = result.replace(pattern, function (match) {
                strings.push(match);
                return '\x00S' + (strings.length - 1) + 'E\x00';
            });
        }
        return { result: result, strings: strings };
    },

    // Helper: Restore string placeholders with highlighted spans
    _restoreStrings: function (text, strings) {
        let result = text;
        for (let i = 0; i < strings.length; i++) {
            result = result.replace('\x00S' + i + 'E\x00', '<span class="hl-string">' + this.escapeHtml(strings[i]) + '</span>');
        }
        return result;
    },

    // Helper: Highlight operators (hl-operator) and punctuation (hl-punct) separately
    _highlightOperators: function (text, extraPatterns) {
        let result = text;
        // Multi-char operators first (=>, ===, !==, &&, ||, etc.)
        const multiOps = /(?:===|!==|=>|&&|\|\||>=|&lt;=|!=|==|-&gt;|::|\+=|-=|\*=|\/=|%=|&amp;&amp;|\|\|)/g;
        result = result.replace(multiOps, function (m) {
            return m.includes('class="') ? m : '<span class="hl-operator">' + m + '</span>';
        });
        for (const pat of (extraPatterns || [])) {
            result = result.replace(pat, function (m) {
                return m.includes('class="') ? m : '<span class="hl-operator">' + m + '</span>';
            });
        }
        return result;
    },

    // Helper: Highlight built-in functions/objects (hl-builtin)
    _highlightBuiltins: function (text, builtins) {
        let result = text;
        builtins.forEach(b => {
            result = result.replace(new RegExp(`\\b(${b})\\b(?![^<]*>)`, 'g'), '<span class="hl-builtin">$1</span>');
        });
        return result;
    },

    // Helper: Highlight dot-access properties (obj.prop but NOT obj.method())
    _highlightProperties: function (text) {
        return text.replace(/\.([a-zA-Z_][\w]*)(?!\s*\()(?![^<]*>)/g, function (m, prop) {
            return '.<span class="hl-property">' + prop + '</span>';
        });
    },

    // ================================
    // HTML HIGHLIGHTING
    // ================================
    highlightHTML: function (code) {
        const styleBlocks = [];
        const scriptBlocks = [];

        // Extract style blocks
        let processedCode = code.replace(/(<style[^>]*>)([\s\S]*?)(<\/style>)/gi, (match, open, content, close) => {
            const placeholder = `___STYLE_BLOCK_${styleBlocks.length}___`;
            styleBlocks.push({ open, content, close });
            return placeholder;
        });

        // Extract script blocks
        processedCode = processedCode.replace(/(<script[^>]*>)([\s\S]*?)(<\/script>)/gi, (match, open, content, close) => {
            const placeholder = `___SCRIPT_BLOCK_${scriptBlocks.length}___`;
            scriptBlocks.push({ open, content, close });
            return placeholder;
        });

        let result = this.escapeHtml(processedCode);

        // Comments
        result = result.replace(/(<!--[\s\S]*?-->)/g, '<span class="hl-comment">$1</span>');

        // DOCTYPE
        result = result.replace(/(&lt;!DOCTYPE\s+)(\w+)(&gt;)/gi,
            '<span class="hl-punct">&lt;!</span><span class="hl-tag">DOCTYPE</span> <span class="hl-keyword">$2</span><span class="hl-punct">&gt;</span>');

        // Tags with attributes
        result = result.replace(/(&lt;)(\/?)(\w+)([^&]*?)(&gt;)/g, (match, open, slash, tag, attrs, close) => {
            let processedAttrs = attrs;
            if (attrs.trim()) {
                processedAttrs = attrs.replace(/\s([\w-]+)(=)("|')([^&]*)("|')/g,
                    ' <span class="hl-attr">$1</span><span class="hl-punct">$2$3</span><span class="hl-string">$4</span><span class="hl-punct">$5</span>'
                );
            }
            return `<span class="hl-punct">${open}${slash}</span><span class="hl-tag">${tag}</span>${processedAttrs}<span class="hl-punct">${close}</span>`;
        });

        // Restore style blocks with CSS highlighting
        styleBlocks.forEach((block, i) => {
            const openTag = `<span class="hl-punct">&lt;</span><span class="hl-tag">style</span><span class="hl-punct">&gt;</span>`;
            const closeTag = `<span class="hl-punct">&lt;/</span><span class="hl-tag">style</span><span class="hl-punct">&gt;</span>`;
            result = result.replace(`___STYLE_BLOCK_${i}___`, openTag + this.highlightCSS(block.content) + closeTag);
        });

        // Restore script blocks with JS highlighting
        scriptBlocks.forEach((block, i) => {
            const openTag = `<span class="hl-punct">&lt;</span><span class="hl-tag">script</span><span class="hl-punct">&gt;</span>`;
            const closeTag = `<span class="hl-punct">&lt;/</span><span class="hl-tag">script</span><span class="hl-punct">&gt;</span>`;
            result = result.replace(`___SCRIPT_BLOCK_${i}___`, openTag + this.highlightJS(block.content) + closeTag);
        });

        return result;
    },

    // ================================
    // CSS HIGHLIGHTING
    // ================================
    highlightCSS: function (code) {
        let result = this.escapeHtml(code);

        // Comments
        result = result.replace(/(\/\*[\s\S]*?\*\/)/g, '<span class="hl-comment">$1</span>');

        const lines = result.split('\n');
        const processedLines = lines.map(line => {
            if (line.includes('hl-comment')) return line;

            // At-rules (@media, @keyframes, @import, etc.)
            if (/^\s*@/.test(line)) {
                return line.replace(/(@[\w-]+)/g, '<span class="hl-keyword">$1</span>');
            }

            // Selectors (lines with { but no :, or lines before {)
            if (line.includes('{') && !line.includes(':')) {
                return line.replace(/^(\s*)([\w.#\-\[\]=\"\s,*:>+~]+)(\s*\{)/g, (m, space, sel, brace) => {
                    // IDs
                    let hl = sel.replace(/(#[a-zA-Z_][\w-]*)/g, '<span class="hl-tag">$1</span>');
                    // Classes
                    hl = hl.replace(/(\.[a-zA-Z_][\w-]*)/g, '<span class="hl-selector">$1</span>');
                    // Pseudo-classes/elements
                    hl = hl.replace(/(:{1,2}[\w-]+)/g, '<span class="hl-attr">$1</span>');
                    // Element selectors
                    hl = hl.replace(/\b([a-zA-Z][\w-]*)\b(?![^<]*>)/g, '<span class="hl-selector">$1</span>');
                    return `${space}${hl}<span class="hl-operator">${brace}</span>`;
                });
            }

            // Properties and values
            if (line.includes(':') && !line.includes('hl-')) {
                return line.replace(/^(\s*)([\w-]+)(\s*:\s*)(.+?)(;?)(\s*)$/g, (m, space, prop, colon, val, semi, end) => {
                    // Highlight values
                    let hv = val;
                    // Hex colors
                    hv = hv.replace(/(#[0-9a-fA-F]{3,8})\b/g, '<span class="hl-number">$1</span>');
                    // Numbers with units
                    hv = hv.replace(/\b(\d+\.?\d*)(px|em|rem|%|vh|vw|vmin|vmax|deg|s|ms|fr|ch|ex)?\b/g,
                        '<span class="hl-number">$1</span><span class="hl-tag">$2</span>');
                    // CSS functions (var, calc, rgb, rgba, hsl, etc.)
                    hv = hv.replace(/\b(var|calc|rgb|rgba|hsl|hsla|linear-gradient|radial-gradient|min|max|clamp|attr|url|env|minmax|repeat)(\s*\()/g,
                        '<span class="hl-function">$1</span><span class="hl-punct">$2</span>');
                    // CSS keywords
                    hv = hv.replace(/\b(inherit|initial|unset|revert|none|auto|normal|bold|italic|solid|dashed|dotted|block|inline|flex|grid|absolute|relative|fixed|sticky|hidden|visible|scroll|wrap|nowrap|center|left|right|top|bottom|start|end|space-between|space-around|stretch|column|row|ease|ease-in|ease-out|ease-in-out|forwards|backwards|infinite|alternate|transparent|currentColor)\b(?![^<]*>)/g,
                        '<span class="hl-value">$1</span>');
                    // !important
                    hv = hv.replace(/(!important)\b/g, '<span class="hl-builtin">$1</span>');
                    // Custom property references (--var-name)
                    hv = hv.replace(/(--[\w-]+)/g, '<span class="hl-attr">$1</span>');

                    return `${space}<span class="hl-property">${prop}</span><span class="hl-operator">${colon}</span>${hv}<span class="hl-punct">${semi}</span>${end}`;
                });
            }

            if (line.trim() === '}') return line.replace(/(\})/, '<span class="hl-operator">$1</span>');
            return line;
        });

        return processedLines.join('\n');
    },

    // ================================
    // JAVASCRIPT / TYPESCRIPT
    // ================================
    highlightJS: function (code) {
        let result = this.escapeHtml(code);

        // Extract strings into placeholders (prevents regex interaction with span attributes)
        const extracted = this._extractStrings(result, [
            /`[^`]*`/g,
            /"[^"]*"/g,
            /'[^']*'/g
        ]);
        result = extracted.result;

        // Comments
        result = result.replace(/(\/\/[^\n]*)/g, '<span class="hl-comment">$1</span>');
        result = result.replace(/(\/\*[\s\S]*?\*\/)/g, '<span class="hl-comment">$1</span>');

        // Keywords (removed this, super, true, false, null, undefined, NaN, Infinity — now hl-value)
        ['const', 'let', 'var', 'function', 'return', 'if', 'else', 'for', 'while', 'class', 'extends',
            'import', 'export', 'from', 'default', 'async', 'await', 'try', 'catch', 'throw', 'new',
            'typeof', 'switch', 'case', 'break', 'continue', 'do', 'in', 'of', 'instanceof',
            'void', 'delete', 'interface', 'type', 'enum', 'private', 'public', 'protected', 'static',
            'readonly', 'abstract', 'implements'].forEach(kw => {
                result = result.replace(new RegExp(`\\b(${kw})\\b(?![^<]*>)`, 'g'), '<span class="hl-keyword">$1</span>');
            });

        // Value literals (special constants)
        result = result.replace(/\b(this|super|true|false|null|undefined|NaN|Infinity)\b(?![^<]*>)/g, '<span class="hl-value">$1</span>');

        // Numbers
        result = result.replace(/\b(\d+\.?\d*)\b(?![^<]*>)/g, '<span class="hl-number">$1</span>');

        // Functions
        result = result.replace(/\b([a-zA-Z_$][\w$]*)(\s*\()/g, (m, fn, p) => m.includes('class="') ? m : `<span class="hl-function">${fn}</span>${p}`);

        // Built-in objects/functions
        result = this._highlightBuiltins(result, [
            'console', 'window', 'document', 'Math', 'JSON', 'Object', 'Array',
            'String', 'Number', 'Boolean', 'Date', 'RegExp', 'Map', 'Set',
            'Promise', 'Symbol', 'WeakMap', 'WeakSet', 'Proxy', 'Reflect',
            'parseInt', 'parseFloat', 'isNaN', 'isFinite', 'setTimeout',
            'setInterval', 'clearTimeout', 'clearInterval', 'fetch',
            'require', 'module', 'exports', 'process', 'Buffer'
        ]);

        // Type names (capitalized identifiers not already highlighted)
        result = result.replace(/\b([A-Z][a-zA-Z0-9_]*)\b(?![^<]*>)/g, '<span class="hl-tag">$1</span>');

        // Dot-access properties
        result = this._highlightProperties(result);

        // Operators
        result = this._highlightOperators(result);

        // Restore strings
        result = this._restoreStrings(result, extracted.strings);

        return result;
    },

    // ================================
    // PYTHON
    // ================================
    highlightPython: function (code) {
        let result = this.escapeHtml(code);

        // Extract strings into placeholders
        const extracted = this._extractStrings(result, [
            /"""[\s\S]*?"""/g,
            /'''[\s\S]*?'''/g,
            /f?"[^"\n]*"/g,
            /f?'[^'\n]*'/g
        ]);
        result = extracted.result;

        // Comments
        result = result.replace(/(#[^\n]*)/g, '<span class="hl-comment">$1</span>');

        // Keywords
        ['def', 'class', 'if', 'elif', 'else', 'for', 'while', 'return', 'import', 'from', 'as',
            'try', 'except', 'finally', 'with', 'lambda', 'yield', 'and', 'or', 'not', 'in', 'is',
            'pass', 'break', 'continue', 'raise', 'global', 'nonlocal', 'assert', 'async', 'await'].forEach(kw => {
                result = result.replace(new RegExp(`\\b(${kw})\\b(?![^<]*>)`, 'g'), '<span class="hl-keyword">$1</span>');
            });

        // Value literals
        result = result.replace(/\b(self|cls|True|False|None)\b(?![^<]*>)/g, '<span class="hl-value">$1</span>');

        // Numbers
        result = result.replace(/\b(\d+\.?\d*)\b(?![^<]*>)/g, '<span class="hl-number">$1</span>');

        // Functions
        result = result.replace(/\b([a-zA-Z_][\w]*)(\s*\()/g, (m, fn, p) => m.includes('class="') ? m : `<span class="hl-function">${fn}</span>${p}`);

        // Built-in functions
        result = this._highlightBuiltins(result, [
            'print', 'len', 'range', 'int', 'str', 'float', 'list', 'dict',
            'tuple', 'set', 'bool', 'type', 'input', 'open', 'super',
            'enumerate', 'zip', 'map', 'filter', 'sorted', 'reversed',
            'abs', 'min', 'max', 'sum', 'round', 'isinstance', 'issubclass',
            'hasattr', 'getattr', 'setattr', 'delattr', 'property',
            'staticmethod', 'classmethod', 'vars', 'dir', 'id', 'hash',
            'next', 'iter', 'repr', 'format', 'chr', 'ord', 'hex', 'oct', 'bin'
        ]);

        // Decorators
        result = result.replace(/(@[\w.]+)/g, '<span class="hl-attr">$1</span>');

        // Type names (capitalized identifiers)
        result = result.replace(/\b([A-Z][a-zA-Z0-9_]*)\b(?![^<]*>)/g, '<span class="hl-tag">$1</span>');

        // Dot-access properties
        result = this._highlightProperties(result);

        // Operators
        result = this._highlightOperators(result);

        // Restore strings
        result = this._restoreStrings(result, extracted.strings);

        return result;
    },

    // ================================
    // JAVA / C# / KOTLIN
    // ================================
    highlightJava: function (code) {
        let result = this.escapeHtml(code);

        // Extract strings into placeholders
        const extracted = this._extractStrings(result, [
            /"[^"]*"/g,
            /'.'/g
        ]);
        result = extracted.result;

        // Comments
        result = result.replace(/(\/\/[^\n]*)/g, '<span class="hl-comment">$1</span>');
        result = result.replace(/(\/\*[\s\S]*?\*\/)/g, '<span class="hl-comment">$1</span>');

        // Keywords (removed this, super, null, true, false — now hl-value)
        ['public', 'private', 'protected', 'static', 'final', 'abstract', 'class', 'interface',
            'extends', 'implements', 'new', 'return', 'if', 'else', 'for', 'while', 'do', 'switch',
            'case', 'break', 'continue', 'try', 'catch', 'finally', 'throw', 'throws', 'import',
            'package', 'void', 'instanceof', 'synchronized', 'volatile',
            'transient', 'native', 'enum', 'assert', 'default', 'goto', 'const', 'var', 'val',
            'fun', 'override', 'open', 'sealed', 'data', 'object', 'companion', 'internal'].forEach(kw => {
                result = result.replace(new RegExp(`\\b(${kw})\\b(?![^<]*>)`, 'g'), '<span class="hl-keyword">$1</span>');
            });

        // Value literals
        result = result.replace(/\b(this|super|true|false|null)\b(?![^<]*>)/g, '<span class="hl-value">$1</span>');

        // Types (capitalized words)
        result = result.replace(/\b([A-Z][a-zA-Z0-9_]*)\b(?![^<]*>)/g, '<span class="hl-tag">$1</span>');

        // Primitives
        ['int', 'long', 'short', 'byte', 'float', 'double', 'boolean', 'char', 'String'].forEach(t => {
            result = result.replace(new RegExp(`\\b(${t})\\b(?![^<]*>)`, 'g'), '<span class="hl-keyword">$1</span>');
        });

        result = result.replace(/\b(\d+\.?\d*[fFdDlL]?)\b(?![^<]*>)/g, '<span class="hl-number">$1</span>');
        result = result.replace(/\b([a-zA-Z_][\w]*)(\s*\()/g, (m, fn, p) => m.includes('class="') ? m : `<span class="hl-function">${fn}</span>${p}`);
        result = result.replace(/(@\w+)/g, '<span class="hl-attr">$1</span>');

        // Dot-access properties
        result = this._highlightProperties(result);

        // Operators
        result = this._highlightOperators(result);

        // Restore strings
        result = this._restoreStrings(result, extracted.strings);

        return result;
    },

    // ================================
    // PHP
    // ================================
    highlightPHP: function (code) {
        let result = this.escapeHtml(code);

        // Extract strings into placeholders
        const extracted = this._extractStrings(result, [
            /"[^"]*"/g,
            /'[^']*'/g
        ]);
        result = extracted.result;

        // Comments
        result = result.replace(/(\/\/[^\n]*)/g, '<span class="hl-comment">$1</span>');
        result = result.replace(/(#[^\n]*)/g, '<span class="hl-comment">$1</span>');
        result = result.replace(/(\/\*[\s\S]*?\*\/)/g, '<span class="hl-comment">$1</span>');

        // PHP tags
        result = result.replace(/(&lt;\?php)/gi, '<span class="hl-tag">$1</span>');
        result = result.replace(/(\?&gt;)/g, '<span class="hl-tag">$1</span>');

        // Variables
        result = result.replace(/(\$[a-zA-Z_][\w]*)/g, '<span class="hl-attr">$1</span>');

        // Keywords (removed true, false, null — now hl-value)
        ['function', 'class', 'public', 'private', 'protected', 'static', 'return', 'if', 'else',
            'elseif', 'for', 'foreach', 'while', 'do', 'switch', 'case', 'break', 'continue', 'try',
            'catch', 'throw', 'new', 'extends', 'implements', 'interface', 'abstract', 'final',
            'namespace', 'use', 'require', 'include', 'require_once', 'include_once', 'echo', 'print',
            'array', 'as', 'global', 'const', 'match', 'fn'].forEach(kw => {
                result = result.replace(new RegExp(`\\b(${kw})\\b(?![^<]*>)`, 'gi'), '<span class="hl-keyword">$1</span>');
            });

        // Value literals
        result = result.replace(/\b(true|false|null)\b(?![^<]*>)/gi, '<span class="hl-value">$1</span>');

        result = result.replace(/\b(\d+\.?\d*)\b(?![^<]*>)/g, '<span class="hl-number">$1</span>');
        result = result.replace(/\b([a-zA-Z_][\w]*)(\s*\()/g, (m, fn, p) => m.includes('class="') ? m : `<span class="hl-function">${fn}</span>${p}`);

        // Arrow property access (->prop but not ->method())
        result = result.replace(/-&gt;([a-zA-Z_][\w]*)(?!\s*\()(?![^<]*>)/g, function (m, prop) {
            return '-&gt;<span class="hl-property">' + prop + '</span>';
        });

        // Operators
        result = this._highlightOperators(result);

        // Restore strings
        result = this._restoreStrings(result, extracted.strings);

        return result;
    },

    // ================================
    // RUBY
    // ================================
    highlightRuby: function (code) {
        let result = this.escapeHtml(code);

        // Extract strings into placeholders
        const extracted = this._extractStrings(result, [
            /"[^"]*"/g,
            /'[^']*'/g
        ]);
        result = extracted.result;

        // Comments
        result = result.replace(/(#[^\n]*)/g, '<span class="hl-comment">$1</span>');

        // Symbols
        result = result.replace(/(:[a-zA-Z_]\w*)/g, '<span class="hl-number">$1</span>');

        // Instance/class variables
        result = result.replace(/(@@?[a-zA-Z_]\w*)/g, '<span class="hl-attr">$1</span>');

        // Keywords (removed self, super, nil, true, false — now hl-value)
        ['def', 'class', 'module', 'end', 'if', 'elsif', 'else', 'unless', 'case', 'when', 'while',
            'until', 'for', 'do', 'begin', 'rescue', 'ensure', 'raise', 'return', 'yield', 'break',
            'next', 'redo', 'retry', 'and', 'or', 'not',
            'in', 'then', 'attr_accessor', 'attr_reader', 'attr_writer', 'require', 'include',
            'extend', 'private', 'protected', 'public', 'lambda', 'proc', 'new'].forEach(kw => {
                result = result.replace(new RegExp(`\\b(${kw})\\b(?![^<]*>)`, 'g'), '<span class="hl-keyword">$1</span>');
            });

        // Value literals
        result = result.replace(/\b(self|super|nil|true|false)\b(?![^<]*>)/g, '<span class="hl-value">$1</span>');

        result = result.replace(/\b(\d+\.?\d*)\b(?![^<]*>)/g, '<span class="hl-number">$1</span>');
        result = result.replace(/\b([a-zA-Z_][\w]*[!?]?)(\s*[\(\{])/g, (m, fn, p) => m.includes('class="') ? m : `<span class="hl-function">${fn}</span>${p}`);

        // Type names (capitalized identifiers)
        result = result.replace(/\b([A-Z][a-zA-Z0-9_]*)\b(?![^<]*>)/g, '<span class="hl-tag">$1</span>');

        // Dot-access properties
        result = this._highlightProperties(result);

        // Operators
        result = this._highlightOperators(result);

        // Restore strings
        result = this._restoreStrings(result, extracted.strings);

        return result;
    },

    // ================================
    // SWIFT
    // ================================
    highlightSwift: function (code) {
        let result = this.escapeHtml(code);

        // Extract strings into placeholders
        const extracted = this._extractStrings(result, [
            /"""[\s\S]*?"""/g,
            /"[^"]*"/g
        ]);
        result = extracted.result;

        // Comments
        result = result.replace(/(\/\/[^\n]*)/g, '<span class="hl-comment">$1</span>');
        result = result.replace(/(\/\*[\s\S]*?\*\/)/g, '<span class="hl-comment">$1</span>');

        // Keywords (removed self, Self, super, nil, true, false — now hl-value)
        ['func', 'class', 'struct', 'enum', 'protocol', 'extension', 'var', 'let', 'return', 'if',
            'else', 'guard', 'switch', 'case', 'default', 'for', 'while', 'repeat', 'break', 'continue',
            'fallthrough', 'in', 'where', 'throw', 'throws', 'rethrows', 'try', 'catch', 'do', 'defer',
            'import', 'typealias', 'associatedtype', 'init', 'deinit', 'subscript', 'operator', 'infix',
            'prefix', 'postfix', 'precedencegroup',
            'as', 'is', 'Any', 'Type', 'static', 'private', 'fileprivate', 'internal', 'public', 'open',
            'mutating', 'nonmutating', 'override', 'final', 'required', 'convenience', 'lazy', 'weak',
            'unowned', 'inout', '@escaping', 'async', 'await'].forEach(kw => {
                result = result.replace(new RegExp(`\\b(${kw})\\b(?![^<]*>)`, 'g'), '<span class="hl-keyword">$1</span>');
            });

        // Value literals
        result = result.replace(/\b(self|Self|super|nil|true|false)\b(?![^<]*>)/g, '<span class="hl-value">$1</span>');

        // Types
        result = result.replace(/\b([A-Z][a-zA-Z0-9_]*)\b(?![^<]*>)/g, '<span class="hl-tag">$1</span>');

        result = result.replace(/\b(\d+\.?\d*)\b(?![^<]*>)/g, '<span class="hl-number">$1</span>');
        result = result.replace(/\b([a-zA-Z_][\w]*)(\s*\()/g, (m, fn, p) => m.includes('class="') ? m : `<span class="hl-function">${fn}</span>${p}`);

        // Dot-access properties
        result = this._highlightProperties(result);

        // Operators
        result = this._highlightOperators(result);

        // Restore strings
        result = this._restoreStrings(result, extracted.strings);

        return result;
    },

    // ================================
    // GO
    // ================================
    highlightGo: function (code) {
        let result = this.escapeHtml(code);

        // Extract strings into placeholders
        const extracted = this._extractStrings(result, [
            /`[^`]*`/g,
            /"[^"]*"/g
        ]);
        result = extracted.result;

        // Comments
        result = result.replace(/(\/\/[^\n]*)/g, '<span class="hl-comment">$1</span>');
        result = result.replace(/(\/\*[\s\S]*?\*\/)/g, '<span class="hl-comment">$1</span>');

        // Keywords
        ['break', 'case', 'chan', 'const', 'continue', 'default', 'defer', 'else', 'fallthrough',
            'for', 'func', 'go', 'goto', 'if', 'import', 'interface', 'map', 'package', 'range',
            'return', 'select', 'struct', 'switch', 'type', 'var'].forEach(kw => {
                result = result.replace(new RegExp(`\\b(${kw})\\b(?![^<]*>)`, 'g'), '<span class="hl-keyword">$1</span>');
            });

        // Built-in types
        ['bool', 'byte', 'complex64', 'complex128', 'error', 'float32', 'float64', 'int', 'int8',
            'int16', 'int32', 'int64', 'rune', 'string', 'uint', 'uint8', 'uint16', 'uint32', 'uint64',
            'uintptr'].forEach(t => {
                result = result.replace(new RegExp(`\\b(${t})\\b(?![^<]*>)`, 'g'), '<span class="hl-keyword">$1</span>');
            });

        // Value literals (moved out of built-in types)
        result = result.replace(/\b(true|false|nil|iota)\b(?![^<]*>)/g, '<span class="hl-value">$1</span>');

        result = result.replace(/\b(\d+\.?\d*)\b(?![^<]*>)/g, '<span class="hl-number">$1</span>');
        result = result.replace(/\b([a-zA-Z_][\w]*)(\s*\()/g, (m, fn, p) => m.includes('class="') ? m : `<span class="hl-function">${fn}</span>${p}`);

        // Type names (capitalized identifiers)
        result = result.replace(/\b([A-Z][a-zA-Z0-9_]*)\b(?![^<]*>)/g, '<span class="hl-tag">$1</span>');

        // Dot-access properties
        result = this._highlightProperties(result);

        // Operators
        result = this._highlightOperators(result, [/:=/g]);

        // Restore strings
        result = this._restoreStrings(result, extracted.strings);

        return result;
    },

    // ================================
    // C / C++
    // ================================
    highlightC: function (code) {
        let result = this.escapeHtml(code);

        // Extract strings into placeholders
        const extracted = this._extractStrings(result, [
            /"[^"]*"/g,
            /'.'/g
        ]);
        result = extracted.result;

        // Comments
        result = result.replace(/(\/\/[^\n]*)/g, '<span class="hl-comment">$1</span>');
        result = result.replace(/(\/\*[\s\S]*?\*\/)/g, '<span class="hl-comment">$1</span>');

        // Preprocessor
        result = result.replace(/^(\s*#\w+[^\n]*)/gm, '<span class="hl-attr">$1</span>');

        // Keywords (removed this, nullptr — this goes to value)
        ['auto', 'break', 'case', 'char', 'const', 'continue', 'default', 'do', 'double', 'else',
            'enum', 'extern', 'float', 'for', 'goto', 'if', 'inline', 'int', 'long', 'register',
            'restrict', 'return', 'short', 'signed', 'sizeof', 'static', 'struct', 'switch', 'typedef',
            'union', 'unsigned', 'void', 'volatile', 'while', '_Bool', '_Complex', '_Imaginary',
            'class', 'public', 'private', 'protected', 'virtual', 'override', 'final', 'explicit',
            'friend', 'mutable', 'namespace', 'new', 'delete', 'operator', 'template',
            'throw', 'try', 'catch', 'using', 'constexpr', 'decltype', 'noexcept',
            'static_assert', 'thread_local', 'alignas', 'alignof'].forEach(kw => {
                result = result.replace(new RegExp(`\\b(${kw})\\b(?![^<]*>)`, 'g'), '<span class="hl-keyword">$1</span>');
            });

        // Value literals
        result = result.replace(/\b(this|true|false|NULL|nullptr)\b(?![^<]*>)/g, '<span class="hl-value">$1</span>');

        // Type names (capitalized identifiers)
        result = result.replace(/\b([A-Z][a-zA-Z0-9_]*)\b(?![^<]*>)/g, '<span class="hl-tag">$1</span>');

        result = result.replace(/\b(0x[0-9a-fA-F]+|\d+\.?\d*[fFlLuU]*)\b(?![^<]*>)/g, '<span class="hl-number">$1</span>');
        result = result.replace(/\b([a-zA-Z_][\w]*)(\s*\()/g, (m, fn, p) => m.includes('class="') ? m : `<span class="hl-function">${fn}</span>${p}`);

        // Dot-access and arrow-access properties
        result = this._highlightProperties(result);
        result = result.replace(/-&gt;([a-zA-Z_][\w]*)(?!\s*\()(?![^<]*>)/g, function (m, prop) {
            return '-&gt;<span class="hl-property">' + prop + '</span>';
        });

        // Operators
        result = this._highlightOperators(result);

        // Restore strings
        result = this._restoreStrings(result, extracted.strings);

        return result;
    },

    // ================================
    // RUST
    // ================================
    highlightRust: function (code) {
        let result = this.escapeHtml(code);

        // Extract strings into placeholders
        const extracted = this._extractStrings(result, [
            /"[^"]*"/g,
            /'[^']*'/g
        ]);
        result = extracted.result;

        // Comments
        result = result.replace(/(\/\/[^\n]*)/g, '<span class="hl-comment">$1</span>');
        result = result.replace(/(\/\*[\s\S]*?\*\/)/g, '<span class="hl-comment">$1</span>');

        // Keywords (removed self — now hl-value)
        ['fn', 'let', 'mut', 'const', 'struct', 'enum', 'impl', 'trait', 'if', 'else', 'match',
            'for', 'while', 'loop', 'return', 'use', 'mod', 'pub', 'super', 'crate',
            'async', 'await', 'move', 'ref', 'where', 'type', 'static', 'unsafe', 'extern',
            'as', 'in', 'break', 'continue', 'dyn', 'abstract', 'macro'].forEach(kw => {
                result = result.replace(new RegExp(`\\b(${kw})\\b(?![^<]*>)`, 'g'), '<span class="hl-keyword">$1</span>');
            });

        // Value literals
        result = result.replace(/\b(self|true|false|Some|None|Ok|Err)\b(?![^<]*>)/g, '<span class="hl-value">$1</span>');

        // Types and special values
        result = result.replace(/\b([A-Z][a-zA-Z0-9_]*)\b(?![^<]*>)/g, '<span class="hl-tag">$1</span>');
        result = result.replace(/\b(\d+\.?\d*)\b(?![^<]*>)/g, '<span class="hl-number">$1</span>');
        result = result.replace(/\b([a-zA-Z_][\w]*)(\s*[\(\!])/g, (m, fn, p) => m.includes('class="') ? m : `<span class="hl-function">${fn}</span>${p}`);

        // Dot-access properties
        result = this._highlightProperties(result);

        // Operators
        result = this._highlightOperators(result);

        // Restore strings
        result = this._restoreStrings(result, extracted.strings);

        return result;
    },

    // ================================
    // SHELL / BASH
    // ================================
    highlightShell: function (code) {
        let result = this.escapeHtml(code);

        // Extract strings into placeholders
        const extracted = this._extractStrings(result, [
            /"[^"]*"/g,
            /'[^']*'/g
        ]);
        result = extracted.result;

        // Comments
        result = result.replace(/(#[^\n]*)/g, '<span class="hl-comment">$1</span>');

        // Keywords
        ['if', 'then', 'else', 'elif', 'fi', 'for', 'while', 'do', 'done', 'case', 'esac',
            'function', 'return', 'exit', 'in', 'select', 'until', 'break', 'continue',
            'local', 'declare', 'readonly', 'export', 'unset', 'shift', 'set'].forEach(kw => {
                result = result.replace(new RegExp(`\\b(${kw})\\b(?![^<]*>)`, 'g'), '<span class="hl-keyword">$1</span>');
            });

        // Built-in commands
        ['echo', 'printf', 'read', 'source', 'alias', 'cd', 'pwd', 'ls', 'rm', 'cp', 'mv',
            'mkdir', 'chmod', 'chown', 'grep', 'sed', 'awk', 'cat', 'head', 'tail', 'find',
            'xargs', 'sort', 'uniq', 'wc', 'curl', 'wget', 'tar', 'ssh', 'sudo',
            'test', 'eval', 'exec', 'trap'].forEach(cmd => {
                result = result.replace(new RegExp(`\\b(${cmd})\\b(?![^<]*>)`, 'g'), '<span class="hl-function">$1</span>');
            });

        // Value literals
        result = result.replace(/\b(true|false)\b(?![^<]*>)/g, '<span class="hl-value">$1</span>');

        // Variables
        result = result.replace(/(\$\{?[a-zA-Z_][\w]*\}?)/g, '<span class="hl-attr">$1</span>');
        result = result.replace(/(\$[0-9@#\?\*!$-])/g, '<span class="hl-attr">$1</span>');

        // Numbers
        result = result.replace(/\b(\d+)\b(?![^<]*>)/g, '<span class="hl-number">$1</span>');

        // Shell operators (pipes, redirects)
        result = result.replace(/((?:\|&amp;|\|{1,2}|&amp;{1,2}|&gt;{1,2}|&lt;{1,2}))(?![^<]*>)/g, '<span class="hl-punct">$1</span>');

        // Restore strings
        result = this._restoreStrings(result, extracted.strings);

        return result;
    },

    // ================================
    // SQL
    // ================================
    highlightSQL: function (code) {
        let result = this.escapeHtml(code);

        // Extract strings into placeholders
        const extracted = this._extractStrings(result, [
            /'[^']*'/g
        ]);
        result = extracted.result;

        // Comments
        result = result.replace(/(--[^\n]*)/g, '<span class="hl-comment">$1</span>');
        result = result.replace(/(\/\*[\s\S]*?\*\/)/g, '<span class="hl-comment">$1</span>');

        // Keywords (case insensitive) — removed TRUE, FALSE, NULL — now hl-value
        ['SELECT', 'FROM', 'WHERE', 'AND', 'OR', 'NOT', 'IN', 'LIKE', 'BETWEEN', 'IS',
            'INSERT', 'INTO', 'VALUES', 'UPDATE', 'SET', 'DELETE', 'CREATE', 'TABLE', 'ALTER', 'DROP',
            'INDEX', 'VIEW', 'DATABASE', 'GRANT', 'REVOKE', 'JOIN', 'INNER', 'LEFT', 'RIGHT', 'OUTER',
            'FULL', 'CROSS', 'ON', 'AS', 'ORDER', 'BY', 'ASC', 'DESC', 'GROUP', 'HAVING', 'DISTINCT',
            'UNION', 'ALL', 'LIMIT', 'OFFSET', 'TOP', 'CASE', 'WHEN', 'THEN', 'ELSE', 'END', 'IF',
            'EXISTS', 'PRIMARY', 'KEY', 'FOREIGN', 'REFERENCES', 'CONSTRAINT', 'UNIQUE', 'DEFAULT',
            'AUTO_INCREMENT', 'INT', 'VARCHAR', 'TEXT', 'DATE', 'DATETIME', 'BOOLEAN', 'FLOAT',
            'COUNT', 'SUM', 'AVG', 'MIN', 'MAX', 'COALESCE', 'CONCAT', 'SUBSTRING'].forEach(kw => {
                result = result.replace(new RegExp(`\\b(${kw})\\b(?![^<]*>)`, 'gi'), '<span class="hl-keyword">$1</span>');
            });

        // Value literals
        result = result.replace(/\b(TRUE|FALSE|NULL|true|false|null)\b(?![^<]*>)/g, '<span class="hl-value">$1</span>');

        result = result.replace(/\b(\d+\.?\d*)\b(?![^<]*>)/g, '<span class="hl-number">$1</span>');
        result = result.replace(/\b([a-zA-Z_][\w]*)(\s*\()/g, (m, fn, p) => m.includes('class="') ? m : `<span class="hl-function">${fn}</span>${p}`);

        // Restore strings
        result = this._restoreStrings(result, extracted.strings);

        return result;
    },

    // ================================
    // JSON
    // ================================
    highlightJSON: function (code) {
        let result = this.escapeHtml(code);

        // Section keys — keys whose value is { or [ (structural/nesting keys)
        // Use hl-function to distinguish from leaf property keys
        result = result.replace(/("[\w-]+")(\s*:\s*(?=[{\[]))/g, '<span class="hl-function">$1</span>$2');

        // Leaf property names (remaining keys not already highlighted)
        result = result.replace(/("[\w-]+")(\s*:)(?![^<]*>)/g, '<span class="hl-property">$1</span>$2');

        // String values after colon (object string values)
        result = result.replace(/:(\s*)("[^"]*")/g, ':<span class="hl-string">$1$2</span>');

        // String values inside arrays (not preceded by colon) — hl-attr for variety
        result = result.replace(/([\[,]\s*)("[^"]*")(?=\s*[,\]])/g, function (m, prefix, str) {
            if (m.indexOf('class="') !== -1) return m;
            return prefix + '<span class="hl-attr">' + str + '</span>';
        });

        // Booleans and null as value literals
        result = result.replace(/\b(true|false|null)\b/g, '<span class="hl-value">$1</span>');

        // Numbers
        result = result.replace(/:\s*(-?\d+\.?\d*)/g, ': <span class="hl-number">$1</span>');

        // Colons — hl-operator so they stand out from other punctuation
        result = result.replace(/(:)(?![^<]*>)/g, '<span class="hl-operator">$1</span>');

        // Array brackets — hl-tag to differentiate from object braces
        result = result.replace(/([\[\]])(?![^<]*>)/g, '<span class="hl-tag">$1</span>');

        // Object braces and commas — hl-punct
        result = result.replace(/([{},])(?![^<]*>)/g, '<span class="hl-punct">$1</span>');

        return result;
    },

    // ================================
    // MAIN HIGHLIGHT FUNCTION
    // ================================
    highlight: function (code, language) {
        const lang = language.toLowerCase();

        switch (lang) {
            case 'html': case 'htm': case 'xml': case 'svg':
                return this.highlightHTML(code);
            case 'css': case 'scss': case 'sass': case 'less':
                return this.highlightCSS(code);
            case 'js': case 'javascript': case 'ts': case 'typescript': case 'jsx': case 'tsx':
                return this.highlightJS(code);
            case 'py': case 'python':
                return this.highlightPython(code);
            case 'java': case 'kt': case 'kotlin': case 'cs': case 'csharp':
                return this.highlightJava(code);
            case 'php':
                return this.highlightPHP(code);
            case 'rb': case 'ruby':
                return this.highlightRuby(code);
            case 'swift':
                return this.highlightSwift(code);
            case 'go': case 'golang':
                return this.highlightGo(code);
            case 'c': case 'cpp': case 'c++': case 'h': case 'hpp':
                return this.highlightC(code);
            case 'rs': case 'rust':
                return this.highlightRust(code);
            case 'sh': case 'bash': case 'shell': case 'zsh':
                return this.highlightShell(code);
            case 'sql': case 'mysql': case 'pgsql': case 'sqlite':
                return this.highlightSQL(code);
            case 'json':
                return this.highlightJSON(code);
            default:
                return this.escapeHtml(code);
        }
    }
};

if (typeof window !== 'undefined') {
    window.SyntaxHighlighter = SyntaxHighlighter;
}
