/**
 * JottNote Weather Mode
 * Fetches weather data via open-meteo API with IP geolocation fallback
 */

const WEATHER_CODES = {
    0: { desc: 'Clear', icon: '☀️', group: 'clear' },
    1: { desc: 'Mainly Clear', icon: '🌤️', group: 'clear' },
    2: { desc: 'Partly Cloudy', icon: '⛅', group: 'cloudy' },
    3: { desc: 'Overcast', icon: '☁️', group: 'cloudy' },
    45: { desc: 'Foggy', icon: '🌫️', group: 'cloudy' },
    48: { desc: 'Rime Fog', icon: '🌫️', group: 'cloudy' },
    51: { desc: 'Light Drizzle', icon: '🌦️', group: 'rain' },
    53: { desc: 'Drizzle', icon: '🌦️', group: 'rain' },
    55: { desc: 'Dense Drizzle', icon: '🌧️', group: 'rain' },
    61: { desc: 'Light Rain', icon: '🌧️', group: 'rain' },
    63: { desc: 'Rain', icon: '🌧️', group: 'rain' },
    65: { desc: 'Heavy Rain', icon: '🌧️', group: 'rain' },
    66: { desc: 'Freezing Rain', icon: '🌨️', group: 'rain' },
    67: { desc: 'Heavy Freezing Rain', icon: '🌨️', group: 'rain' },
    71: { desc: 'Light Snow', icon: '🌨️', group: 'snow' },
    73: { desc: 'Snow', icon: '❄️', group: 'snow' },
    75: { desc: 'Heavy Snow', icon: '❄️', group: 'snow' },
    77: { desc: 'Snow Grains', icon: '❄️', group: 'snow' },
    80: { desc: 'Light Showers', icon: '🌦️', group: 'rain' },
    81: { desc: 'Showers', icon: '🌧️', group: 'rain' },
    82: { desc: 'Heavy Showers', icon: '🌧️', group: 'rain' },
    85: { desc: 'Snow Showers', icon: '🌨️', group: 'snow' },
    86: { desc: 'Heavy Snow Showers', icon: '🌨️', group: 'snow' },
    95: { desc: 'Thunderstorm', icon: '⛈️', group: 'thunder' },
    96: { desc: 'Thunderstorm + Hail', icon: '⛈️', group: 'thunder' },
    99: { desc: 'Severe Thunderstorm', icon: '⛈️', group: 'thunder' }
};

const GRADIENTS = {
    clear: 'linear-gradient(160deg, #56CCF2 0%, #2F80ED 100%)',
    cloudy: 'linear-gradient(160deg, #8E9EAB 0%, #606C88 100%)',
    rain: 'linear-gradient(160deg, #4B6CB7 0%, #182848 100%)',
    snow: 'linear-gradient(160deg, #E6DADA 0%, #274046 100%)',
    thunder: 'linear-gradient(160deg, #373B44 0%, #4286f4 100%)',
    night_clear: 'linear-gradient(160deg, #0F2027 0%, #203A43 50%, #2C5364 100%)',
    night_cloudy: 'linear-gradient(160deg, #1a1a2e 0%, #16213e 100%)',
    night_rain: 'linear-gradient(160deg, #0f0c29 0%, #302b63 50%, #24243e 100%)',
    night_snow: 'linear-gradient(160deg, #1c1c2e 0%, #3a3a5c 100%)',
    night_thunder: 'linear-gradient(160deg, #0d0d2b 0%, #1a1a4e 50%, #2d1b69 100%)',
    sunrise_clear: 'linear-gradient(160deg, #FFB75E 0%, #ED8F03 100%)',
    sunrise_cloudy: 'linear-gradient(160deg, #F09819 0%, #EDDE5D 100%)',
    sunrise_rain: 'linear-gradient(160deg, #8A2387 0%, #E94057 50%, #F27121 100%)',
    sunrise_snow: 'linear-gradient(160deg, #E6DADA 0%, #FBB03B 100%)',
    sunrise_thunder: 'linear-gradient(160deg, #4B1248 0%, #F0C27B 100%)',
    sunset_clear: 'linear-gradient(160deg, #FF512F 0%, #F09819 100%)',
    sunset_cloudy: 'linear-gradient(160deg, #e65c00 0%, #F9D423 100%)',
    sunset_rain: 'linear-gradient(160deg, #1f4037 0%, #99f2c8 100%)',
    sunset_snow: 'linear-gradient(160deg, #7b4397 0%, #dc2430 100%)',
    sunset_thunder: 'linear-gradient(160deg, #2c3e50 0%, #fd746c 100%)'
};

/**
 * Resolve lat/lon/name from city or IP
 */
async function resolveLocation(city) {
    if (city && city.trim()) {
        const geoUrl = `https://geocoding-api.open-meteo.com/v1/search?name=${encodeURIComponent(city.trim())}&count=1&language=en&format=json`;
        const geoRes = await fetch(geoUrl);
        const geoData = await geoRes.json();
        if (!geoData.results || geoData.results.length === 0) {
            throw new Error(`City "${city}" not found`);
        }
        const loc = geoData.results[0];
        return {
            lat: loc.latitude,
            lon: loc.longitude,
            name: loc.name + (loc.admin1 ? `, ${loc.admin1}` : ''),
            timezone: loc.timezone || Intl.DateTimeFormat().resolvedOptions().timeZone
        };
    }
    const ipRes = await fetch('https://get.geojs.io/v1/ip/geo.json');
    const ipData = await ipRes.json();
    return {
        lat: parseFloat(ipData.latitude),
        lon: parseFloat(ipData.longitude),
        name: ipData.city + (ipData.region ? `, ${ipData.region}` : ''),
        timezone: ipData.timezone || Intl.DateTimeFormat().resolvedOptions().timeZone
    };
}

function getLocalTimeInfo(timezone) {
    const now = new Date();
    let localTimeStr = '';
    let localHour = now.getHours();

    try {
        const options = { timeZone: timezone, hour: 'numeric', minute: '2-digit', hour12: true };
        localTimeStr = now.toLocaleTimeString('en-US', options);

        const parts = new Intl.DateTimeFormat('en-US', { timeZone: timezone, hour: 'numeric', hourCycle: 'h23' }).formatToParts(now);
        const hourPart = parts.find(p => p.type === 'hour');
        if (hourPart) localHour = parseInt(hourPart.value, 10);
    } catch (e) {
        localTimeStr = now.toLocaleTimeString('en-US', { hour: 'numeric', minute: '2-digit', hour12: true });
    }

    return { timeStr: localTimeStr, hour: localHour };
}

function getGradient(code, hour) {
    const info = WEATHER_CODES[code] || { group: 'clear' };
    if (hour >= 5 && hour < 8) return GRADIENTS['sunrise_' + info.group] || GRADIENTS.sunrise_clear;
    if (hour >= 17 && hour < 19) return GRADIENTS['sunset_' + info.group] || GRADIENTS.sunset_clear;
    if (hour >= 19 || hour < 5) return GRADIENTS['night_' + info.group] || GRADIENTS.night_clear;
    return GRADIENTS[info.group] || GRADIENTS.clear;
}

function getInfo(code, precip = 0) {
    const base = WEATHER_CODES[code] || { desc: 'Unknown', icon: '🌡️', group: 'clear' };

    // Add accumulation info for rain or snow
    if (precip > 0 && (base.group === 'rain' || base.group === 'snow' || base.group === 'thunder')) {
        return {
            ...base,
            desc: `${base.desc} ${precip.toFixed(1)}"`
        };
    }

    return base;
}

function isNightHour(hour) {
    return hour >= 19 || hour < 6;
}

/**
 * Fetch current weather with hourly forecast
 */
export async function fetchWeather(city) {
    const loc = await resolveLocation(city);
    const url = `https://api.open-meteo.com/v1/forecast?latitude=${loc.lat}&longitude=${loc.lon}&current=temperature_2m,apparent_temperature,weather_code,wind_speed_10m,relative_humidity_2m,precipitation&hourly=temperature_2m,weather_code&temperature_unit=fahrenheit&precipitation_unit=inch&forecast_days=1&timezone=auto`;
    const res = await fetch(url);
    const data = await res.json();

    const current = data.current;
    const code = current.weather_code;
    const precip = current.precipitation || 0;
    const info = getInfo(code, precip);

    const localInfo = getLocalTimeInfo(loc.timezone);
    const isNight = isNightHour(localInfo.hour);

    // Build hourly forecast (next 6 hours from current hour)
    const hourly = [];
    if (data.hourly) {
        for (let i = 0; i < 6; i++) {
            const h = (localInfo.hour + i) % 24;
            const idx = data.hourly.time.findIndex(t => new Date(t).getHours() === h);
            if (idx >= 0) {
                const hCode = data.hourly.weather_code[idx];
                hourly.push({
                    hour: h === 0 ? '12AM' : h < 12 ? `${h}AM` : h === 12 ? '12PM' : `${h - 12}PM`,
                    temp: Math.round(data.hourly.temperature_2m[idx]),
                    icon: getInfo(hCode).icon,
                    isNight: isNightHour(h)
                });
            }
        }
    }

    return {
        type: 'current',
        temperature: Math.round(current.temperature_2m),
        feelsLike: Math.round(current.apparent_temperature),
        unit: '°',
        description: info.desc,
        icon: info.icon,
        windSpeed: Math.round(current.wind_speed_10m),
        humidity: current.relative_humidity_2m,
        location: loc.name,
        localTime: localInfo.timeStr,
        isNight,
        gradient: getGradient(code, localInfo.hour),
        hourly
    };
}

/**
 * Fetch tomorrow's weather
 */
export async function fetchWeatherTomorrow(city) {
    const loc = await resolveLocation(city);
    const url = `https://api.open-meteo.com/v1/forecast?latitude=${loc.lat}&longitude=${loc.lon}&daily=temperature_2m_max,temperature_2m_min,weather_code,apparent_temperature_max,wind_speed_10m_max,precipitation_sum&hourly=temperature_2m,weather_code&temperature_unit=fahrenheit&precipitation_unit=inch&forecast_days=2&timezone=auto`;
    const res = await fetch(url);
    const data = await res.json();

    const tomorrow = 1; // index 1 = tomorrow
    const code = data.daily.weather_code[tomorrow];
    const precip = data.daily.precipitation_sum[tomorrow] || 0;
    const info = getInfo(code, precip);

    const localInfo = getLocalTimeInfo(loc.timezone);

    // Build hourly for tomorrow (8AM–8PM spread)
    const hourly = [];
    if (data.hourly) {
        const tomorrowDate = data.daily.time[tomorrow];
        const hoursToShow = [6, 9, 12, 15, 18, 21];
        for (const h of hoursToShow) {
            const timeStr = `${tomorrowDate}T${String(h).padStart(2, '0')}:00`;
            const idx = data.hourly.time.indexOf(timeStr);
            if (idx >= 0) {
                const hCode = data.hourly.weather_code[idx];
                hourly.push({
                    hour: h === 0 ? '12AM' : h < 12 ? `${h}AM` : h === 12 ? '12PM' : `${h - 12}PM`,
                    temp: Math.round(data.hourly.temperature_2m[idx]),
                    icon: getInfo(hCode).icon,
                    isNight: isNightHour(h)
                });
            }
        }
    }

    return {
        type: 'tomorrow',
        tempHigh: Math.round(data.daily.temperature_2m_max[tomorrow]),
        tempLow: Math.round(data.daily.temperature_2m_min[tomorrow]),
        feelsLike: Math.round(data.daily.apparent_temperature_max[tomorrow]),
        unit: '°',
        description: info.desc,
        icon: info.icon,
        windSpeed: Math.round(data.daily.wind_speed_10m_max[tomorrow]),
        location: loc.name,
        localTime: localInfo.timeStr,
        gradient: getGradient(code, 12), // default to daytime gradient for tomorrow
        hourly
    };
}

/**
 * Fetch 7-day daily forecast
 */
export async function fetchWeatherDaily(city) {
    const loc = await resolveLocation(city);
    const url = `https://api.open-meteo.com/v1/forecast?latitude=${loc.lat}&longitude=${loc.lon}&current=temperature_2m,apparent_temperature,weather_code,precipitation&daily=temperature_2m_max,temperature_2m_min,weather_code,precipitation_sum&temperature_unit=fahrenheit&precipitation_unit=inch&forecast_days=7&timezone=auto`;
    const res = await fetch(url);
    const data = await res.json();

    const current = data.current;
    const code = current.weather_code;
    const precip = current.precipitation || 0;
    const info = getInfo(code, precip);

    const localInfo = getLocalTimeInfo(loc.timezone);
    const isNight = isNightHour(localInfo.hour);

    const days = [];
    const dayNames = ['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat'];
    for (let i = 0; i < data.daily.time.length; i++) {
        const d = new Date(data.daily.time[i] + 'T12:00:00');
        const dCode = data.daily.weather_code[i];
        const dPrecip = data.daily.precipitation_sum[i] || 0;
        const dInfo = getInfo(dCode, dPrecip);

        days.push({
            day: i === 0 ? 'Today' : dayNames[d.getDay()],
            high: Math.round(data.daily.temperature_2m_max[i]),
            low: Math.round(data.daily.temperature_2m_min[i]),
            icon: dInfo.icon,
            desc: dInfo.desc

        });
    }

    return {
        type: 'daily',
        temperature: Math.round(current.temperature_2m),
        feelsLike: Math.round(current.apparent_temperature),
        unit: '°',
        description: info.desc,
        icon: info.icon,
        location: loc.name,
        localTime: localInfo.timeStr,
        isNight,
        gradient: getGradient(code, localInfo.hour),
        days
    };
}

/**
 * Parse weather command to determine mode and city.
 * Supports: weather, weather:city, weather:daily, weather:tomorrow,
 *           weather:daily:city, weather:city:daily, etc.
 * @returns {{ mode: 'current'|'tomorrow'|'daily', city: string|null }}
 */
export function parseWeatherCommand(content) {
    const firstLine = (content.split('\n')[0] || '').trim();
    const colonIdx = firstLine.indexOf(':');

    if (colonIdx < 0) {
        return { mode: 'current', city: null };
    }

    // Split all parts after "weather:"
    const parts = firstLine.substring(colonIdx + 1).split(':').map(p => p.trim());
    const MODE_WORDS = { daily: 'daily', '7day': 'daily', '7d': 'daily', tomorrow: 'tomorrow', tmr: 'tomorrow' };

    let mode = 'current';
    let city = null;

    for (const part of parts) {
        if (!part) continue;
        const lower = part.toLowerCase();
        if (MODE_WORDS[lower]) {
            mode = MODE_WORDS[lower];
        } else {
            // Preserve original case for city name
            city = part;
        }
    }

    return { mode, city };
}
