/**
 * JottNote UUID & Password Generator Mode
 * Generates UUIDs, passwords, hex strings, and base64 strings
 * Uses crypto.getRandomValues() for secure randomness
 */

/**
 * Process UUID command and return results
 * @param {string} content - Full note content
 * @returns {Object} { type, results: string[], description: string }
 */
export function processUuidCommand(content) {
    const lines = content.split('\n');
    const firstLine = (lines[0] || '').trim().toLowerCase();

    // Just "uuid" alone = help
    if (firstLine === 'uuid' && lines.length <= 1) {
        return { type: 'help', results: [], description: '' };
    }

    // "uuid" with Enter (second line exists) = generate 1 UUID
    if (firstLine === 'uuid' && lines.length > 1) {
        return { type: 'uuid', results: [generateUUIDv4()], description: 'UUID v4' };
    }

    // uuid:N — generate N UUIDs
    const countMatch = firstLine.match(/^uuid:(\d+)$/);
    if (countMatch) {
        const n = Math.min(parseInt(countMatch[1], 10), 50);
        const results = [];
        for (let i = 0; i < n; i++) results.push(generateUUIDv4());
        return { type: 'uuid', results, description: `${n} UUID${n > 1 ? 's' : ''} v4` };
    }

    // uuid:pass or uuid:pass:N
    const passMatch = firstLine.match(/^uuid:pass(?::(\d+))?$/);
    if (passMatch) {
        const length = passMatch[1] ? Math.min(parseInt(passMatch[1], 10), 128) : 16;
        return { type: 'password', results: [generatePassword(length)], description: `${length}-char password` };
    }

    // uuid:hex
    if (firstLine === 'uuid:hex') {
        return { type: 'hex', results: [generateHex(32)], description: '32-char hex string' };
    }

    // uuid:base64
    if (firstLine === 'uuid:base64') {
        return { type: 'base64', results: [generateBase64(32)], description: 'Base64 string' };
    }

    return { type: 'help', results: [], description: '' };
}

/**
 * Generate a UUID v4 string
 * @returns {string}
 */
function generateUUIDv4() {
    const bytes = new Uint8Array(16);
    crypto.getRandomValues(bytes);
    // Set version (4) and variant (10)
    bytes[6] = (bytes[6] & 0x0F) | 0x40;
    bytes[8] = (bytes[8] & 0x3F) | 0x80;

    const hex = Array.from(bytes, b => b.toString(16).padStart(2, '0')).join('');
    return `${hex.slice(0, 8)}-${hex.slice(8, 12)}-${hex.slice(12, 16)}-${hex.slice(16, 20)}-${hex.slice(20)}`;
}

/**
 * Generate a random password
 * @param {number} length
 * @returns {string}
 */
function generatePassword(length) {
    const chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789!@#$%^&*_-+=';
    const bytes = new Uint8Array(length);
    crypto.getRandomValues(bytes);
    return Array.from(bytes, b => chars[b % chars.length]).join('');
}

/**
 * Generate a random hex string
 * @param {number} length - Number of hex characters
 * @returns {string}
 */
function generateHex(length) {
    const bytes = new Uint8Array(Math.ceil(length / 2));
    crypto.getRandomValues(bytes);
    return Array.from(bytes, b => b.toString(16).padStart(2, '0')).join('').slice(0, length);
}

/**
 * Generate a random base64 string
 * @param {number} byteLength - Number of random bytes
 * @returns {string}
 */
function generateBase64(byteLength) {
    const bytes = new Uint8Array(byteLength);
    crypto.getRandomValues(bytes);
    const binary = Array.from(bytes, b => String.fromCharCode(b)).join('');
    return btoa(binary);
}
