/**
 * JottNote Timer Module
 * Handles stopwatch, countdown, and pomodoro timers
 */

// Timer state
let timerState = {
    type: null,         // 'stopwatch', 'countdown', 'pomodoro'
    running: false,
    paused: false,
    startTime: null,
    pauseStartedAt: null, // When current pause began
    duration: 0,        // For countdown (in seconds)
    remaining: 0,       // For countdown (in seconds)
    title: null,
    // Pomodoro specific
    workDuration: 0,
    breakDuration: 0,
    isBreak: false,
    completedRounds: 0, // Number of completed work+break cycles
    intervalId: null
};

let onTickCallback = null;
let onCompleteCallback = null;

/**
 * Parse timer command from text
 * @param {string} text - Timer command (e.g., "timer 5", "timer pomo")
 * @returns {Object|null} Parsed timer config
 */
export function parseTimerCommand(text) {
    const trimmed = text.trim().toLowerCase();
    const originalText = text.trim(); // Keep original case for title

    // Check if it starts with "timer"
    if (!trimmed.startsWith('timer')) {
        return null;
    }

    const parts = trimmed.substring(5).trim();
    const originalParts = originalText.substring(5).trim();

    // Just "timer" - start stopwatch
    if (!parts) {
        return { type: 'stopwatch' };
    }

    // Stopwatch with laps: "timer:stopwatch" or "timer stopwatch"
    const swMatch = parts.match(/^:?stopwatch(?:\s+(\w))?$/);
    if (swMatch) {
        const sub = swMatch[1]; // p, l, s, r or undefined
        if (!sub) return { type: 'lapwatch', action: 'start' };
        if (sub === 'p') return { type: 'lapwatch', action: 'pause' };
        if (sub === 'l') return { type: 'lapwatch', action: 'lap' };
        if (sub === 's') return { type: 'lapwatch', action: 'stop' };
        if (sub === 'r') return { type: 'lapwatch', action: 'reset' };
    }

    // Control commands
    if (parts === 'p') return { type: 'control', action: 'pause' };
    if (parts === 'r') return { type: 'control', action: 'restart' };
    if (parts === 's' || parts === '0') return { type: 'control', action: 'stop' };
    if (parts === 'breathe') return { type: 'breathe' };

    // Standard pomodoro
    if (parts === 'pomo') {
        return { type: 'pomodoro', work: 25 * 60, break: 5 * 60 };
    }

    // Parse time values
    // Check for title in format "timer 5: Do laundry" or "timer 5:30: Do laundry"
    let timeStr = parts;
    let title = null;

    // Match pattern: number(s) followed by colon and space, then title
    // e.g., "5: Do laundry" or "5:30: Do laundry"
    const titleMatch = originalParts.match(/^([\d.:]+):\s+(.+)$/);
    if (titleMatch) {
        timeStr = titleMatch[1].toLowerCase();
        title = titleMatch[2];
    } else {
        // Also support " : " separator format
        const colonTitleIndex = parts.indexOf(' : ');
        if (colonTitleIndex > 0) {
            timeStr = parts.substring(0, colonTitleIndex).trim();
            title = originalParts.substring(colonTitleIndex + 3).trim();
        }
    }

    // Check for pomodoro format (e.g., "timer 5 1" for 5 min work, 1 min rest)
    const spaceParts = timeStr.split(/\s+/);
    if (spaceParts.length === 2) {
        const work = parseTimeValue(spaceParts[0]);
        const breakTime = parseTimeValue(spaceParts[1]);
        if (work > 0 && breakTime > 0) {
            return { type: 'pomodoro', work: work * 60, break: breakTime * 60, title };
        }
    }

    // Single countdown value
    const duration = parseTimeValue(timeStr);
    if (duration > 0) {
        return { type: 'countdown', duration: duration * 60, title };
    }

    return null;
}

/**
 * Parse time value (e.g., "5", "5.5", "5:30")
 * @param {string} str
 * @returns {number} Minutes
 */
function parseTimeValue(str) {
    // Format: "5:30" (5 minutes 30 seconds)
    if (str.includes(':')) {
        const [min, sec] = str.split(':').map(Number);
        if (!isNaN(min) && !isNaN(sec)) {
            return min + sec / 60;
        }
    }

    // Format: "5.5" or "5" (minutes)
    const num = parseFloat(str);
    if (!isNaN(num) && num > 0) {
        return num;
    }

    return 0;
}

/**
 * Start a timer
 * @param {Object} config - Timer configuration
 * @param {Function} onTick - Callback on each tick with { time, progress }
 * @param {Function} onComplete - Callback when timer completes
 */
export function startTimer(config, onTick, onComplete) {
    stopTimer();

    onTickCallback = onTick;
    onCompleteCallback = onComplete;

    timerState = {
        ...timerState,
        type: config.type,
        running: true,
        paused: false,
        startTime: Date.now(),
        pauseStartedAt: null,
        title: config.title || null
    };

    if (config.type === 'stopwatch') {
        timerState.intervalId = setInterval(tickStopwatch, 1000);
        tickStopwatch();
    } else if (config.type === 'countdown') {
        timerState.duration = config.duration;
        timerState.remaining = config.duration;
        timerState.intervalId = setInterval(tickCountdown, 1000);
        tickCountdown();
    } else if (config.type === 'pomodoro') {
        timerState.workDuration = config.work;
        timerState.breakDuration = config.break;
        timerState.duration = config.work;
        timerState.remaining = config.work;
        timerState.isBreak = false;
        timerState.completedRounds = 0;
        timerState.intervalId = setInterval(tickPomodoro, 1000);
        tickPomodoro();
    }
}

/**
 * Tick for stopwatch
 */
function tickStopwatch() {
    if (!timerState.running || timerState.paused) return;

    const elapsed = Math.floor((Date.now() - timerState.startTime) / 1000);

    if (onTickCallback) {
        onTickCallback({
            type: 'stopwatch',
            time: formatTime(elapsed),
            seconds: elapsed,
            progress: null
        });
    }
}

/**
 * Tick for countdown
 */
function tickCountdown() {
    if (!timerState.running || timerState.paused) return;

    const elapsed = Math.floor((Date.now() - timerState.startTime) / 1000);
    timerState.remaining = Math.max(0, timerState.duration - elapsed);

    const progress = 1 - (timerState.remaining / timerState.duration);

    if (onTickCallback) {
        onTickCallback({
            type: 'countdown',
            time: formatTime(timerState.remaining),
            seconds: timerState.remaining,
            progress,
            title: timerState.title
        });
    }

    if (timerState.remaining <= 0) {
        stopTimer();
        if (onCompleteCallback) {
            onCompleteCallback({ type: 'countdown', title: timerState.title });
        }
    }
}

/**
 * Tick for pomodoro
 */
function tickPomodoro() {
    if (!timerState.running || timerState.paused) return;

    const elapsed = Math.floor((Date.now() - timerState.startTime) / 1000);
    timerState.remaining = Math.max(0, timerState.duration - elapsed);

    const progress = 1 - (timerState.remaining / timerState.duration);

    // Determine progress bar direction based on phase
    // Work phase: top to bottom (normal)
    // Break phase: bottom to top (reversed)
    const reverseProgress = timerState.isBreak;

    if (onTickCallback) {
        onTickCallback({
            type: 'pomodoro',
            time: formatTime(timerState.remaining),
            seconds: timerState.remaining,
            progress,
            isBreak: timerState.isBreak,
            completedRounds: timerState.completedRounds,
            reverseProgress,
            title: timerState.title
        });
    }

    if (timerState.remaining <= 0) {
        // Switch between work and break
        const wasBreak = timerState.isBreak;
        timerState.isBreak = !timerState.isBreak;

        // Increment round counter when transitioning from break to work
        if (wasBreak && !timerState.isBreak) {
            timerState.completedRounds++;
        }

        timerState.duration = timerState.isBreak ? timerState.breakDuration : timerState.workDuration;
        timerState.remaining = timerState.duration;
        timerState.startTime = Date.now();
        timerState.pauseStartedAt = null;

        if (onCompleteCallback) {
            onCompleteCallback({
                type: 'pomodoro',
                phase: timerState.isBreak ? 'break' : 'work',
                completedRounds: timerState.completedRounds,
                title: timerState.title
            });
        }
    }
}

/**
 * Format seconds to MM:SS
 * @param {number} seconds
 * @returns {string}
 */
export function formatTime(seconds) {
    const mins = Math.floor(seconds / 60);
    const secs = seconds % 60;
    return `${mins.toString().padStart(2, '0')}:${secs.toString().padStart(2, '0')}`;
}

/**
 * Pause/resume timer
 */
export function togglePause() {
    if (!timerState.running) return;

    if (timerState.paused) {
        // Resume — shift startTime forward by how long we were paused
        const pauseDuration = Date.now() - timerState.pauseStartedAt;
        timerState.startTime += pauseDuration;
        timerState.paused = false;
        timerState.pauseStartedAt = null;
    } else {
        // Pause — record when the pause began
        timerState.paused = true;
        timerState.pauseStartedAt = Date.now();
    }

    return timerState.paused;
}

/**
 * Restart timer
 */
export function restartTimer() {
    if (!timerState.running) return;

    timerState.startTime = Date.now();
    timerState.pauseStartedAt = null;
    timerState.paused = false;

    if (timerState.type === 'countdown' || timerState.type === 'pomodoro') {
        timerState.remaining = timerState.duration;
    }
}

/**
 * Stop timer
 */
export function stopTimer() {
    if (timerState.intervalId) {
        clearInterval(timerState.intervalId);
        timerState.intervalId = null;
    }

    timerState.running = false;
    timerState.paused = false;
    timerState.type = null;

    if (onTickCallback) {
        onTickCallback(null);
    }
}

/**
 * Check if timer is running
 * @returns {boolean}
 */
export function isTimerRunning() {
    return timerState.running;
}

/**
 * Check if timer is paused
 * @returns {boolean}
 */
export function isTimerPaused() {
    return timerState.paused;
}

/**
 * Get current timer state
 * @returns {Object}
 */
export function getTimerState() {
    return { ...timerState };
}
