/**
 * JottNote Themes
 * System: Strict Theme Styling (8-Color Palette + Text/BG)
 * Theme Structure:
 * - Background Color
 * - Text Primary Color (for theme name/UI text)
 * - 8 Palette Accent Colors (displayed as dots):
 *   C1: Text Secondary
 *   C2: Accent 1 (Math)
 *   C3: Accent 2 (List)
 *   C4: Accent 3 (Sum/Bookmarks)
 *   C5: Accent 4 (Code/Timer)
 *   C6: Accent 5 (Count/Sort)
 *   C7: Accent 6
 *   C8: Border
 */

export const THEMES = {
    vancouver: {
        name: 'Vancouver',
        colors: {
            '--bg-primary': '#1B1C1E',
            '--text-primary': '#FFFFFF',
            '--text-secondary': '#0E4B62',
            '--accent-1-main': '#008B8B',
            '--accent-2-main': '#4E7E6B',
            '--accent-3-main': '#BD56FF',
            '--accent-4-main': '#4BA524',
            '--accent-5-main': '#FFA041',
            '--accent-6-main': '#FF3F3F',
            '--border': '#FFF0D4',
            /* Derived Defaults */
            '--bg-secondary': '#2A2B2D',
            '--bg-tertiary': '#2A2B2D',
            '--text-muted': '#0E4B62',
            '--accent': '#008B8B',
            '--accent-light': '#4E7E6B'
        }
    },
    gundam: {
        name: 'Gundam Wing',
        colors: {
            '--bg-primary': '#FFFFFF',
            '--text-primary': '#000000',
            '--text-secondary': '#0066CC',
            '--accent-1-main': '#003D7A',
            '--accent-2-main': '#4E84E1',
            '--accent-3-main': '#FF3B30',
            '--accent-4-main': '#FFCC33',
            '--accent-5-main': '#666666',
            '--accent-6-main': '#3A82F6',
            '--border': '#FFD644',
            /* Derived */
            '--bg-secondary': '#F5F5F5',
            '--bg-tertiary': '#F5F5F5',
            '--text-muted': '#0066CC',
            '--accent': '#003D7A',
            '--accent-light': '#4E84E1'
        }
    },
    totoro: {
        name: 'Totoro',
        colors: {
            '--bg-primary': '#F3F3F3',
            '--text-primary': '#4B4B4B',
            '--text-secondary': '#5A8471',
            '--accent-1-main': '#3E5F52',
            '--accent-2-main': '#8EB16B',
            '--accent-3-main': '#D3B98C',
            '--accent-4-main': '#FFA6A6',
            '--accent-5-main': '#C69E7C',
            '--accent-6-main': '#87B364',
            '--border': '#FDE49E',
            /* Derived */
            '--bg-secondary': '#E8E8E8',
            '--bg-tertiary': '#E8E8E8',
            '--text-muted': '#5A8471',
            '--accent': '#3E5F52',
            '--accent-light': '#8EB16B'
        }
    },
    piccolo: {
        name: 'Piccolo',
        colors: {
            '--bg-primary': '#F1F8F1',
            '--text-primary': '#1A1A1A',
            '--text-secondary': '#A626B5',
            '--accent-1-main': '#8E52A2',
            '--accent-2-main': '#4CB050',
            '--accent-3-main': '#52D03E',
            '--accent-4-main': '#FF527F',
            '--accent-5-main': '#6A96E4',
            '--accent-6-main': '#FF3F3F',
            '--border': '#ACFF3F',
            /* Derived */
            '--bg-secondary': '#E5F2E5',
            '--bg-tertiary': '#E5F2E5',
            '--text-muted': '#A626B5',
            '--accent': '#8E52A2',
            '--accent-light': '#4CB050'
        }
    },
    tartan: {
        name: 'Tartan',
        colors: {
            '--bg-primary': '#F9F5EF',
            '--text-primary': '#2D2926',
            '--text-secondary': '#FF7C2C',
            '--accent-1-main': '#CC5521',
            '--accent-2-main': '#FFA352',
            '--accent-3-main': '#59A85B',
            '--accent-4-main': '#7793AD',
            '--accent-5-main': '#5A391A',
            '--accent-6-main': '#3B2118',
            '--border': '#F5C8AD',
            /* Derived */
            '--bg-secondary': '#F0EBE5',
            '--bg-tertiary': '#F0EBE5',
            '--text-muted': '#FF7C2C',
            '--accent': '#CC5521',
            '--accent-light': '#FFA352'
        }
    },
    muaddib: {
        name: "Muad'Dib",
        colors: {
            '--bg-primary': '#E5D5B5',
            '--text-primary': '#5A3E2B',
            '--text-secondary': '#A66226',
            '--accent-1-main': '#6A401A',
            '--accent-2-main': '#3B2110',
            '--accent-3-main': '#41A2F6',
            '--accent-4-main': '#31413F',
            '--accent-5-main': '#D3C382',
            '--accent-6-main': '#422518',
            '--border': '#F4DC76',
            /* Derived */
            '--bg-secondary': '#D9C9A9',
            '--bg-tertiary': '#D9C9A9',
            '--text-muted': '#A66226',
            '--accent': '#6A401A',
            '--accent-light': '#3B2110'
        }
    },
    sanrio: {
        name: 'Sanrio',
        colors: {
            '--bg-primary': '#FFFFFF',
            '--text-primary': '#E874B2',
            '--text-secondary': '#91BCE3',
            '--accent-1-main': '#B0C4E8',
            '--accent-2-main': '#BDD0D9',
            '--accent-3-main': '#96C9A2',
            '--accent-4-main': '#A1C0B0',
            '--accent-5-main': '#F1C5CB',
            '--accent-6-main': '#D6A6D1',
            '--border': '#D1E2D6',
            /* Derived */
            '--bg-secondary': '#F9F9F9',
            '--bg-tertiary': '#F9F9F9',
            '--text-muted': '#91BCE3',
            '--accent': '#B0C4E8',
            '--accent-light': '#BDD0D9'
        }
    },
    tokyo: {
        name: 'Tokyo Drift',
        colors: {
            '--bg-primary': '#1A1A1A',
            '--text-primary': '#FFFFFF',
            '--text-secondary': '#66CCFF',
            '--accent-1-main': '#008B8B',
            '--accent-2-main': '#789B99',
            '--accent-3-main': '#C492F2',
            '--accent-4-main': '#32CD32',
            '--accent-5-main': '#FF9B3F',
            '--accent-6-main': '#FF7575',
            '--border': '#7A6F56',
            /* Derived */
            '--bg-secondary': '#242424',
            '--bg-tertiary': '#242424',
            '--text-muted': '#66CCFF',
            '--accent': '#008B8B',
            '--accent-light': '#789B99'
        }
    },
    vendetta: {
        name: 'Vendetta',
        colors: {
            '--bg-primary': '#121212',
            '--text-primary': '#FFFFFF',
            '--text-secondary': '#C61A1A',
            '--accent-1-main': '#FBE5C3',
            '--accent-2-main': '#88A69E',
            '--accent-3-main': '#3B3B3B',
            '--accent-4-main': '#D65518',
            '--accent-5-main': '#FF4500',
            '--accent-6-main': '#FF1A1A',
            '--border': '#8B1A1A',
            /* Derived */
            '--bg-secondary': '#1E1E1E',
            '--bg-tertiary': '#1E1E1E',
            '--text-muted': '#C61A1A',
            '--accent': '#FBE5C3',
            '--accent-light': '#88A69E'
        }
    },
    mononoke: {
        name: 'Mononoke',
        colors: {
            '--bg-primary': '#05120B',
            '--text-primary': '#FFFFFF',
            '--text-secondary': '#228B22',
            '--accent-1-main': '#76D7C4',
            '--accent-2-main': '#5D7B7E',
            '--accent-3-main': '#8B4513',
            '--accent-4-main': '#E74C3C',
            '--accent-5-main': '#998E7B',
            '--accent-6-main': '#329932',
            '--border': '#FF6347',
            /* Derived */
            '--bg-secondary': '#0A1F14',
            '--bg-tertiary': '#0A1F14',
            '--text-muted': '#228B22',
            '--accent': '#76D7C4',
            '--accent-light': '#5D7B7E'
        }
    },
    a24: {
        name: 'A24',
        colors: {
            '--bg-primary': '#141A18',
            '--text-primary': '#D3D3D3',
            '--text-secondary': '#5B7B72',
            '--accent-1-main': '#234D42',
            '--accent-2-main': '#4A6293',
            '--accent-3-main': '#8B2323',
            '--accent-4-main': '#D3622D',
            '--accent-5-main': '#E5B13A',
            '--accent-6-main': '#50E3C2',
            '--border': '#433B5A',
            /* Derived */
            '--bg-secondary': '#1A2220',
            '--bg-tertiary': '#1A2220',
            '--text-muted': '#5B7B72',
            '--accent': '#234D42',
            '--accent-light': '#4A6293'
        }
    },
    agrabah: {
        name: 'Agrabah',
        colors: {
            '--bg-primary': '#0B0B24',
            '--text-primary': '#FFFFFF',
            '--text-secondary': '#FFEA00',
            '--accent-1-main': '#D4D400',
            '--accent-2-main': '#FFF59D',
            '--accent-3-main': '#9D50FF',
            '--accent-4-main': '#00E5FF',
            '--accent-5-main': '#FF6BB5',
            '--accent-6-main': '#FFA326',
            '--border': '#A32650',
            /* Derived */
            '--bg-secondary': '#141433',
            '--bg-tertiary': '#141433',
            '--text-muted': '#FFEA00',
            '--accent': '#D4D400',
            '--accent-light': '#FFF59D'
        }
    },
    knight: {
        name: 'Knight',
        colors: {
            '--bg-primary': '#1C1C1E',
            '--text-primary': '#8E8E93',
            '--text-secondary': '#3D5A80',
            '--accent-1-main': '#24315E',
            '--accent-2-main': '#6E85B7',
            '--accent-3-main': '#32CD32',
            '--accent-4-main': '#FF3B30',
            '--accent-5-main': '#FFD700',
            '--accent-6-main': '#9932CC',
            '--border': '#1B4F72',
            /* Derived */
            '--bg-secondary': '#2A2A2C',
            '--bg-tertiary': '#2A2A2C',
            '--text-muted': '#3D5A80',
            '--accent': '#24315E',
            '--accent-light': '#6E85B7'
        }
    },
    shadow: {
        name: 'Shadow Moses',
        colors: {
            '--bg-primary': '#242929',
            '--text-primary': '#D0D0D0',
            '--text-secondary': '#A3C6D0',
            '--accent-1-main': '#6E9197',
            '--accent-2-main': '#405659',
            '--accent-3-main': '#A1141E',
            '--accent-4-main': '#729B8C',
            '--accent-5-main': '#9B8C80',
            '--accent-6-main': '#FFFFFF',
            '--border': '#7DBE8B',
            /* Derived */
            '--bg-secondary': '#2E3333',
            '--bg-tertiary': '#2E3333',
            '--text-muted': '#A3C6D0',
            '--accent': '#6E9197',
            '--accent-light': '#405659'
        }
    }
};

export const CUSTOM_THEME_KEY = 'jottnote_custom_theme';

const REQUIRED_COLOR_KEYS = [
    '--bg-primary', '--text-primary', '--text-secondary',
    '--accent-1-main', '--accent-2-main', '--accent-3-main',
    '--accent-4-main', '--accent-5-main', '--accent-6-main',
    '--border'
];

/**
 * Validate a custom theme object from imported JSON
 * @param {Object} themeData - parsed JSON
 * @returns {{ valid: boolean, errors: string[] }}
 */
export function validateCustomTheme(themeData) {
    const errors = [];

    if (!themeData || typeof themeData !== 'object') {
        return { valid: false, errors: ['Invalid theme data'] };
    }

    if (!themeData.name || typeof themeData.name !== 'string' || !themeData.name.trim()) {
        errors.push('Theme must have a non-empty "name" string');
    }

    if (!themeData.colors || typeof themeData.colors !== 'object') {
        errors.push('Theme must have a "colors" object');
        return { valid: false, errors };
    }

    for (const key of REQUIRED_COLOR_KEYS) {
        if (!themeData.colors[key]) {
            errors.push(`Missing required color: ${key}`);
        } else if (!/^#[0-9A-Fa-f]{6}$/.test(themeData.colors[key])) {
            errors.push(`Invalid hex color for ${key}: ${themeData.colors[key]}`);
        }
    }

    return { valid: errors.length === 0, errors };
}
