/**
 * JottNote Storage Service
 * Handles note persistence using Chrome's storage API
 */

const STORAGE_KEY = 'jottnote_notes';
const CURRENT_NOTE_KEY = 'jottnote_current';
const SETTINGS_KEY = 'jottnote_settings';
const TEMPLATES_KEY = 'jottnote_templates';

/**
 * Promise-based write queue to serialize read-modify-write operations
 * Prevents race conditions when multiple writes happen concurrently
 */
let _storageQueue = Promise.resolve();
function withStorageLock(fn) {
    _storageQueue = _storageQueue.then(fn, fn);
    return _storageQueue;
}

/**
 * Note schema:
 * {
 *   id: string (UUID),
 *   content: string,
 *   createdAt: string (ISO date),
 *   modifiedAt: string (ISO date)
 * }
 */

/**
 * Generate a UUID for new notes
 */
function generateId() {
  return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, function (c) {
    const r = Math.random() * 16 | 0;
    const v = c === 'x' ? r : (r & 0x3 | 0x8);
    return v.toString(16);
  });
}

/**
 * Get all notes from storage
 * @returns {Promise<Array>} Array of notes sorted by modifiedAt (newest first)
 */
export async function getNotes() {
  try {
    const result = await chrome.storage.local.get(STORAGE_KEY);
    const notes = result[STORAGE_KEY] || [];
    // Sort by modifiedAt descending (newest first)
    return notes.sort((a, b) => new Date(b.modifiedAt) - new Date(a.modifiedAt));
  } catch (error) {
    console.error('Error getting notes:', error);
    return [];
  }
}

/**
 * Get notes sorted by creation order (newest created first) — stable for navigation
 * Falls back to modifiedAt for notes missing createdAt
 * @returns {Promise<Array>}
 */
export async function getNotesByCreation() {
  try {
    const result = await chrome.storage.local.get(STORAGE_KEY);
    const notes = result[STORAGE_KEY] || [];
    return notes.sort((a, b) => {
      const aTime = new Date(a.createdAt || a.modifiedAt);
      const bTime = new Date(b.createdAt || b.modifiedAt);
      return bTime - aTime;
    });
  } catch (error) {
    console.error('Error getting notes:', error);
    return [];
  }
}

/**
 * Get a single note by ID
 * @param {string} id - Note ID
 * @returns {Promise<Object|null>} Note object or null if not found
 */
export async function getNote(id) {
  const notes = await getNotes();
  return notes.find(note => note.id === id) || null;
}

/**
 * Save or update a note
 * @param {Object} note - Note object (must have id)
 * @returns {Promise<Object>} Saved note
 */
export function saveNote(note) {
  return withStorageLock(async () => {
    try {
      const notes = await getNotes();
      const existingIndex = notes.findIndex(n => n.id === note.id);

      const now = new Date().toISOString();
      const updatedNote = {
        ...note,
        modifiedAt: now
      };

      if (existingIndex >= 0) {
        notes[existingIndex] = updatedNote;
      } else {
        updatedNote.createdAt = now;
        notes.unshift(updatedNote);
      }

      await chrome.storage.local.set({ [STORAGE_KEY]: notes });
      return updatedNote;
    } catch (error) {
      console.error('Error saving note:', error);
      throw error;
    }
  });
}

/**
 * Create a new note
 * @param {string} content - Initial content (default empty)
 * @returns {Promise<Object>} New note
 */
export async function createNote(content = '') {
  const now = new Date().toISOString();
  const note = {
    id: generateId(),
    content,
    createdAt: now,
    modifiedAt: now
  };

  return await saveNote(note);
}

/**
 * Delete a note by ID
 * @param {string} id - Note ID to delete
 * @returns {Promise<boolean>} True if deleted
 */
export function deleteNote(id) {
  return withStorageLock(async () => {
    try {
      const notes = await getNotes();
      const filtered = notes.filter(note => note.id !== id);
      await chrome.storage.local.set({ [STORAGE_KEY]: filtered });
      return true;
    } catch (error) {
      console.error('Error deleting note:', error);
      return false;
    }
  });
}

/**
 * Get the current note ID
 * @returns {Promise<string|null>} Current note ID
 */
export async function getCurrentNoteId() {
  try {
    const result = await chrome.storage.local.get(CURRENT_NOTE_KEY);
    return result[CURRENT_NOTE_KEY] || null;
  } catch (error) {
    console.error('Error getting current note ID:', error);
    return null;
  }
}

/**
 * Set the current note ID
 * @param {string} id - Note ID
 */
export async function setCurrentNoteId(id) {
  try {
    await chrome.storage.local.set({ [CURRENT_NOTE_KEY]: id });
  } catch (error) {
    console.error('Error setting current note ID:', error);
  }
}

/**
 * Get settings
 * @returns {Promise<Object>} Settings object
 */
export async function getSettings() {
  try {
    const result = await chrome.storage.local.get(SETTINGS_KEY);
    return result[SETTINGS_KEY] || {
      theme: 'dark',
      textSize: 'm',
      checkTrigger: '/x',
      skipKeywordsOnCopy: true,
      skipCheckTriggersOnCopy: true,
      stripLeadingSpaces: true,
      stripListNumbers: true,
      stripBullets: true,
      stripMarkdown: false,
      stripEmptyLines: false,
      disableLinkShortening: false,
      exportOmitKeyword: false,
      exportUseFirstLine: false,
      autoLaunch: false,
      newNoteOnLaunch: false,
      autoDeletePeriod: 4, // 0=Today, 1=1week, 2=1month, 3=1year, 4=Never
      autoCreatePeriod: 5, // 0=Always, 1=3mins, 2=30mins, 3=1hr, 4=1day, 5=Never
      showNoteCount: false,
      showTagCloud: true,
      backupCount: 12,
      backupInterval: 8, // 0=10mins, 1=30mins, 2=1hr, 3=3hrs, 4=6hrs, 5=12hrs, 6=1day, 7=1week, 8=Never
      // Keyword settings
      listKeywords: '',
      listMainKeyword: 'list',
      listCheckKeywords: '/x, done',
      listMainChecker: '/x',
      mathKeywords: '',
      mathMainKeyword: 'math',
      mathSigDigits: 2,
      mathThousandsComma: true,
      mathUpdateCurrencies: true,
      mathCustomRates: '',
      mathPrimarySymbol: '$',
      mathPrimaryCurrency: 'USD',
      mathSecondaryCurrency: 'CAD',
      sumKeywords: '',
      sumMainKeyword: 'sum',
      avgKeywords: '',
      avgMainKeyword: 'avg',
      countKeywords: '',
      countMainKeyword: 'count',
      codeKeywords: '',
      codeMainKeyword: 'code',
      timerKeywords: '',
      timerMainKeyword: 'timer',
      timerPauseOnClose: false,
      dateKeywords: '',
      dateMainKeyword: 'date',
      dateFormat: 'MMM D, YYYY',
      disableAllKeywords: false,
      disableAllLinks: false,
      defaultCodeLanguage: 'js',
      darkSyntaxTheme: 'one-dark',
      lightSyntaxTheme: 'github-light'
    };
  } catch (error) {
    console.error('Error getting settings:', error);
    return {};
  }
}

/**
 * Save settings
 * @param {Object} settings - Settings to save
 */
export function saveSettings(settings) {
  return withStorageLock(async () => {
    try {
      const current = await getSettings();
      const updated = { ...current, ...settings };
      await chrome.storage.local.set({ [SETTINGS_KEY]: updated });
      return updated;
    } catch (error) {
      console.error('Error saving settings:', error);
      throw error;
    }
  });
}

/**
 * Get note index in the sorted list
 * @param {string} id - Note ID
 * @returns {Promise<number>} Index or -1 if not found
 */
export async function getNoteIndex(id) {
  const notes = await getNotes();
  return notes.findIndex(note => note.id === id);
}

/**
 * Promote a note to the front (make it most recent)
 * @param {string} id - Note ID
 * @returns {Promise<Object>} Updated note
 */
export async function promoteNote(id) {
  const note = await getNote(id);
  if (note) {
    return await saveNote(note); // Updates modifiedAt, making it the most recently modified
  }
  return null;
}

/**
 * Template schema:
 * {
 *   id: string (UUID),
 *   name: string (lowercase, trimmed),
 *   content: string,
 *   createdAt: string (ISO date),
 *   modifiedAt: string (ISO date)
 * }
 */

/**
 * Get all templates from storage
 * @returns {Promise<Array>} Array of templates sorted alphabetically by name
 */
export async function getTemplates() {
  try {
    const result = await chrome.storage.local.get(TEMPLATES_KEY);
    const templates = result[TEMPLATES_KEY] || [];
    // Sort alphabetically by name
    return templates.sort((a, b) => a.name.localeCompare(b.name));
  } catch (error) {
    console.error('Error getting templates:', error);
    return [];
  }
}

/**
 * Get a single template by name (case-insensitive)
 * @param {string} name - Template name
 * @returns {Promise<Object|null>} Template object or null if not found
 */
export async function getTemplate(name) {
  const templates = await getTemplates();
  const normalizedName = name.toLowerCase().trim();
  return templates.find(t => t.name === normalizedName) || null;
}

/**
 * Save or update a template
 * @param {string} name - Template name
 * @param {string} content - Template content
 * @returns {Promise<Object>} Saved template
 */
export async function saveTemplate(name, content) {
  try {
    const templates = await getTemplates();
    const normalizedName = name.toLowerCase().trim();
    const existingIndex = templates.findIndex(t => t.name === normalizedName);

    const now = new Date().toISOString();
    const template = {
      id: existingIndex >= 0 ? templates[existingIndex].id : generateId(),
      name: normalizedName,
      content: content,
      modifiedAt: now
    };

    if (existingIndex >= 0) {
      templates[existingIndex] = template;
    } else {
      template.createdAt = now;
      templates.push(template);
    }

    await chrome.storage.local.set({ [TEMPLATES_KEY]: templates });
    return template;
  } catch (error) {
    console.error('Error saving template:', error);
    throw error;
  }
}

/**
 * Delete a template by name
 * @param {string} name - Template name to delete
 * @returns {Promise<boolean>} True if deleted
 */
export async function deleteTemplate(name) {
  try {
    const templates = await getTemplates();
    const normalizedName = name.toLowerCase().trim();
    const filtered = templates.filter(t => t.name !== normalizedName);
    await chrome.storage.local.set({ [TEMPLATES_KEY]: filtered });
    return true;
  } catch (error) {
    console.error('Error deleting template:', error);
    return false;
  }
}

const PLACEHOLDERS_KEY = 'jottnote_placeholders';

/**
 * Get all saved placeholders
 * @returns {Promise<Object>} Map of { key: value }
 */
export async function getPlaceholders() {
  try {
    const result = await chrome.storage.local.get(PLACEHOLDERS_KEY);
    return result[PLACEHOLDERS_KEY] || {};
  } catch (error) {
    console.error('Error getting placeholders:', error);
    return {};
  }
}

/**
 * Save placeholders (merges with existing)
 * @param {Object} map - { key: value } pairs to save
 * @returns {Promise<Object>} Updated placeholders map
 */
export async function savePlaceholders(map) {
  try {
    const existing = await getPlaceholders();
    const merged = { ...existing, ...map };
    await chrome.storage.local.set({ [PLACEHOLDERS_KEY]: merged });
    return merged;
  } catch (error) {
    console.error('Error saving placeholders:', error);
    throw error;
  }
}

/**
 * Delete a single placeholder by key
 * @param {string} key - Placeholder key to delete
 * @returns {Promise<boolean>} True if deleted
 */
export async function deletePlaceholder(key) {
  try {
    const placeholders = await getPlaceholders();
    delete placeholders[key];
    await chrome.storage.local.set({ [PLACEHOLDERS_KEY]: placeholders });
    return true;
  } catch (error) {
    console.error('Error deleting placeholder:', error);
    return false;
  }
}
