/**
 * JottNote Stopwatch Mode
 * Independent stopwatch with lap tracking
 */

let state = {
    running: false,
    paused: false,
    startTime: 0,
    elapsed: 0,
    pauseStart: 0,
    totalPaused: 0,
    laps: [],
    intervalId: null,
    onTick: null
};

/**
 * Parse stopwatch command from text
 * @param {string} text - Command text (e.g., "stopwatch", "stopwatch p")
 * @returns {Object|null} { action: 'start'|'pause'|'lap'|'stop'|'reset' }
 */
export function parseStopwatchCommand(text) {
    const trimmed = text.trim().toLowerCase();
    if (trimmed === 'stopwatch' || trimmed === 'stopwatch start') {
        return { action: 'start' };
    }
    if (trimmed === 'stopwatch p' || trimmed === 'stopwatch pause') {
        return { action: 'pause' };
    }
    if (trimmed === 'stopwatch l' || trimmed === 'stopwatch lap') {
        return { action: 'lap' };
    }
    if (trimmed === 'stopwatch s' || trimmed === 'stopwatch stop') {
        return { action: 'stop' };
    }
    if (trimmed === 'stopwatch r' || trimmed === 'stopwatch reset') {
        return { action: 'reset' };
    }
    return null;
}

/**
 * Start the stopwatch
 * @param {Function} onTick - Callback with { time, ms, elapsed, laps }
 */
export function startStopwatch(onTick) {
    if (state.running && !state.paused) return;

    if (state.paused) {
        // Resume
        state.totalPaused += Date.now() - state.pauseStart;
        state.paused = false;
        state.onTick = onTick || state.onTick;
        tick();
        return;
    }

    // Fresh start
    state.running = true;
    state.paused = false;
    state.startTime = Date.now();
    state.elapsed = 0;
    state.totalPaused = 0;
    state.laps = [];
    state.onTick = onTick;

    if (state.intervalId) clearInterval(state.intervalId);
    state.intervalId = setInterval(tick, 50);
    tick();
}

/**
 * Stop the stopwatch
 */
export function stopStopwatch() {
    state.running = false;
    state.paused = false;
    if (state.intervalId) {
        clearInterval(state.intervalId);
        state.intervalId = null;
    }
}

/**
 * Toggle pause
 * @returns {boolean} Whether now paused
 */
export function toggleStopwatchPause() {
    if (!state.running) return false;

    if (state.paused) {
        state.totalPaused += Date.now() - state.pauseStart;
        state.paused = false;
        if (!state.intervalId) {
            state.intervalId = setInterval(tick, 50);
        }
        tick();
        return false;
    } else {
        state.paused = true;
        state.pauseStart = Date.now();
        if (state.intervalId) {
            clearInterval(state.intervalId);
            state.intervalId = null;
        }
        return true;
    }
}

/**
 * Record a lap
 * @returns {Object|null} Lap info { index, time, delta }
 */
export function lapStopwatch() {
    if (!state.running) return null;

    const elapsed = getCurrentElapsed();
    const prevLapTime = state.laps.length > 0 ? state.laps[state.laps.length - 1].elapsed : 0;
    const delta = elapsed - prevLapTime;

    const lap = {
        index: state.laps.length + 1,
        elapsed,
        delta,
        timeStr: formatTime(elapsed),
        deltaStr: formatTime(delta)
    };

    state.laps.push(lap);
    tick();
    return lap;
}

/**
 * Reset the stopwatch
 */
export function resetStopwatch() {
    stopStopwatch();
    state.elapsed = 0;
    state.laps = [];
    state.totalPaused = 0;
}

/**
 * Check if stopwatch is running
 * @returns {boolean}
 */
export function isStopwatchRunning() {
    return state.running;
}

/**
 * Check if stopwatch is paused
 * @returns {boolean}
 */
export function isStopwatchPaused() {
    return state.paused;
}

/**
 * Get current laps
 * @returns {Array}
 */
export function getStopwatchLaps() {
    return [...state.laps];
}

function getCurrentElapsed() {
    if (!state.running) return state.elapsed;
    const pausedTime = state.paused ? (Date.now() - state.pauseStart) : 0;
    return Date.now() - state.startTime - state.totalPaused - pausedTime;
}

function formatTime(ms) {
    const totalSecs = Math.floor(ms / 1000);
    const mins = Math.floor(totalSecs / 60).toString().padStart(2, '0');
    const secs = (totalSecs % 60).toString().padStart(2, '0');
    const centis = Math.floor((ms % 1000) / 10).toString().padStart(2, '0');
    return { mins, secs, centis, display: `${mins}:${secs}.${centis}` };
}

function tick() {
    if (!state.running || !state.onTick) return;

    const elapsed = getCurrentElapsed();
    state.elapsed = elapsed;
    const time = formatTime(elapsed);

    state.onTick({
        time: `${time.mins}:${time.secs}`,
        ms: `.${time.centis}`,
        elapsed,
        laps: [...state.laps]
    });
}
