/**
 * JottNote Stats Mode
 * Handles sum, average, and count calculations
 */

/**
 * Extract all numbers from content (excluding first line if it's a keyword)
 * @param {string} content - Note content
 * @returns {number[]} Array of numbers found
 */
export function extractNumbers(content) {
    const lines = content.split('\n');

    // Check if first line is a keyword
    const firstLine = lines[0].trim().toLowerCase();
    const isKeyword = ['sum', 'avg', 'average', 'count', 'math', 'list', 'code', 'timer'].some(
        kw => firstLine === kw || firstLine.startsWith(kw + ':')
    );

    // Skip first line if it's a keyword
    const contentLines = isKeyword ? lines.slice(1) : lines;
    const text = contentLines.join('\n');

    // Find all numbers (including decimals and negatives)
    // Ignore numbers that are part of comments
    const numbers = [];

    for (const line of contentLines) {
        // Skip comment lines
        if (line.trim().startsWith('//')) continue;

        // Match numbers: -123, 123.45, .5, etc.
        const matches = line.match(/-?\d+\.?\d*|\.\d+/g);
        if (matches) {
            for (const match of matches) {
                const num = parseFloat(match);
                if (!isNaN(num)) {
                    numbers.push(num);
                }
            }
        }
    }

    return numbers;
}

/**
 * Calculate sum of numbers
 * @param {number[]} numbers
 * @returns {number}
 */
export function calculateSum(numbers) {
    return numbers.reduce((acc, n) => acc + n, 0);
}

/**
 * Calculate average of numbers
 * @param {number[]} numbers
 * @returns {number}
 */
export function calculateAverage(numbers) {
    if (numbers.length === 0) return 0;
    return calculateSum(numbers) / numbers.length;
}

/**
 * Count syllables in a word (approximate)
 * @param {string} word
 * @returns {number}
 */
function countSyllables(word) {
    word = word.toLowerCase().trim();

    // Remove non-letters
    word = word.replace(/[^a-z]/g, '');
    if (word.length === 0) return 0;
    if (word.length <= 3) return 1;

    // Count vowel groups
    const vowelGroups = word.match(/[aeiouy]+/g);
    let syllables = vowelGroups ? vowelGroups.length : 0;

    // Adjust for silent e at end — but only if the word is long enough
    // and the e isn't the only vowel sound (e.g. "the", "more", "store")
    if (word.endsWith('e') && !word.endsWith('le') && word.length > 4) {
        // Only subtract if there are other vowel groups to carry the sound
        if (syllables > 1) syllables--;
    }

    // Adjust for consonant+le endings (e.g. "table", "simple", "little")
    if (word.endsWith('le') && word.length > 3) {
        const prev = word[word.length - 3];
        if (prev && !/[aeiouy]/.test(prev)) syllables++;
    }

    // Common suffixes that add syllables
    if (word.endsWith('tion') || word.endsWith('sion')) {
        // Already counted by vowel groups, no adjustment needed
    } else if (word.endsWith('ed') && word.length > 3) {
        // "ed" is silent in most cases (walked, played) but not after t/d (wanted, needed)
        const beforeEd = word[word.length - 3];
        if (beforeEd !== 't' && beforeEd !== 'd') {
            if (syllables > 1) syllables--;
        }
    }

    // Minimum 1 syllable per word
    return Math.max(1, syllables);
}

/**
 * Count content statistics with readability scores
 * @param {string} content - Note content
 * @returns {Object} { items, words, chars, sentences, fleschReadingEase, fleschKincaidGrade }
 */
export function countStats(content) {
    const lines = content.split('\n');

    // Check if first line is a keyword
    const firstLine = lines[0].trim().toLowerCase();
    const isKeyword = ['sum', 'avg', 'average', 'count', 'math', 'list', 'code', 'timer'].some(
        kw => firstLine === kw || firstLine.startsWith(kw + ':')
    );

    // Skip first line if it's a keyword
    const contentLines = isKeyword ? lines.slice(1) : lines;
    const text = contentLines.join(' '); // Join with spaces for proper sentence counting

    // Count non-empty lines (items)
    const items = contentLines.filter(l => l.trim().length > 0).length;

    // Count words (split on whitespace, filter out pure numbers/symbols)
    const allTokens = text.trim().split(/\s+/).filter(w => w.length > 0);
    const words = allTokens.length;

    // For syllable counting, only use tokens that contain letters
    const wordArray = allTokens.filter(w => /[a-zA-Z]/.test(w));

    // Count all characters (including spaces)
    const chars = contentLines.join('').length;

    // Count sentences — match sentence-ending punctuation
    // Each . ! or ? (or cluster like "?!" or "...") ends one sentence
    const sentenceEnders = text.match(/[.!?]+/g);
    let sentences = sentenceEnders ? sentenceEnders.length : 0;
    // If there's text but no punctuation, treat the whole thing as 1 sentence
    if (sentences === 0 && words > 0) sentences = 1;

    // Count syllables
    let totalSyllables = 0;
    for (const word of wordArray) {
        totalSyllables += countSyllables(word);
    }

    // Calculate Flesch Reading Ease Score
    // Formula: 206.835 - 1.015 * (words/sentences) - 84.6 * (syllables/words)
    // Use wordArray.length for syllable ratio (only actual words, not numbers/symbols)
    let fleschReadingEase = 0;
    const syllableWordCount = wordArray.length;
    if (words > 0 && sentences > 0 && syllableWordCount > 0) {
        fleschReadingEase = 206.835 - 1.015 * (words / sentences) - 84.6 * (totalSyllables / syllableWordCount);
        fleschReadingEase = Math.max(0, Math.min(100, fleschReadingEase)); // Clamp between 0-100
    }

    // Calculate Flesch-Kincaid Grade Level
    // Formula: 0.39 * (words/sentences) + 11.8 * (syllables/words) - 15.59
    let fleschKincaidGrade = 0;
    if (words > 0 && sentences > 0 && syllableWordCount > 0) {
        fleschKincaidGrade = 0.39 * (words / sentences) + 11.8 * (totalSyllables / syllableWordCount) - 15.59;
        fleschKincaidGrade = Math.max(0, fleschKincaidGrade); // Don't go below 0
    }

    return {
        items,
        words,
        chars,
        sentences,
        fleschReadingEase: fleschReadingEase.toFixed(2),
        fleschKincaidGrade: fleschKincaidGrade.toFixed(2)
    };
}

/**
 * Format number with thousands separator
 * @param {number} num
 * @param {number} decimals
 * @returns {string}
 */
function formatNumber(num, decimals = 2) {
    // Round to specified decimals
    const rounded = Math.round(num * Math.pow(10, decimals)) / Math.pow(10, decimals);

    // Format with commas
    const parts = rounded.toString().split('.');
    parts[0] = parts[0].replace(/\B(?=(\d{3})+(?!\d))/g, ',');

    return parts.join('.');
}

/**
 * Get stats result for display
 * @param {string} mode - 'sum', 'avg', or 'count'
 * @param {string} content - Note content
 * @returns {Object} { label, value, detail }
 */
export function getStatsResult(mode, content) {
    if (mode === 'sum') {
        const numbers = extractNumbers(content);
        const sum = calculateSum(numbers);
        return {
            label: 'Total',
            value: formatNumber(sum),
            detail: `${numbers.length} number${numbers.length === 1 ? '' : 's'}`
        };
    }

    if (mode === 'avg' || mode === 'average') {
        const numbers = extractNumbers(content);
        const avg = calculateAverage(numbers);
        return {
            label: 'Average',
            value: formatNumber(avg),
            detail: `${numbers.length} number${numbers.length === 1 ? '' : 's'}`
        };
    }

    if (mode === 'count') {
        const stats = countStats(content);
        return {
            label: 'Count',
            value: `${stats.items} lines`,
            detail: `${stats.words} words  •  ${stats.chars} chars`
        };
    }

    return null;
}
