/**
 * JottNote Spell Check Module
 * Uses Typo.js library for spell checking
 */

let typo = null;
let spellCheckEnabled = false;
let isInitializing = false;

/**
 * Initialize spell check with dictionary
 * @param {boolean} enabled - Whether spell check is enabled
 */
export async function initSpellCheck(enabled) {
    spellCheckEnabled = enabled;

    if (enabled && !typo && !isInitializing) {
        isInitializing = true;
        try {
            // Check if Typo is available (loaded via script tag in popup.html)
            if (typeof Typo === 'undefined') {
                console.warn('Typo.js not loaded yet, spell check unavailable');
                spellCheckEnabled = false;
                isInitializing = false;
                return;
            }

            // Load dictionary and affix files
            const affUrl = chrome.runtime.getURL('dictionaries/en_US-web.aff');
            const dicUrl = chrome.runtime.getURL('dictionaries/en_US-web.dic');

            // Fetch both files
            const [affResponse, dicResponse] = await Promise.all([
                fetch(affUrl),
                fetch(dicUrl)
            ]);

            const affData = await affResponse.text();
            const dicData = await dicResponse.text();

            // Create Typo instance with preloaded data
            typo = new Typo("en_US", affData, dicData, { platform: 'any' });

        } catch (error) {
            console.error('Failed to initialize spell checker:', error);
            typo = null;
            spellCheckEnabled = false;
        } finally {
            isInitializing = false;
        }
    }
}

/**
 * Set spell check enabled state
 * @param {boolean} enabled
 */
export async function setSpellCheckEnabled(enabled) {
    spellCheckEnabled = enabled;
    if (enabled && !typo) {
        await initSpellCheck(true);
    }
}

/**
 * Get spell check enabled state
 * @returns {boolean}
 */
export function isSpellCheckEnabled() {
    return spellCheckEnabled && typo !== null;
}

/**
 * Check if a word should be skipped
 * @param {string} word
 * @returns {boolean}
 */
function shouldSkipWord(word) {
    // Skip words that are:
    // - Too short (1-2 chars)
    // - Numbers
    // - URLs
    // - Email addresses
    // - All caps (likely acronyms)
    // - Have numbers mixed in
    if (word.length <= 2) return true;
    if (/^\d+$/.test(word)) return true;
    if (/^https?:\/\//.test(word)) return true;
    if (/@/.test(word)) return true;
    if (word === word.toUpperCase() && word.length <= 5) return true;
    if (/\d/.test(word)) return true;

    // Only check alphabetic words
    if (!/^[a-zA-Z]+$/.test(word)) return true;

    return false;
}

/**
 * Find all misspelled words in content
 * @param {string} content
 * @returns {Array<{word: string, start: number, end: number}>}
 */
export function findMisspelledWords(content) {
    if (!spellCheckEnabled || !typo) return [];

    const misspelled = [];
    const words = content.split(/\b/);
    let position = 0;

    for (const word of words) {
        const trimmed = word.trim();
        if (trimmed && !shouldSkipWord(trimmed)) {
            // Check with Typo.js
            if (!typo.check(trimmed)) {
                misspelled.push({
                    word: trimmed,
                    start: position,
                    end: position + word.length
                });
            }
        }
        position += word.length;
    }

    return misspelled;
}

/**
 * Get spelling suggestions for a word
 * @param {string} word
 * @returns {Array<string>}
 */
export function getSpellingSuggestions(word) {
    if (!typo) return [];

    try {
        const suggestions = typo.suggest(word);
        // Return max 5 suggestions
        return suggestions ? suggestions.slice(0, 5) : [];
    } catch (error) {
        console.error('Error getting suggestions:', error);
        return [];
    }
}
