/**
 * JottNote Sort Mode
 * Sorts lines based on different criteria
 */

/**
 * Parse sort command
 * @param {string} content - Note content
 * @returns {Object} { type, hasSortCommand, isActive, shouldExecute }
 */
export function parseSortCommand(content) {
    const lines = content.split('\n');
    const firstLine = lines[0] || '';
    const trimmed = firstLine.trim().toLowerCase();

    // Check for "sort:end" anywhere in content
    const hasEnd = lines.some(line => line.trim().toLowerCase() === 'sort:end');

    // Check for "sort" alone (help)
    if (trimmed === 'sort' && lines.length === 1) {
        return { type: 'help', hasSortCommand: true, isActive: false, shouldExecute: false };
    }

    // Check for sort commands
    let sortType = null;
    if (trimmed === 'sort:asc') {
        sortType = 'asc';
    } else if (trimmed === 'sort:desc') {
        sortType = 'desc';
    } else if (trimmed === 'sort:length') {
        sortType = 'length';
    } else if (trimmed === 'sort:abc') {
        sortType = 'abc';
    }

    if (sortType) {
        return {
            type: sortType,
            hasSortCommand: true,
            isActive: !hasEnd, // Active until sort:end is typed
            shouldExecute: hasEnd // Only execute when sort:end is present
        };
    }

    return { type: null, hasSortCommand: false, isActive: false, shouldExecute: false };
}

/**
 * Sort lines by ascending numeric value
 * @param {Array<string>} lines
 * @returns {Array<string>}
 */
function sortAscending(lines) {
    return [...lines].sort((a, b) => {
        const numA = parseFloat(a.trim());
        const numB = parseFloat(b.trim());

        // Handle non-numeric lines
        if (isNaN(numA) && isNaN(numB)) return 0;
        if (isNaN(numA)) return 1;
        if (isNaN(numB)) return -1;

        return numA - numB;
    });
}

/**
 * Sort lines by descending numeric value
 * @param {Array<string>} lines
 * @returns {Array<string>}
 */
function sortDescending(lines) {
    return [...lines].sort((a, b) => {
        const numA = parseFloat(a.trim());
        const numB = parseFloat(b.trim());

        // Handle non-numeric lines
        if (isNaN(numA) && isNaN(numB)) return 0;
        if (isNaN(numA)) return 1;
        if (isNaN(numB)) return -1;

        return numB - numA;
    });
}

/**
 * Sort lines by length
 * @param {Array<string>} lines
 * @returns {Array<string>}
 */
function sortByLength(lines) {
    return [...lines].sort((a, b) => a.length - b.length);
}

/**
 * Sort lines alphabetically
 * @param {Array<string>} lines
 * @returns {Array<string>}
 */
function sortAlphabetically(lines) {
    return [...lines].sort((a, b) => a.trim().localeCompare(b.trim()));
}

/**
 * Apply sort to content
 * @param {string} content - Note content
 * @param {string} type - Sort type (asc, desc, length, abc)
 * @returns {string} Sorted content
 */
export function applySortToContent(content, type) {
    const lines = content.split('\n');

    // Find the sort:end line
    const endLineIndex = lines.findIndex(line => line.trim().toLowerCase() === 'sort:end');

    if (endLineIndex === -1) {
        // No sort:end found, don't sort
        return content;
    }

    // Skip first line (keyword) and exclude sort:end line
    const firstLine = lines[0];
    const contentLines = lines.slice(1, endLineIndex);

    // Separate non-empty lines for sorting, track empty line positions
    const nonEmptyLines = [];
    const emptyLineIndices = [];
    contentLines.forEach((line, i) => {
        if (line.trim() === '') {
            emptyLineIndices.push(i);
        } else {
            nonEmptyLines.push(line);
        }
    });

    let sortedLines;
    switch (type) {
        case 'asc':
            sortedLines = sortAscending(nonEmptyLines);
            break;
        case 'desc':
            sortedLines = sortDescending(nonEmptyLines);
            break;
        case 'length':
            sortedLines = sortByLength(nonEmptyLines);
            break;
        case 'abc':
            sortedLines = sortAlphabetically(nonEmptyLines);
            break;
        default:
            sortedLines = nonEmptyLines;
    }

    // Append empty lines at the end (preserves them without index mismatch)
    const result = [...sortedLines, ...emptyLineIndices.map(() => '')];

    // Return sorted content (remove sort:end line and anything after it)
    return [firstLine, ...result].join('\n');
}

/**
 * Check if should show sort help
 * @param {string} content
 * @returns {boolean}
 */
export function shouldShowSortHelp(content) {
    const lines = content.split('\n');
    const firstLine = lines[0] || '';
    const trimmed = firstLine.trim().toLowerCase();

    // Show help only when first line is exactly "sort" with no other content
    // Note: Sort doesn't support custom keywords yet
    return trimmed === 'sort' && lines.length === 1;
}
