/**
 * Session Management Module
 * Handles saving, loading, and managing browser tab sessions
 */

const SESSIONS_STORAGE_KEY = 'jottnote_sessions';

/**
 * Check if a URL uses a safe protocol (http or https)
 */
function isSafeUrl(url) {
    try {
        const parsed = new URL(url);
        return parsed.protocol === 'http:' || parsed.protocol === 'https:';
    } catch {
        return false;
    }
}

/**
 * Get all sessions from storage
 */
export async function getSessions() {
    const result = await chrome.storage.local.get(SESSIONS_STORAGE_KEY);
    return result[SESSIONS_STORAGE_KEY] || [];
}

/**
 * Get a single session by ID
 */
export async function getSession(id) {
    const sessions = await getSessions();
    return sessions.find(s => s.id === id);
}

/**
 * Save a session to storage
 */
export async function saveSession(session) {
    const sessions = await getSessions();

    // Ensure session has required fields
    if (!session.id) {
        session.id = generateSessionId();
    }
    if (!session.dateCreated) {
        session.dateCreated = new Date().toISOString();
    }
    session.dateModified = new Date().toISOString();

    // Check if session already exists (update) or is new
    const existingIndex = sessions.findIndex(s => s.id === session.id);

    if (existingIndex !== -1) {
        // Update existing session
        sessions[existingIndex] = session;
    } else {
        // Add new session to the beginning
        sessions.unshift(session);
    }

    await chrome.storage.local.set({ [SESSIONS_STORAGE_KEY]: sessions });
    return session;
}

/**
 * Delete a session
 */
export async function deleteSession(id) {
    const sessions = await getSessions();
    const filtered = sessions.filter(s => s.id !== id);
    await chrome.storage.local.set({ [SESSIONS_STORAGE_KEY]: filtered });
}

/**
 * Update an existing session
 */
export async function updateSession(id, updates) {
    const sessions = await getSessions();
    const index = sessions.findIndex(s => s.id === id);

    if (index === -1) {
        throw new Error('Session not found');
    }

    sessions[index] = {
        ...sessions[index],
        ...updates,
        dateModified: new Date().toISOString()
    };

    await chrome.storage.local.set({ [SESSIONS_STORAGE_KEY]: sessions });
    return sessions[index];
}

/**
 * Capture current browser tabs and save as a session
 */
export async function saveCurrentSession(title, tags = []) {
    // Get all tabs from all windows
    const tabs = await chrome.tabs.query({});

    // Filter to only HTTP/HTTPS URLs (skip chrome://, extensions, etc.)
    const validTabs = tabs.filter(tab => {
        const url = tab.url || '';
        // Only HTTP/HTTPS
        if (!url.startsWith('http://') && !url.startsWith('https://')) {
            return false;
        }
        // Skip JottNote extension itself
        if (url.includes('chrome-extension://') || url.includes('/popup.html')) {
            return false;
        }
        return true;
    });

    // Create session object
    const session = {
        id: generateSessionId(),
        title: title || `Session ${new Date().toLocaleDateString()}`,
        urls: validTabs.map(tab => ({
            url: tab.url,
            title: tab.title || extractTitleFromUrl(tab.url)
        })),
        tags: validateTags(tags),
        dateCreated: new Date().toISOString(),
        dateModified: new Date().toISOString(),
        tabCount: validTabs.length
    };

    return await saveSession(session);
}

/**
 * Open a session - opens all URLs in new tabs
 */
export async function openSession(sessionId) {
    const session = await getSession(sessionId);

    if (!session) {
        throw new Error('Session not found');
    }

    // Open each URL in a new tab (validate protocol)
    for (const item of session.urls) {
        const urlString = typeof item === 'string' ? item : item.url;
        if (urlString && isSafeUrl(urlString)) {
            await chrome.tabs.create({ url: urlString, active: false });
        }
    }

    // Update last accessed time
    await updateSession(sessionId, {
        lastAccessed: new Date().toISOString()
    });
}

/**
 * Search sessions by title, tags, or URLs
 */
export async function searchSessions(query) {
    const sessions = await getSessions();

    if (!query || query.trim() === '') {
        return sessions;
    }

    const lowerQuery = query.toLowerCase();

    return sessions.filter(session => {
        // Search in title
        if (session.title.toLowerCase().includes(lowerQuery)) {
            return true;
        }

        // Search in tags
        if (session.tags && session.tags.some(tag => tag.toLowerCase().includes(lowerQuery))) {
            return true;
        }

        // Search in URLs
        if (session.urls && session.urls.some(item =>
            item.url.toLowerCase().includes(lowerQuery) ||
            (item.title && item.title.toLowerCase().includes(lowerQuery))
        )) {
            return true;
        }

        return false;
    });
}

/**
 * Get all unique tags from all sessions
 */
export async function getAllSessionTags() {
    const sessions = await getSessions();
    const tagSet = new Set();

    sessions.forEach(session => {
        if (session.tags) {
            session.tags.forEach(tag => tagSet.add(tag));
        }
    });

    return Array.from(tagSet).sort();
}

/**
 * Export sessions to JSON format (for download)
 */
export function exportSessionsToJSON(sessions) {
    return JSON.stringify(sessions, null, 2);
}

/**
 * Import sessions from JSON data
 */
export async function importSessionsFromJSON(jsonData) {
    try {
        const importedSessions = JSON.parse(jsonData);

        if (!Array.isArray(importedSessions)) {
            throw new Error('Invalid session data format');
        }

        const sessions = await getSessions();
        const results = {
            imported: 0,
            skipped: 0,
            errors: []
        };

        for (const session of importedSessions) {
            try {
                // Validate session structure
                if (!session.title || !session.urls || !Array.isArray(session.urls)) {
                    results.errors.push(`Invalid session: ${session.title || 'Untitled'}`);
                    results.skipped++;
                    continue;
                }

                // Check for duplicate (same title and similar content)
                const isDuplicate = sessions.some(existing =>
                    existing.title === session.title &&
                    existing.urls.length === session.urls.length
                );

                if (isDuplicate) {
                    results.skipped++;
                    continue;
                }

                // Generate new ID and update dates
                const newSession = {
                    ...session,
                    id: generateSessionId(),
                    dateCreated: session.dateCreated || new Date().toISOString(),
                    dateModified: new Date().toISOString(),
                    tags: validateTags(session.tags || [])
                };

                await saveSession(newSession);
                results.imported++;

            } catch (err) {
                results.errors.push(`Error importing session: ${err.message}`);
                results.skipped++;
            }
        }

        return results;

    } catch (err) {
        throw new Error(`Failed to parse session data: ${err.message}`);
    }
}

/**
 * Generate a unique session ID
 */
function generateSessionId() {
    return `session_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
}

/**
 * Extract title from URL if not provided
 */
function extractTitleFromUrl(url) {
    try {
        const urlObj = new URL(url);
        const hostname = urlObj.hostname.replace('www.', '');
        const path = urlObj.pathname.split('/').filter(p => p).join(' > ');
        return path ? `${hostname} - ${path}` : hostname;
    } catch {
        return url;
    }
}

/**
 * Validate and clean tags
 */
function validateTags(tags) {
    if (!Array.isArray(tags)) {
        return [];
    }

    const reserved = ['add', 'search', 'edit', 'end', 'save', 'help', 'delete', 'open', 'import'];

    return tags
        .map(tag => tag.toString().trim().toLowerCase())
        .filter(tag => tag.length > 0 && !reserved.includes(tag))
        .filter((tag, index, self) => self.indexOf(tag) === index); // Remove duplicates
}
