/**
 * JottNote Navigation Service
 * Handles note navigation (prev/next/jump/promote)
 */

import { getNotes, getNotesByCreation, getNote, createNote, deleteNote, getCurrentNoteId, setCurrentNoteId, getNoteIndex } from './storage.js';

/**
 * Navigate to the previous note (older, higher index in sorted array)
 * @returns {Promise<Object|null>} Previous note or null if already at oldest
 */
export async function goToPrevious() {
    const currentId = await getCurrentNoteId();
    const notes = await getNotesByCreation();

    if (!currentId || notes.length === 0) {
        return null;
    }

    const currentIndex = notes.findIndex(n => n.id === currentId);
    if (currentIndex === -1) return null;

    // Previous = older note = higher index in creation-sorted array
    if (currentIndex < notes.length - 1) {
        const prevNote = notes[currentIndex + 1];
        await setCurrentNoteId(prevNote.id);
        return prevNote;
    }

    // Already at the oldest note
    return null;
}

/**
 * Navigate to the next note (newer, lower index in sorted array)
 * If already at the newest note, creates a new blank note
 * @param {string} currentContent - Current note content (for empty note cleanup)
 * @returns {Promise<Object>} Next or new note
 */
export async function goToNext(currentContent = '') {
    const currentId = await getCurrentNoteId();
    let notes = await getNotesByCreation();

    if (!currentId || notes.length === 0) {
        const newNote = await createNote();
        await setCurrentNoteId(newNote.id);
        return newNote;
    }

    const currentIndex = notes.findIndex(n => n.id === currentId);
    if (currentIndex === -1) {
        // Current note not found — go to the newest created note
        const frontNote = notes[0];
        await setCurrentNoteId(frontNote.id);
        return frontNote;
    }

    // Next = newer note = lower index in creation-sorted array
    if (currentIndex > 0) {
        const nextNote = notes[currentIndex - 1];
        await setCurrentNoteId(nextNote.id);
        return nextNote;
    }

    // Already at the newest note (index 0) — create a new note
    // But if the current note is empty, just stay (don't create another blank)
    if (currentContent.trim() === '') {
        return null;
    }

    const newNote = await createNote();
    await setCurrentNoteId(newNote.id);
    return newNote;
}

/**
 * Jump to the front (most recent note)
 * @returns {Promise<Object|null>} Most recent note or null
 */
export async function jumpToFront() {
    const notes = await getNotes();

    if (notes.length === 0) {
        const newNote = await createNote();
        await setCurrentNoteId(newNote.id);
        return newNote;
    }

    const frontNote = notes[0];
    await setCurrentNoteId(frontNote.id);
    return frontNote;
}

/**
 * Promote current note to front
 * @param {string} id - Note ID to promote
 * @returns {Promise<Object|null>} Promoted note
 */
export async function promoteNote(id) {
    const note = await getNote(id);
    if (note) {
        const { saveNote } = await import('./storage.js');
        return await saveNote(note);
    }
    return null;
}

/**
 * Delete current note and navigate to appropriate note
 * @param {string} id - Note ID to delete
 * @returns {Promise<Object|null>} New current note
 */
export async function deleteCurrentNote(id) {
    const notes = await getNotesByCreation();
    const currentIndex = notes.findIndex(n => n.id === id);

    await deleteNote(id);

    const updatedNotes = await getNotesByCreation();

    if (updatedNotes.length === 0) {
        // No notes left, create new one
        const newNote = await createNote();
        await setCurrentNoteId(newNote.id);
        return newNote;
    }

    // Navigate to the next newer note, or the previous if at front
    const targetIndex = Math.min(currentIndex, updatedNotes.length - 1);
    const targetNote = updatedNotes[Math.max(0, targetIndex - 1)] || updatedNotes[0];
    await setCurrentNoteId(targetNote.id);
    return targetNote;
}

/**
 * Get navigation info for current note
 * @returns {Promise<Object>} { currentIndex, totalNotes, isFirst, isLast }
 */
export async function getNavigationInfo() {
    const currentId = await getCurrentNoteId();
    const notes = await getNotesByCreation();

    if (!currentId || notes.length === 0) {
        return { currentIndex: 0, totalNotes: 0, isFirst: true, isLast: true };
    }

    const currentIndex = notes.findIndex(n => n.id === currentId);
    if (currentIndex === -1) {
        return { currentIndex: 0, totalNotes: notes.length, isFirst: true, isLast: true };
    }
    return {
        currentIndex: currentIndex + 1,
        totalNotes: notes.length,
        isFirst: currentIndex === 0,
        isLast: currentIndex === notes.length - 1
    };
}
