/**
 * JottNote Math Mode
 * Handles calculations, variables, and conversions (Unit & Currency)
 */

import { isComment } from './keywords.js';

// --- Variable Storage ---
let variables = {};

/**
 * Reset variables (call when switching notes)
 */
export function resetVariables() {
    variables = {};
}

// --- Unit Database ---
const UNITS = {
    length: {
        base: 'm',
        units: {
            m: 1, meter: 1, metre: 1, meters: 1, metres: 1,
            km: 1000, kilometer: 1000, kilometre: 1000, kilometers: 1000, kilometres: 1000,
            cm: 0.01, centimeter: 0.01, centimetre: 0.01, centimeters: 0.01, centimetres: 0.01,
            mm: 0.001, millimeter: 0.001, mullimetre: 0.001, millimeters: 0.001, mullimetres: 0.001,
            mi: 1609.344, mile: 1609.344, miles: 1609.344,
            yd: 0.9144, yard: 0.9144, yards: 0.9144,
            ft: 0.3048, foot: 0.3048, feet: 0.3048, "'": 0.3048, "′": 0.3048,
            in: 0.0254, inch: 0.0254, inches: 0.0254, '"': 0.0254, "″": 0.0254, "''": 0.0254
        }
    },
    mass: {
        base: 'kg',
        units: {
            kg: 1, kilogram: 1, kilograms: 1,
            g: 0.001, gram: 0.001, grams: 0.001,
            mg: 0.000001, milligram: 0.000001, milligrams: 0.000001,
            lb: 0.45359237, lbs: 0.45359237, pound: 0.45359237, pounds: 0.45359237,
            oz: 0.0283495, ounce: 0.0283495, ounces: 0.0283495,
            stone: 6.35029, stones: 6.35029, st: 6.35029,
            ton: 907.185, tons: 907.185, tonne: 1000, tonnes: 1000
        }
    },
    volume: {
        base: 'l',
        units: {
            l: 1, liter: 1, litre: 1, liters: 1, litres: 1,
            ml: 0.001, milliliter: 0.001, millilitre: 0.001, milliliters: 0.001, millilitres: 0.001,
            gal: 3.78541, gallon: 3.78541, gallons: 3.78541,
            qt: 0.946353, quart: 0.946353, quarts: 0.946353,
            pt: 0.473176, pint: 0.473176, pints: 0.473176,
            cup: 0.236588, cups: 0.236588,
            floz: 0.0295735, "fl oz": 0.0295735,
            tbsp: 0.0147868, tablespoon: 0.0147868, tablespoons: 0.0147868,
            tsp: 0.00492892, teaspoon: 0.00492892, teaspoons: 0.00492892
        }
    },
    area: {
        base: 'sqm',
        units: {
            sqm: 1, "m²": 1, "sq m": 1, "square meter": 1, "square meters": 1,
            sqcm: 0.0001, "cm²": 0.0001, "sq cm": 0.0001, "square centimeter": 0.0001, "square centimeters": 0.0001,
            sqkm: 1000000, "km²": 1000000, "sq km": 1000000, "square kilometer": 1000000, "square kilometers": 1000000,
            sqft: 0.092903, "ft²": 0.092903, "sq ft": 0.092903, "square foot": 0.092903, "square feet": 0.092903,
            sqin: 0.00064516, "in²": 0.00064516, "sq in": 0.00064516, "square inch": 0.00064516, "square inches": 0.00064516,
            acre: 4046.86, acres: 4046.86,
            ha: 10000, hectare: 10000, hectares: 10000
        }
    },
    temperature: {
        special: true,
        units: ['c', 'celsius', 'f', 'fahrenheit', 'k', 'kelvin']
    }
};

// --- Currency System with Caching (Fiat + Crypto) ---
const Currency = {
    rates: null,
    cryptoRates: null,
    base: 'USD',
    timestamp: 0,
    cryptoTimestamp: 0,
    CACHE_DURATION: 24 * 60 * 60 * 1000, // 24 hours
    CRYPTO_CACHE_DURATION: 5 * 60 * 1000, // 5 minutes (crypto prices change faster)

    async init() {
        // Try loading fiat rates from storage first
        try {
            const data = await chrome.storage.local.get(['currencyRates', 'currencyTimestamp', 'cryptoRates', 'cryptoTimestamp']);

            // Load fiat rates
            if (data.currencyRates && data.currencyTimestamp) {
                if (Date.now() - data.currencyTimestamp < this.CACHE_DURATION) {
                    this.rates = data.currencyRates;
                    this.timestamp = data.currencyTimestamp;
                } else {
                    this.fetchRates();
                }
            } else {
                this.fetchRates();
            }

            // Load crypto rates
            if (data.cryptoRates && data.cryptoTimestamp) {
                if (Date.now() - data.cryptoTimestamp < this.CRYPTO_CACHE_DURATION) {
                    this.cryptoRates = data.cryptoRates;
                    this.cryptoTimestamp = data.cryptoTimestamp;
                } else {
                    this.fetchCryptoRates();
                }
            } else {
                this.fetchCryptoRates();
            }
        } catch (e) {
            // Fetch new rates if storage fails
            this.fetchRates();
            this.fetchCryptoRates();
        }
    },

    async fetchRates() {
        try {
            // Using a free API (ExchangeRate-API supports 160+ currencies)
            const response = await fetch('https://api.exchangerate-api.com/v4/latest/USD');
            if (response.ok) {
                const data = await response.json();
                this.rates = data.rates;
                this.timestamp = Date.now();

                // Cache it
                await chrome.storage.local.set({
                    currencyRates: this.rates,
                    currencyTimestamp: this.timestamp
                });
            }
        } catch (e) {
            console.error("Failed to fetch currency rates:", e);
        }
    },

    async fetchCryptoRates() {
        try {
            // Using CoinGecko's free API (no key required)
            // Get prices for major cryptocurrencies in USD
            const cryptoIds = [
                'bitcoin', 'ethereum', 'tether', 'binancecoin', 'solana', 'ripple',
                'usd-coin', 'cardano', 'dogecoin', 'tron', 'avalanche-2', 'polkadot',
                'matic-network', 'litecoin', 'chainlink', 'bitcoin-cash', 'shiba-inu',
                'uniswap', 'cosmos', 'stellar', 'monero', 'ethereum-classic', 'algorand',
                'filecoin', 'aptos', 'vechain', 'hedera-hashgraph', 'internet-computer',
                'near', 'arbitrum'
            ];

            const response = await fetch(
                `https://api.coingecko.com/api/v3/simple/price?ids=${cryptoIds.join(',')}&vs_currencies=usd`
            );

            if (response.ok) {
                const data = await response.json();

                // Map CoinGecko IDs to our crypto codes
                const idToCode = {
                    'bitcoin': 'BTC', 'ethereum': 'ETH', 'tether': 'USDT',
                    'binancecoin': 'BNB', 'solana': 'SOL', 'ripple': 'XRP',
                    'usd-coin': 'USDC', 'cardano': 'ADA', 'dogecoin': 'DOGE',
                    'tron': 'TRX', 'avalanche-2': 'AVAX', 'polkadot': 'DOT',
                    'matic-network': 'MATIC', 'litecoin': 'LTC', 'chainlink': 'LINK',
                    'bitcoin-cash': 'BCH', 'shiba-inu': 'SHIB', 'uniswap': 'UNI',
                    'cosmos': 'ATOM', 'stellar': 'XLM', 'monero': 'XMR',
                    'ethereum-classic': 'ETC', 'algorand': 'ALGO', 'filecoin': 'FIL',
                    'aptos': 'APT', 'vechain': 'VET', 'hedera-hashgraph': 'HBAR',
                    'internet-computer': 'ICP', 'near': 'NEAR', 'arbitrum': 'ARB'
                };

                this.cryptoRates = {};
                for (const [id, code] of Object.entries(idToCode)) {
                    if (data[id] && data[id].usd) {
                        this.cryptoRates[code] = data[id].usd;
                    }
                }

                this.cryptoTimestamp = Date.now();

                // Cache it
                await chrome.storage.local.set({
                    cryptoRates: this.cryptoRates,
                    cryptoTimestamp: this.cryptoTimestamp
                });
            }
        } catch (e) {
            console.error("Failed to fetch crypto rates:", e);
        }
    },

    isCrypto(code) {
        return this.cryptoRates && this.cryptoRates[code.toUpperCase()] !== undefined;
    },

    convert(amount, from, to) {
        const fromUpper = from.toUpperCase();
        const toUpper = to.toUpperCase();

        // Check if either is crypto
        const fromIsCrypto = this.isCrypto(fromUpper);
        const toIsCrypto = this.isCrypto(toUpper);

        // Crypto to Crypto
        if (fromIsCrypto && toIsCrypto) {
            if (!this.cryptoRates) return null;
            const fromPriceUSD = this.cryptoRates[fromUpper];
            const toPriceUSD = this.cryptoRates[toUpper];
            if (fromPriceUSD && toPriceUSD) {
                return (amount * fromPriceUSD) / toPriceUSD;
            }
            return null;
        }

        // Crypto to Fiat
        if (fromIsCrypto && !toIsCrypto) {
            if (!this.cryptoRates || !this.rates) return null;
            const cryptoPriceUSD = this.cryptoRates[fromUpper];
            const fiatRate = this.rates[toUpper];
            if (cryptoPriceUSD && fiatRate) {
                // Convert crypto to USD, then USD to target fiat
                const usdAmount = amount * cryptoPriceUSD;
                return usdAmount * fiatRate;
            }
            return null;
        }

        // Fiat to Crypto
        if (!fromIsCrypto && toIsCrypto) {
            if (!this.rates || !this.cryptoRates) return null;
            const fiatRate = this.rates[fromUpper];
            const cryptoPriceUSD = this.cryptoRates[toUpper];
            if (fiatRate && cryptoPriceUSD) {
                // Convert fiat to USD, then USD to crypto
                const usdAmount = amount / fiatRate;
                return usdAmount / cryptoPriceUSD;
            }
            return null;
        }

        // Fiat to Fiat (original logic)
        if (!this.rates) return null;
        const fromRate = this.rates[fromUpper];
        const toRate = this.rates[toUpper];
        if (fromRate && toRate) {
            return (amount / fromRate) * toRate;
        }

        return null;
    }
};

// Initialize currency immediately
Currency.init();


// --- Logic ---

/**
 * Handle Unit Conversions
 * @param {string} fromUnit 
 * @param {string} toUnit 
 * @param {number} value 
 * @returns {number|null}
 */
function convertUnits(value, fromUnit, toUnit) {
    const from = fromUnit.toLowerCase();
    const to = toUnit.toLowerCase();

    // Check normal categories
    for (const category of Object.values(UNITS)) {
        if (category.special) continue;

        if (category.units[from] !== undefined && category.units[to] !== undefined) {
            const factorFrom = category.units[from]; // value relative to base
            const factorTo = category.units[to];

            // Convert to base: val * factorFrom
            // Convert to target: base / factorTo
            return (value * factorFrom) / factorTo;
        }
    }

    // Check Temperature
    const t = UNITS.temperature;
    if (t.units.includes(from) && t.units.includes(to)) {
        let celsius;
        // To Celsius
        if (from === 'c' || from === 'celsius') celsius = value;
        else if (from === 'f' || from === 'fahrenheit') celsius = (value - 32) * 5 / 9;
        else if (from === 'k' || from === 'kelvin') celsius = value - 273.15;

        // From Celsius
        if (to === 'c' || to === 'celsius') return celsius;
        else if (to === 'f' || to === 'fahrenheit') return (celsius * 9 / 5) + 32;
        else if (to === 'k' || to === 'kelvin') return celsius + 273.15;
    }

    // Check Currency
    const result = Currency.convert(value, from, to);
    if (result !== null) return result;

    return null; // No conversion found
}

// --- Mixed-Unit Arithmetic Helpers ---

const CURRENCY_SYMBOLS = { '$': 'USD', '€': 'EUR', '£': 'GBP', '¥': 'JPY', '₿': 'BTC' };

/**
 * Resolve a unit string to its category and metadata
 * @param {string} unitStr
 * @returns {{ category: string, name: string, factor?: number }|null}
 */
function resolveUnit(unitStr) {
    if (!unitStr) return null;

    // Check currency symbols
    if (CURRENCY_SYMBOLS[unitStr]) {
        return { category: 'currency', name: CURRENCY_SYMBOLS[unitStr] };
    }

    const lower = unitStr.toLowerCase();
    const upper = unitStr.toUpperCase();

    // Check physical units
    for (const [catName, category] of Object.entries(UNITS)) {
        if (category.special) {
            // Temperature
            if (category.units.includes(lower)) {
                return { category: catName, name: lower };
            }
            continue;
        }
        if (category.units[lower] !== undefined) {
            return { category: catName, name: lower, factor: category.units[lower] };
        }
    }

    // Check fiat currency
    if (Currency.rates && Currency.rates[upper]) {
        return { category: 'currency', name: upper };
    }

    // Check crypto
    if (Currency.cryptoRates && Currency.cryptoRates[upper]) {
        return { category: 'currency', name: upper };
    }

    return null;
}

/**
 * Extract target unit clause from end of expression
 * e.g., "5ft + 3in in cm" → { targetUnit: "cm", exprWithout: "5ft + 3in" }
 * @param {string} expr
 * @returns {{ targetUnit: string, exprWithout: string }|null}
 */
function extractTargetUnit(expr) {
    const match = expr.match(/\s+(?:to|in|->)\s+([a-zA-Z$€£¥₿]+)$/i);
    if (match) {
        const unitStr = match[1];
        const resolved = resolveUnit(unitStr);
        if (resolved) {
            return { targetUnit: unitStr, exprWithout: expr.slice(0, match.index).trim() };
        }
    }
    return null;
}

/**
 * Convert a value to its category's base unit
 * @param {number} value
 * @param {{ category: string, name: string, factor?: number }} unitInfo
 * @returns {number|null}
 */
function convertToBase(value, unitInfo) {
    if (unitInfo.category === 'temperature') {
        const name = unitInfo.name;
        if (name === 'c' || name === 'celsius') return value;
        if (name === 'f' || name === 'fahrenheit') return (value - 32) * 5 / 9;
        if (name === 'k' || name === 'kelvin') return value - 273.15;
        return null;
    }
    if (unitInfo.category === 'currency') {
        return Currency.convert(value, unitInfo.name, 'USD');
    }
    // Physical units: factor converts to base
    return value * unitInfo.factor;
}

/**
 * Convert a value from base unit to target unit
 * @param {number} value
 * @param {{ category: string, name: string, factor?: number }} unitInfo
 * @returns {number|null}
 */
function convertFromBase(value, unitInfo) {
    if (unitInfo.category === 'temperature') {
        const name = unitInfo.name;
        if (name === 'c' || name === 'celsius') return value;
        if (name === 'f' || name === 'fahrenheit') return (value * 9 / 5) + 32;
        if (name === 'k' || name === 'kelvin') return value + 273.15;
        return null;
    }
    if (unitInfo.category === 'currency') {
        return Currency.convert(value, 'USD', unitInfo.name);
    }
    return value / unitInfo.factor;
}

/**
 * Tokenize an expression into typed tokens (numbers, value+unit, operators)
 * @param {string} expr
 * @returns {Array<{ type: string, value?: number, unit?: string, op?: string }>}
 */
function tokenizeWithUnits(expr) {
    const tokens = [];
    expr = expr.replace(/×/g, '*').replace(/÷/g, '/');

    let i = 0;
    const len = expr.length;

    while (i < len) {
        // Skip whitespace
        if (/\s/.test(expr[i])) { i++; continue; }

        // Currency symbol + number (+ optional trailing code)
        if (CURRENCY_SYMBOLS[expr[i]]) {
            const sym = expr[i];
            const numMatch = expr.slice(i + 1).match(/^([\d,]+\.?\d*)/);
            if (numMatch) {
                const val = parseFloat(numMatch[1].replace(/,/g, ''));
                i += 1 + numMatch[1].length;
                // Check for trailing currency code like "USD" after $100
                const codeMatch = expr.slice(i).match(/^\s*([a-zA-Z]+)/);
                let unit = CURRENCY_SYMBOLS[sym];
                if (codeMatch) {
                    const resolved = resolveUnit(codeMatch[1]);
                    if (resolved && resolved.category === 'currency') {
                        unit = resolved.name;
                        i += codeMatch[0].length;
                    }
                }
                tokens.push({ type: 'value_unit', value: val, unit });
                continue;
            }
            i++;
            continue;
        }

        // Number (possibly followed by unit)
        const numMatch = expr.slice(i).match(/^([\d,]+\.?\d*(?:[eE][-+]?\d+)?)/);
        if (numMatch) {
            const val = parseFloat(numMatch[1].replace(/,/g, ''));
            if (!isNaN(val)) {
                const afterNum = i + numMatch[1].length;
                // Check for immediately adjacent unit (no space) first
                const adjUnit = expr.slice(afterNum).match(/^([a-zA-Z]+)/);
                if (adjUnit && resolveUnit(adjUnit[1])) {
                    tokens.push({ type: 'value_unit', value: val, unit: adjUnit[1] });
                    i = afterNum + adjUnit[1].length;
                    continue;
                }
                tokens.push({ type: 'number', value: val });
                i = afterNum;
                continue;
            }
        }

        // Power operator **
        if (expr[i] === '*' && expr[i + 1] === '*') {
            tokens.push({ type: 'op', op: '^' });
            i += 2;
            continue;
        }

        // Operators
        if ('+-*/%^()'.includes(expr[i])) {
            tokens.push({ type: 'op', op: expr[i] });
            i++;
            continue;
        }

        // 'x' as multiplication between values
        if (expr[i].toLowerCase() === 'x') {
            const prev = tokens.length > 0 ? tokens[tokens.length - 1] : null;
            if (prev && (prev.type === 'number' || prev.type === 'value_unit' || (prev.type === 'op' && prev.op === ')'))) {
                tokens.push({ type: 'op', op: '*' });
                i++;
                continue;
            }
        }

        // Skip unknown characters
        i++;
    }

    return tokens;
}

/**
 * Unit-aware recursive descent parser for mixed-unit arithmetic
 * @param {string} expr - Expression with units (target already stripped)
 * @param {string} targetUnitStr - Target unit string
 * @returns {{ value: number, unit: string }|null}
 */
function calculateWithUnits(expr, targetUnitStr) {
    const targetInfo = resolveUnit(targetUnitStr);
    if (!targetInfo) return null;

    const tokens = tokenizeWithUnits(expr);
    if (tokens.length === 0) return null;

    // Check if any unit tokens exist
    const hasUnits = tokens.some(t => t.type === 'value_unit');

    // Validate all unit tokens are same category as target
    if (hasUnits) {
        for (const t of tokens) {
            if (t.type === 'value_unit') {
                const info = resolveUnit(t.unit);
                if (!info || info.category !== targetInfo.category) return null;
            }
        }
    }

    // Convert all value_unit tokens to base unit values
    const processed = tokens.map(t => {
        if (t.type === 'value_unit') {
            const info = resolveUnit(t.unit);
            const baseVal = convertToBase(t.value, info);
            if (baseVal === null) return null;
            return { type: 'number', value: baseVal };
        }
        return t;
    });
    if (processed.includes(null)) return null;

    // Recursive descent parser
    let pos = 0;
    const peek = () => processed[pos];
    const consume = () => processed[pos++];

    const parsePrimary = () => {
        const token = consume();
        if (!token) throw new Error("Unexpected end");
        if (token.type === 'number') return token.value;
        if (token.type === 'op' && token.op === '(') {
            const val = parseExpr();
            const closing = consume();
            if (!closing || closing.op !== ')') throw new Error("Expected )");
            return val;
        }
        if (token.type === 'op' && token.op === '-') return -parsePrimary();
        if (token.type === 'op' && token.op === '+') return parsePrimary();
        throw new Error("Unexpected token");
    };

    const parsePower = () => {
        let left = parsePrimary();
        while (peek() && peek().type === 'op' && peek().op === '^') {
            consume();
            left = Math.pow(left, parsePrimary());
        }
        return left;
    };

    const parseFactor = () => {
        let left = parsePower();
        while (peek() && peek().type === 'op' && (peek().op === '*' || peek().op === '/' || peek().op === '%')) {
            const op = consume().op;
            const right = parsePower();
            if (op === '*') left *= right;
            else if (op === '/') left /= right;
            else left %= right;
        }
        return left;
    };

    const parseExpr = () => {
        let left = parseFactor();
        while (peek() && peek().type === 'op' && (peek().op === '+' || peek().op === '-')) {
            const op = consume().op;
            const right = parseFactor();
            if (op === '+') left += right;
            else left -= right;
        }
        return left;
    };

    try {
        const result = parseExpr();
        if (pos < processed.length) return null;

        if (hasUnits) {
            const finalVal = convertFromBase(result, targetInfo);
            if (finalVal === null) return null;
            return { value: finalVal, unit: targetUnitStr };
        } else {
            // Scalar expression — just label result with target unit
            return { value: result, unit: targetUnitStr };
        }
    } catch (e) {
        return null;
    }
}

/**
 * Format a result value with its unit
 * @param {string} formatted - Already-formatted number string
 * @param {string} unit - Unit string
 * @returns {string}
 */
function formatWithUnit(formatted, unit) {
    const upper = unit.toUpperCase();
    // Currency symbol prefixes
    const prefixMap = { 'USD': '$', 'EUR': '€', 'GBP': '£', 'JPY': '¥' };
    if (prefixMap[upper]) {
        return `${prefixMap[upper]}${formatted} ${upper}`;
    }
    // Crypto or other fiat — suffix
    if (Currency.isCrypto(upper) || (Currency.rates && Currency.rates[upper])) {
        return `${formatted} ${upper}`;
    }
    // Physical units — suffix
    return `${formatted} ${unit}`;
}

/**
 * Parse and calculate a math expression safely (no eval)
 * Uses a recursive descent parser
 * @param {string} expression
 * @returns {number|null}
 */
export function calculate(expression) {
    try {
        // Clean the expression
        let expr = expression.trim();
        if (expr.endsWith('=')) {
            expr = expr.slice(0, -1).trim();
        }
        if (!expr) return null;

        // Substitute variables FIRST
        for (const [name, value] of Object.entries(variables)) {
            // Sort variables by length descending to avoid prefix collision
            const escapedName = name.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
            const regex = new RegExp('\\b' + escapedName + '\\b', 'gi');
            expr = expr.replace(regex, String(value));
        }

        // --- Extract target unit if present ---
        const targetResult = extractTargetUnit(expr);
        let targetUnit = null;
        if (targetResult) {
            targetUnit = targetResult.targetUnit;
            expr = targetResult.exprWithout;
        }

        // --- Simple conversion: "10 km" with target ---
        if (targetUnit) {
            const cleanExpr = expr.replace(/,/g, '');
            // Number + unit (e.g., "10 km", "100f")
            const simpleMatch = cleanExpr.match(/^([\d.]+)\s*([a-zA-Z"']+)$/);
            if (simpleMatch) {
                const val = parseFloat(simpleMatch[1]);
                const from = simpleMatch[2];
                if (!isNaN(val)) {
                    const result = convertUnits(val, from, targetUnit);
                    if (result !== null) return { value: result, unit: targetUnit };
                }
            }
            // Currency symbol + number + optional code (e.g., "$100", "€50 EUR")
            const currMatch = cleanExpr.match(/^([$€£¥₿])([\d.]+)\s*([a-zA-Z]+)?$/);
            if (currMatch) {
                const val = parseFloat(currMatch[2]);
                const from = currMatch[3] || CURRENCY_SYMBOLS[currMatch[1]];
                if (!isNaN(val) && from) {
                    const result = convertUnits(val, from, targetUnit);
                    if (result !== null) return { value: result, unit: targetUnit };
                }
            }
        }

        // --- Percentage Operations (only without target unit) ---
        if (!targetUnit) {
            // "what % of X is Y" or "what percent of X is Y"
            const whatPercentMatch = expr.match(/^what\s*%?\s*(?:percent(?:age)?)?\s*(?:of)\s+([\d,.]+)\s+is\s+([\d,.]+)$/i);
            if (whatPercentMatch) {
                const total = parseFloat(whatPercentMatch[1].replace(/,/g, ''));
                const part = parseFloat(whatPercentMatch[2].replace(/,/g, ''));
                if (!isNaN(total) && !isNaN(part) && total !== 0) {
                    return { value: (part / total) * 100, unit: null };
                }
            }

            // "X + Y%" → X * (1 + Y/100), "X - Y%" → X * (1 - Y/100)
            const percentOpMatch = expr.match(/^([\d,.]+)\s*([+\-])\s*([\d,.]+)\s*%$/);
            if (percentOpMatch) {
                const base = parseFloat(percentOpMatch[1].replace(/,/g, ''));
                const op = percentOpMatch[2];
                const pct = parseFloat(percentOpMatch[3].replace(/,/g, ''));
                if (!isNaN(base) && !isNaN(pct)) {
                    if (op === '+') return { value: base * (1 + pct / 100), unit: null };
                    if (op === '-') return { value: base * (1 - pct / 100), unit: null };
                }
            }
        }

        // --- Unit-aware calculation with target ---
        if (targetUnit) {
            const unitResult = calculateWithUnits(expr, targetUnit);
            if (unitResult) return unitResult;
        }

        // --- Standard Math Parser ---
        // Clean up symbols
        expr = expr
            .replace(/×/g, '*')
            .replace(/÷/g, '/')
            .replace(/(\d)\s*x\s*(\d)/gi, '$1*$2') // Handle 'x' between numbers
            .replace(/\^/g, '**')
            .replace(/[$€£¥₹₽₩₺₫฿₴₦₲₵₡₭₾₱₪₸]/g, ''); // Strip currency

        // Helper Tokenizer
        const tokens = [];
        // Adjusted regex to handle scientific notation
        const tokenRegex = /\s*([0-9]*\.?[0-9]+([eE][-+]?[0-9]+)?|[a-zA-Z_]\w*|[+\-*/%^()])/g;
        let match;

        while ((match = tokenRegex.exec(expr)) !== null) {
            if (match[0].trim()) tokens.push(match[0].trim());
        }

        // Parser state
        let pos = 0;
        const peek = () => tokens[pos];
        const consume = () => tokens[pos++];

        const parsePrimary = () => {
            const token = consume();
            if (!token) throw new Error("Unexpected end");

            // Number
            if (!isNaN(parseFloat(token)) && isFinite(token)) {
                return parseFloat(token);
            }

            // Parentheses
            if (token === '(') {
                const val = parseExpression();
                if (consume() !== ')') throw new Error("Expected )");
                return val;
            }

            // Math Constants/Functions
            const lowerKey = token.toLowerCase();
            if (lowerKey === 'pi') return Math.PI;
            if (lowerKey === 'e') return Math.E;

            // Math Functions like sqrt(4)
            if (['sqrt', 'cos', 'sin', 'tan', 'abs', 'round', 'floor', 'ceil', 'log', 'exp'].includes(lowerKey)) {
                if (peek() === '(') {
                    consume(); // eat (
                    const val = parseExpression();
                    if (consume() !== ')') throw new Error("Expected )");
                    return Math[lowerKey](val);
                }
            }

            throw new Error("Unknown token: " + token);
        };

        const parsePower = () => {
            let left = parsePrimary();
            while (peek() === '^' || peek() === '**') {
                consume();
                const right = parsePrimary(); // Right associative? usually, allowing left for simple
                left = Math.pow(left, right);
            }
            return left;
        };

        const parseFactor = () => {
            let left = parsePower();
            while (true) {
                const op = peek();
                if (op === '*' || op === '/' || op === '%') {
                    consume();
                    const right = parsePower();
                    if (op === '*') left *= right;
                    else if (op === '/') left /= right;
                    else if (op === '%') left %= right;
                } else if (op === '(' || (!isNaN(parseFloat(op)) && isFinite(op))) {
                    // Implicit multiplication: 5(2) or (2)(3)
                    const right = parsePower();
                    left *= right;
                } else {
                    break;
                }
            }
            return left;
        };

        const parseExpression = () => {
            let left = parseFactor();
            while (peek() === '+' || peek() === '-') {
                const op = consume();
                const right = parseFactor();
                if (op === '+') left += right;
                else if (op === '-') left -= right;
            }
            return left;
        };

        const result = parseExpression();

        // Ensure all tokens consumed
        if (pos < tokens.length) return null;

        return { value: result, unit: null };

    } catch (error) {
        return null;
    }
}

/**
 * Format a number result
 * @param {number} value
 * @param {Object} options
 * @returns {string}
 */
export function formatNumber(value, options = {}) {
    const { significantDigits = 2, thousandsSeparator = true } = options;

    let formatted = value.toFixed(significantDigits);

    // Remove trailing zeros after decimal
    if (formatted.includes('.')) {
        formatted = formatted.replace(/\.?0+$/, '');
    }

    // Add thousands separator
    if (thousandsSeparator) {
        const parts = formatted.split('.');
        parts[0] = parts[0].replace(/\B(?=(\d{3})+(?!\d))/g, ',');
        formatted = parts.join('.');
    }

    return formatted;
}

/**
 * Parse a variable assignment
 * @param {string} line
 * @returns {Object|null} { name, value } or null
 */
export function parseVariable(line) {
    const colonIndex = line.indexOf(':');

    if (colonIndex > 0 && !line.includes('=')) {
        const name = line.substring(0, colonIndex).trim();
        const valueStr = line.substring(colonIndex + 1).trim();

        // Don't treat keywords as variables
        const keywords = ['list', 'math', 'sum', 'avg', 'count', 'code', 'timer', 'date'];
        if (keywords.includes(name.toLowerCase())) {
            return null;
        }

        const calcResult = calculate(valueStr);
        const value = parseFloat(valueStr) || (calcResult ? calcResult.value : null);

        if (value !== null && !isNaN(value)) {
            return { name, value };
        }
    }

    return null;
}

/**
 * Process math content
 * @param {string} content
 * @param {Object} options
 * @returns {Array<Object>} Processed lines with results
 */
export function processMathContent(content, options = {}) {
    const lines = content.split('\n');
    const results = [];

    // Reset variables for fresh processing
    resetVariables();

    // Skip first line (keyword)
    for (let i = 1; i < lines.length; i++) {
        const line = lines[i];
        const trimmed = line.trim();

        if (!trimmed) {
            results.push({ type: 'empty', lineIndex: i });
            continue;
        }

        if (isComment(line)) {
            results.push({
                type: 'comment',
                lineIndex: i,
                text: trimmed.substring(2).trim()
            });
            continue;
        }

        // Check for variable assignment
        const variable = parseVariable(line);
        if (variable) {
            variables[variable.name.toLowerCase()] = variable.value;
            results.push({
                type: 'variable',
                lineIndex: i,
                name: variable.name,
                value: variable.value,
                formatted: formatNumber(variable.value, options)
            });
            continue;
        }

        // Check for calculation (ends with =)
        if (trimmed.endsWith('=')) {
            const calcResult = calculate(trimmed);
            const result = calcResult ? calcResult.value : null;

            // Detect "what % of" pattern and append % to result
            const isPercentQuery = /^what\s*%?\s*(?:percent(?:age)?)?\s*(?:of)/i.test(trimmed);
            let formatted = result !== null ? formatNumber(result, options) : 'Error';
            if (calcResult && calcResult.unit) {
                formatted = formatWithUnit(formatted, calcResult.unit);
            } else if (isPercentQuery && result !== null) {
                formatted += '%';
            }

            results.push({
                type: 'calculation',
                lineIndex: i,
                expression: trimmed,
                result,
                formatted
            });
            continue;
        }

        // Regular text line
        results.push({
            type: 'text',
            lineIndex: i,
            text: trimmed
        });
    }

    return results;
}

/**
 * Render math results as HTML
 * @param {Array<Object>} results
 * @returns {string}
 */
export function renderMathHTML(results) {
    let html = '';

    for (const item of results) {
        switch (item.type) {
            case 'empty':
                html += '<div class="math-line empty"></div>';
                break;

            case 'comment':
                html += `<div class="math-line comment-line">// ${escapeHtml(item.text)}</div>`;
                break;

            case 'variable':
                html += `
          <div class="math-line variable-line">
            <span class="variable-name">${escapeHtml(item.name)}</span>
            <span class="variable-sep">:</span>
            <span class="variable-value">${item.formatted}</span>
          </div>
        `;
                break;

            case 'calculation':
                html += `
          <div class="math-line calculation-line">
            <span class="math-expression">${escapeHtml(item.expression)}</span>
            <span class="math-result" data-value="${item.result}">${item.formatted}</span>
          </div>
        `;
                break;

            case 'text':
                html += `<div class="math-line">${escapeHtml(item.text)}</div>`;
                break;
        }
    }

    return html;
}

/**
 * Escape HTML
 * @param {string} text
 * @returns {string}
 */
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}
