/**
 * JottNote Simple Markdown
 * Handles basic markdown formatting
 */

/**
 * Parse and render simple markdown
 * @param {string} text
 * @returns {string} HTML
 */
export function renderMarkdown(text) {
    let html = escapeHtml(text);

    // Bold: **text** or __text__
    html = html.replace(/\*\*(.+?)\*\*/g, '<strong>$1</strong>');

    // Underline: __text__
    html = html.replace(/__(.+?)__/g, '<u>$1</u>');

    // Italic: _text_ or *text*
    html = html.replace(/(?<!\*)\*(?!\*)(.+?)(?<!\*)\*(?!\*)/g, '<em>$1</em>');
    html = html.replace(/(?<!_)_(?!_)(.+?)(?<!_)_(?!_)/g, '<em>$1</em>');

    // Strikethrough: ~text~
    html = html.replace(/~(.+?)~/g, '<del>$1</del>');

    // Inline code: `text`
    html = html.replace(/`([^`]+)`/g, '<code>$1</code>');

    return html;
}

/**
 * Render a line with heading support
 * @param {string} line
 * @returns {Object} { html, isHeading, level }
 */
export function renderLine(line) {
    const trimmed = line.trimStart();

    // Check for headings
    if (trimmed.startsWith('### ')) {
        return {
            html: `<span class="heading-3">${renderMarkdown(trimmed.substring(4))}</span>`,
            isHeading: true,
            level: 3
        };
    }

    if (trimmed.startsWith('## ')) {
        return {
            html: `<span class="heading-2">${renderMarkdown(trimmed.substring(3))}</span>`,
            isHeading: true,
            level: 2
        };
    }

    if (trimmed.startsWith('# ')) {
        return {
            html: `<span class="heading-1">${renderMarkdown(trimmed.substring(2))}</span>`,
            isHeading: true,
            level: 1
        };
    }

    // Check for comment
    if (trimmed.startsWith('//')) {
        return {
            html: `<span class="comment-line">${escapeHtml(line)}</span>`,
            isComment: true
        };
    }

    // Regular line with markdown
    return {
        html: renderMarkdown(line),
        isHeading: false
    };
}

/**
 * Strip markdown from text (for clipboard)
 * @param {string} text
 * @returns {string}
 */
export function stripMarkdown(text) {
    return text
        .replace(/\*\*(.+?)\*\*/g, '$1')  // Bold
        .replace(/__(.+?)__/g, '$1')       // Underline
        .replace(/\*(.+?)\*/g, '$1')       // Italic
        .replace(/_(.+?)_/g, '$1')         // Italic
        .replace(/~(.+?)~/g, '$1')         // Strikethrough
        .replace(/`(.+?)`/g, '$1')         // Inline code
        .replace(/^#+\s+/gm, '');          // Headings
}

/**
 * Escape HTML characters
 * @param {string} text
 * @returns {string}
 */
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}
