/**
 * JottNote Lorem Ipsum Generator Mode
 * Generates placeholder text in paragraph, word, or sentence modes
 */

const LOREM_SENTENCES = [
    'Lorem ipsum dolor sit amet, consectetur adipiscing elit.',
    'Sed do eiusmod tempor incididunt ut labore et dolore magna aliqua.',
    'Ut enim ad minim veniam, quis nostrud exercitation ullamco laboris.',
    'Nisi ut aliquip ex ea commodo consequat.',
    'Duis aute irure dolor in reprehenderit in voluptate velit esse cillum dolore.',
    'Eu fugiat nulla pariatur.',
    'Excepteur sint occaecat cupidatat non proident.',
    'Sunt in culpa qui officia deserunt mollit anim id est laborum.',
    'Curabitur pretium tincidunt lacus sed porttitor.',
    'Nunc sagittis sem a tortor tempus pellentesque.',
    'Maecenas egestas arcu quis ligula mattis placerat.',
    'Donec at orci non arcu pharetra vestibulum.',
    'Pellentesque habitant morbi tristique senectus et netus et malesuada fames.',
    'Ac turpis egestas integer eget aliquet nibh praesent tristique magna.',
    'Viverra justo nec ultrices dui sapien eget mi proin.',
    'Faucibus pulvinar elementum integer enim neque volutpat ac tincidunt vitae.',
    'Morbi enim nunc faucibus a pellentesque sit amet porttitor eget.',
    'Aenean sed adipiscing diam donec adipiscing tristique risus nec feugiat.',
    'Eget nunc scelerisque viverra mauris in aliquam sem fringilla.',
    'Neque aliquam vestibulum morbi blandit cursus risus at ultrices mi.',
    'Tempus imperdiet nulla malesuada pellentesque elit eget gravida.',
    'Blandit turpis cursus in hac habitasse platea dictumst.',
    'Vitae tortor condimentum lacinia quis vel eros donec ac odio.',
    'Accumsan sit amet nulla facilisi morbi tempus iaculis urna.',
    'Id volutpat lacus laoreet non curabitur gravida arcu ac tortor.',
    'Dignissim suspendisse in est ante in nibh mauris cursus mattis.',
    'Molestie a iaculis at erat pellentesque adipiscing commodo elit.',
    'Non blandit massa enim nec dui nunc mattis enim ut tellus.',
    'Elementum sagittis vitae et leo duis ut diam quam nulla.',
    'Porttitor rhoncus dolor purus non enim praesent elementum facilisis.',
    'Amet cursus sit amet dictum sit amet justo donec enim.',
    'Diam vulputate ut pharetra sit amet aliquam id diam maecenas.',
    'Ultricies mi quis hendrerit dolor magna eget est lorem ipsum.',
    'Dolor sit amet consectetur adipiscing elit pellentesque habitant morbi.',
    'Tristique senectus et netus et malesuada fames ac turpis egestas.',
    'Sed risus pretium quam vulputate dignissim suspendisse in est.',
    'Ante in nibh mauris cursus mattis molestie a iaculis at.',
    'Erat pellentesque adipiscing commodo elit at imperdiet dui accumsan.',
    'Sit amet nulla facilisi morbi tempus iaculis urna id volutpat.',
    'Lacus laoreet non curabitur gravida arcu ac tortor dignissim.',
    'Convallis a cras semper auctor neque vitae tempus quam pellentesque.',
    'Nec nam aliquam sem et tortor consequat id porta nibh.',
    'Venenatis tellus in metus vulputate eu scelerisque felis imperdiet proin.',
    'Fermentum et sollicitudin ac orci phasellus egestas tellus rutrum.',
    'Tellus in hac habitasse platea dictumst vestibulum rhoncus est.',
    'Pellentesque nec nam aliquam sem et tortor consequat.',
    'Id interdum velit laoreet id donec ultrices tincidunt arcu.',
    'Non pulvinar neque laoreet suspendisse interdum consectetur libero id.',
    'Faucibus nisl tincidunt eget nullam non nisi est sit amet.',
    'Facilisis magna etiam tempor orci eu lobortis elementum nibh tellus.'
];

/**
 * Process lorem command and return generated text
 * @param {string} content - Full note content
 * @returns {Object} { text: string, type: string, help: boolean }
 */
export function processLoremCommand(content) {
    const firstLine = (content.split('\n')[0] || '').trim().toLowerCase();

    // Just "lorem" alone = help
    if (firstLine === 'lorem' && content.split('\n').length <= 1) {
        return { text: '', type: 'help', help: true };
    }

    // "lorem" + Enter = 1 paragraph
    if (firstLine === 'lorem') {
        return { text: generateParagraphs(1), type: 'paragraphs', help: false };
    }

    // lorem:N — N paragraphs
    const paraMatch = firstLine.match(/^lorem:(\d+)$/);
    if (paraMatch) {
        const n = Math.min(parseInt(paraMatch[1], 10), 20);
        return { text: generateParagraphs(n), type: 'paragraphs', help: false };
    }

    // lorem:words:N
    const wordsMatch = firstLine.match(/^lorem:words:(\d+)$/);
    if (wordsMatch) {
        const n = Math.min(parseInt(wordsMatch[1], 10), 1000);
        return { text: generateWords(n), type: 'words', help: false };
    }

    // lorem:sentences:N
    const sentMatch = firstLine.match(/^lorem:sentences:(\d+)$/);
    if (sentMatch) {
        const n = Math.min(parseInt(sentMatch[1], 10), 100);
        return { text: generateSentences(n), type: 'sentences', help: false };
    }

    return { text: '', type: 'help', help: true };
}

function shuffled(arr) {
    const a = [...arr];
    for (let i = a.length - 1; i > 0; i--) {
        const j = Math.floor(Math.random() * (i + 1));
        [a[i], a[j]] = [a[j], a[i]];
    }
    return a;
}

function generateParagraphs(count) {
    const paragraphs = [];
    for (let p = 0; p < count; p++) {
        const sentences = shuffled(LOREM_SENTENCES).slice(0, 4 + Math.floor(Math.random() * 4));
        // First paragraph always starts with "Lorem ipsum..."
        if (p === 0 && sentences[0] !== LOREM_SENTENCES[0]) {
            sentences.unshift(LOREM_SENTENCES[0]);
        }
        paragraphs.push(sentences.join(' '));
    }
    return paragraphs.join('\n\n');
}

function generateWords(count) {
    const allWords = LOREM_SENTENCES.join(' ').split(/\s+/);
    const result = [];
    // Start with "Lorem ipsum..."
    result.push('Lorem', 'ipsum');
    let idx = 2;
    while (result.length < count) {
        const shuffledWords = shuffled(allWords);
        for (const word of shuffledWords) {
            if (result.length >= count) break;
            result.push(word);
        }
    }
    return result.slice(0, count).join(' ') + '.';
}

function generateSentences(count) {
    const result = [];
    let pool = [...LOREM_SENTENCES];
    while (result.length < count) {
        if (pool.length === 0) pool = shuffled(LOREM_SENTENCES);
        result.push(pool.shift());
    }
    return result.join(' ');
}
