/**
 * JottNote Links Service
 * Handles URL detection, shortening, and interactions
 */

// URL regex pattern
const URL_REGEX = /https?:\/\/[^\s<>"{}|\\^`[\]]+/gi;

// Shortened URL display length
const SHORT_URL_LENGTH = 25;

// Map to track shortened URLs to their full URLs
const shortenedUrlMap = new Map();

/**
 * Detect URLs in text
 * @param {string} text
 * @returns {Array<Object>} { url, start, end }
 */
export function detectUrls(text) {
    const urls = [];
    let match;

    const regex = new RegExp(URL_REGEX);
    while ((match = regex.exec(text)) !== null) {
        urls.push({
            url: match[0],
            start: match.index,
            end: match.index + match[0].length
        });
    }

    return urls;
}

/**
 * Shorten a URL for display with + markers
 * @param {string} url
 * @returns {string}
 */
export function shortenUrl(url) {
    try {
        const parsed = new URL(url);
        let display = parsed.hostname.replace(/^www\./, '');

        // Add path if exists
        if (parsed.pathname && parsed.pathname !== '/') {
            display += parsed.pathname;
        }

        // Shorten if too long - keep beginning and end, replace middle with ...
        if (display.length > SHORT_URL_LENGTH) {
            const keepChars = Math.floor((SHORT_URL_LENGTH - 3) / 2);
            display = display.substring(0, keepChars) + '...' + display.substring(display.length - keepChars);
        }

        const shortened = '+' + display + '+';

        // Store mapping for later retrieval
        shortenedUrlMap.set(shortened, url);
        shortenedUrlMap.set(display, url); // Also store without + markers

        return shortened;
    } catch {
        // If URL parsing fails, just truncate
        let display = url;
        if (url.length > SHORT_URL_LENGTH) {
            const keepChars = Math.floor((SHORT_URL_LENGTH - 3) / 2);
            display = url.substring(0, keepChars) + '...' + url.substring(url.length - keepChars);
        }
        const shortened = '+' + display + '+';
        shortenedUrlMap.set(shortened, url);
        shortenedUrlMap.set(display, url);
        return shortened;
    }
}

/**
 * Get full URL from shortened version
 * @param {string} shortened - Shortened URL (with or without + markers)
 * @returns {string|null}
 */
export function getFullUrlFromShortened(shortened) {
    // Try with + markers
    if (shortenedUrlMap.has(shortened)) {
        return shortenedUrlMap.get(shortened);
    }

    // Try without + markers
    const withoutMarkers = shortened.replace(/^\+|\+$/g, '');
    if (shortenedUrlMap.has(withoutMarkers)) {
        return shortenedUrlMap.get(withoutMarkers);
    }

    // If it's already a full URL, return it
    if (shortened.match(/^https?:\/\//)) {
        return shortened;
    }

    return null;
}

/**
 * Get full URL wrapped with + markers
 * @param {string} url
 * @returns {string}
 */
export function getFullUrl(url) {
    return '+' + url + '+';
}

/**
 * Render text with shortened URLs
 * @param {string} text
 * @param {boolean} shortenLinks
 * @returns {string} HTML with clickable links
 */
export function renderLinksHTML(text, shortenLinks = true) {
    const urls = detectUrls(text);

    if (urls.length === 0) {
        return escapeHtml(text);
    }

    let result = '';
    let lastEnd = 0;

    for (const { url, start, end } of urls) {
        // Add text before this URL
        result += escapeHtml(text.substring(lastEnd, start));

        // Add the link
        const display = shortenLinks ? shortenUrl(url) : url;
        result += `<a href="${escapeHtml(url)}" class="shortened-link" data-full-url="${escapeHtml(url)}" target="_blank" rel="noopener">${escapeHtml(display)}</a>`;

        lastEnd = end;
    }

    // Add remaining text
    result += escapeHtml(text.substring(lastEnd));

    return result;
}

/**
 * Handle link click (Ctrl/Cmd + Click to open)
 * @param {Event} event
 */
export function handleLinkClick(event) {
    const link = event.target.closest('.shortened-link');
    if (!link) return;

    event.preventDefault();

    // Ctrl/Cmd + Click opens in new tab
    if (event.ctrlKey || event.metaKey) {
        const url = link.dataset.fullUrl || link.href;
        window.open(url, '_blank');
        return;
    }

    // Ctrl/Cmd + Shift + Click toggles expansion
    if (event.shiftKey && (event.ctrlKey || event.metaKey)) {
        toggleLinkExpansion(link);
        return;
    }
}

/**
 * Toggle link between shortened and full view
 * @param {HTMLElement} link
 */
export function toggleLinkExpansion(link) {
    const fullUrl = link.dataset.fullUrl;
    const isExpanded = link.classList.contains('expanded');

    if (isExpanded) {
        link.textContent = shortenUrl(fullUrl);
        link.classList.remove('expanded');
    } else {
        link.textContent = fullUrl;
        link.classList.add('expanded');
    }
}

/**
 * Expand all links in a container
 * @param {HTMLElement} container
 */
export function expandAllLinks(container) {
    const links = container.querySelectorAll('.shortened-link');
    links.forEach(link => {
        link.textContent = link.dataset.fullUrl;
        link.classList.add('expanded');
    });
}

/**
 * Get full URL from text for clipboard
 * @param {string} text
 * @returns {string} Text with URLs expanded
 */
export function expandUrlsInText(text) {
    // URLs in the raw text are already full, this is for display->clipboard
    return text;
}

/**
 * Auto-shorten URLs in editor when space or enter is pressed
 * @param {HTMLTextAreaElement} editor
 * @param {boolean} enabled - Whether shortening is enabled
 */
export function autoShortenUrls(editor, enabled = true) {
    if (!enabled) return;

    const content = editor.value;
    const cursorPos = editor.selectionStart;

    // Find URLs in entire content
    const urls = detectUrls(content);

    if (urls.length === 0) return;

    // Get the last URL that's before or right at cursor
    // Account for space/enter that was just typed (cursor is 1-2 chars after URL)
    let lastUrl = null;
    for (const url of urls) {
        if (url.end <= cursorPos && cursorPos <= url.end + 2) {
            lastUrl = url;
        }
    }

    if (!lastUrl) return;

    // Check if this URL is already shortened (has + markers)
    const urlText = content.substring(lastUrl.start, lastUrl.end);
    if (urlText.startsWith('+') && urlText.endsWith('+')) return;

    // Shorten the URL
    const shortened = shortenUrl(lastUrl.url);

    // Replace in content
    const before = content.substring(0, lastUrl.start);
    const after = content.substring(lastUrl.end);
    editor.value = before + shortened + after;

    // Update cursor position (maintain same relative position)
    const cursorOffset = cursorPos - lastUrl.end;
    editor.selectionStart = editor.selectionEnd = lastUrl.start + shortened.length + cursorOffset;

    // Trigger input event to save
    editor.dispatchEvent(new Event('input', { bubbles: true }));
}

/**
 * Toggle link expansion/shortening at cursor position
 * @param {HTMLTextAreaElement} editor
 */
export function toggleLinkAtCursor(editor) {
    const content = editor.value;
    const cursorPos = editor.selectionStart;

    // Find all links (both shortened and full)
    const linkPattern = /\+[^\+]+\+/g;
    let match;
    let linkToToggle = null;

    while ((match = linkPattern.exec(content)) !== null) {
        const start = match.index;
        const end = start + match[0].length;

        // Check if cursor is within this link
        if (cursorPos >= start && cursorPos <= end) {
            linkToToggle = {
                text: match[0],
                start: start,
                end: end
            };
            break;
        }
    }

    if (!linkToToggle) return;

    // Remove + markers to get the link text
    const linkText = linkToToggle.text.substring(1, linkToToggle.text.length - 1);

    let replacement;

    // Check if this is a full URL (starts with http)
    if (linkText.match(/^https?:\/\//)) {
        // It's expanded - shorten it
        replacement = shortenUrl(linkText);
    } else {
        // It's shortened - expand it
        // Try to get full URL from map
        const fullUrl = getFullUrlFromShortened(linkText);
        if (fullUrl) {
            replacement = '+' + fullUrl + '+';
        } else {
            // Can't find full URL, keep as is
            replacement = linkToToggle.text;
        }
    }

    // Replace in content
    const before = content.substring(0, linkToToggle.start);
    const after = content.substring(linkToToggle.end);
    editor.value = before + replacement + after;

    // Update cursor position
    editor.selectionStart = editor.selectionEnd = linkToToggle.start + replacement.length;

    // Trigger input event to save
    editor.dispatchEvent(new Event('input', { bubbles: true }));
}

/**
 * Clear the URL map (call when switching notes or clearing editor)
 */
export function clearUrlMap() {
    shortenedUrlMap.clear();
}

/**
 * Rebuild the shortened URL map from existing content
 * Call this when loading a note to restore the mappings
 * @param {string} content - Note content
 */
export function rebuildUrlMap(content) {
    // Clear existing map
    shortenedUrlMap.clear();

    // Find all shortened links in content (+text+)
    const shortenedPattern = /\+([^\+]+)\+/g;
    let match;

    while ((match = shortenedPattern.exec(content)) !== null) {
        const linkText = match[1];

        // Check if this looks like a shortened URL (not a full URL)
        if (!linkText.match(/^https?:\/\//)) {
            // Try to reconstruct the full URL
            // Remove any ... and try to build a valid URL
            let reconstructed = linkText;

            // If it doesn't have a protocol, try adding https://
            if (!reconstructed.match(/^https?:\/\//)) {
                // Check if it looks like a domain
                if (reconstructed.match(/^[a-zA-Z0-9.-]+/) || reconstructed.includes('...')) {
                    // Remove ... if present (it's middle-truncated)
                    const withoutDots = reconstructed.replace(/\.\.\./, '');

                    // Try to build URL - prefer https
                    let fullUrl = null;

                    // If it has path components, construct full URL
                    if (withoutDots.match(/\//)) {
                        fullUrl = 'https://' + withoutDots;
                    } else {
                        // Just a domain
                        fullUrl = 'https://' + withoutDots;
                    }

                    // Store the mapping
                    if (fullUrl) {
                        shortenedUrlMap.set(linkText, fullUrl);
                        shortenedUrlMap.set(match[0], fullUrl); // Also with + markers
                    }
                }
            }
        } else {
            // It's already a full URL in +url+ format
            // Store it as expanded
            shortenedUrlMap.set(linkText, linkText);
            shortenedUrlMap.set(match[0], linkText);
        }
    }

    // Also find any raw URLs and pre-populate them
    const rawUrls = detectUrls(content);
    for (const urlInfo of rawUrls) {
        const url = urlInfo.url;
        const shortened = shortenUrl(url);
        // shortenUrl already adds to map, so we're good
    }
}

/**
 * Escape HTML characters
 * @param {string} text
 * @returns {string}
 */
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}
