/**
 * JottNote License Module
 * Manages Pro/Free status via ExtensionPay
 *
 * Usage:
 *   import { initLicense, isPro, openUpgrade } from './js/license.js';
 *   await initLicense();
 *   if (isPro()) { ... }
 */

const LICENSE_KEY = 'jottnote_pro_status';
const REVERIFY_INTERVAL = 30 * 24 * 60 * 60 * 1000; // 30 days

let _cachedIsPro = false;
let _extpay = null;
let _pollTimer = null;
let _onStatusChange = null;

/**
 * Initialize the license system. Called once at startup.
 * Reads cached Pro status from storage, then tries to verify with ExtensionPay.
 * @param {Function} onStatusChange - Optional callback when Pro status is updated/verified
 */
export async function initLicense(onStatusChange) {
    _onStatusChange = onStatusChange;

    // 1. Load cached status from storage (instant, works offline)
    try {
        const stored = await chrome.storage.local.get(LICENSE_KEY);
        if (stored[LICENSE_KEY]) {
            _cachedIsPro = stored[LICENSE_KEY].isPro === true;
        }
    } catch (e) {
        console.warn('License: failed to read cached status', e);
    }

    // 2. Initialize ExtensionPay if available
    try {
        if (typeof ExtPay !== 'undefined') {
            _extpay = ExtPay('jottnote');
            _extpay.startBackground();

            // 3. Register onPaid callback for instant payment detection
            // This fires when the content script on extensionpay.com relays
            // the payment success message back to the extension.
            try {
                _extpay.onPaid.addListener(() => {
                    _cachedIsPro = true;
                    chrome.storage.local.set({
                        [LICENSE_KEY]: {
                            isPro: true,
                            verifiedAt: Date.now()
                        }
                    });
                    if (typeof _onStatusChange === 'function') {
                        _onStatusChange(true);
                    }
                    // Stop any active polling since we've confirmed payment
                    if (_pollTimer) {
                        clearInterval(_pollTimer);
                        _pollTimer = null;
                    }
                });
            } catch (e) {
                // onPaid requires content_scripts in manifest — log but don't fail
                console.warn('License: onPaid listener setup failed', e);
            }

            // 4. Listen for storage changes from background script's polling
            // When background detects payment via poll_user_paid(), it stores
            // extensionpay_user to storage.sync which we can detect here.
            try {
                chrome.storage.onChanged.addListener((changes, areaName) => {
                    if (areaName === 'sync' && changes.extensionpay_user) {
                        const newUser = changes.extensionpay_user.newValue;
                        if (newUser && newUser.paidAt && !_cachedIsPro) {
                            _cachedIsPro = true;
                            chrome.storage.local.set({
                                [LICENSE_KEY]: {
                                    isPro: true,
                                    verifiedAt: Date.now()
                                }
                            });
                            if (typeof _onStatusChange === 'function') {
                                _onStatusChange(true);
                            }
                            if (_pollTimer) {
                                clearInterval(_pollTimer);
                                _pollTimer = null;
                            }
                        }
                    }
                });
            } catch (e) {
                console.warn('License: storage listener setup failed', e);
            }

            // 5. Try online verification (non-blocking)
            refreshProStatus().then(isPro => {
                if (typeof onStatusChange === 'function') {
                    onStatusChange(isPro);
                }
            }).catch(err => {
                console.warn('License: online verification failed (offline?)', err);
            });
        }
    } catch (e) {
        console.warn('License: ExtPay init failed', e);
    }
}

/**
 * Synchronous check — zero-cost, used at every gate point.
 * @returns {boolean}
 */
export function isPro() {
    return _cachedIsPro === true;
}

/**
 * Refresh Pro status from ExtensionPay server.
 * Updates both memory cache and chrome.storage.local.
 */
export async function refreshProStatus() {
    if (!_extpay) return _cachedIsPro;

    try {
        const user = await _extpay.getUser();
        _cachedIsPro = user.paid === true;

        await chrome.storage.local.set({
            [LICENSE_KEY]: {
                isPro: _cachedIsPro,
                verifiedAt: Date.now(),
                email: user.email || null
            }
        });

        return _cachedIsPro;
    } catch (e) {
        console.warn('License: refresh failed', e);
        return _cachedIsPro; // Return cached value on failure
    }
}

/**
 * Open the ExtensionPay payment page and poll for completion.
 * Checks if user is already Pro first (e.g. paid in a previous session).
 * @param {Function} onSuccess - Called when payment is confirmed
 */
export async function openUpgrade(onSuccess) {
    if (!_extpay) {
        console.error('License: ExtPay not initialized');
        return;
    }

    // Check if already Pro before opening payment page
    // (handles case where user paid previously but popup didn't detect it)
    try {
        const alreadyPro = await refreshProStatus();
        if (alreadyPro) {
            if (typeof onSuccess === 'function') onSuccess();
            return;
        }
    } catch (e) {
        // Continue to payment page if check fails
    }

    _extpay.openPaymentPage();

    // Poll for payment completion every 1 second
    let elapsed = 0;
    const POLL_INTERVAL = 1000;
    const MAX_POLL_TIME = 10 * 60 * 1000; // 10 minutes

    if (_pollTimer) clearInterval(_pollTimer);

    _pollTimer = setInterval(async () => {
        elapsed += POLL_INTERVAL;

        if (elapsed >= MAX_POLL_TIME) {
            clearInterval(_pollTimer);
            _pollTimer = null;
            return;
        }

        try {
            const nowPro = await refreshProStatus();
            if (nowPro) {
                clearInterval(_pollTimer);
                _pollTimer = null;
                if (typeof onSuccess === 'function') onSuccess();
            }
        } catch (e) {
            // Keep polling
        }
    }, POLL_INTERVAL);
}

/**
 * List of free theme keys (4 themes).
 */
export const FREE_THEMES = ['vancouver', 'totoro', 'a24', 'shadow', 'sanrio'];

/**
 * Check if a theme is available on the free tier.
 * @param {string} themeKey
 * @returns {boolean}
 */
export function isThemeFree(themeKey) {
    return FREE_THEMES.includes(themeKey);
}

/**
 * Pro-only keyword modes.
 */
export const PRO_MODES = ['code', 'timer', 'bookmarks', 'session', 'game', 'date', 'sort', 'template', 'placeholder', 'define', 'related'];

/**
 * Check if a keyword mode requires Pro.
 * @param {string} mode
 * @returns {boolean}
 */
export function isProMode(mode) {
    return PRO_MODES.includes(mode);
}

/**
 * Pro-only slash command keywords (filtered from popup for free users).
 */
export const PRO_KEYWORDS = ['code', 'bookmarks', 'timer', 'date', 'template', 'sort', 'session', 'game', 'placeholder', 'uuid', 'lorem', 'weather', 'define', 'related'];
