/**
 * JottNote Keywords Service
 * Detects and handles keyword modes
 */

// Default keyword mappings
const DEFAULT_KEYWORDS = {
    math: 'math',
    sum: 'sum',
    avg: 'avg',
    average: 'avg',
    count: 'count',
    code: 'code',
    timer: 'timer',
    list: 'list',
    currency: 'currency',
    template: 'template',
    sort: 'sort',
    game: 'game',
    bookmarks: 'bookmarks',
    session: 'session',
    date: 'date',
    placeholder: 'placeholder',
    uuid: 'uuid',
    lorem: 'lorem',
    weather: 'weather',
    define: 'define',
    related: 'related'
};

// Current keywords (starts with defaults, updated by settings)
let currentKeywords = { ...DEFAULT_KEYWORDS };

// Configuration from settings
let keywordsConfig = {
    disableAllKeywords: false,
    disableLinkShortening: false,
    disableAllLinks: false
};

/**
 * Update keywords configuration from settings
 * @param {Object} settings
 */
export function updateKeywordsConfig(settings) {
    if (!settings) return;

    // Update flags
    keywordsConfig.disableAllKeywords = settings.disableAllKeywords || false;
    keywordsConfig.disableLinkShortening = settings.disableLinkShortening || false;
    keywordsConfig.disableAllLinks = settings.disableAllLinks || false;

    // Reset to defaults first
    currentKeywords = { ...DEFAULT_KEYWORDS };

    // Helper to add custom keywords
    const addCustom = (inputStr, mode) => {
        if (!inputStr) return;
        const keys = inputStr.split(',').map(k => k.trim().toLowerCase()).filter(k => k);
        keys.forEach(key => {
            currentKeywords[key] = mode;
        });
    };

    // Helper to override main keyword
    const overrideMain = (key, mode) => {
        if (key && key !== mode) {
            // Remove old mapping if it exists (optional, but safer to keep defaults working too?)
            // actually, usually main keyword REPLACES the default one if it's different?
            // currentKeywords[mode] is the default.
            // If user selects "todo" as main keyword for "list", then "todo" -> "list".
            currentKeywords[key.toLowerCase()] = mode;
        }
    };

    // Apply custom keywords from settings
    addCustom(settings.listKeywords, 'list');
    addCustom(settings.mathKeywords, 'math');
    addCustom(settings.sumKeywords, 'sum');
    addCustom(settings.avgKeywords, 'avg');
    addCustom(settings.countKeywords, 'count');
    addCustom(settings.codeKeywords, 'code');
    addCustom(settings.timerKeywords, 'timer');
    addCustom(settings.dateKeywords, 'date');

    // Apply main keyword overrides (ensure the selected main keyword maps to the mode)
    if (settings.listMainKeyword) currentKeywords[settings.listMainKeyword] = 'list';
    if (settings.mathMainKeyword) currentKeywords[settings.mathMainKeyword] = 'math';
    if (settings.sumMainKeyword) currentKeywords[settings.sumMainKeyword] = 'sum';
    if (settings.avgMainKeyword) currentKeywords[settings.avgMainKeyword] = 'avg';
    if (settings.countMainKeyword) currentKeywords[settings.countMainKeyword] = 'count';
    if (settings.codeMainKeyword) currentKeywords[settings.codeMainKeyword] = 'code';
    if (settings.timerMainKeyword) currentKeywords[settings.timerMainKeyword] = 'timer';
    if (settings.dateMainKeyword) currentKeywords[settings.dateMainKeyword] = 'date';
}

/**
 * Parse first line for keyword and optional title
 * @param {string} content - Note content
 * @returns {Object} { keyword, title, mode }
 */
export function parseKeyword(content) {
    if (keywordsConfig.disableAllKeywords) {
        return {
            keyword: null,
            mode: null,
            title: null,
            hasKeyword: false
        };
    }

    const firstLine = content.split('\n')[0] || '';
    const trimmed = firstLine.trim().toLowerCase();

    // Check for keyword with title (e.g., "list: My Tasks")
    const colonIndex = firstLine.indexOf(':');
    if (colonIndex > 0) {
        const keywordPart = firstLine.substring(0, colonIndex).trim().toLowerCase();
        const titlePart = firstLine.substring(colonIndex + 1).trim();

        if (currentKeywords[keywordPart]) {
            return {
                keyword: keywordPart,
                mode: currentKeywords[keywordPart],
                title: titlePart || null,
                hasKeyword: true
            };
        }
    }

    // Check for standalone keyword
    if (currentKeywords[trimmed]) {
        return {
            keyword: trimmed,
            mode: currentKeywords[trimmed],
            title: null,
            hasKeyword: true
        };
    }

    return {
        keyword: null,
        mode: null,
        title: null,
        hasKeyword: false
    };
}

/**
 * Check if slash command should show
 * @param {string} content
 * @param {number} cursorPosition
 * @returns {boolean}
 */
export function shouldShowSlashCommand(content, cursorPosition) {
    if (keywordsConfig.disableAllKeywords) return false;

    // Find the start of the current line
    const beforeCursor = content.substring(0, cursorPosition);
    const lastNewline = beforeCursor.lastIndexOf('\n');
    const lineStart = lastNewline + 1;
    const currentLineContent = beforeCursor.substring(lineStart);

    // Show if line starts with just "/"
    return currentLineContent === '/';
}

/**
 * Get available keywords for slash command
 * @returns {Array}
 */
export function getSlashKeywords() {
    return [
        { key: 'list', label: 'list', description: 'Enable checklist' },
        { key: 'math', label: 'math', description: 'Calculate, convert, & variables' },
        { key: 'sum', label: 'sum', description: 'Add all numbers' },
        { key: 'avg', label: 'avg', description: 'Average all numbers' },
        { key: 'count', label: 'count', description: 'Count items, words, chars' },
        { key: 'code', label: 'code', description: 'Syntax highlighting' },
        { key: 'bookmarks', label: 'bookmarks', description: 'Save and manage bookmarks' },
        { key: 'timer', label: 'timer', description: 'Start a timer or countdown' },
        { key: 'template', label: 'template', description: 'Create and load templates' },
        { key: 'sort', label: 'sort', description: 'Sort lines alphabetically' },
        { key: 'currency', label: 'currency', description: 'View supported currencies' },
        { key: 'session', label: 'session', description: 'Save and manage browser sessions' },
        { key: 'date', label: 'date', description: 'Date calculations & insertion' },
        { key: 'game', label: 'game', description: 'Mini-games' },
        { key: 'placeholder', label: 'placeholder', description: 'Create & manage template placeholders' },
        { key: 'uuid', label: 'uuid', description: 'Generate UUIDs & passwords' },
        { key: 'lorem', label: 'lorem', description: 'Generate placeholder text' },
        { key: 'weather', label: 'weather', description: 'Current weather conditions' },
        { key: 'define', label: 'define', description: 'Dictionary lookup' },
        { key: 'related', label: 'related', description: 'Find synonyms & antonyms' }
    ];
}

/**
 * Insert keyword from slash command
 * @param {HTMLTextAreaElement} editor
 * @param {string} keyword
 */
export function insertKeyword(editor, keyword) {
    const content = editor.value;
    const cursorPosition = editor.selectionStart;

    // Find the "/" character to replace
    const beforeCursor = content.substring(0, cursorPosition);
    const lastNewline = beforeCursor.lastIndexOf('\n');
    const lineStart = lastNewline + 1;

    // Replace "/" with keyword (no trailing newline so help guide shows)
    const before = content.substring(0, lineStart);
    const after = content.substring(cursorPosition);

    editor.value = before + keyword + after;
    editor.selectionStart = editor.selectionEnd = before.length + keyword.length;

    // Trigger change event
    editor.dispatchEvent(new Event('input', { bubbles: true }));
}

/**
 * Get lines for mode processing (excluding first keyword line)
 * @param {string} content
 * @returns {Array<{index: number, text: string, originalIndex: number}>}
 */
export function getContentLines(content) {
    const lines = content.split('\n');
    const result = [];

    // Skip first line if it's a keyword
    const { hasKeyword } = parseKeyword(content);
    const startIndex = hasKeyword ? 1 : 0;

    for (let i = startIndex; i < lines.length; i++) {
        result.push({
            index: result.length,
            originalIndex: i,
            text: lines[i]
        });
    }

    return result;
}

/**
 * Check if a line is a comment
 * @param {string} line
 * @returns {boolean}
 */
export function isComment(line) {
    return line.trim().startsWith('//');
}

/**
 * Check if a line is a heading
 * @param {string} line
 * @returns {number} Heading level (1-3) or 0 if not a heading
 */
export function getHeadingLevel(line) {
    const trimmed = line.trimStart();
    if (trimmed.startsWith('### ')) return 3;
    if (trimmed.startsWith('## ')) return 2;
    if (trimmed.startsWith('# ')) return 1;
    return 0;
}

/**
 * Parse template command from content
 * @param {string} content - Note content
 * @returns {Object} { type, templateName, hasTemplateCommand }
 *   type: 'help' | 'create' | 'end' | 'load' | null
 *   templateName: string or null
 */
export function parseTemplateCommand(content) {
    if (keywordsConfig.disableAllKeywords) return { type: null, templateName: null, hasTemplateCommand: false };

    const lines = content.split('\n');
    const firstLine = lines[0] || '';
    const trimmed = firstLine.trim().toLowerCase();

    // Check for "template:end" anywhere in content (CHECK THIS FIRST!)
    const hasEnd = lines.some(line => line.trim().toLowerCase() === 'template:end');
    if (hasEnd) {
        return { type: 'end', templateName: null, hasTemplateCommand: true };
    }

    // Check for "template" alone (help)
    if (trimmed === 'template' && lines.length === 1) {
        return { type: 'help', templateName: null, hasTemplateCommand: true };
    }

    // Check for "template:create" on first line
    if (trimmed === 'template:create') {
        return { type: 'create', templateName: null, hasTemplateCommand: true };
    }

    // Check for "template:name" (load template) on first line
    if (trimmed.startsWith('template:')) {
        const templateName = trimmed.substring(9).trim();
        if (templateName && templateName !== 'create' && templateName !== 'end') {
            return { type: 'load', templateName, hasTemplateCommand: true };
        }
    }

    return { type: null, templateName: null, hasTemplateCommand: false };
}

/**
 * Parse game command from content
 * @param {string} content - Note content
 * @returns {Object} { type, gameName, hasGameCommand }
 */
export function parseGameCommand(content) {
    if (keywordsConfig.disableAllKeywords) return { type: null, gameName: null, hasGameCommand: false };

    const lines = content.split('\n');
    const firstLine = lines[0] || '';
    const trimmed = firstLine.trim().toLowerCase();

    // Check for "game" alone (help)
    if (trimmed === 'game' && lines.length === 1) {
        return { type: 'help', gameName: null, hasGameCommand: true };
    }

    // Check for "game:tic" (Tic-Tac-Toe)
    if (trimmed === 'game:tic') {
        return { type: 'tic', gameName: 'tic', hasGameCommand: true };
    }

    // Check for "game:hangman" (Hangman)
    if (trimmed === 'game:hangman') {
        return { type: 'hangman', gameName: 'hangman', hasGameCommand: true };
    }

    // Check for "game:quiz" (Quiz)
    if (trimmed === 'game:quiz') {
        return { type: 'quiz', gameName: 'quiz', hasGameCommand: true };
    }

    // Check for "game:type" (Typing Speed Test)
    if (trimmed === 'game:type') {
        return { type: 'type', gameName: 'type', hasGameCommand: true };
    }

    // Check for "game:react" (Reaction Time)
    if (trimmed === 'game:react') {
        return { type: 'react', gameName: 'react', hasGameCommand: true };
    }

    // Check for "game:mines" (Minesweeper)
    if (trimmed === 'game:mines') {
        return { type: 'mines', gameName: 'mines', hasGameCommand: true };
    }

    // Check for "game:memory" (Memory Match)
    if (trimmed === 'game:memory') {
        return { type: 'memory', gameName: 'memory', hasGameCommand: true };
    }

    // Check for "game:wordle" (Wordle)
    if (trimmed === 'game:wordle') {
        return { type: 'wordle', gameName: 'wordle', hasGameCommand: true };
    }

    return { type: null, gameName: null, hasGameCommand: false };
}

/**
 * Parse bookmarks command from content
 * @param {string} content - Note content
 * @returns {Object} { type, tag, query, bookmarkId, hasBookmarksCommand }
 *   type: 'help' | 'add' | 'filter' | 'search' | 'edit' | null
 *   tag: string or null (for filter)
 *   query: string or null (for search)
 *   bookmarkId: string or null (for edit)
 */
export function parseBookmarksCommand(content) {
    if (keywordsConfig.disableAllKeywords) return { type: null, hasBookmarksCommand: false };

    const lines = content.split('\n');
    const firstLine = lines[0] || '';
    const trimmed = firstLine.trim().toLowerCase();

    // Check for "bookmarks:end" or "bookmarks:save" anywhere in content
    const hasEnd = lines.some(line => {
        const l = line.trim().toLowerCase();
        return l === 'bookmarks:end' || l === 'bookmarks:save';
    });

    if (hasEnd) {
        return { type: 'end', hasBookmarksCommand: true };
    }

    // Check for "bookmarks" alone (help/show all)
    if (trimmed === 'bookmarks' && lines.length === 1) {
        return { type: 'help', hasBookmarksCommand: true };
    }

    // Check for "bookmarks:add"
    if (trimmed === 'bookmarks:add') {
        return { type: 'add', hasBookmarksCommand: true };
    }

    // Check for "bookmarks:edit:id"
    if (trimmed.startsWith('bookmarks:edit:')) {
        const bookmarkId = trimmed.substring(15).trim();
        return { type: 'edit', bookmarkId, hasBookmarksCommand: true };
    }

    // Check for "bookmarks:import"
    if (trimmed === 'bookmarks:import') {
        return { type: 'import', hasBookmarksCommand: true };
    }

    // Check for "bookmarks:search query"
    if (trimmed.startsWith('bookmarks:search ')) {
        const query = trimmed.substring(17).trim();
        return { type: 'search', query, hasBookmarksCommand: true };
    }

    // Check for "bookmarks:tag" (filter by tag)
    if (trimmed.startsWith('bookmarks:')) {
        const tag = trimmed.substring(10).trim();
        if (tag && tag !== 'add' && tag !== 'search' && tag !== 'end' && tag !== 'save' && tag !== 'import' && !tag.startsWith('edit:')) {
            return { type: 'filter', tag, hasBookmarksCommand: true };
        }
    }

    return { type: null, hasBookmarksCommand: false };
}

/**
 * Parse session command from content
 * @param {string} content - Note content
 * @returns {Object} { type, tag, query, sessionId, hasSessionCommand }
 *   type: 'help' | 'add' | 'filter' | 'search' | 'edit' | 'end' | null
 *   tag: string or null (for filter)
 *   query: string or null (for search)
 *   sessionId: string or null (for edit)
 */
export function parseSessionCommand(content) {
    if (keywordsConfig.disableAllKeywords) return { type: null, hasSessionCommand: false };

    const lines = content.split('\n');
    const firstLine = lines[0] || '';
    const trimmed = firstLine.trim().toLowerCase();

    // Check for "session:end" or "session:save" anywhere in content
    const hasEnd = lines.some(line => {
        const l = line.trim().toLowerCase();
        return l === 'session:end' || l === 'session:save';
    });

    if (hasEnd) {
        return { type: 'end', hasSessionCommand: true };
    }

    // Check for "session" alone (help/show all)
    if (trimmed === 'session' && lines.length === 1) {
        return { type: 'help', hasSessionCommand: true };
    }

    // Check for "session:add"
    if (trimmed === 'session:add') {
        return { type: 'add', hasSessionCommand: true };
    }

    // Check for "session:edit:id"
    if (trimmed.startsWith('session:edit:')) {
        const sessionId = trimmed.substring(13).trim();
        return { type: 'edit', sessionId, hasSessionCommand: true };
    }

    // Check for "session:import"
    if (trimmed === 'session:import') {
        return { type: 'import', hasSessionCommand: true };
    }

    // Check for "session:search query"
    if (trimmed.startsWith('session:search ')) {
        const query = trimmed.substring(15).trim();
        return { type: 'search', query, hasSessionCommand: true };
    }

    // Check for "session:tag" (filter by tag)
    if (trimmed.startsWith('session:')) {
        const tag = trimmed.substring(8).trim();
        if (tag && tag !== 'add' && tag !== 'search' && tag !== 'end' && tag !== 'save' && tag !== 'import' && !tag.startsWith('edit:')) {
            return { type: 'filter', tag, hasSessionCommand: true };
        }
    }

    return { type: null, hasSessionCommand: false };
}

/**
 * Parse placeholders command from content
 * @param {string} content - Note content
 * @returns {Object} { type, hasPlaceholdersCommand }
 *   type: 'help' | 'create' | 'end' | null
 */
export function parsePlaceholdersCommand(content) {
    if (keywordsConfig.disableAllKeywords) return { type: null, hasPlaceholdersCommand: false };

    const lines = content.split('\n');
    const firstLine = lines[0] || '';
    const trimmed = firstLine.trim().toLowerCase();

    // Check for "placeholder:end" anywhere in content (CHECK THIS FIRST!)
    const hasEnd = lines.some(line => line.trim().toLowerCase() === 'placeholder:end');
    if (hasEnd) {
        return { type: 'end', hasPlaceholdersCommand: true };
    }

    // Check for "placeholder" alone (help)
    if (trimmed === 'placeholder' && lines.length === 1) {
        return { type: 'help', hasPlaceholdersCommand: true };
    }

    // Check for "placeholder:create" on first line
    if (trimmed === 'placeholder:create') {
        return { type: 'create', hasPlaceholdersCommand: true };
    }

    return { type: null, hasPlaceholdersCommand: false };
}

/**
 * Parse dictionary or related command
 * @param {string} content - Note content
 * @returns {Object} { word, hasDefineCommand }
 */
export function parseDictionaryCommand(content) {
    if (keywordsConfig.disableAllKeywords) return { word: null, hasDefineCommand: false };

    const firstLine = (content.split('\n')[0] || '').trim();
    const colonIdx = firstLine.indexOf(':');

    if (colonIdx < 0) {
        return { word: null, hasDefineCommand: true };
    }

    const word = firstLine.substring(colonIdx + 1).trim();
    return { word, hasDefineCommand: true };
}
