/**
 * Wordle Game Module
 */

const WORD_LENGTH = 5;
const MAX_GUESSES = 6;

// 5-letter word list
const WORDS = [
    'ABOUT', 'ABOVE', 'ABUSE', 'ACTOR', 'ACUTE', 'ADMIT', 'ADOPT', 'ADULT', 'AFTER', 'AGAIN',
    'AGENT', 'AGREE', 'AHEAD', 'ALARM', 'ALBUM', 'ALERT', 'ALIEN', 'ALIGN', 'ALIVE', 'ALLEY',
    'ALLOW', 'ALONE', 'ALONG', 'ALTER', 'AMONG', 'ANGEL', 'ANGER', 'ANGLE', 'ANGRY', 'APART',
    'APPLE', 'APPLY', 'ARENA', 'ARGUE', 'ARISE', 'ARMOR', 'ARRAY', 'ASIDE', 'ASSET', 'AVOID',
    'AWARD', 'AWARE', 'BADLY', 'BASIC', 'BASIS', 'BEACH', 'BEAST', 'BEGIN', 'BEING', 'BELOW',
    'BENCH', 'BIBLE', 'BIRTH', 'BLACK', 'BLADE', 'BLAME', 'BLAND', 'BLANK', 'BLAST', 'BLAZE',
    'BLEED', 'BLEND', 'BLESS', 'BLIND', 'BLOCK', 'BLOOD', 'BLOOM', 'BLOWN', 'BOARD', 'BONUS',
    'BOOST', 'BOUND', 'BRAIN', 'BRAND', 'BRAVE', 'BREAD', 'BREAK', 'BREED', 'BRICK', 'BRIDE',
    'BRIEF', 'BRING', 'BROAD', 'BROKE', 'BROWN', 'BRUSH', 'BUILD', 'BUILT', 'BUNCH', 'BURST',
    'BUYER', 'CABIN', 'CABLE', 'CANDY', 'CARRY', 'CATCH', 'CAUSE', 'CHAIN', 'CHAIR', 'CHAOS',
    'CHARM', 'CHART', 'CHASE', 'CHEAP', 'CHECK', 'CHEEK', 'CHEST', 'CHIEF', 'CHILD', 'CHINA',
    'CHOIR', 'CHUNK', 'CLAIM', 'CLASS', 'CLEAN', 'CLEAR', 'CLERK', 'CLICK', 'CLIFF', 'CLIMB',
    'CLING', 'CLOCK', 'CLONE', 'CLOSE', 'CLOTH', 'CLOUD', 'COACH', 'COAST', 'COLOR', 'COMET',
    'CORAL', 'COUNT', 'COURT', 'COVER', 'CRACK', 'CRAFT', 'CRANE', 'CRASH', 'CRAZY', 'CREAM',
    'CREEK', 'CRIME', 'CRISP', 'CROSS', 'CROWD', 'CROWN', 'CRUEL', 'CRUSH', 'CURVE', 'CYCLE',
    'DAILY', 'DANCE', 'DEBUT', 'DECAY', 'DELAY', 'DELTA', 'DENSE', 'DEPTH', 'DEVIL', 'DIARY',
    'DIRTY', 'DOUBT', 'DOZEN', 'DRAFT', 'DRAIN', 'DRAMA', 'DRANK', 'DRAWN', 'DREAM', 'DRESS',
    'DRIED', 'DRIFT', 'DRILL', 'DRINK', 'DRIVE', 'DRONE', 'DROWN', 'DRUGS', 'DRUNK', 'DYING',
    'EAGER', 'EAGLE', 'EARLY', 'EARTH', 'EATEN', 'EIGHT', 'ELDER', 'ELECT', 'ELITE', 'EMPTY',
    'ENEMY', 'ENJOY', 'ENTER', 'ENTRY', 'EQUAL', 'ERROR', 'EVENT', 'EVERY', 'EXACT', 'EXILE',
    'EXIST', 'EXTRA', 'FAINT', 'FAITH', 'FALSE', 'FANCY', 'FATAL', 'FAULT', 'FEAST', 'FENCE',
    'FEVER', 'FIBER', 'FIELD', 'FIFTH', 'FIFTY', 'FIGHT', 'FINAL', 'FIRST', 'FIXED', 'FLAME',
    'FLASH', 'FLEET', 'FLESH', 'FLICK', 'FLIES', 'FLOAT', 'FLOOD', 'FLOOR', 'FLOUR', 'FLUID',
    'FLUSH', 'FLUTE', 'FOCAL', 'FOCUS', 'FORCE', 'FORGE', 'FORTH', 'FORUM', 'FOUND', 'FRAME',
    'FRANK', 'FRAUD', 'FRESH', 'FRONT', 'FROST', 'FRUIT', 'FULLY', 'FUNNY', 'GHOST', 'GIANT',
    'GIVEN', 'GLASS', 'GLOBE', 'GLOOM', 'GLORY', 'GLOVE', 'GOING', 'GRACE', 'GRADE',
    'GRAIN', 'GRAND', 'GRANT', 'GRAPH', 'GRASP', 'GRASS', 'GRAVE', 'GREAT', 'GREEN', 'GREET',
    'GRIEF', 'GRILL', 'GRIND', 'GROAN', 'GROSS', 'GROUP', 'GROWN', 'GUARD', 'GUESS', 'GUEST',
    'GUIDE', 'GUILT', 'HAPPY', 'HARSH', 'HEART', 'HEAVY', 'HENCE', 'HERBS', 'HONOR', 'HORSE',
    'HOTEL', 'HOUSE', 'HUMAN', 'HUMOR', 'HURRY', 'IDEAL', 'IMAGE', 'IMPLY', 'INDEX', 'INDIE',
    'INNER', 'INPUT', 'IRONY', 'ISSUE', 'IVORY', 'JEWEL', 'JOINT', 'JOKER', 'JUDGE', 'JUICE',
    'KNIFE', 'KNOCK', 'KNOWN', 'LABEL', 'LANCE', 'LARGE', 'LASER', 'LATER', 'LAUGH', 'LAYER',
    'LEARN', 'LEASE', 'LEAVE', 'LEGAL', 'LEMON', 'LEVEL', 'LEVER', 'LIGHT', 'LIMIT', 'LINEN',
    'LIVER', 'LOCAL', 'LODGE', 'LOGIC', 'LOOSE', 'LOVER', 'LOWER', 'LOYAL', 'LUCKY',
    'LUNCH', 'LYING', 'MAGIC', 'MAJOR', 'MAKER', 'MANOR', 'MAPLE', 'MARCH', 'MATCH', 'MAYBE',
    'MAYOR', 'MEDIA', 'MERIT', 'METAL', 'METER', 'MIDST', 'MIGHT', 'MINOR', 'MINUS', 'MIXED',
    'MODEL', 'MONEY', 'MONTH', 'MORAL', 'MOTOR', 'MOUNT', 'MOUSE', 'MOUTH', 'MOVED', 'MOVIE',
    'MUSIC', 'NAMED', 'NERVE', 'NEVER', 'NEWLY', 'NIGHT', 'NOBLE', 'NOISE', 'NORTH', 'NOTED',
    'NOVEL', 'NURSE', 'OCCUR', 'OCEAN', 'OFFER', 'OFTEN', 'OLIVE', 'ONSET', 'OPERA', 'ORBIT',
    'ORDER', 'OTHER', 'OUGHT', 'OUTER', 'OWNED', 'OWNER', 'OXIDE', 'OZONE', 'PAINT', 'PANEL',
    'PANIC', 'PAPER', 'PARTY', 'PASTA', 'PATCH', 'PAUSE', 'PEACE', 'PEACH', 'PEARL', 'PENNY',
    'PHASE', 'PHONE', 'PHOTO', 'PIANO', 'PIECE', 'PILOT', 'PITCH', 'PIZZA', 'PLACE', 'PLAIN',
    'PLANE', 'PLANT', 'PLATE', 'PLAZA', 'PLEAD', 'PLUMB', 'POINT', 'POKER', 'POLAR', 'POUND',
    'POWER', 'PRESS', 'PRICE', 'PRIDE', 'PRIME', 'PRINT', 'PRIOR', 'PRIZE', 'PROBE', 'PROOF',
    'PROSE', 'PROUD', 'PROVE', 'PSALM', 'PULSE', 'PUNCH', 'PUPIL', 'PURSE', 'QUEEN', 'QUEST',
    'QUEUE', 'QUICK', 'QUIET', 'QUOTA', 'QUOTE', 'RADAR', 'RADIO', 'RAISE', 'RALLY', 'RANGE',
    'RAPID', 'RATIO', 'REACH', 'REACT', 'READY', 'REALM', 'REBEL', 'REFER', 'REIGN', 'RELAX',
    'REPLY', 'RIDER', 'RIDGE', 'RIFLE', 'RIGHT', 'RIGID', 'RISKY', 'RIVAL', 'RIVER', 'ROAST',
    'ROBIN', 'ROBOT', 'ROCKY', 'ROGER', 'ROGUE', 'ROMAN', 'ROUGH', 'ROUND', 'ROUTE', 'ROVER',
    'ROYAL', 'RUGBY', 'RULER', 'RURAL', 'SADLY', 'SAINT', 'SALAD', 'SAUCE', 'SCALE', 'SCENE',
    'SCOPE', 'SCORE', 'SCOUT', 'SENSE', 'SERVE', 'SEVEN', 'SHALL', 'SHAME', 'SHAPE', 'SHARE',
    'SHARK', 'SHARP', 'SHEER', 'SHEET', 'SHELF', 'SHELL', 'SHIFT', 'SHINE', 'SHIRT', 'SHOCK',
    'SHOOT', 'SHORT', 'SHOUT', 'SIGHT', 'SILLY', 'SINCE', 'SIXTH', 'SIXTY', 'SIZED', 'SKILL',
    'SKULL', 'SLASH', 'SLEEP', 'SLICE', 'SLIDE', 'SLOPE', 'SMALL', 'SMART', 'SMELL', 'SMILE',
    'SMITH', 'SMOKE', 'SNAKE', 'SOLAR', 'SOLID', 'SOLVE', 'SORRY', 'SOUND', 'SOUTH', 'SPACE',
    'SPARE', 'SPARK', 'SPEAK', 'SPEED', 'SPEND', 'SPENT', 'SPICE', 'SPINE', 'SPLIT', 'SPOKE',
    'SPOON', 'SPORT', 'SPRAY', 'SQUAD', 'STACK', 'STAFF', 'STAGE', 'STAIN', 'STAKE', 'STALE',
    'STALL', 'STAMP', 'STAND', 'STARK', 'START', 'STATE', 'STAYS', 'STEAK', 'STEAL', 'STEAM',
    'STEEL', 'STEEP', 'STEER', 'STERN', 'STICK', 'STIFF', 'STILL', 'STOCK', 'STONE', 'STOOD',
    'STORE', 'STORM', 'STORY', 'STOVE', 'STRAP', 'STRAW', 'STRIP', 'STUCK', 'STUDY', 'STUFF',
    'STYLE', 'SUGAR', 'SUITE', 'SUPER', 'SURGE', 'SWAMP', 'SWEAR', 'SWEEP', 'SWEET', 'SWIFT',
    'SWING', 'SWORD', 'SWORE', 'SWORN', 'SYRUP', 'TABLE', 'TASTE', 'TEACH', 'TEETH', 'TEMPO',
    'THEME', 'THICK', 'THIEF', 'THING', 'THINK', 'THIRD', 'THORN', 'THREE', 'THREW', 'THROW',
    'THUMB', 'TIDAL', 'TIGER', 'TIGHT', 'TIMER', 'TIRED', 'TITLE', 'TODAY', 'TOKEN', 'TOPIC',
    'TOTAL', 'TOUCH', 'TOUGH', 'TOWEL', 'TOWER', 'TOXIC', 'TRACE', 'TRACK', 'TRADE', 'TRAIL',
    'TRAIN', 'TRAIT', 'TRASH', 'TREAT', 'TREND', 'TRIAL', 'TRIBE', 'TRICK', 'TRIED', 'TROOP',
    'TRUCK', 'TRULY', 'TRUMP', 'TRUNK', 'TRUST', 'TRUTH', 'TUMOR', 'TWICE', 'TWIST', 'ULTRA',
    'UNCLE', 'UNDER', 'UNION', 'UNITE', 'UNITY', 'UNTIL', 'UPPER', 'UPSET', 'URBAN', 'USAGE',
    'USUAL', 'UTTER', 'VALID', 'VALUE', 'VAPOR', 'VAULT', 'VIDEO', 'VIGOR', 'VINYL', 'VIRAL',
    'VIRUS', 'VISIT', 'VITAL', 'VIVID', 'VOCAL', 'VODKA', 'VOICE', 'VOTER', 'WAIST', 'WASTE',
    'WATCH', 'WATER', 'WEARY', 'WEAVE', 'WEDGE', 'WEIGH', 'WEIRD', 'WHEAT', 'WHEEL', 'WHERE',
    'WHICH', 'WHILE', 'WHITE', 'WHOLE', 'WHOSE', 'WIDTH', 'WITCH', 'WOMAN', 'WOMEN', 'WORLD',
    'WORRY', 'WORSE', 'WORST', 'WORTH', 'WOULD', 'WOUND', 'WRATH', 'WRIST', 'WRITE', 'WRONG',
    'WROTE', 'YACHT', 'YIELD', 'YOUNG', 'YOUTH', 'ZEBRA'
];

// Game state
let gameContainer = null;
let targetWord = '';
let guesses = [];       // array of submitted guess strings
let currentGuess = '';
let gameOver = false;
let gameWon = false;
let showHelp = false;
let helpChecked = false;
let letterStates = {};  // letter -> 'correct' | 'present' | 'absent'

const HELP_KEY = 'game_help_seen_wordle';
const INSTRUCTIONS = 'Guess the 5-letter word in 6 tries. Type a word and press Enter. Green = correct spot, yellow = wrong spot, gray = not in word.';

/**
 * Initialize the game
 */
export async function initWordle(container) {
    gameContainer = container;
    if (!helpChecked) {
        helpChecked = true;
        try {
            const data = await chrome.storage.local.get(HELP_KEY);
            showHelp = !data[HELP_KEY];
        } catch (e) { showHelp = true; }
    }
    resetGame();
    renderGame();
    document.removeEventListener('keydown', handleKeyDown);
    document.addEventListener('keydown', handleKeyDown);
}

/**
 * Reset game state
 */
function resetGame() {
    targetWord = WORDS[Math.floor(Math.random() * WORDS.length)];
    guesses = [];
    currentGuess = '';
    gameOver = false;
    gameWon = false;
    letterStates = {};
}

/**
 * Evaluate a guess against the target
 * Returns array of 'correct' | 'present' | 'absent'
 */
function evaluateGuess(guess) {
    const result = Array(WORD_LENGTH).fill('absent');
    const targetChars = targetWord.split('');
    const guessChars = guess.split('');

    // First pass: correct positions
    for (let i = 0; i < WORD_LENGTH; i++) {
        if (guessChars[i] === targetChars[i]) {
            result[i] = 'correct';
            targetChars[i] = null;
            guessChars[i] = null;
        }
    }

    // Second pass: present but wrong position
    for (let i = 0; i < WORD_LENGTH; i++) {
        if (guessChars[i] === null) continue;
        const idx = targetChars.indexOf(guessChars[i]);
        if (idx !== -1) {
            result[i] = 'present';
            targetChars[idx] = null;
        }
    }

    return result;
}

/**
 * Submit current guess
 */
function submitGuess() {
    if (currentGuess.length !== WORD_LENGTH) return;

    const guess = currentGuess.toUpperCase();
    const result = evaluateGuess(guess);

    guesses.push({ word: guess, result });

    // Update letter states for keyboard
    for (let i = 0; i < WORD_LENGTH; i++) {
        const letter = guess[i];
        const state = result[i];
        // Only upgrade: absent -> present -> correct
        if (state === 'correct') {
            letterStates[letter] = 'correct';
        } else if (state === 'present' && letterStates[letter] !== 'correct') {
            letterStates[letter] = 'present';
        } else if (!letterStates[letter]) {
            letterStates[letter] = 'absent';
        }
    }

    if (guess === targetWord) {
        gameWon = true;
        gameOver = true;
    } else if (guesses.length >= MAX_GUESSES) {
        gameOver = true;
    }

    currentGuess = '';
    renderGame();
}

/**
 * Handle keyboard input
 */
function handleKeyDown(e) {
    if (!gameContainer || gameOver) return;
    if (!gameContainer.querySelector('.wordle-game')) return;

    // Don't intercept keys when user is typing in the editor or any input/textarea
    const tag = e.target.tagName;
    const editable = e.target.isContentEditable;
    if (tag === 'INPUT' || tag === 'TEXTAREA' || editable) return;

    if (e.key === 'Enter') {
        e.preventDefault();
        submitGuess();
    } else if (e.key === 'Backspace') {
        e.preventDefault();
        currentGuess = currentGuess.slice(0, -1);
        renderGame();
    } else if (/^[a-zA-Z]$/.test(e.key) && currentGuess.length < WORD_LENGTH) {
        e.preventDefault();
        currentGuess += e.key.toUpperCase();
        renderGame();
    }
}

/**
 * Handle on-screen keyboard click
 */
function handleKeyClick(key) {
    if (gameOver) return;

    if (key === 'ENTER') {
        submitGuess();
    } else if (key === 'BACK') {
        currentGuess = currentGuess.slice(0, -1);
        renderGame();
    } else if (currentGuess.length < WORD_LENGTH) {
        currentGuess += key;
        renderGame();
    }
}

/**
 * Render the game board
 */
function renderGame() {
    if (!gameContainer) return;

    let statusText = 'Wordle';
    if (gameWon) statusText = 'You Win!';
    else if (gameOver) statusText = `Answer: ${targetWord}`;

    // Build grid
    let gridHtml = '';
    for (let row = 0; row < MAX_GUESSES; row++) {
        for (let col = 0; col < WORD_LENGTH; col++) {
            let letter = '';
            let cls = 'wordle-cell';

            if (row < guesses.length) {
                // Submitted row
                letter = guesses[row].word[col];
                cls += ` wordle-${guesses[row].result[col]}`;
            } else if (row === guesses.length) {
                // Current input row
                letter = col < currentGuess.length ? currentGuess[col] : '';
                if (letter) cls += ' wordle-cell-filled';
            }
            // Empty rows stay empty

            gridHtml += `<div class="${cls}">${letter}</div>`;
        }
    }

    // Build keyboard
    const kbRows = [
        ['Q', 'W', 'E', 'R', 'T', 'Y', 'U', 'I', 'O', 'P'],
        ['A', 'S', 'D', 'F', 'G', 'H', 'J', 'K', 'L'],
        ['ENTER', 'Z', 'X', 'C', 'V', 'B', 'N', 'M', 'BACK']
    ];

    let kbHtml = '';
    for (const row of kbRows) {
        kbHtml += '<div class="wordle-kb-row">';
        for (const key of row) {
            const state = letterStates[key] || '';
            const wide = (key === 'ENTER' || key === 'BACK') ? ' wordle-key-wide' : '';
            const display = key === 'BACK' ? '\u232B' : key;
            kbHtml += `<button class="wordle-key${wide}${state ? ` wordle-key-${state}` : ''}" data-key="${key}">${display}</button>`;
        }
        kbHtml += '</div>';
    }

    const helpBtn = `<button class="game-help-btn" id="wordle-help-btn" aria-label="Instructions">?</button>`;
    const header = `<div class="game-header"><div class="game-status">${statusText}</div>${helpBtn}</div>`;
    const overlay = showHelp ? `
        <div class="game-instructions">
            <div class="game-instructions-title">How to Play</div>
            <div class="game-instructions-text">${INSTRUCTIONS}</div>
            <button class="game-reset-btn game-instructions-dismiss" id="wordle-dismiss">Got it</button>
        </div>` : '';

    gameContainer.innerHTML = `
        <div class="wordle-game">
            ${header}
            <div class="wordle-grid">${gridHtml}</div>
            <div class="wordle-keyboard">${kbHtml}</div>
            <button class="game-reset-btn" id="wordle-reset">New Game</button>
            ${overlay}
        </div>
    `;

    // Keyboard click listeners
    const keys = gameContainer.querySelectorAll('.wordle-key');
    keys.forEach(k => {
        k.addEventListener('click', () => handleKeyClick(k.dataset.key));
    });

    const resetBtn = gameContainer.querySelector('#wordle-reset');
    if (resetBtn) {
        resetBtn.addEventListener('click', () => {
            resetGame();
            renderGame();
        });
    }

    const helpBtnEl = gameContainer.querySelector('#wordle-help-btn');
    if (helpBtnEl) {
        helpBtnEl.addEventListener('click', () => { showHelp = !showHelp; renderGame(); });
    }

    const dismissEl = gameContainer.querySelector('#wordle-dismiss');
    if (dismissEl) {
        dismissEl.addEventListener('click', () => { showHelp = false; try { chrome.storage.local.set({ [HELP_KEY]: true }); } catch(e){} renderGame(); });
    }
}
