/**
 * Typing Speed Test Game Module
 */

// Passage bank (~30-50 words each)
const PASSAGES = [
    "The quick brown fox jumps over the lazy dog near the riverbank. A gentle breeze carries the scent of wildflowers across the meadow while birds sing their morning songs from the treetops above.",
    "Programming is the art of telling a computer what to do. Every line of code is a small instruction that builds toward something larger. The best programs are written with clarity and purpose.",
    "The ocean stretches endlessly toward the horizon where sky meets water in a thin silver line. Waves crash against the rocky shore sending salt spray into the cool morning air above the cliffs.",
    "Coffee shops have become the modern workspace for many people around the world. The hum of conversation and clinking cups creates a backdrop that somehow helps with focus and creativity.",
    "Mountains rise sharply from the valley floor covered in dense green forest. Snow caps their peaks even in summer creating a striking contrast against the deep blue sky and drifting white clouds.",
    "Reading books opens doors to worlds that exist only in the imagination. Each page turned reveals new characters and ideas that change the way we think about our own lives and experiences.",
    "The city comes alive at night with neon signs and streetlights casting colorful reflections on the wet pavement. People hurry along the sidewalks heading to restaurants and theaters downtown.",
    "Music has the power to change our mood in an instant. A single melody can transport us back to a specific moment in time bringing with it all the emotions we felt that day.",
    "Technology evolves at a pace that would have seemed impossible just decades ago. What once filled entire rooms now fits in our pockets connecting us to billions of people around the globe.",
    "A good meal brings people together like nothing else can. The shared experience of preparing and eating food creates bonds that last a lifetime across cultures and generations everywhere.",
    "Rain patters softly against the window as thunder rumbles in the distance. The world outside looks gray and blurred but inside the warm glow of a lamp makes everything feel safe and cozy.",
    "Stars fill the night sky like scattered diamonds on dark velvet. Far from city lights every constellation stands out clearly telling ancient stories that humans have shared for thousands of years.",
    "The garden grows wilder each day as spring turns to summer. Tomato plants climb their stakes while sunflowers stretch toward the light and bees drift lazily between the rows of lavender.",
    "Learning a new skill takes patience and consistent practice over time. The first attempts are always clumsy and frustrating but gradual improvement reveals itself to those who refuse to give up.",
    "Autumn leaves drift down from the trees painting the ground in shades of gold and crimson. The air turns crisp and cool carrying the faint smell of wood smoke from nearby chimneys.",
    "The library is a sanctuary of knowledge where shelves stretch from floor to ceiling packed with volumes covering every subject imaginable. Quiet footsteps echo through the aisles between ancient texts.",
    "Creativity flows best when we stop trying to force it. A walk outside or a conversation with a friend can unlock ideas that hours of staring at a blank page never could.",
    "Sunlight breaks through the clouds after days of rain turning every puddle into a mirror. Children splash through them laughing while dogs shake water from their fur on the sidewalk nearby.",
    "Writing code is like building with invisible blocks. Each function snaps into place connecting inputs to outputs until a complex system emerges from hundreds of small logical decisions made along the way.",
    "The train rumbles through the countryside passing farms and small towns. Passengers gaze out the windows watching the landscape change from flat plains to rolling hills as the journey continues onward."
];

// Game state
let gameContainer = null;
let currentPassage = '';
let startTime = null;
let endTime = null;
let timerInterval = null;
let gameFinished = false;
let showHelp = false;
let helpChecked = false;

const HELP_KEY = 'game_help_seen_type';
const INSTRUCTIONS = 'Type the displayed passage as fast and accurately as you can. The timer starts on your first keystroke. Results show your WPM and accuracy when finished.';

/**
 * Initialize the game
 */
export async function initTypingTest(container) {
    gameContainer = container;
    if (!helpChecked) {
        helpChecked = true;
        try {
            const data = await chrome.storage.local.get(HELP_KEY);
            showHelp = !data[HELP_KEY];
        } catch (e) { showHelp = true; }
    }
    resetGame();
    pickPassage();
    renderGame();
}

/**
 * Reset game state
 */
function resetGame() {
    startTime = null;
    endTime = null;
    gameFinished = false;
    if (timerInterval) {
        clearInterval(timerInterval);
        timerInterval = null;
    }
}

/**
 * Pick a random passage
 */
function pickPassage() {
    currentPassage = PASSAGES[Math.floor(Math.random() * PASSAGES.length)];
}

/**
 * Render the game
 */
function renderGame() {
    if (!gameContainer) return;

    const helpBtn = `<button class="game-help-btn" id="type-help-btn" aria-label="Instructions">?</button>`;
    const header = `<div class="game-header"><div class="game-status">Typing Speed Test</div>${helpBtn}</div>`;
    const overlay = showHelp ? `
        <div class="game-instructions">
            <div class="game-instructions-title">How to Play</div>
            <div class="game-instructions-text">${INSTRUCTIONS}</div>
            <button class="game-reset-btn game-instructions-dismiss" id="type-dismiss">Got it</button>
        </div>` : '';

    gameContainer.innerHTML = `
        <div class="typing-game">
            ${header}
            <div class="typing-stats">
                <span class="typing-stat" id="type-timer">0.0s</span>
                <span class="typing-stat" id="type-wpm">0 WPM</span>
                <span class="typing-stat" id="type-accuracy">100%</span>
            </div>
            <div class="typing-passage" id="type-passage">${renderPassage('')}</div>
            <textarea class="typing-input" id="type-input" placeholder="Start typing here..." spellcheck="false" autocomplete="off" autocorrect="off" autocapitalize="off">${gameFinished ? currentPassage : ''}</textarea>
            <div class="typing-buttons">
                <button class="game-reset-btn" id="type-retry">Try Again</button>
                <button class="game-reset-btn" id="type-new">New Passage</button>
            </div>
            ${overlay}
        </div>
    `;

    const input = gameContainer.querySelector('#type-input');
    const retryBtn = gameContainer.querySelector('#type-retry');
    const newBtn = gameContainer.querySelector('#type-new');

    const helpBtnEl = gameContainer.querySelector('#type-help-btn');
    if (helpBtnEl) {
        helpBtnEl.addEventListener('click', () => { showHelp = !showHelp; renderGame(); });
    }

    const dismissEl = gameContainer.querySelector('#type-dismiss');
    if (dismissEl) {
        dismissEl.addEventListener('click', () => { showHelp = false; try { chrome.storage.local.set({ [HELP_KEY]: true }); } catch(e){} renderGame(); });
    }

    if (!gameFinished) {
        input.addEventListener('input', handleInput);
        input.focus();
    } else {
        input.setAttribute('readonly', 'true');
        showResults();
    }

    retryBtn.addEventListener('click', () => {
        resetGame();
        renderGame();
    });

    newBtn.addEventListener('click', () => {
        resetGame();
        pickPassage();
        renderGame();
    });
}

/**
 * Render passage with character-level highlighting
 */
function renderPassage(typed) {
    let html = '';
    let wordChars = '';
    for (let i = 0; i <= currentPassage.length; i++) {
        const char = i < currentPassage.length ? currentPassage[i] : null;

        if (char === ' ' || char === null) {
            // Flush accumulated word as an inline-block group
            if (wordChars) {
                html += `<span class="typing-word">${wordChars}</span>`;
                wordChars = '';
            }
            if (char === ' ') {
                // Render space character outside word spans so line can break here
                if (i < typed.length) {
                    const cls = typed[i] === ' ' ? 'typing-correct' : 'typing-error';
                    html += `<span class="${cls}"> </span>`;
                } else if (i === typed.length) {
                    html += `<span class="typing-cursor"> </span>`;
                } else {
                    html += `<span class="typing-pending"> </span>`;
                }
            }
        } else {
            const displayChar = escapeChar(char);
            if (i < typed.length) {
                const cls = typed[i] === char ? 'typing-correct' : 'typing-error';
                wordChars += `<span class="${cls}">${displayChar}</span>`;
            } else if (i === typed.length) {
                wordChars += `<span class="typing-cursor">${displayChar}</span>`;
            } else {
                wordChars += `<span class="typing-pending">${displayChar}</span>`;
            }
        }
    }
    return html;
}

/**
 * Escape a character for safe HTML insertion
 */
function escapeChar(ch) {
    if (ch === '&') return '&amp;';
    if (ch === '<') return '&lt;';
    if (ch === '>') return '&gt;';
    return ch;
}

/**
 * Handle typing input
 */
function handleInput(e) {
    if (gameFinished) return;

    const typed = e.target.value;

    // Start timer on first keystroke
    if (!startTime && typed.length > 0) {
        startTime = Date.now();
        timerInterval = setInterval(updateTimer, 100);
    }

    // Update passage highlighting
    const passageEl = gameContainer.querySelector('#type-passage');
    if (passageEl) {
        passageEl.innerHTML = renderPassage(typed);
    }

    // Update live stats
    updateLiveStats(typed);

    // Check if finished
    if (typed.length >= currentPassage.length) {
        endTime = Date.now();
        gameFinished = true;
        if (timerInterval) {
            clearInterval(timerInterval);
            timerInterval = null;
        }
        showResults();

        const input = gameContainer.querySelector('#type-input');
        if (input) input.setAttribute('readonly', 'true');
    }
}

/**
 * Update the timer display
 */
function updateTimer() {
    if (!startTime) return;
    const elapsed = (Date.now() - startTime) / 1000;
    const timerEl = gameContainer.querySelector('#type-timer');
    if (timerEl) {
        timerEl.textContent = elapsed.toFixed(1) + 's';
    }
}

/**
 * Update live WPM and accuracy while typing
 */
function updateLiveStats(typed) {
    if (!startTime || typed.length === 0) return;

    const elapsed = (Date.now() - startTime) / 1000;
    const minutes = elapsed / 60;

    // WPM: standard word = 5 characters
    const wordsTyped = typed.length / 5;
    const wpm = minutes > 0 ? Math.round(wordsTyped / minutes) : 0;

    // Accuracy: correct characters / total typed
    let correct = 0;
    for (let i = 0; i < typed.length; i++) {
        if (typed[i] === currentPassage[i]) correct++;
    }
    const accuracy = Math.round((correct / typed.length) * 100);

    const wpmEl = gameContainer.querySelector('#type-wpm');
    const accEl = gameContainer.querySelector('#type-accuracy');
    if (wpmEl) wpmEl.textContent = wpm + ' WPM';
    if (accEl) accEl.textContent = accuracy + '%';
}

/**
 * Show final results
 */
function showResults() {
    if (!startTime || !endTime) return;

    const elapsed = (endTime - startTime) / 1000;
    const minutes = elapsed / 60;
    const wordsTyped = currentPassage.length / 5;
    const wpm = Math.round(wordsTyped / minutes);

    // Count correct characters from the input
    const input = gameContainer.querySelector('#type-input');
    const typed = input ? input.value : '';
    let correct = 0;
    for (let i = 0; i < currentPassage.length; i++) {
        if (i < typed.length && typed[i] === currentPassage[i]) correct++;
    }
    const accuracy = Math.round((correct / currentPassage.length) * 100);

    // Net WPM (penalize errors)
    const grossWpm = wpm;
    const errors = currentPassage.length - correct;
    const netWpm = Math.max(0, Math.round(grossWpm - (errors / minutes)));

    // Rating
    let rating;
    if (netWpm >= 80) rating = 'Blazing Fast';
    else if (netWpm >= 60) rating = 'Excellent';
    else if (netWpm >= 40) rating = 'Good';
    else if (netWpm >= 25) rating = 'Average';
    else rating = 'Keep Practicing';

    const timerEl = gameContainer.querySelector('#type-timer');
    const wpmEl = gameContainer.querySelector('#type-wpm');
    const accEl = gameContainer.querySelector('#type-accuracy');
    const statusEl = gameContainer.querySelector('.game-status');

    if (timerEl) timerEl.textContent = elapsed.toFixed(1) + 's';
    if (wpmEl) wpmEl.textContent = netWpm + ' WPM';
    if (accEl) accEl.textContent = accuracy + '%';
    if (statusEl) statusEl.textContent = rating;
}
