/**
 * Tic-Tac-Toe Game Module
 */

// Game state
let gameBoard = ['', '', '', '', '', '', '', '', ''];
let currentPlayer = 'X';
let gameActive = true;
let gameContainer = null;
let showHelp = false;
let helpChecked = false;

const HELP_KEY = 'game_help_seen_tic';
const INSTRUCTIONS = 'Play as X against the computer. Click any empty cell to place your mark. Get 3 in a row horizontally, vertically, or diagonally to win.';

// Win conditions
const WIN_CONDITIONS = [
    [0, 1, 2], [3, 4, 5], [6, 7, 8], // Rows
    [0, 3, 6], [1, 4, 7], [2, 5, 8], // Columns
    [0, 4, 8], [2, 4, 6]             // Diagonals
];

/**
 * Initialize the game
 */
export async function initTicTacToe(container) {
    gameContainer = container;
    if (!helpChecked) {
        helpChecked = true;
        try {
            const data = await chrome.storage.local.get(HELP_KEY);
            showHelp = !data[HELP_KEY];
        } catch (e) { showHelp = true; }
    }
    resetGame();
    renderGame();
}

/**
 * Reset game state
 */
function resetGame() {
    gameBoard = ['', '', '', '', '', '', '', '', ''];
    currentPlayer = 'X';
    gameActive = true;
}

/**
 * Render the game board
 */
function renderGame() {
    if (!gameContainer) return;

    const status = getGameStatus();

    const helpBtn = `<button class="game-help-btn" id="tic-help-btn" aria-label="Instructions">?</button>`;
    const header = `<div class="game-header"><div class="game-status">${status}</div>${helpBtn}</div>`;
    const overlay = showHelp ? `
        <div class="game-instructions">
            <div class="game-instructions-title">How to Play</div>
            <div class="game-instructions-text">${INSTRUCTIONS}</div>
            <button class="game-reset-btn game-instructions-dismiss" id="tic-dismiss">Got it</button>
        </div>` : '';

    gameContainer.innerHTML = `
        <div class="tic-tac-toe-game">
            ${header}
            <div class="game-board">
                ${renderBoard()}
            </div>
            <button class="game-reset-btn" id="tic-reset">New Game</button>
            ${overlay}
        </div>
    `;

    // Add event listeners
    const cells = gameContainer.querySelectorAll('.game-cell');
    cells.forEach((cell, index) => {
        cell.addEventListener('click', () => handleCellClick(index));
    });

    const resetBtn = gameContainer.querySelector('#tic-reset');
    if (resetBtn) {
        resetBtn.addEventListener('click', () => {
            resetGame();
            renderGame();
        });
    }

    const helpBtnEl = gameContainer.querySelector('#tic-help-btn');
    if (helpBtnEl) {
        helpBtnEl.addEventListener('click', () => { showHelp = !showHelp; renderGame(); });
    }

    const dismissEl = gameContainer.querySelector('#tic-dismiss');
    if (dismissEl) {
        dismissEl.addEventListener('click', () => { showHelp = false; try { chrome.storage.local.set({ [HELP_KEY]: true }); } catch(e){} renderGame(); });
    }
}

/**
 * Render the game board grid
 */
function renderBoard() {
    let html = '';
    for (let i = 0; i < 9; i++) {
        const value = gameBoard[i];
        const winningCell = isWinningCell(i);
        html += `<div class="game-cell ${winningCell ? 'winning' : ''}" data-index="${i}">${value}</div>`;
    }
    return html;
}

/**
 * Handle cell click
 */
function handleCellClick(index) {
    if (!gameActive || gameBoard[index] !== '' || currentPlayer !== 'X') return;

    // Player move (X)
    gameBoard[index] = 'X';

    if (checkWin()) {
        gameActive = false;
        renderGame();
        return;
    }

    if (checkDraw()) {
        gameActive = false;
        renderGame();
        return;
    }

    // Computer's turn (O)
    currentPlayer = 'O';
    renderGame();

    setTimeout(() => {
        if (gameActive) {
            makeComputerMove();
        }
    }, 300);
}

/**
 * Make computer move (O)
 */
function makeComputerMove() {
    // Simple AI: Try to win, block player, or take random cell
    const move = getBestMove();

    if (move !== -1) {
        gameBoard[move] = 'O';

        if (checkWin()) {
            gameActive = false;
        } else if (checkDraw()) {
            gameActive = false;
        } else {
            currentPlayer = 'X';
        }

        renderGame();
    }
}

/**
 * Get best move for computer
 */
function getBestMove() {
    // 1. Try to win
    for (let i = 0; i < 9; i++) {
        if (gameBoard[i] === '') {
            gameBoard[i] = 'O';
            if (checkWin()) {
                gameBoard[i] = '';
                return i;
            }
            gameBoard[i] = '';
        }
    }

    // 2. Block player from winning
    for (let i = 0; i < 9; i++) {
        if (gameBoard[i] === '') {
            gameBoard[i] = 'X';
            if (checkWin()) {
                gameBoard[i] = '';
                return i;
            }
            gameBoard[i] = '';
        }
    }

    // 3. Take center if available
    if (gameBoard[4] === '') return 4;

    // 4. Take a corner
    const corners = [0, 2, 6, 8];
    const availableCorners = corners.filter(i => gameBoard[i] === '');
    if (availableCorners.length > 0) {
        return availableCorners[Math.floor(Math.random() * availableCorners.length)];
    }

    // 5. Take any available cell
    for (let i = 0; i < 9; i++) {
        if (gameBoard[i] === '') return i;
    }

    return -1;
}

/**
 * Check for win
 */
function checkWin() {
    return WIN_CONDITIONS.some(condition => {
        const [a, b, c] = condition;
        return gameBoard[a] !== '' &&
            gameBoard[a] === gameBoard[b] &&
            gameBoard[a] === gameBoard[c];
    });
}

/**
 * Check for draw
 */
function checkDraw() {
    return gameBoard.every(cell => cell !== '');
}

/**
 * Check if cell is part of winning combination
 */
function isWinningCell(index) {
    if (!checkWin()) return false;

    return WIN_CONDITIONS.some(condition => {
        const [a, b, c] = condition;
        if (gameBoard[a] !== '' && gameBoard[a] === gameBoard[b] && gameBoard[a] === gameBoard[c]) {
            return condition.includes(index);
        }
        return false;
    });
}

/**
 * Get game status message
 */
function getGameStatus() {
    if (!gameActive) {
        if (checkWin()) {
            // Find who won by checking the board
            const winner = gameBoard.find(cell => cell !== '');
            for (const condition of WIN_CONDITIONS) {
                const [a, b, c] = condition;
                if (gameBoard[a] !== '' && gameBoard[a] === gameBoard[b] && gameBoard[a] === gameBoard[c]) {
                    return `${gameBoard[a]} Wins!`;
                }
            }
        }
        return "It's a Draw!";
    }
    return currentPlayer === 'X' ? 'Your Turn (X)' : 'Computer Thinking...';
}
