/**
 * Reaction Time Game Module
 */

const TOTAL_ROUNDS = 5;
const MIN_DELAY = 2000;
const MAX_DELAY = 5000;

// Game state
let gameContainer = null;
let currentRound = 0;
let results = [];
let phase = 'idle'; // idle | waiting | ready | too-early | result | finished
let greenTime = null;
let delayTimeout = null;
let showHelp = false;
let helpChecked = false;

const HELP_KEY = 'game_help_seen_react';
const INSTRUCTIONS = 'Wait for the zone to turn green, then click as fast as you can. Complete 5 rounds to see your average reaction time. Don\'t click early or you\'ll have to retry the round!';

/**
 * Initialize the game
 */
export async function initReactionTime(container) {
    gameContainer = container;
    if (!helpChecked) {
        helpChecked = true;
        try {
            const data = await chrome.storage.local.get(HELP_KEY);
            showHelp = !data[HELP_KEY];
        } catch (e) { showHelp = true; }
    }
    resetGame();
    renderGame();
}

/**
 * Reset game state
 */
function resetGame() {
    currentRound = 0;
    results = [];
    phase = 'idle';
    greenTime = null;
    if (delayTimeout) {
        clearTimeout(delayTimeout);
        delayTimeout = null;
    }
}

/**
 * Get a random delay between min and max
 */
function randomDelay() {
    return MIN_DELAY + Math.random() * (MAX_DELAY - MIN_DELAY);
}

/**
 * Render the game
 */
function renderGame() {
    if (!gameContainer) return;

    let zoneClass = '';
    let zoneText = '';
    let showBtn = false;
    let btnText = '';

    switch (phase) {
        case 'idle':
            zoneClass = 'react-zone-idle';
            zoneText = 'Test your reflexes!<br><span class="react-subtitle">Click Start when ready</span>';
            showBtn = true;
            btnText = 'Start';
            break;
        case 'waiting':
            zoneClass = 'react-zone-waiting';
            zoneText = 'Wait for green...';
            break;
        case 'ready':
            zoneClass = 'react-zone-ready';
            zoneText = 'Click!';
            break;
        case 'too-early':
            zoneClass = 'react-zone-early';
            zoneText = 'Too early!<br><span class="react-subtitle">Click to try again</span>';
            break;
        case 'result': {
            const lastResult = results[results.length - 1];
            zoneClass = 'react-zone-result';
            zoneText = `${lastResult} ms<br><span class="react-subtitle">Round ${currentRound} of ${TOTAL_ROUNDS} — Click to continue</span>`;
            break;
        }
        case 'finished':
            zoneClass = 'react-zone-idle';
            zoneText = '';
            break;
    }

    let roundIndicators = '';
    for (let i = 0; i < TOTAL_ROUNDS; i++) {
        let cls = 'react-round-dot';
        if (i < results.length) {
            cls += ' react-round-done';
        } else if (i === currentRound && (phase === 'waiting' || phase === 'ready')) {
            cls += ' react-round-active';
        }
        roundIndicators += `<span class="${cls}"></span>`;
    }

    const helpBtnHtml = `<button class="game-help-btn" id="react-help-btn" aria-label="Instructions">?</button>`;
    const header = `<div class="game-header"><div class="game-status">Reaction Time</div>${helpBtnHtml}</div>`;
    const overlay = showHelp ? `
        <div class="game-instructions">
            <div class="game-instructions-title">How to Play</div>
            <div class="game-instructions-text">${INSTRUCTIONS}</div>
            <button class="game-reset-btn game-instructions-dismiss" id="react-dismiss">Got it</button>
        </div>` : '';

    let html = `
        <div class="react-game">
            ${header}
            <div class="react-rounds">${roundIndicators}</div>
            <div class="react-zone ${zoneClass}" id="react-zone">${zoneText}</div>
    `;

    if (phase === 'finished') {
        html += renderResults();
    }

    if (showBtn) {
        html += `<button class="game-reset-btn" id="react-start">${btnText}</button>`;
    }

    if (phase === 'finished') {
        html += `<button class="game-reset-btn" id="react-again">Play Again</button>`;
    }

    html += overlay + `</div>`;

    gameContainer.innerHTML = html;

    // Event listeners
    const zone = gameContainer.querySelector('#react-zone');
    if (zone) {
        zone.addEventListener('click', handleZoneClick);
    }

    const startBtn = gameContainer.querySelector('#react-start');
    if (startBtn) {
        startBtn.addEventListener('click', startRound);
    }

    const againBtn = gameContainer.querySelector('#react-again');
    if (againBtn) {
        againBtn.addEventListener('click', () => {
            resetGame();
            renderGame();
        });
    }

    const helpBtnEl = gameContainer.querySelector('#react-help-btn');
    if (helpBtnEl) {
        helpBtnEl.addEventListener('click', () => { showHelp = !showHelp; renderGame(); });
    }

    const dismissEl = gameContainer.querySelector('#react-dismiss');
    if (dismissEl) {
        dismissEl.addEventListener('click', () => { showHelp = false; try { chrome.storage.local.set({ [HELP_KEY]: true }); } catch(e){} renderGame(); });
    }
}

/**
 * Start a new round
 */
function startRound() {
    phase = 'waiting';
    greenTime = null;
    renderGame();

    const delay = randomDelay();
    delayTimeout = setTimeout(() => {
        phase = 'ready';
        greenTime = Date.now();
        renderGame();
    }, delay);
}

/**
 * Handle click on the reaction zone
 */
function handleZoneClick() {
    if (phase === 'waiting') {
        // Clicked too early
        if (delayTimeout) {
            clearTimeout(delayTimeout);
            delayTimeout = null;
        }
        phase = 'too-early';
        renderGame();
    } else if (phase === 'ready') {
        // Record reaction time
        const reactionTime = Date.now() - greenTime;
        results.push(reactionTime);
        currentRound++;

        if (currentRound >= TOTAL_ROUNDS) {
            phase = 'finished';
        } else {
            phase = 'result';
        }
        renderGame();
    } else if (phase === 'too-early') {
        // Retry same round
        startRound();
    } else if (phase === 'result') {
        // Next round
        startRound();
    }
}

/**
 * Render final results
 */
function renderResults() {
    const avg = Math.round(results.reduce((a, b) => a + b, 0) / results.length);
    const best = Math.min(...results);

    let rating;
    if (avg < 200) rating = 'Incredible';
    else if (avg < 300) rating = 'Fast';
    else if (avg < 400) rating = 'Average';
    else rating = 'Keep Practicing';

    let roundsList = results.map((r, i) =>
        `<span class="react-result-round">R${i + 1}: <strong>${r} ms</strong></span>`
    ).join('');

    return `
        <div class="react-results">
            <div class="react-rating">${rating}</div>
            <div class="react-avg">${avg} ms <span class="react-avg-label">average</span></div>
            <div class="react-best">Best: ${best} ms</div>
            <div class="react-result-rounds">${roundsList}</div>
        </div>
    `;
}
