/**
 * Quiz Game Module
 */

// Question bank - ~60 questions per category
const QUIZ_CATEGORIES = {
    general: {
        name: 'General Knowledge',
        questions: [
            { q: 'What is the largest ocean on Earth?', a: ['Pacific', 'Pacific Ocean'] },
            { q: 'How many continents are there?', a: ['7', 'Seven'] },
            { q: 'What is the smallest country in the world?', a: ['Vatican', 'Vatican City'] },
            { q: 'What year did World War II end?', a: ['1945'] },
            { q: 'What is the chemical symbol for gold?', a: ['Au'] },
            { q: 'How many days are in a leap year?', a: ['366'] },
            { q: 'What is the capital of Australia?', a: ['Canberra'] },
            { q: 'What language is spoken in Brazil?', a: ['Portuguese'] },
            { q: 'How many sides does a hexagon have?', a: ['6', 'Six'] },
            { q: 'What is the largest planet in our solar system?', a: ['Jupiter'] },
            { q: 'What is the hardest natural substance on Earth?', a: ['Diamond'] },
            { q: 'Which planet is known as the Red Planet?', a: ['Mars'] },
            { q: 'What is the main ingredient in guacamole?', a: ['Avocado'] },
            { q: 'What currency is used in Japan?', a: ['Yen'] },
            { q: 'What is the largest animal on Earth?', a: ['Blue Whale', 'Whale'] },
            { q: 'How many stars are on the US flag?', a: ['50', 'Fifty'] },
            { q: 'Who painted the Mona Lisa?', a: ['Leonardo da Vinci', 'Da Vinci'] },
            { q: 'What is the boiling point of water in Celsius?', a: ['100'] },
            { q: 'Which fruit is known for keeping the doctor away?', a: ['Apple'] },
            { q: 'What is the most consumed drink in the world after water?', a: ['Tea'] },
            { q: 'What is the chemical formula for water?', a: ['H2O'] },
            { q: 'What do pandas eat?', a: ['Bamboo'] },
            { q: 'Which organ pumps blood in the human body?', a: ['Heart'] },
            { q: 'What is the color of an emerald?', a: ['Green'] },
            { q: 'How many legs does a spider have?', a: ['8', 'Eight'] },
            { q: 'What is the freezing point of water in Celsius?', a: ['0'] },
            { q: 'What is the name of the toy cowboy in Toy Story?', a: ['Woody'] },
            { q: 'Which country invented pizza?', a: ['Italy'] },
            { q: 'What is the capital of Spain?', a: ['Madrid'] },
            { q: 'How many letters are in the English alphabet?', a: ['26'] },
            { q: 'What is the tallest animal in the world?', a: ['Giraffe'] },
            { q: 'What gas do we breathe out?', a: ['Carbon Dioxide', 'CO2'] },
            { q: 'How many colors are in a rainbow?', a: ['7', 'Seven'] },
            { q: 'What is the national flower of Japan?', a: ['Cherry Blossom', 'Sakura'] },
            { q: 'What instrument has 88 keys?', a: ['Piano'] },
            { q: 'What is baby cow called?', a: ['Calf'] },
            { q: 'How many weeks are in a year?', a: ['52'] },
            { q: 'What is the fastest land animal?', a: ['Cheetah'] },
            { q: 'Which metal is liquid at room temperature?', a: ['Mercury'] },
            { q: 'What color do you get mixing blue and yellow?', a: ['Green'] },
            { q: 'How many zeros are in a million?', a: ['6', 'Six'] },
            { q: 'What is the longest bone in the human body?', a: ['Femur'] },
            { q: 'What fruit do raisins come from?', a: ['Grape', 'Grapes'] },
            { q: 'What is the largest continent?', a: ['Asia'] },
            { q: 'How many strings does a standard guitar have?', a: ['6', 'Six'] },
            { q: 'What is the square root of 144?', a: ['12', 'Twelve'] },
            { q: 'Which blood type is the universal donor?', a: ['O', 'O Negative', 'O-'] },
            { q: 'How many chambers does the human heart have?', a: ['4', 'Four'] },
            { q: 'What does AM stand for in time?', a: ['Ante Meridiem'] },
            { q: 'What is the smallest prime number?', a: ['2', 'Two'] },
            { q: 'How many decades are in a century?', a: ['10', 'Ten'] },
            { q: 'What is the largest internal organ in the human body?', a: ['Liver'] },
            { q: 'What shape has five sides?', a: ['Pentagon'] },
            { q: 'What color is a school bus in the USA?', a: ['Yellow'] },
            { q: 'How many planets are in our solar system?', a: ['8', 'Eight'] },
            { q: 'What is the currency of the United Kingdom?', a: ['Pound', 'Pound Sterling', 'GBP'] },
            { q: 'What is the opposite of "nocturnal"?', a: ['Diurnal'] },
            { q: 'How many bones does a shark have?', a: ['0', 'Zero', 'None'] },
            { q: 'What is the most spoken language in the world?', a: ['English', 'Mandarin'] },
            { q: 'Which vitamin does the sun provide?', a: ['Vitamin D', 'D'] }
        ]
    },
    science: {
        name: 'Science',
        questions: [
            { q: 'What is H2O commonly known as?', a: ['Water'] },
            { q: 'What planet is known as the Red Planet?', a: ['Mars'] },
            { q: 'What is the speed of light in km/s? (rounded)', a: ['300000', '300,000'] },
            { q: 'What gas do plants absorb from the atmosphere?', a: ['Carbon Dioxide', 'CO2'] },
            { q: 'What is the powerhouse of the cell?', a: ['Mitochondria'] },
            { q: 'How many bones are in the human body?', a: ['206'] },
            { q: 'What is the center of an atom called?', a: ['Nucleus'] },
            { q: 'What force keeps us on the ground?', a: ['Gravity'] },
            { q: 'What particle has a positive charge?', a: ['Proton'] },
            { q: 'What particle has a negative charge?', a: ['Electron'] },
            { q: 'What is the heaviest naturally occurring element?', a: ['Uranium'] },
            { q: 'What is the study of living organisms called?', a: ['Biology'] },
            { q: 'Which planet is closest to the sun?', a: ['Mercury'] },
            { q: 'What is the largest organ in the human body?', a: ['Skin'] },
            { q: 'What gas do humans need to breathe to survive?', a: ['Oxygen'] },
            { q: 'What is the unit of electrical resistance?', a: ['Ohm'] },
            { q: 'Who developed the theory of relativity?', a: ['Einstein', 'Albert Einstein'] },
            { q: 'What planet has the most rings?', a: ['Saturn'] },
            { q: 'What is the chemical symbol for Iron?', a: ['Fe'] },
            { q: 'What is the nearest star to Earth?', a: ['Sun'] },
            { q: 'How many teeth does an adult human have?', a: ['32'] },
            { q: 'What is the most abundant gas in Earth\'s atmosphere?', a: ['Nitrogen'] },
            { q: 'What protects Earth from the Sun\'s UV radiation?', a: ['Ozone', 'Ozone Layer'] },
            { q: 'What is the chemical symbol for Oxygen?', a: ['O'] },
            { q: 'What part of the plant conducts photosynthesis?', a: ['Leaf', 'Leaves', 'Chloroplast'] },
            { q: 'What is the largest bone in the human body?', a: ['Femur', 'Thigh Bone'] },
            { q: 'What cloud type produces thunderstorms?', a: ['Cumulonimbus'] },
            { q: 'What is the chemical symbol for Sodium?', a: ['Na'] },
            { q: 'What is the pH of pure water?', a: ['7'] },
            { q: 'What type of rock is formed from cooled lava?', a: ['Igneous'] },
            { q: 'What is the study of earthquakes called?', a: ['Seismology'] },
            { q: 'How many chromosomes do humans have?', a: ['46'] },
            { q: 'What vitamin is produced when skin is exposed to sunlight?', a: ['Vitamin D', 'D'] },
            { q: 'What organ produces insulin?', a: ['Pancreas'] },
            { q: 'What is the chemical symbol for Potassium?', a: ['K'] },
            { q: 'What is the study of fungi called?', a: ['Mycology'] },
            { q: 'What planet is the hottest in our solar system?', a: ['Venus'] },
            { q: 'What is the smallest unit of matter?', a: ['Atom'] },
            { q: 'What type of animal is a frog?', a: ['Amphibian'] },
            { q: 'What is the process of water turning to vapor called?', a: ['Evaporation'] },
            { q: 'What is the chemical formula for table salt?', a: ['NaCl'] },
            { q: 'How many moons does Mars have?', a: ['2', 'Two'] },
            { q: 'What element has the atomic number 1?', a: ['Hydrogen'] },
            { q: 'What is the study of weather called?', a: ['Meteorology'] },
            { q: 'What is the hardest mineral on the Mohs scale?', a: ['Diamond'] },
            { q: 'What is the main gas found in the Sun?', a: ['Hydrogen'] },
            { q: 'What unit measures electric current?', a: ['Ampere', 'Amp'] },
            { q: 'What blood cells fight infection?', a: ['White Blood Cells', 'White'] },
            { q: 'What is the chemical symbol for Lead?', a: ['Pb'] },
            { q: 'What part of the brain controls balance?', a: ['Cerebellum'] },
            { q: 'What is the process by which plants make food?', a: ['Photosynthesis'] },
            { q: 'What planet has the Great Red Spot?', a: ['Jupiter'] },
            { q: 'What is the unit of frequency?', a: ['Hertz', 'Hz'] },
            { q: 'What layer of Earth is made of liquid iron?', a: ['Outer Core'] },
            { q: 'What is absolute zero in Celsius?', a: ['-273', '-273.15'] },
            { q: 'What type of bond shares electrons?', a: ['Covalent'] },
            { q: 'How many lungs does the human body have?', a: ['2', 'Two'] },
            { q: 'What is the chemical symbol for Silver?', a: ['Ag'] },
            { q: 'What scale measures mineral hardness?', a: ['Mohs'] },
            { q: 'What is the name for molten rock below Earth\'s surface?', a: ['Magma'] }
        ]
    },
    history: {
        name: 'History',
        questions: [
            { q: 'Who was the first President of the United States?', a: ['George Washington', 'Washington'] },
            { q: 'In what year did the Titanic sink?', a: ['1912'] },
            { q: 'What ancient wonder was located in Egypt?', a: ['Pyramids', 'Great Pyramid', 'Pyramid of Giza'] },
            { q: 'Who discovered America in 1492?', a: ['Christopher Columbus', 'Columbus'] },
            { q: 'What wall fell in 1989?', a: ['Berlin Wall', 'Berlin'] },
            { q: 'Who was the first man on the moon?', a: ['Neil Armstrong', 'Armstrong'] },
            { q: 'What year did World War I begin?', a: ['1914'] },
            { q: 'Who wrote Romeo and Juliet?', a: ['William Shakespeare', 'Shakespeare'] },
            { q: 'What empire was ruled by Julius Caesar?', a: ['Roman', 'Roman Empire', 'Rome'] },
            { q: 'Which country gifted the Statue of Liberty to the USA?', a: ['France'] },
            { q: 'Who was the first female Prime Minister of the UK?', a: ['Margaret Thatcher', 'Thatcher'] },
            { q: 'What year did the US Declaration of Independence get signed?', a: ['1776'] },
            { q: 'Which war was fought between the North and South in the US?', a: ['Civil War', 'American Civil War'] },
            { q: 'Who was the British Prime Minister during WWII?', a: ['Winston Churchill', 'Churchill'] },
            { q: 'What ancient civilization built Machu Picchu?', a: ['Inca', 'Incan'] },
            { q: 'Who was the first Emperor of Rome?', a: ['Augustus', 'Augustus Caesar'] },
            { q: 'Who is known as the Maid of Orleans?', a: ['Joan of Arc'] },
            { q: 'Which ship carried the Pilgrims to America in 1620?', a: ['Mayflower'] },
            { q: 'What was the name of the first space shuttle?', a: ['Columbia'] },
            { q: 'Who assassinated Abraham Lincoln?', a: ['John Wilkes Booth', 'Booth'] },
            { q: 'Which country was Cleopatra queen of?', a: ['Egypt'] },
            { q: 'What year did the first World War end?', a: ['1918'] },
            { q: 'Who invented the lightbulb?', a: ['Thomas Edison', 'Edison'] },
            { q: 'What year did man land on the moon?', a: ['1969'] },
            { q: 'Who was the leader of the Soviet Union during WWII?', a: ['Stalin', 'Joseph Stalin'] },
            { q: 'What is the name of the period of rebirth in Europe?', a: ['Renaissance'] },
            { q: 'Which king had six wives?', a: ['Henry VIII', 'Henry the Eighth'] },
            { q: 'Where was Napoleon born?', a: ['Corsica'] },
            { q: 'What document did the Magna Carta inspire?', a: ['Constitution', 'Bill of Rights'] },
            { q: 'Who was the youngest US president?', a: ['Theodore Roosevelt', 'JFK', 'Kennedy'] },
            { q: 'What year did the Great Fire of London occur?', a: ['1666'] },
            { q: 'Who painted the ceiling of the Sistine Chapel?', a: ['Michelangelo'] },
            { q: 'What city was destroyed by Mount Vesuvius in 79 AD?', a: ['Pompeii'] },
            { q: 'Who invented the printing press?', a: ['Gutenberg', 'Johannes Gutenberg'] },
            { q: 'What country was formerly known as Persia?', a: ['Iran'] },
            { q: 'Who led India to independence from Britain?', a: ['Gandhi', 'Mahatma Gandhi'] },
            { q: 'What year did the French Revolution begin?', a: ['1789'] },
            { q: 'What was the longest war in US history?', a: ['Afghanistan', 'War in Afghanistan'] },
            { q: 'Who was the Egyptian god of the sun?', a: ['Ra'] },
            { q: 'Which explorer first circumnavigated the globe?', a: ['Magellan', 'Ferdinand Magellan'] },
            { q: 'What ancient civilization invented democracy?', a: ['Greece', 'Greek', 'Athens'] },
            { q: 'Who wrote the Communist Manifesto?', a: ['Karl Marx', 'Marx'] },
            { q: 'What was the last country to abolish slavery?', a: ['Mauritania'] },
            { q: 'What year did the Cold War end?', a: ['1991'] },
            { q: 'Who was the first woman in space?', a: ['Valentina Tereshkova', 'Tereshkova'] },
            { q: 'What empire built the Colosseum?', a: ['Roman', 'Roman Empire'] },
            { q: 'What disease killed a third of Europe in the 1300s?', a: ['Black Death', 'Plague', 'Bubonic Plague'] },
            { q: 'Who was the first explorer to reach the South Pole?', a: ['Amundsen', 'Roald Amundsen'] },
            { q: 'What year was the United Nations founded?', a: ['1945'] },
            { q: 'Which US president freed the slaves?', a: ['Abraham Lincoln', 'Lincoln'] },
            { q: 'What was the name of the ship Darwin sailed on?', a: ['Beagle', 'HMS Beagle'] },
            { q: 'Who built the Great Wall of China?', a: ['Qin Shi Huang', 'China'] },
            { q: 'What year did the Chernobyl disaster occur?', a: ['1986'] },
            { q: 'What treaty ended World War I?', a: ['Treaty of Versailles', 'Versailles'] },
            { q: 'Who discovered penicillin?', a: ['Alexander Fleming', 'Fleming'] },
            { q: 'What year was the Berlin Wall built?', a: ['1961'] },
            { q: 'Which queen ruled England for 63 years in the 1800s?', a: ['Victoria', 'Queen Victoria'] },
            { q: 'What civilization built the Parthenon?', a: ['Greek', 'Greece', 'Athens'] },
            { q: 'Who was the first black president of South Africa?', a: ['Nelson Mandela', 'Mandela'] },
            { q: 'What year did Pearl Harbor get attacked?', a: ['1941'] }
        ]
    },
    geography: {
        name: 'Geography',
        questions: [
            { q: 'What is the capital of France?', a: ['Paris'] },
            { q: 'What is the longest river in the world?', a: ['Nile', 'Nile River'] },
            { q: 'What is the tallest mountain in the world?', a: ['Mount Everest', 'Everest'] },
            { q: 'What country has the most population?', a: ['China', 'India'] },
            { q: 'What is the largest desert in the world?', a: ['Sahara', 'Sahara Desert', 'Antarctic'] },
            { q: 'What ocean is between America and Europe?', a: ['Atlantic', 'Atlantic Ocean'] },
            { q: 'What is the capital of Japan?', a: ['Tokyo'] },
            { q: 'What country is home to the kangaroo?', a: ['Australia'] },
            { q: 'What is the smallest continent?', a: ['Australia'] },
            { q: 'What country has the Eiffel Tower?', a: ['France'] },
            { q: 'What is the capital of Italy?', a: ['Rome'] },
            { q: 'Which continent is the Sahara Desert located in?', a: ['Africa'] },
            { q: 'What is the largest country by land area?', a: ['Russia'] },
            { q: 'What is the capital of Canada?', a: ['Ottawa'] },
            { q: 'Which river flows through London?', a: ['Thames'] },
            { q: 'What is the capital of Egypt?', a: ['Cairo'] },
            { q: 'Which country is known as the Land of the Rising Sun?', a: ['Japan'] },
            { q: 'Which US state is the largest by area?', a: ['Alaska'] },
            { q: 'What is the capital of Germany?', a: ['Berlin'] },
            { q: 'Which country has the most islands?', a: ['Sweden'] },
            { q: 'What is the longest river in South America?', a: ['Amazon'] },
            { q: 'What is the capital of Brazil?', a: ['Brasilia', 'Brasília'] },
            { q: 'What is the smallest country in South America?', a: ['Suriname'] },
            { q: 'What is the capital of China?', a: ['Beijing'] },
            { q: 'Which mountain range separates Europe and Asia?', a: ['Ural', 'Urals'] },
            { q: 'Which ocean lies to the east of Africa?', a: ['Indian', 'Indian Ocean'] },
            { q: 'What is the nickname of New York City?', a: ['Big Apple', 'The Big Apple'] },
            { q: 'What is the capital of India?', a: ['New Delhi', 'Delhi'] },
            { q: 'What is the deepest ocean trench?', a: ['Mariana Trench', 'Mariana'] },
            { q: 'What river runs through Paris?', a: ['Seine'] },
            { q: 'What is the capital of South Korea?', a: ['Seoul'] },
            { q: 'Which country has the shape of a boot?', a: ['Italy'] },
            { q: 'What is the largest lake in Africa?', a: ['Victoria', 'Lake Victoria'] },
            { q: 'What is the capital of Argentina?', a: ['Buenos Aires'] },
            { q: 'Which strait separates Europe from Africa?', a: ['Gibraltar', 'Strait of Gibraltar'] },
            { q: 'What is the driest continent?', a: ['Antarctica'] },
            { q: 'What is the capital of Turkey?', a: ['Ankara'] },
            { q: 'What is the highest waterfall in the world?', a: ['Angel Falls', 'Angel'] },
            { q: 'What country is the Great Barrier Reef in?', a: ['Australia'] },
            { q: 'What is the capital of Mexico?', a: ['Mexico City'] },
            { q: 'Which two countries share the longest border?', a: ['USA and Canada', 'Canada and USA', 'US Canada'] },
            { q: 'What is the capital of Russia?', a: ['Moscow'] },
            { q: 'What mountain is the tallest in North America?', a: ['Denali', 'Mount Denali', 'McKinley'] },
            { q: 'What is the largest island in the world?', a: ['Greenland'] },
            { q: 'What is the capital of Thailand?', a: ['Bangkok'] },
            { q: 'Which river is the longest in Europe?', a: ['Volga'] },
            { q: 'What is the capital of Nigeria?', a: ['Abuja'] },
            { q: 'What is the smallest US state by area?', a: ['Rhode Island'] },
            { q: 'What sea is bordered by Europe, Africa, and Asia?', a: ['Mediterranean', 'Mediterranean Sea'] },
            { q: 'What is the capital of Kenya?', a: ['Nairobi'] },
            { q: 'What country is home to the Taj Mahal?', a: ['India'] },
            { q: 'What is the capital of Peru?', a: ['Lima'] },
            { q: 'Which continent has the most countries?', a: ['Africa'] },
            { q: 'What US state has the most volcanoes?', a: ['Alaska'] },
            { q: 'What is the capital of Poland?', a: ['Warsaw'] },
            { q: 'What is the longest mountain range in the world?', a: ['Andes'] },
            { q: 'What country has the most time zones?', a: ['France'] },
            { q: 'What is the capital of Norway?', a: ['Oslo'] },
            { q: 'What is the saltiest body of water on Earth?', a: ['Dead Sea'] },
            { q: 'What is the capital of Vietnam?', a: ['Hanoi'] }
        ]
    },
    technology: {
        name: 'Technology',
        questions: [
            { q: 'Who founded Apple Computer?', a: ['Steve Jobs', 'Jobs'] },
            { q: 'What does CPU stand for?', a: ['Central Processing Unit'] },
            { q: 'What year was the first iPhone released?', a: ['2007'] },
            { q: 'What does HTML stand for?', a: ['HyperText Markup Language'] },
            { q: 'Who founded Microsoft?', a: ['Bill Gates', 'Gates'] },
            { q: 'What does WWW stand for?', a: ['World Wide Web'] },
            { q: 'What company created the Android operating system?', a: ['Google'] },
            { q: 'What does USB stand for?', a: ['Universal Serial Bus'] },
            { q: 'What programming language is known for web development?', a: ['JavaScript', 'JS'] },
            { q: 'What company owns Instagram?', a: ['Meta', 'Facebook'] },
            { q: 'What does RAM stand for?', a: ['Random Access Memory'] },
            { q: 'What is the name of Apple\'s voice assistant?', a: ['Siri'] },
            { q: 'Who is the CEO of Tesla?', a: ['Elon Musk', 'Musk'] },
            { q: 'What does GPS stand for?', a: ['Global Positioning System'] },
            { q: 'What is the most popular search engine?', a: ['Google'] },
            { q: 'What does PDF stand for?', a: ['Portable Document Format'] },
            { q: 'Who founded Amazon?', a: ['Jeff Bezos', 'Bezos'] },
            { q: 'What does AI stand for?', a: ['Artificial Intelligence'] },
            { q: 'What company created Windows?', a: ['Microsoft'] },
            { q: 'What does URL stand for?', a: ['Uniform Resource Locator'] },
            { q: 'What year was Facebook founded?', a: ['2004'] },
            { q: 'What is the main circuit board in a computer called?', a: ['Motherboard'] },
            { q: 'Which company makes the PlayStation?', a: ['Sony'] },
            { q: 'What is the name of the Linux penguin mascot?', a: ['Tux'] },
            { q: 'What does VR stand for?', a: ['Virtual Reality'] },
            { q: 'What programming language is named after a snake?', a: ['Python'] },
            { q: 'Who created the World Wide Web?', a: ['Tim Berners-Lee', 'Berners-Lee'] },
            { q: 'What does SSD stand for?', a: ['Solid State Drive'] },
            { q: 'What year was YouTube founded?', a: ['2005'] },
            { q: 'What does API stand for?', a: ['Application Programming Interface'] },
            { q: 'What company developed the Java language?', a: ['Sun Microsystems', 'Sun'] },
            { q: 'What does LAN stand for?', a: ['Local Area Network'] },
            { q: 'What is the name of Google\'s web browser?', a: ['Chrome'] },
            { q: 'What does IoT stand for?', a: ['Internet of Things'] },
            { q: 'Who co-founded Apple with Steve Jobs?', a: ['Steve Wozniak', 'Wozniak'] },
            { q: 'What does HTTPS stand for?', a: ['HyperText Transfer Protocol Secure'] },
            { q: 'What year was the first email sent?', a: ['1971'] },
            { q: 'What does BIOS stand for?', a: ['Basic Input Output System'] },
            { q: 'What company makes the Xbox?', a: ['Microsoft'] },
            { q: 'What does CSS stand for?', a: ['Cascading Style Sheets'] },
            { q: 'What social media platform is known for short videos?', a: ['TikTok'] },
            { q: 'What is the name of Amazon\'s voice assistant?', a: ['Alexa'] },
            { q: 'What does DNS stand for?', a: ['Domain Name System'] },
            { q: 'What year was Bitcoin created?', a: ['2009'] },
            { q: 'What does SQL stand for?', a: ['Structured Query Language'] },
            { q: 'Who is the creator of Linux?', a: ['Linus Torvalds', 'Torvalds'] },
            { q: 'What company acquired GitHub in 2018?', a: ['Microsoft'] },
            { q: 'What does FTP stand for?', a: ['File Transfer Protocol'] },
            { q: 'What was Google\'s original name?', a: ['Backrub'] },
            { q: 'What does GPU stand for?', a: ['Graphics Processing Unit'] },
            { q: 'What year was Twitter launched?', a: ['2006'] },
            { q: 'What does NFC stand for?', a: ['Near Field Communication'] },
            { q: 'Who invented the telephone?', a: ['Alexander Graham Bell', 'Bell'] },
            { q: 'What does IP stand for in networking?', a: ['Internet Protocol'] },
            { q: 'What company makes the Kindle?', a: ['Amazon'] },
            { q: 'What does CAPTCHA help prevent?', a: ['Bots', 'Spam', 'Robots'] },
            { q: 'What year was Spotify launched?', a: ['2008'] },
            { q: 'What does OLED stand for?', a: ['Organic Light Emitting Diode'] },
            { q: 'What company developed the C programming language?', a: ['Bell Labs'] },
            { q: 'What does VPN stand for?', a: ['Virtual Private Network'] }
        ]
    },
    movies: {
        name: 'Movies & TV',
        questions: [
            { q: 'Who played Iron Man in the Marvel movies?', a: ['Robert Downey Jr', 'Robert Downey Junior', 'RDJ'] },
            { q: 'What movie features a character named Jack Dawson?', a: ['Titanic'] },
            { q: 'What is the name of Harry Potter\'s owl?', a: ['Hedwig'] },
            { q: 'What TV show features the character Walter White?', a: ['Breaking Bad'] },
            { q: 'Who directed Jurassic Park?', a: ['Steven Spielberg', 'Spielberg'] },
            { q: 'What is the highest-grossing film of all time?', a: ['Avatar'] },
            { q: 'What planet is Superman from?', a: ['Krypton'] },
            { q: 'What is the name of the lion in The Lion King?', a: ['Simba'] },
            { q: 'What TV show is set in the fictional town of Hawkins?', a: ['Stranger Things'] },
            { q: 'Who played Jack Sparrow in Pirates of the Caribbean?', a: ['Johnny Depp', 'Depp'] },
            { q: 'Who is the voice of Woody in Toy Story?', a: ['Tom Hanks', 'Hanks'] },
            { q: 'What is the name of the hobbit played by Elijah Wood?', a: ['Frodo', 'Frodo Baggins'] },
            { q: 'Which movie features the song "Let It Go"?', a: ['Frozen'] },
            { q: 'Who is the villain in The Dark Knight?', a: ['Joker', 'The Joker'] },
            { q: 'Which Star Wars character is Luke Skywalker\'s father?', a: ['Darth Vader', 'Vader', 'Anakin'] },
            { q: 'Who played Rachel Green in Friends?', a: ['Jennifer Aniston', 'Aniston'] },
            { q: 'What is the name of the fictional city in Batman?', a: ['Gotham', 'Gotham City'] },
            { q: 'What color pill does Neo take in The Matrix?', a: ['Red'] },
            { q: 'Who directed Pulp Fiction?', a: ['Quentin Tarantino', 'Tarantino'] },
            { q: 'Which TV show is set in Westeros?', a: ['Game of Thrones'] },
            { q: 'Who played Forrest Gump?', a: ['Tom Hanks', 'Hanks'] },
            { q: 'What is the name of the snowman in Frozen?', a: ['Olaf'] },
            { q: 'Which house does Harry Potter belong to?', a: ['Gryffindor'] },
            { q: 'Who directed Titanic?', a: ['James Cameron', 'Cameron'] },
            { q: 'What is the name of the ring in Lord of the Rings?', a: ['One Ring', 'The One Ring'] },
            { q: 'Which Marvel superhero is from Wakanda?', a: ['Black Panther'] },
            { q: 'What is the name of the toy spaceman in Toy Story?', a: ['Buzz', 'Buzz Lightyear'] },
            { q: 'What kind of fish is Nemo?', a: ['Clownfish'] },
            { q: 'What is the name of Shrek\'s wife?', a: ['Fiona', 'Princess Fiona'] },
            { q: 'Who plays Wolverine in X-Men?', a: ['Hugh Jackman', 'Jackman'] },
            { q: 'What TV show takes place in a paper company in Scranton?', a: ['The Office'] },
            { q: 'Who directed Inception?', a: ['Christopher Nolan', 'Nolan'] },
            { q: 'What is the name of Thor\'s hammer?', a: ['Mjolnir'] },
            { q: 'Who played the Joker in The Dark Knight?', a: ['Heath Ledger', 'Ledger'] },
            { q: 'What movie features a DeLorean time machine?', a: ['Back to the Future'] },
            { q: 'What is the name of the dragon in How to Train Your Dragon?', a: ['Toothless'] },
            { q: 'Who played Hermione in the Harry Potter films?', a: ['Emma Watson', 'Watson'] },
            { q: 'What animated movie features a rat that cooks?', a: ['Ratatouille'] },
            { q: 'What TV show features dragons and White Walkers?', a: ['Game of Thrones'] },
            { q: 'Who plays Captain America?', a: ['Chris Evans', 'Evans'] },
            { q: 'What is the name of Batman\'s butler?', a: ['Alfred'] },
            { q: 'What movie franchise features Dominic Toretto?', a: ['Fast and Furious', 'Fast & Furious'] },
            { q: 'Who directed The Godfather?', a: ['Francis Ford Coppola', 'Coppola'] },
            { q: 'What TV show features the character Eleven?', a: ['Stranger Things'] },
            { q: 'What is the name of the clown in IT?', a: ['Pennywise'] },
            { q: 'Who played the Terminator?', a: ['Arnold Schwarzenegger', 'Schwarzenegger'] },
            { q: 'What Pixar movie features emotions as characters?', a: ['Inside Out'] },
            { q: 'What is the name of the ship in Alien?', a: ['Nostromo'] },
            { q: 'Who directed Interstellar?', a: ['Christopher Nolan', 'Nolan'] },
            { q: 'What TV show features a chemistry teacher turned drug lord?', a: ['Breaking Bad'] },
            { q: 'What movie features the quote "Here\'s looking at you, kid"?', a: ['Casablanca'] },
            { q: 'Who plays Spider-Man in the MCU (newest)?', a: ['Tom Holland', 'Holland'] },
            { q: 'What is the name of the island in Jurassic Park?', a: ['Isla Nublar'] },
            { q: 'What movie features a volleyball named Wilson?', a: ['Cast Away'] },
            { q: 'Who played Gandalf in Lord of the Rings?', a: ['Ian McKellen', 'McKellen'] },
            { q: 'What animated movie features a fish looking for his son?', a: ['Finding Nemo'] },
            { q: 'What TV show is about a zombie apocalypse?', a: ['The Walking Dead'] },
            { q: 'Who directed Schindler\'s List?', a: ['Steven Spielberg', 'Spielberg'] },
            { q: 'What is Indiana Jones\' real first name?', a: ['Henry'] },
            { q: 'What movie features the quote "May the Force be with you"?', a: ['Star Wars'] }
        ]
    },
    sports: {
        name: 'Sports',
        questions: [
            { q: 'How many players are on a soccer team?', a: ['11', 'Eleven'] },
            { q: 'What sport is played at Wimbledon?', a: ['Tennis'] },
            { q: 'How many points is a touchdown in American football?', a: ['6', 'Six'] },
            { q: 'What country won the 2018 FIFA World Cup?', a: ['France'] },
            { q: 'How many rings are on the Olympic flag?', a: ['5', 'Five'] },
            { q: 'What sport does LeBron James play?', a: ['Basketball'] },
            { q: 'How many holes are played in a round of golf?', a: ['18', 'Eighteen'] },
            { q: 'What is the national sport of Canada?', a: ['Hockey', 'Ice Hockey', 'Lacrosse'] },
            { q: 'How many Grand Slam tennis tournaments are there?', a: ['4', 'Four'] },
            { q: 'What color is the center of an archery target?', a: ['Yellow', 'Gold'] },
            { q: 'In what sport would you perform a slam dunk?', a: ['Basketball'] },
            { q: 'How long is a marathon in miles?', a: ['26.2', '26'] },
            { q: 'What sport uses a shuttlecock?', a: ['Badminton'] },
            { q: 'In bowling, what is it called when you knock down all pins?', a: ['Strike'] },
            { q: 'What country invented cricket?', a: ['England'] },
            { q: 'How many quarters are in an NFL game?', a: ['4', 'Four'] },
            { q: 'What sport is known as "The Beautiful Game"?', a: ['Soccer', 'Football'] },
            { q: 'Who holds the record for most Olympic gold medals?', a: ['Michael Phelps', 'Phelps'] },
            { q: 'What is the diameter of a basketball hoop in inches?', a: ['18'] },
            { q: 'Which country has won the most World Cups?', a: ['Brazil'] },
            { q: 'In baseball, how many strikes make an out?', a: ['3', 'Three'] },
            { q: 'What sport involves a scrum?', a: ['Rugby'] },
            { q: 'What surface is the French Open played on?', a: ['Clay'] },
            { q: 'Which boxer was known as "The Greatest"?', a: ['Muhammad Ali', 'Ali'] },
            { q: 'How many players are on a baseball team on the field?', a: ['9', 'Nine'] },
            { q: 'What is the distance of a sprint race in Olympics (shortest)?', a: ['100m', '100'] },
            { q: 'What object is hit in ice hockey?', a: ['Puck'] },
            { q: 'How many periods are in an NHL hockey game?', a: ['3', 'Three'] },
            { q: 'What sport does Tiger Woods play?', a: ['Golf'] },
            { q: 'How many players are on a basketball team on the court?', a: ['5', 'Five'] },
            { q: 'What is the highest possible score in a single bowling game?', a: ['300'] },
            { q: 'What country hosts the Tour de France?', a: ['France'] },
            { q: 'How many innings are in a standard baseball game?', a: ['9', 'Nine'] },
            { q: 'What sport is played with a shuttlecock and racket?', a: ['Badminton'] },
            { q: 'Which athlete has the most Super Bowl wins as quarterback?', a: ['Tom Brady', 'Brady'] },
            { q: 'What is the term for three strikes in a row in bowling?', a: ['Turkey'] },
            { q: 'How many dimples are on a standard golf ball?', a: ['336'] },
            { q: 'What martial art became an Olympic sport in 2000?', a: ['Taekwondo'] },
            { q: 'What country is the sport of sumo wrestling from?', a: ['Japan'] },
            { q: 'How many players are on a volleyball team on the court?', a: ['6', 'Six'] },
            { q: 'What is the term for zero in tennis?', a: ['Love'] },
            { q: 'What sport does Serena Williams play?', a: ['Tennis'] },
            { q: 'How many laps are in a standard NASCAR race at Daytona?', a: ['200'] },
            { q: 'What sport uses the term "birdie"?', a: ['Golf'] },
            { q: 'What is a perfect score in gymnastics?', a: ['10'] },
            { q: 'How long is an Olympic swimming pool in meters?', a: ['50'] },
            { q: 'What color is the number 1 ball in pool?', a: ['Yellow'] },
            { q: 'What country invented table tennis?', a: ['England'] },
            { q: 'How many sets does a man need to win in a Grand Slam tennis match?', a: ['3', 'Three'] },
            { q: 'What is the only sport played on the moon?', a: ['Golf'] },
            { q: 'What NFL team has won the most Super Bowls?', a: ['New England Patriots', 'Patriots', 'Pittsburgh Steelers', 'Steelers'] },
            { q: 'What sport uses a foil, epee, or sabre?', a: ['Fencing'] },
            { q: 'How many points is a field goal in football?', a: ['3', 'Three'] },
            { q: 'What country won the first FIFA World Cup in 1930?', a: ['Uruguay'] },
            { q: 'What is the width of a soccer goal in feet?', a: ['24'] },
            { q: 'What sport does Usain Bolt compete in?', a: ['Track', 'Sprinting', 'Track and Field'] },
            { q: 'How many minutes is each half of a soccer match?', a: ['45'] },
            { q: 'What is the penalty for a false start in sprinting?', a: ['Disqualification'] },
            { q: 'What NBA team plays in Los Angeles (purple and gold)?', a: ['Lakers', 'LA Lakers'] },
            { q: 'How many red cards result in ejection from a soccer game?', a: ['1', 'One'] }
        ]
    },
    literature: {
        name: 'Literature',
        questions: [
            { q: 'Who wrote "1984"?', a: ['George Orwell', 'Orwell'] },
            { q: 'Who wrote "Harry Potter"?', a: ['J.K. Rowling', 'Rowling'] },
            { q: 'What play contains "To be, or not to be"?', a: ['Hamlet'] },
            { q: 'Who wrote "Pride and Prejudice"?', a: ['Jane Austen', 'Austen'] },
            { q: 'What is the name of Sherlock Holmes\' assistant?', a: ['Watson', 'Dr. Watson'] },
            { q: 'Who wrote "The Great Gatsby"?', a: ['F. Scott Fitzgerald', 'Fitzgerald'] },
            { q: 'What is the prequel to "The Lord of the Rings"?', a: ['The Hobbit'] },
            { q: 'Who wrote "Moby Dick"?', a: ['Herman Melville', 'Melville'] },
            { q: 'What is the pen name of Samuel Clemens?', a: ['Mark Twain', 'Twain'] },
            { q: 'Who wrote "The Catcher in the Rye"?', a: ['J.D. Salinger', 'Salinger'] },
            { q: 'What epic poem was written by Homer?', a: ['The Odyssey', 'Odyssey', 'The Iliad', 'Iliad'] },
            { q: 'Who wrote "To Kill a Mockingbird"?', a: ['Harper Lee', 'Lee'] },
            { q: 'What is the name of the pig in "Charlotte\'s Web"?', a: ['Wilbur'] },
            { q: 'Who wrote "The Chronicles of Narnia"?', a: ['C.S. Lewis', 'Lewis'] },
            { q: 'What is the first book of the Bible?', a: ['Genesis'] },
            { q: 'Who wrote "Alice in Wonderland"?', a: ['Lewis Carroll', 'Carroll'] },
            { q: 'Who is the author of "Frankenstein"?', a: ['Mary Shelley', 'Shelley'] },
            { q: 'What Shakespeare play features "Et tu, Brute"?', a: ['Julius Caesar'] },
            { q: 'Who wrote "The Hunger Games"?', a: ['Suzanne Collins', 'Collins'] },
            { q: 'What is the title of the first Harry Potter book (US)?', a: ['Sorcerer\'s Stone', 'Harry Potter and the Sorcerer\'s Stone'] },
            { q: 'Who wrote "War and Peace"?', a: ['Tolstoy', 'Leo Tolstoy'] },
            { q: 'What is the name of Captain Ahab\'s ship?', a: ['Pequod'] },
            { q: 'Who wrote "Crime and Punishment"?', a: ['Dostoevsky'] },
            { q: 'Who creates the Ring in Lord of the Rings?', a: ['Sauron'] },
            { q: 'Who wrote "A Tale of Two Cities"?', a: ['Charles Dickens', 'Dickens'] },
            { q: 'Who wrote "The Hobbit"?', a: ['Tolkien', 'J.R.R. Tolkien'] },
            { q: 'What color is the scarlet letter?', a: ['Red', 'Scarlet'] },
            { q: 'Who wrote "Of Mice and Men"?', a: ['John Steinbeck', 'Steinbeck'] },
            { q: 'Who is the protagonist of "The Hunger Games"?', a: ['Katniss', 'Katniss Everdeen'] },
            { q: 'Who wrote "Brave New World"?', a: ['Aldous Huxley', 'Huxley'] },
            { q: 'What is the name of Don Quixote\'s horse?', a: ['Rocinante'] },
            { q: 'Who wrote "The Raven"?', a: ['Edgar Allan Poe', 'Poe'] },
            { q: 'What is the last book of the Bible?', a: ['Revelation', 'Revelations'] },
            { q: 'Who wrote "Fahrenheit 451"?', a: ['Ray Bradbury', 'Bradbury'] },
            { q: 'What novel begins "Call me Ishmael"?', a: ['Moby Dick'] },
            { q: 'Who wrote "The Picture of Dorian Gray"?', a: ['Oscar Wilde', 'Wilde'] },
            { q: 'What is the name of Harry Potter\'s best friend?', a: ['Ron', 'Ron Weasley'] },
            { q: 'Who wrote "Lord of the Flies"?', a: ['William Golding', 'Golding'] },
            { q: 'What dystopian novel features "Big Brother"?', a: ['1984'] },
            { q: 'Who wrote "The Da Vinci Code"?', a: ['Dan Brown', 'Brown'] },
            { q: 'What novel features a character named Atticus Finch?', a: ['To Kill a Mockingbird'] },
            { q: 'Who wrote "Dracula"?', a: ['Bram Stoker', 'Stoker'] },
            { q: 'What is the name of the wizard school in Harry Potter?', a: ['Hogwarts'] },
            { q: 'Who wrote "The Odyssey"?', a: ['Homer'] },
            { q: 'What novel features a character named Jay Gatsby?', a: ['The Great Gatsby'] },
            { q: 'Who wrote "Animal Farm"?', a: ['George Orwell', 'Orwell'] },
            { q: 'What is the name of Sherlock Holmes\' brother?', a: ['Mycroft'] },
            { q: 'Who wrote "Jane Eyre"?', a: ['Charlotte Bronte', 'Bronte'] },
            { q: 'What book features the land of Lilliput?', a: ['Gulliver\'s Travels'] },
            { q: 'Who wrote "The Old Man and the Sea"?', a: ['Ernest Hemingway', 'Hemingway'] },
            { q: 'What Shakespeare play features fairies and a donkey?', a: ['A Midsummer Night\'s Dream'] },
            { q: 'Who wrote "Wuthering Heights"?', a: ['Emily Bronte', 'Bronte'] },
            { q: 'What is the name of Long John Silver\'s parrot?', a: ['Captain Flint'] },
            { q: 'Who wrote "The Alchemist"?', a: ['Paulo Coelho', 'Coelho'] },
            { q: 'What novel is set in the fictional country of Oceania?', a: ['1984'] },
            { q: 'Who wrote "Les Miserables"?', a: ['Victor Hugo', 'Hugo'] },
            { q: 'What is the name of the whale in Moby Dick?', a: ['Moby Dick'] },
            { q: 'Who wrote "The Shining"?', a: ['Stephen King', 'King'] },
            { q: 'What Greek hero had a vulnerable heel?', a: ['Achilles'] },
            { q: 'Who wrote "Gone with the Wind"?', a: ['Margaret Mitchell', 'Mitchell'] }
        ]
    }
};

// Game state
let gameContainer = null;
let currentCategory = null;
let currentQuestionIndex = 0;
let score = 0;
let gameActive = false;
let currentRoundQuestions = [];
// Track recently used question indices per category to prevent repeats
let recentQuestions = {};
let historyLoaded = false;
let showHelp = false;
let helpChecked = false;

const HELP_KEY = 'game_help_seen_quiz';
const INSTRUCTIONS = 'Pick a category, then answer 10 trivia questions by typing your answer and pressing Enter or Submit. Answers are case-insensitive with typo tolerance.';

const QUIZ_STORAGE_KEY = 'quiz_recent_questions';

/**
 * Load recent question history from storage
 */
async function loadHistory() {
    if (historyLoaded) return;
    try {
        const data = await chrome.storage.local.get(QUIZ_STORAGE_KEY);
        if (data[QUIZ_STORAGE_KEY] && typeof data[QUIZ_STORAGE_KEY] === 'object') {
            recentQuestions = data[QUIZ_STORAGE_KEY];
        }
    } catch (e) { /* ignore storage errors */ }
    historyLoaded = true;
}

/**
 * Save recent question history to storage
 */
function saveHistory() {
    try {
        chrome.storage.local.set({ [QUIZ_STORAGE_KEY]: recentQuestions });
    } catch (e) { /* ignore storage errors */ }
}

/**
 * Pick questions for a round, avoiding recently used ones
 */
function pickRoundQuestions(categoryKey, count) {
    const allQuestions = QUIZ_CATEGORIES[categoryKey].questions;
    const totalAvailable = allQuestions.length;

    // Initialize tracking for this category if needed
    if (!recentQuestions[categoryKey]) {
        recentQuestions[categoryKey] = [];
    }

    const recent = recentQuestions[categoryKey];

    // Get indices not recently used
    let availableIndices = [];
    for (let i = 0; i < totalAvailable; i++) {
        if (!recent.includes(i)) availableIndices.push(i);
    }

    // If not enough available, reset history but keep last round's questions out
    if (availableIndices.length < count) {
        const lastRound = recent.slice(-count);
        recentQuestions[categoryKey] = lastRound;
        availableIndices = [];
        for (let i = 0; i < totalAvailable; i++) {
            if (!lastRound.includes(i)) availableIndices.push(i);
        }
    }

    // Shuffle available indices and pick
    for (let i = availableIndices.length - 1; i > 0; i--) {
        const j = Math.floor(Math.random() * (i + 1));
        [availableIndices[i], availableIndices[j]] = [availableIndices[j], availableIndices[i]];
    }

    const pickedIndices = availableIndices.slice(0, count);

    // Record these as recently used
    recentQuestions[categoryKey].push(...pickedIndices);

    // Trim history to ~60% of pool
    const maxHistory = Math.floor(totalAvailable * 0.6);
    if (recentQuestions[categoryKey].length > maxHistory) {
        recentQuestions[categoryKey] = recentQuestions[categoryKey].slice(-maxHistory);
    }

    saveHistory();

    // Return the actual question objects in random order
    return pickedIndices.map(i => allQuestions[i]);
}

/**
 * Initialize the game
 */
export async function initQuiz(container) {
    gameContainer = container;
    await loadHistory();
    if (!helpChecked) {
        helpChecked = true;
        try {
            const data = await chrome.storage.local.get(HELP_KEY);
            showHelp = !data[HELP_KEY];
        } catch (e) { showHelp = true; }
    }
    showCategorySelection();
}

/**
 * Show category selection screen
 */
function showCategorySelection() {
    if (!gameContainer) return;

    gameActive = false;
    currentCategory = null;
    currentQuestionIndex = 0;
    score = 0;
    currentRoundQuestions = [];

    const helpBtn = `<button class="game-help-btn" id="quiz-help-btn" aria-label="Instructions">?</button>`;
    const header = `<div class="game-header"><div class="game-status">Choose a Category</div>${helpBtn}</div>`;
    const overlay = showHelp ? `
        <div class="game-instructions">
            <div class="game-instructions-title">How to Play</div>
            <div class="game-instructions-text">${INSTRUCTIONS}</div>
            <button class="game-reset-btn game-instructions-dismiss" id="quiz-dismiss">Got it</button>
        </div>` : '';

    let html = '<div class="quiz-game">' + header + '<div class="quiz-categories">';

    for (const [key, cat] of Object.entries(QUIZ_CATEGORIES)) {
        html += `<button class="quiz-category-btn" data-category="${key}">${cat.name}</button>`;
    }

    html += '</div>' + overlay + '</div>';
    gameContainer.innerHTML = html;

    // Add event listeners
    const btns = gameContainer.querySelectorAll('.quiz-category-btn');
    btns.forEach(btn => {
        btn.addEventListener('click', () => startQuiz(btn.dataset.category));
    });

    const helpBtnEl = gameContainer.querySelector('#quiz-help-btn');
    if (helpBtnEl) {
        helpBtnEl.addEventListener('click', () => { showHelp = !showHelp; showCategorySelection(); });
    }

    const dismissEl = gameContainer.querySelector('#quiz-dismiss');
    if (dismissEl) {
        dismissEl.addEventListener('click', () => { showHelp = false; try { chrome.storage.local.set({ [HELP_KEY]: true }); } catch(e){} showCategorySelection(); });
    }
}

/**
 * Start quiz with selected category
 */
function startQuiz(categoryKey) {
    currentCategory = categoryKey;
    currentQuestionIndex = 0;
    score = 0;
    gameActive = true;

    // Pick 10 questions avoiding recent repeats
    currentRoundQuestions = pickRoundQuestions(categoryKey, 10);

    showQuestion();
}

/**
 * Show current question
 */
function showQuestion() {
    if (!gameContainer || !currentCategory || !currentRoundQuestions.length) return;

    const category = QUIZ_CATEGORIES[currentCategory];
    const question = currentRoundQuestions[currentQuestionIndex];
    const questionNum = currentQuestionIndex + 1;
    const totalQuestions = currentRoundQuestions.length;

    gameContainer.innerHTML = `
        <div class="quiz-game">
            <div class="quiz-header">
                <div class="quiz-category-name">${category.name}</div>
                <div class="quiz-score">Score: ${score}/${totalQuestions}</div>
            </div>
            <div class="quiz-question-num">Question ${questionNum} of ${totalQuestions}</div>
            <div class="quiz-question">${question.q}</div>
            <div class="quiz-answer-input">
                <input type="text" id="quiz-answer" placeholder="Type your answer..." autocomplete="off">
                <button id="quiz-submit">Submit</button>
            </div>
            <div class="quiz-feedback" id="quiz-feedback"></div>
        </div>
    `;

    const input = gameContainer.querySelector('#quiz-answer');
    const submitBtn = gameContainer.querySelector('#quiz-submit');

    if (input && submitBtn) {
        input.focus();
        submitBtn.addEventListener('click', () => checkAnswer(input.value));
        input.addEventListener('keypress', (e) => {
            if (e.key === 'Enter') checkAnswer(input.value);
        });
    }
}

/**
 * Normalize a string for comparison: lowercase, strip punctuation/articles/extra spaces
 */
function normalize(str) {
    return str.toLowerCase()
        .replace(/[''"".,!?;:\-()]/g, '')  // strip punctuation
        .replace(/\b(the|a|an|of|and|in)\b/g, ' ')  // strip common filler words
        .replace(/\s+/g, ' ')  // collapse spaces
        .trim();
}

/**
 * Levenshtein distance between two strings
 */
function editDistance(a, b) {
    const m = a.length, n = b.length;
    const dp = Array.from({ length: m + 1 }, () => Array(n + 1).fill(0));
    for (let i = 0; i <= m; i++) dp[i][0] = i;
    for (let j = 0; j <= n; j++) dp[0][j] = j;
    for (let i = 1; i <= m; i++) {
        for (let j = 1; j <= n; j++) {
            dp[i][j] = a[i - 1] === b[j - 1]
                ? dp[i - 1][j - 1]
                : 1 + Math.min(dp[i - 1][j], dp[i][j - 1], dp[i - 1][j - 1]);
        }
    }
    return dp[m][n];
}

/**
 * Check if user's answer loosely matches any accepted answer.
 * Accepts: exact match, last-word match (e.g. last name), contains match,
 * starts-with match, and typo tolerance via edit distance.
 */
function isLooseMatch(userInput, acceptedAnswers) {
    const user = normalize(userInput);
    if (!user) return false;

    for (const ans of acceptedAnswers) {
        const target = normalize(ans);
        if (!target) continue;

        // Exact match after normalization
        if (user === target) return true;

        // User's answer matches the last word (last name, e.g. "Einstein")
        const targetWords = target.split(' ');
        if (targetWords.length > 1 && user === targetWords[targetWords.length - 1]) return true;

        // User's answer contains the full target or vice versa
        if (user.includes(target) || target.includes(user)) {
            // Only accept if the shorter string is at least 3 chars to avoid false positives
            const shorter = Math.min(user.length, target.length);
            if (shorter >= 3) return true;
        }

        // Typo tolerance: allow edit distance based on word length
        // Short words (<=4): must be exact; 5-7 chars: 1 typo; 8+: 2 typos
        const maxDist = target.length <= 4 ? 0 : target.length <= 7 ? 1 : 2;
        if (maxDist > 0 && editDistance(user, target) <= maxDist) return true;

        // Also check typo tolerance against last word for multi-word answers
        if (targetWords.length > 1) {
            const lastName = targetWords[targetWords.length - 1];
            const lastMaxDist = lastName.length <= 4 ? 0 : lastName.length <= 7 ? 1 : 2;
            if (lastMaxDist > 0 && editDistance(user, lastName) <= lastMaxDist) return true;
        }
    }

    return false;
}

/**
 * Check answer
 */
function checkAnswer(userAnswer) {
    if (!currentCategory) return;

    const question = currentRoundQuestions[currentQuestionIndex];
    const feedback = gameContainer.querySelector('#quiz-feedback');
    const input = gameContainer.querySelector('#quiz-answer');
    const submitBtn = gameContainer.querySelector('#quiz-submit');

    const isCorrect = isLooseMatch(userAnswer, question.a);

    if (isCorrect) {
        score++;
        feedback.textContent = '✓ Correct!';
        feedback.className = 'quiz-feedback correct';
    } else {
        feedback.textContent = `✗ Wrong! Answer: ${question.a[0]}`;
        feedback.className = 'quiz-feedback incorrect';
    }

    input.disabled = true;
    submitBtn.disabled = true;

    // Move to next question or show results
    setTimeout(() => {
        currentQuestionIndex++;
        if (currentQuestionIndex < currentRoundQuestions.length) {
            showQuestion();
        } else {
            showResults();
        }
    }, 1500);
}

/**
 * Show final results
 */
function showResults() {
    if (!gameContainer || !currentCategory) return;

    const totalQuestions = currentRoundQuestions.length;
    const percentage = Math.round((score / totalQuestions) * 100);

    let message = '';
    if (percentage >= 80) message = 'Excellent! 🎉';
    else if (percentage >= 60) message = 'Good job! 👍';
    else if (percentage >= 40) message = 'Not bad! 👌';
    else message = 'Keep practicing! 💪';

    gameContainer.innerHTML = `
        <div class="quiz-game">
            <div class="game-status">${message}</div>
            <div class="quiz-results">
                <div class="quiz-final-score">${score} / ${totalQuestions}</div>
                <div class="quiz-percentage">${percentage}%</div>
            </div>
            <button class="game-reset-btn" id="quiz-new">New Quiz</button>
            <button class="game-reset-btn" id="quiz-category">Change Category</button>
        </div>
    `;

    const newBtn = gameContainer.querySelector('#quiz-new');
    const catBtn = gameContainer.querySelector('#quiz-category');

    if (newBtn) newBtn.addEventListener('click', () => startQuiz(currentCategory));
    if (catBtn) catBtn.addEventListener('click', () => showCategorySelection());
}
