/**
 * Minesweeper Game Module
 */

const ROWS = 9;
const COLS = 9;
const MINE_COUNT = 10;

// Game state
let gameContainer = null;
let board = [];       // { mine, revealed, flagged, adjacentMines }
let gameOver = false;
let gameWon = false;
let firstClick = true;
let flagCount = 0;
let revealedCount = 0;
let showHelp = false;
let helpChecked = false;
let timerInterval = null;
let elapsed = 0;

const HELP_KEY = 'game_help_seen_mines';
const INSTRUCTIONS = 'Click cells to reveal them. Numbers show how many adjacent mines exist. Right-click or long-press to flag suspected mines. Reveal all safe cells to win. Don\'t click a mine!';

/**
 * Initialize the game
 */
export async function initMinesweeper(container) {
    gameContainer = container;
    if (!helpChecked) {
        helpChecked = true;
        try {
            const data = await chrome.storage.local.get(HELP_KEY);
            showHelp = !data[HELP_KEY];
        } catch (e) { showHelp = true; }
    }
    resetGame();
    renderGame();
}

/**
 * Reset game state
 */
function resetGame() {
    board = [];
    for (let r = 0; r < ROWS; r++) {
        board[r] = [];
        for (let c = 0; c < COLS; c++) {
            board[r][c] = { mine: false, revealed: false, flagged: false, adjacentMines: 0 };
        }
    }
    gameOver = false;
    gameWon = false;
    firstClick = true;
    flagCount = 0;
    revealedCount = 0;
    elapsed = 0;
    if (timerInterval) {
        clearInterval(timerInterval);
        timerInterval = null;
    }
}

/**
 * Place mines avoiding the first-clicked cell and its neighbors
 */
function placeMines(safeR, safeC) {
    const safeCells = new Set();
    for (let dr = -1; dr <= 1; dr++) {
        for (let dc = -1; dc <= 1; dc++) {
            const nr = safeR + dr;
            const nc = safeC + dc;
            if (nr >= 0 && nr < ROWS && nc >= 0 && nc < COLS) {
                safeCells.add(nr * COLS + nc);
            }
        }
    }

    let placed = 0;
    while (placed < MINE_COUNT) {
        const r = Math.floor(Math.random() * ROWS);
        const c = Math.floor(Math.random() * COLS);
        if (!board[r][c].mine && !safeCells.has(r * COLS + c)) {
            board[r][c].mine = true;
            placed++;
        }
    }

    // Calculate adjacent mine counts
    for (let r = 0; r < ROWS; r++) {
        for (let c = 0; c < COLS; c++) {
            if (board[r][c].mine) continue;
            let count = 0;
            for (let dr = -1; dr <= 1; dr++) {
                for (let dc = -1; dc <= 1; dc++) {
                    const nr = r + dr;
                    const nc = c + dc;
                    if (nr >= 0 && nr < ROWS && nc >= 0 && nc < COLS && board[nr][nc].mine) {
                        count++;
                    }
                }
            }
            board[r][c].adjacentMines = count;
        }
    }
}

/**
 * Reveal a cell (flood-fill for zeros)
 */
function reveal(r, c) {
    if (r < 0 || r >= ROWS || c < 0 || c >= COLS) return;
    const cell = board[r][c];
    if (cell.revealed || cell.flagged) return;

    cell.revealed = true;
    revealedCount++;

    if (cell.mine) {
        gameOver = true;
        cell.exploded = true;
        // Reveal all mines
        for (let rr = 0; rr < ROWS; rr++) {
            for (let cc = 0; cc < COLS; cc++) {
                if (board[rr][cc].mine) board[rr][cc].revealed = true;
            }
        }
        if (timerInterval) {
            clearInterval(timerInterval);
            timerInterval = null;
        }
        return;
    }

    // Flood fill if no adjacent mines
    if (cell.adjacentMines === 0) {
        for (let dr = -1; dr <= 1; dr++) {
            for (let dc = -1; dc <= 1; dc++) {
                if (dr === 0 && dc === 0) continue;
                reveal(r + dr, c + dc);
            }
        }
    }

    // Check win
    if (revealedCount === ROWS * COLS - MINE_COUNT) {
        gameWon = true;
        gameOver = true;
        if (timerInterval) {
            clearInterval(timerInterval);
            timerInterval = null;
        }
    }
}

/**
 * Handle cell left-click
 */
function handleClick(r, c) {
    if (gameOver) return;
    const cell = board[r][c];
    if (cell.flagged || cell.revealed) return;

    if (firstClick) {
        firstClick = false;
        placeMines(r, c);
        elapsed = 0;
        timerInterval = setInterval(() => {
            elapsed++;
            const timerEl = gameContainer.querySelector('#mines-timer');
            if (timerEl) timerEl.textContent = elapsed + 's';
        }, 1000);
    }

    reveal(r, c);
    renderGame();
}

/**
 * Handle cell right-click (flag)
 */
function handleFlag(r, c) {
    if (gameOver) return;
    const cell = board[r][c];
    if (cell.revealed) return;

    cell.flagged = !cell.flagged;
    flagCount += cell.flagged ? 1 : -1;
    renderGame();
}

/**
 * Get display content for a cell
 */
function cellContent(cell) {
    if (!cell.revealed) {
        return cell.flagged ? '\u2691' : '';
    }
    if (cell.mine) return '\u2739';
    if (cell.adjacentMines === 0) return '';
    return cell.adjacentMines;
}

/**
 * Get CSS class for a cell
 */
function cellClass(cell) {
    let cls = 'mines-cell';
    if (!cell.revealed) {
        cls += cell.flagged ? ' mines-cell-flagged' : ' mines-cell-hidden';
    } else if (cell.mine) {
        cls += cell.exploded ? ' mines-cell-exploded' : ' mines-cell-mine';
    } else {
        cls += ' mines-cell-revealed';
        if (cell.adjacentMines > 0) cls += ` mines-${cell.adjacentMines}`;
    }
    return cls;
}

/**
 * Render the game
 */
function renderGame() {
    if (!gameContainer) return;

    let statusText = 'Minesweeper';
    if (gameWon) statusText = 'You Win!';
    else if (gameOver) statusText = 'Game Over';

    let boardHtml = '';
    for (let r = 0; r < ROWS; r++) {
        for (let c = 0; c < COLS; c++) {
            const cell = board[r][c];
            boardHtml += `<button class="${cellClass(cell)}" data-r="${r}" data-c="${c}">${cellContent(cell)}</button>`;
        }
    }

    const remaining = MINE_COUNT - flagCount;

    const helpBtn = `<button class="game-help-btn" id="mines-help-btn" aria-label="Instructions">?</button>`;
    const header = `<div class="game-header"><div class="game-status">${statusText}</div>${helpBtn}</div>`;
    const overlay = showHelp ? `
        <div class="game-instructions">
            <div class="game-instructions-title">How to Play</div>
            <div class="game-instructions-text">${INSTRUCTIONS}</div>
            <button class="game-reset-btn game-instructions-dismiss" id="mines-dismiss">Got it</button>
        </div>` : '';

    gameContainer.innerHTML = `
        <div class="mines-game">
            ${header}
            <div class="mines-info">
                <div class="mines-info-box">
                    <div class="mines-info-label">Mines</div>
                    <div class="mines-info-value">${remaining}</div>
                </div>
                <div class="mines-info-box">
                    <div class="mines-info-label">Time</div>
                    <div class="mines-info-value" id="mines-timer">${elapsed}s</div>
                </div>
            </div>
            <div class="mines-board">${boardHtml}</div>
            <div class="mines-hint">Right-click to flag</div>
            <button class="game-reset-btn" id="mines-reset">New Game</button>
            ${overlay}
        </div>
    `;

    // Prevent browser context menu on the entire board
    const boardEl = gameContainer.querySelector('.mines-board');
    if (boardEl) {
        boardEl.addEventListener('contextmenu', (e) => e.preventDefault());
    }

    // Cell click listeners
    const cells = gameContainer.querySelectorAll('.mines-cell');
    cells.forEach(cell => {
        const r = parseInt(cell.dataset.r);
        const c = parseInt(cell.dataset.c);
        let pressTimer = null;
        let longPressed = false;

        // Right-click to flag
        cell.addEventListener('contextmenu', (e) => {
            e.preventDefault();
            e.stopPropagation();
            handleFlag(r, c);
        });

        // Left-click to reveal (skip if long-press just flagged)
        cell.addEventListener('click', (e) => {
            e.preventDefault();
            if (longPressed) {
                longPressed = false;
                return;
            }
            handleClick(r, c);
        });

        // Long-press for flag on touch devices
        cell.addEventListener('touchstart', (e) => {
            longPressed = false;
            pressTimer = setTimeout(() => {
                longPressed = true;
                handleFlag(r, c);
            }, 500);
        }, { passive: true });
        cell.addEventListener('touchend', () => {
            if (pressTimer) { clearTimeout(pressTimer); pressTimer = null; }
        });
        cell.addEventListener('touchmove', () => {
            if (pressTimer) { clearTimeout(pressTimer); pressTimer = null; }
        });
    });

    const resetBtn = gameContainer.querySelector('#mines-reset');
    if (resetBtn) {
        resetBtn.addEventListener('click', () => {
            resetGame();
            renderGame();
        });
    }

    const helpBtnEl = gameContainer.querySelector('#mines-help-btn');
    if (helpBtnEl) {
        helpBtnEl.addEventListener('click', () => { showHelp = !showHelp; renderGame(); });
    }

    const dismissEl = gameContainer.querySelector('#mines-dismiss');
    if (dismissEl) {
        dismissEl.addEventListener('click', () => { showHelp = false; try { chrome.storage.local.set({ [HELP_KEY]: true }); } catch(e){} renderGame(); });
    }
}
