/**
 * Memory Match Game Module
 */

// 24 cards (12 pairs) on a 6x4 grid
const TOTAL_PAIRS = 12;
const COLS = 6;
const ROWS = 4;

const SYMBOLS = [
    '\u2660', '\u2665', '\u2666', '\u2663', // spade, heart, diamond, club
    '\u2605', '\u263A', '\u266B', '\u2602',  // star, smiley, music, umbrella
    '\u2622', '\u2708', '\u2744', '\u269B',  // radiation, plane, snowflake, atom
    '\u2618', '\u2691', '\u2693', '\u2694'   // shamrock, flag, anchor, swords
];

// Game state
let gameContainer = null;
let cards = [];
let flipped = [];      // indices of currently flipped cards (max 2)
let matched = new Set();
let moves = 0;
let lockBoard = false;
let showHelp = false;
let helpChecked = false;

const HELP_KEY = 'game_help_seen_memory';
const INSTRUCTIONS = 'Click cards to flip them over. Match all pairs by remembering card positions. Only two cards can be flipped at a time. Find all pairs in as few moves as possible.';

/**
 * Initialize the game
 */
export async function initMemory(container) {
    gameContainer = container;
    if (!helpChecked) {
        helpChecked = true;
        try {
            const data = await chrome.storage.local.get(HELP_KEY);
            showHelp = !data[HELP_KEY];
        } catch (e) { showHelp = true; }
    }
    resetGame();
    renderGame();
}

/**
 * Reset game state
 */
function resetGame() {
    // Pick TOTAL_PAIRS symbols and duplicate
    const picked = SYMBOLS.slice(0, TOTAL_PAIRS);
    const deck = [...picked, ...picked];

    // Shuffle
    for (let i = deck.length - 1; i > 0; i--) {
        const j = Math.floor(Math.random() * (i + 1));
        [deck[i], deck[j]] = [deck[j], deck[i]];
    }

    cards = deck;
    flipped = [];
    matched = new Set();
    moves = 0;
    lockBoard = false;
}

/**
 * Handle card click
 */
function handleCardClick(index) {
    if (lockBoard) return;
    if (flipped.includes(index)) return;
    if (matched.has(index)) return;

    flipped.push(index);
    renderGame();

    if (flipped.length === 2) {
        moves++;
        lockBoard = true;
        const [a, b] = flipped;

        if (cards[a] === cards[b]) {
            // Match found
            matched.add(a);
            matched.add(b);
            flipped = [];
            lockBoard = false;
            renderGame();
        } else {
            // No match — flip back after delay
            setTimeout(() => {
                flipped = [];
                lockBoard = false;
                renderGame();
            }, 800);
        }
    }
}

/**
 * Check if game is won
 */
function isWon() {
    return matched.size === cards.length;
}

/**
 * Render the game
 */
function renderGame() {
    if (!gameContainer) return;

    const won = isWon();
    let statusText = 'Memory Match';
    if (won) statusText = 'You Win!';

    let boardHtml = '';
    for (let i = 0; i < cards.length; i++) {
        const isFlipped = flipped.includes(i);
        const isMatched = matched.has(i);
        const show = isFlipped || isMatched;
        const cls = `memory-cell${isMatched ? ' memory-cell-matched' : ''}${isFlipped ? ' memory-cell-flipped' : ''}${!show ? ' memory-cell-hidden' : ''}`;
        const content = show ? cards[i] : '';
        boardHtml += `<button class="${cls}" data-index="${i}">${content}</button>`;
    }

    const helpBtn = `<button class="game-help-btn" id="memory-help-btn" aria-label="Instructions">?</button>`;
    const header = `<div class="game-header"><div class="game-status">${statusText}</div>${helpBtn}</div>`;
    const overlay = showHelp ? `
        <div class="game-instructions">
            <div class="game-instructions-title">How to Play</div>
            <div class="game-instructions-text">${INSTRUCTIONS}</div>
            <button class="game-reset-btn game-instructions-dismiss" id="memory-dismiss">Got it</button>
        </div>` : '';

    gameContainer.innerHTML = `
        <div class="memory-game">
            ${header}
            <div class="memory-info">Moves: ${moves}</div>
            <div class="memory-board">${boardHtml}</div>
            <button class="game-reset-btn" id="memory-reset">New Game</button>
            ${overlay}
        </div>
    `;

    // Cell click listeners
    const cellEls = gameContainer.querySelectorAll('.memory-cell');
    cellEls.forEach(cell => {
        const idx = parseInt(cell.dataset.index);
        cell.addEventListener('click', () => handleCardClick(idx));
    });

    const resetBtn = gameContainer.querySelector('#memory-reset');
    if (resetBtn) {
        resetBtn.addEventListener('click', () => {
            resetGame();
            renderGame();
        });
    }

    const helpBtnEl = gameContainer.querySelector('#memory-help-btn');
    if (helpBtnEl) {
        helpBtnEl.addEventListener('click', () => { showHelp = !showHelp; renderGame(); });
    }

    const dismissEl = gameContainer.querySelector('#memory-dismiss');
    if (dismissEl) {
        dismissEl.addEventListener('click', () => { showHelp = false; try { chrome.storage.local.set({ [HELP_KEY]: true }); } catch(e){} renderGame(); });
    }
}
