/**
 * Hangman Game Module
 */

// Word lists organized by category for variety
const WORD_CATEGORIES = {
    technology: [
        'JAVASCRIPT', 'PYTHON', 'CODING', 'COMPUTER', 'KEYBOARD', 'MONITOR', 'PROGRAM',
        'FUNCTION', 'VARIABLE', 'ALGORITHM', 'DATABASE', 'NETWORK', 'BROWSER', 'WEBSITE',
        'SOFTWARE', 'HARDWARE', 'INTERNET', 'DOWNLOAD', 'UPLOAD', 'SERVER', 'CLIENT',
        'DEVELOPER', 'ENGINEER', 'DESIGNER', 'PROJECT', 'TESTING', 'DEBUGGING', 'VERSION',
        'FRAMEWORK', 'LIBRARY', 'PACKAGE', 'MODULE', 'COMPONENT', 'INTERFACE', 'TERMINAL',
        'COMMAND', 'SYNTAX', 'COMPILER', 'RUNTIME', 'MEMORY', 'STORAGE', 'PROCESSOR',
        'GRAPHICS', 'DISPLAY', 'RESOLUTION', 'PIXEL', 'VECTOR', 'RASTER', 'FORMAT',
        'EXTENSION', 'DIRECTORY', 'FOLDER', 'DOCUMENT', 'SPREADSHEET', 'PRESENTATION',
        'BANDWIDTH', 'BLUETOOTH', 'CACHE', 'COOKIE', 'CURSOR', 'DOMAIN', 'ENCRYPT',
        'FIREWALL', 'GATEWAY', 'HOSTING', 'KERNEL', 'LAPTOP', 'MALWARE', 'PROTOCOL',
        'QUANTUM', 'ROUTER', 'SCRIPT', 'TABLET', 'VIRTUAL', 'WIDGET', 'BINARY',
        'BOOLEAN', 'BUFFER', 'CHIPSET', 'CLUSTER', 'CONSOLE', 'DAEMON', 'ETHERNET',
        'FIRMWARE', 'GIGABYTE', 'HASHMAP', 'INSTANCE', 'ITERATE', 'LATENCY', 'MAINFRAME',
        'NAVIGATE', 'OPTIMIZE', 'PAYLOAD', 'REFACTOR', 'SANDBOX', 'TEMPLATE', 'UNICODE'
    ],
    animals: [
        'ELEPHANT', 'GIRAFFE', 'PENGUIN', 'DOLPHIN', 'BUTTERFLY', 'KANGAROO', 'CHEETAH',
        'FLAMINGO', 'GORILLA', 'HAMSTER', 'IGUANA', 'JAGUAR', 'LEOPARD', 'MUSTANG',
        'NARWHAL', 'OCTOPUS', 'PANTHER', 'RACCOON', 'SPARROW', 'TORTOISE', 'VULTURE',
        'WALRUS', 'FALCON', 'BUFFALO', 'CHINCHILLA', 'GAZELLE', 'HEDGEHOG', 'LOBSTER',
        'MACKEREL', 'PELICAN', 'SALMON', 'TOUCAN', 'WARTHOG', 'ZEBRA', 'ALLIGATOR',
        'ANTELOPE', 'BARRACUDA', 'CHAMELEON', 'DRAGONFLY', 'STALLION', 'GOLDFISH',
        'HUMMINGBIRD', 'JELLYFISH', 'KINGFISHER', 'MONGOOSE', 'PARROT', 'ROOSTER',
        'SCORPION', 'STARFISH', 'STINGRAY', 'SWORDFISH', 'TADPOLE', 'WOODPECKER',
        'ARMADILLO', 'CHIPMUNK', 'COYOTE', 'CRICKET', 'FERRET', 'MEERKAT', 'PEACOCK'
    ],
    food: [
        'AVOCADO', 'BROCCOLI', 'CHOCOLATE', 'DUMPLING', 'ESPRESSO', 'FOCACCIA',
        'GUACAMOLE', 'HAMBURGER', 'JALAPENO', 'LASAGNA', 'MACARONI', 'NECTARINE',
        'PANCAKE', 'QUESADILLA', 'RASPBERRY', 'SANDWICH', 'TIRAMISU', 'VANILLA',
        'ZUCCHINI', 'BLUEBERRY', 'CINNAMON', 'CROISSANT', 'EGGPLANT', 'MUSHROOM',
        'PINEAPPLE', 'SPAGHETTI', 'STRAWBERRY', 'TANGERINE', 'WATERMELON', 'ALMOND',
        'BAGUETTE', 'BURRITO', 'CASHEW', 'COCONUT', 'CRANBERRY', 'CUCUMBER', 'GRAPEFRUIT',
        'HAZELNUT', 'MOZZARELLA', 'OMELETTE', 'PISTACHIO', 'PORRIDGE', 'PUMPKIN',
        'RAVIOLI', 'RISOTTO', 'SAUSAGE', 'TRUFFLE', 'WAFFLE', 'CABBAGE', 'ARTICHOKE',
        'CILANTRO', 'MANDARIN', 'PARSLEY', 'PRETZEL', 'RADISH', 'SPINACH', 'TURNIP'
    ],
    geography: [
        'AMAZON', 'BERMUDA', 'CAIRO', 'DENMARK', 'ETHIOPIA', 'FLORENCE', 'GUATEMALA',
        'HELSINKI', 'ICELAND', 'JAMAICA', 'KILIMANJARO', 'LONDON', 'MOROCCO', 'NEPTUNE',
        'OKLAHOMA', 'PORTUGAL', 'SAHARA', 'TOKYO', 'UKRAINE', 'VIENNA', 'WYOMING',
        'ZANZIBAR', 'ATLANTIC', 'BUDAPEST', 'COLUMBIA', 'EVEREST', 'GALAPAGOS',
        'HONOLULU', 'ISTANBUL', 'JUPITER', 'KENTUCKY', 'MICHIGAN', 'NIAGARA',
        'PATAGONIA', 'SINGAPORE', 'TENNESSEE', 'VANCOUVER', 'ZIMBABWE', 'AMSTERDAM',
        'BARCELONA', 'CAMBODIA', 'EDINBURGH', 'MADAGASCAR', 'MISSISSIPPI', 'MONGOLIA',
        'NICARAGUA', 'PAKISTAN', 'STOCKHOLM', 'VENEZUELA', 'ALABAMA', 'BAHAMAS',
        'COLORADO', 'JAKARTA', 'MALAYSIA', 'MELBOURNE', 'PHOENIX', 'SEATTLE', 'TORONTO'
    ],
    science: [
        'ASTEROID', 'BACTERIA', 'CATALYST', 'DINOSAUR', 'ELECTRON', 'FORMULA',
        'GRAVITY', 'HYDROGEN', 'ISOTOPE', 'MOLECULE', 'NEUTRON', 'ORGANISM',
        'PHOTON', 'RADIATION', 'SPECTRUM', 'TITANIUM', 'URANIUM', 'VELOCITY',
        'CALCIUM', 'CARBON', 'CELSIUS', 'CHLORINE', 'CHROMOSOME', 'COMPOUND',
        'CRYSTAL', 'DENSITY', 'ECLIPSE', 'ELEMENT', 'ENZYME', 'EROSION', 'FOSSIL',
        'FRACTURE', 'GALAXY', 'GENOME', 'HABITAT', 'HELIUM', 'KINETIC', 'MAGNET',
        'MINERAL', 'NUCLEUS', 'ORGANIC', 'OSMOSIS', 'OXYGEN', 'PLASMA', 'POLYMER',
        'PROTEIN', 'PROTON', 'QUANTUM', 'REACTOR', 'SILICON', 'SOLVENT', 'THERMAL',
        'VOLTAGE', 'LITHIUM', 'NEPTUNE', 'NITROGEN', 'NUCLEUS', 'SULFUR', 'SYNAPSE'
    ],
    sports: [
        'BASEBALL', 'BASKETBALL', 'BOWLING', 'CHAMPION', 'CRICKET', 'CYCLING',
        'DEFENSE', 'DRIBBLE', 'FOOTBALL', 'FREESTYLE', 'GOALIE', 'GYMNASTICS',
        'HANDBALL', 'HOCKEY', 'HURDLES', 'JAVELIN', 'KAYAKING', 'LACROSSE',
        'MARATHON', 'OFFENSE', 'PENALTY', 'PLAYOFF', 'QUARTER', 'REFEREE',
        'ROWING', 'RUGBY', 'SAILING', 'SKATING', 'SOCCER', 'SOFTBALL', 'SPRINT',
        'STADIUM', 'SURFING', 'TENNIS', 'TIMEOUT', 'TROPHY', 'UMPIRE', 'VICTORY',
        'VOLLEY', 'WRESTLE', 'ARCHERY', 'ATHLETE', 'BATTING', 'CAPTAIN', 'FENCING',
        'FITNESS', 'HALFTIME', 'INNINGS', 'PITCHER', 'REBOUND', 'SCOREBOARD',
        'SHUTOUT', 'STRIKER', 'SWIMMING', 'TACTICS', 'TRIATHLON', 'TURNOVER'
    ],
    music: [
        'ACOUSTIC', 'AMPLIFIER', 'BARITONE', 'CADENCE', 'CLARINET', 'CONCERT',
        'DRUMBEAT', 'ENSEMBLE', 'FALSETTO', 'HARMONY', 'JUKEBOX', 'KARAOKE',
        'MANDOLIN', 'MELODY', 'OCTAVE', 'ORCHESTRA', 'PIANIST', 'PLAYLIST',
        'RHYTHM', 'SERENADE', 'SOPRANO', 'SYMPHONY', 'TAMBOURINE', 'TEMPO',
        'TROMBONE', 'TRUMPET', 'UKULELE', 'VIBRATO', 'XYLOPHONE', 'ANTHEM',
        'BALLAD', 'BASSOON', 'CELLIST', 'CHORUS', 'CYMBAL', 'DRUMMER', 'FIDDLER',
        'FLUTIST', 'GUITAR', 'LYRICAL', 'MAESTRO', 'MUSICAL', 'QUARTET', 'RECITAL',
        'SOLOIST', 'SONATA', 'STACCATO', 'STRINGS', 'TUNING', 'VOCALIST'
    ],
    everyday: [
        'ADVENTURE', 'BLANKET', 'CALENDAR', 'CURTAIN', 'DIAMOND', 'ELEVATOR',
        'FOUNTAIN', 'GARDEN', 'HAMMOCK', 'ISLAND', 'JOURNEY', 'KITCHEN',
        'LANTERN', 'MAGAZINE', 'NAPKIN', 'ORIGAMI', 'PACKAGE', 'QUARTER',
        'RAINBOW', 'SANDCASTLE', 'TREASURE', 'UMBRELLA', 'VACATION', 'WHISPER',
        'ANTIQUE', 'BALLOON', 'BISCUIT', 'BLANKET', 'CANDLE', 'CHIMNEY',
        'COMPASS', 'COSTUME', 'DOORBELL', 'ENVELOPE', 'FEATHER', 'GLASSES',
        'HAMMOCK', 'LANTERN', 'LIBRARY', 'MIRROR', 'NECKLACE', 'PAINTING',
        'PERFUME', 'PHOTOGRAPH', 'PILLOW', 'POCKET', 'PUZZLE', 'RIBBON',
        'SHELTER', 'SLIPPER', 'STAIRCASE', 'SUITCASE', 'SUNRISE', 'SWEATER',
        'TELESCOPE', 'THIMBLE', 'VILLAGE', 'WARDROBE', 'WINDOW', 'WREATH'
    ]
};

// Flatten all words into a single list
const ALL_WORDS = Object.values(WORD_CATEGORIES).flat();

// Hangman ASCII art stages (plain ASCII for consistent alignment)
const HANGMAN_STAGES = [
    // Stage 0 - Empty
    `  +---+
  |
  |
  |
  |
=====`,
    // Stage 1 - Head
    `  +---+
  |   O
  |
  |
  |
=====`,
    // Stage 2 - Body
    `  +---+
  |   O
  |   |
  |
  |
=====`,
    // Stage 3 - Left arm
    `  +---+
  |   O
  |  /|
  |
  |
=====`,
    // Stage 4 - Right arm
    `  +---+
  |   O
  |  /|\\
  |
  |
=====`,
    // Stage 5 - Left leg
    `  +---+
  |   O
  |  /|\\
  |  /
  |
=====`,
    // Stage 6 - Right leg (Game Over)
    `  +---+
  |   O
  |  /|\\
  |  / \\
  |
=====`
];

// Game state
let gameContainer = null;
let currentWord = '';
let guessedLetters = [];
let wrongGuesses = 0;
let maxWrongGuesses = 6;
let gameActive = true;
let recentWords = []; // Track recently used words to prevent repeats
let historyLoaded = false;
let showHelp = false;
let helpChecked = false;

const HELP_KEY = 'game_help_seen_hangman';
const INSTRUCTIONS = 'Guess the hidden word one letter at a time. Click letters on the keyboard or type to guess. You can also guess the full word using the input field. You have 6 wrong guesses before the game ends.';

const HANGMAN_STORAGE_KEY = 'hangman_recent_words';
const MAX_HISTORY = Math.floor(ALL_WORDS.length * 0.4);

/**
 * Load recent word history from storage
 */
async function loadHistory() {
    if (historyLoaded) return;
    try {
        const data = await chrome.storage.local.get(HANGMAN_STORAGE_KEY);
        if (data[HANGMAN_STORAGE_KEY] && Array.isArray(data[HANGMAN_STORAGE_KEY])) {
            recentWords = data[HANGMAN_STORAGE_KEY];
        }
    } catch (e) { /* ignore storage errors */ }
    historyLoaded = true;
}

/**
 * Save recent word history to storage
 */
function saveHistory() {
    try {
        chrome.storage.local.set({ [HANGMAN_STORAGE_KEY]: recentWords });
    } catch (e) { /* ignore storage errors */ }
}

/**
 * Pick a word that hasn't been used recently
 */
function pickWord() {
    // Filter out recently used words
    let available = ALL_WORDS.filter(w => !recentWords.includes(w));

    // If somehow exhausted, reset history
    if (available.length === 0) {
        recentWords = [];
        available = ALL_WORDS;
    }

    const word = available[Math.floor(Math.random() * available.length)];
    recentWords.push(word);

    // Trim history if it gets too long
    if (recentWords.length > MAX_HISTORY) {
        recentWords = recentWords.slice(-MAX_HISTORY);
    }

    saveHistory();
    return word;
}

/**
 * Initialize the game
 */
export async function initHangman(container) {
    gameContainer = container;
    await loadHistory();
    if (!helpChecked) {
        helpChecked = true;
        try {
            const data = await chrome.storage.local.get(HELP_KEY);
            showHelp = !data[HELP_KEY];
        } catch (e) { showHelp = true; }
    }
    resetGame();
    renderGame();
}

/**
 * Reset game state
 */
function resetGame() {
    currentWord = pickWord();
    guessedLetters = [];
    wrongGuesses = 0;
    gameActive = true;
}

/**
 * Render the game
 */
function renderGame() {
    if (!gameContainer) return;

    const wordDisplay = getWordDisplay();
    const status = getGameStatus();
    const hangmanArt = HANGMAN_STAGES[wrongGuesses];

    const helpBtn = `<button class="game-help-btn" id="hangman-help-btn" aria-label="Instructions">?</button>`;
    const header = `<div class="game-header"><div class="game-status">${status}</div>${helpBtn}</div>`;
    const overlay = showHelp ? `
        <div class="game-instructions">
            <div class="game-instructions-title">How to Play</div>
            <div class="game-instructions-text">${INSTRUCTIONS}</div>
            <button class="game-reset-btn game-instructions-dismiss" id="hangman-dismiss">Got it</button>
        </div>` : '';

    gameContainer.innerHTML = `
        <div class="hangman-game">
            ${header}
            <div class="hangman-art">${hangmanArt}</div>
            <div class="hangman-word">${wordDisplay}</div>
            <div class="hangman-info">
                <div>Guessed: ${guessedLetters.join(', ') || 'None'}</div>
                <div>Remaining: ${maxWrongGuesses - wrongGuesses} tries</div>
            </div>
            <div class="hangman-guess-input">
                <input type="text" id="word-guess-input" placeholder="Guess word..." maxlength="20" ${!gameActive ? 'disabled' : ''}>
                <button id="word-guess-btn" ${!gameActive ? 'disabled' : ''}>Guess</button>
            </div>
            <div class="hangman-keyboard">
                ${renderKeyboard()}
            </div>
            <button class="game-reset-btn" id="hangman-reset">New Game</button>
            ${overlay}
        </div>
    `;

    // Add event listeners
    const keys = gameContainer.querySelectorAll('.hangman-key');
    keys.forEach(key => {
        key.addEventListener('click', () => handleLetterGuess(key.dataset.letter));
    });

    const resetBtn = gameContainer.querySelector('#hangman-reset');
    if (resetBtn) {
        resetBtn.addEventListener('click', () => {
            resetGame();
            renderGame();
        });
    }

    // Word guess input handlers
    const wordInput = gameContainer.querySelector('#word-guess-input');
    const wordBtn = gameContainer.querySelector('#word-guess-btn');

    if (wordBtn && wordInput) {
        wordBtn.addEventListener('click', () => handleWordGuess(wordInput.value));
        wordInput.addEventListener('keypress', (e) => {
            if (e.key === 'Enter') {
                handleWordGuess(wordInput.value);
            }
        });
    }

    const helpBtnEl = gameContainer.querySelector('#hangman-help-btn');
    if (helpBtnEl) {
        helpBtnEl.addEventListener('click', () => { showHelp = !showHelp; renderGame(); });
    }

    const dismissEl = gameContainer.querySelector('#hangman-dismiss');
    if (dismissEl) {
        dismissEl.addEventListener('click', () => { showHelp = false; try { chrome.storage.local.set({ [HELP_KEY]: true }); } catch(e){} renderGame(); });
    }
}

/**
 * Handle word guess
 */
function handleWordGuess(guess) {
    if (!gameActive || !guess) return;

    const guessUpper = guess.trim().toUpperCase();

    if (guessUpper === currentWord) {
        // Correct! Win the game
        gameActive = false;
        // Fill in all letters
        currentWord.split('').forEach(letter => {
            if (!guessedLetters.includes(letter)) {
                guessedLetters.push(letter);
            }
        });
    } else {
        // Wrong! Instant loss
        wrongGuesses = maxWrongGuesses;
        gameActive = false;
    }

    renderGame();
}

/**
 * Render keyboard
 */
function renderKeyboard() {
    const alphabet = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';
    let html = '';

    for (let i = 0; i < alphabet.length; i++) {
        const letter = alphabet[i];
        const disabled = guessedLetters.includes(letter) || !gameActive;
        html += `<button class="hangman-key ${disabled ? 'disabled' : ''}"
                        data-letter="${letter}"
                        ${disabled ? 'disabled' : ''}>${letter}</button>`;
    }

    return html;
}

/**
 * Get word display with blanks
 */
function getWordDisplay() {
    return currentWord
        .split('')
        .map(letter => guessedLetters.includes(letter) ? letter : '_')
        .join(' ');
}

/**
 * Handle letter guess
 */
function handleLetterGuess(letter) {
    if (!gameActive || guessedLetters.includes(letter)) return;

    guessedLetters.push(letter);

    if (!currentWord.includes(letter)) {
        wrongGuesses++;
    }

    // Check win/loss
    if (wrongGuesses >= maxWrongGuesses) {
        gameActive = false;
    } else if (currentWord.split('').every(l => guessedLetters.includes(l))) {
        gameActive = false;
    }

    renderGame();
}

/**
 * Get game status message
 */
function getGameStatus() {
    if (!gameActive) {
        if (wrongGuesses >= maxWrongGuesses) {
            return `Game Over! The word was: ${currentWord}`;
        }
        return 'You Win! 🎉';
    }
    return 'Guess the word!';
}
