/**
 * 2048 Game Module
 */

const SIZE = 4;

// Game state
let gameContainer = null;
let grid = [];
let score = 0;
let bestScore = 0;
let gameOver = false;
let hasWon = false;
let keepPlaying = false;
let showHelp = true;

const INSTRUCTIONS = 'Use arrow keys to slide all tiles. When two tiles with the same number collide, they merge into one. Reach the 2048 tile to win! Swipe gestures also work.';

/**
 * Initialize the game
 */
export function init2048(container) {
    gameContainer = container;
    resetGame();
    renderGame();
}

/**
 * Reset game state
 */
function resetGame() {
    grid = Array.from({ length: SIZE }, () => Array(SIZE).fill(0));
    score = 0;
    gameOver = false;
    hasWon = false;
    keepPlaying = false;
    spawnTile();
    spawnTile();
}

/**
 * Spawn a new tile (2 or 4) in a random empty cell
 */
function spawnTile() {
    const empty = [];
    for (let r = 0; r < SIZE; r++) {
        for (let c = 0; c < SIZE; c++) {
            if (grid[r][c] === 0) empty.push({ r, c });
        }
    }
    if (empty.length === 0) return;
    const cell = empty[Math.floor(Math.random() * empty.length)];
    grid[cell.r][cell.c] = Math.random() < 0.9 ? 2 : 4;
}

/**
 * Slide and merge a single row to the left
 * Returns { row, mergeScore }
 */
function slideRow(row) {
    // Remove zeros
    let filtered = row.filter(v => v !== 0);
    let mergeScore = 0;

    // Merge adjacent equal tiles
    for (let i = 0; i < filtered.length - 1; i++) {
        if (filtered[i] === filtered[i + 1]) {
            filtered[i] *= 2;
            mergeScore += filtered[i];
            filtered[i + 1] = 0;
            i++; // skip merged tile
        }
    }

    // Remove zeros again after merge
    filtered = filtered.filter(v => v !== 0);

    // Pad with zeros to original length
    while (filtered.length < SIZE) filtered.push(0);

    return { row: filtered, mergeScore };
}

/**
 * Rotate grid 90 degrees clockwise
 */
function rotateGrid(g) {
    const rotated = Array.from({ length: SIZE }, () => Array(SIZE).fill(0));
    for (let r = 0; r < SIZE; r++) {
        for (let c = 0; c < SIZE; c++) {
            rotated[c][SIZE - 1 - r] = g[r][c];
        }
    }
    return rotated;
}

/**
 * Move tiles in a direction
 * Returns true if anything moved
 */
function move(direction) {
    if (gameOver) return false;

    // Normalize: rotate so we always slide left, then rotate back
    let rotations = 0;
    switch (direction) {
        case 'left': rotations = 0; break;
        case 'up': rotations = 1; break;
        case 'right': rotations = 2; break;
        case 'down': rotations = 3; break;
    }

    let g = grid.map(row => [...row]);

    // Rotate to normalize
    for (let i = 0; i < rotations; i++) {
        g = rotateGrid(g);
    }

    // Slide each row left
    let moved = false;
    let moveScore = 0;
    for (let r = 0; r < SIZE; r++) {
        const { row, mergeScore } = slideRow(g[r]);
        if (row.some((v, i) => v !== g[r][i])) moved = true;
        g[r] = row;
        moveScore += mergeScore;
    }

    // Rotate back
    for (let i = 0; i < (4 - rotations) % 4; i++) {
        g = rotateGrid(g);
    }

    if (!moved) return false;

    grid = g;
    score += moveScore;
    if (score > bestScore) bestScore = score;

    // Check for 2048 win
    if (!hasWon && !keepPlaying) {
        for (let r = 0; r < SIZE; r++) {
            for (let c = 0; c < SIZE; c++) {
                if (grid[r][c] === 2048) {
                    hasWon = true;
                }
            }
        }
    }

    spawnTile();

    // Check game over
    if (!canMove()) {
        gameOver = true;
    }

    return true;
}

/**
 * Check if any move is possible
 */
function canMove() {
    for (let r = 0; r < SIZE; r++) {
        for (let c = 0; c < SIZE; c++) {
            if (grid[r][c] === 0) return true;
            if (c < SIZE - 1 && grid[r][c] === grid[r][c + 1]) return true;
            if (r < SIZE - 1 && grid[r][c] === grid[r + 1][c]) return true;
        }
    }
    return false;
}

/**
 * Get CSS class for tile value
 */
function tileClass(value) {
    if (value === 0) return 'tile-empty';
    if (value <= 2048) return `tile-${value}`;
    return 'tile-super';
}

/**
 * Render the game
 */
function renderGame() {
    if (!gameContainer) return;

    let statusText = 'Slide tiles to merge!';
    if (gameOver) statusText = 'Game Over';
    if (hasWon && !keepPlaying) statusText = 'You Win!';

    let gridHtml = '';
    for (let r = 0; r < SIZE; r++) {
        for (let c = 0; c < SIZE; c++) {
            const val = grid[r][c];
            const display = val || '';
            gridHtml += `<div class="g2048-cell ${tileClass(val)}">${display}</div>`;
        }
    }

    let overlayHtml = '';
    if (gameOver) {
        overlayHtml = `<div class="g2048-overlay"><span>Game Over</span></div>`;
    } else if (hasWon && !keepPlaying) {
        overlayHtml = `
            <div class="g2048-overlay g2048-overlay-win">
                <span>2048!</span>
                <button class="game-reset-btn" id="g2048-continue">Keep Going</button>
            </div>`;
    }

    const helpBtn = `<button class="game-help-btn" id="g2048-help-btn" aria-label="Instructions">?</button>`;
    const header = `<div class="game-header"><div class="game-status">${statusText}</div>${helpBtn}</div>`;
    const overlay = showHelp ? `
        <div class="game-instructions">
            <div class="game-instructions-title">How to Play</div>
            <div class="game-instructions-text">${INSTRUCTIONS}</div>
            <button class="game-reset-btn game-instructions-dismiss" id="g2048-dismiss">Got it</button>
        </div>` : '';

    gameContainer.innerHTML = `
        <div class="g2048-game" tabindex="0" id="g2048-container">
            ${header}
            <div class="g2048-scores">
                <div class="g2048-score-box">
                    <div class="g2048-score-label">Score</div>
                    <div class="g2048-score-value">${score}</div>
                </div>
                <div class="g2048-score-box">
                    <div class="g2048-score-label">Best</div>
                    <div class="g2048-score-value">${bestScore}</div>
                </div>
            </div>
            <div class="g2048-board">
                ${gridHtml}
                ${overlayHtml}
            </div>
            <div class="g2048-hint">Use arrow keys to slide</div>
            <button class="game-reset-btn" id="g2048-reset">New Game</button>
            ${overlay}
        </div>
    `;

    // Focus for keyboard input
    const containerEl = gameContainer.querySelector('#g2048-container');
    if (containerEl) {
        containerEl.addEventListener('keydown', handleKeyDown);
        containerEl.focus();
    }

    // Touch support
    setupTouch(containerEl);

    const resetBtn = gameContainer.querySelector('#g2048-reset');
    if (resetBtn) {
        resetBtn.addEventListener('click', (e) => {
            e.stopPropagation();
            resetGame();
            renderGame();
        });
    }

    const continueBtn = gameContainer.querySelector('#g2048-continue');
    if (continueBtn) {
        continueBtn.addEventListener('click', (e) => {
            e.stopPropagation();
            keepPlaying = true;
            renderGame();
        });
    }

    const helpBtnEl = gameContainer.querySelector('#g2048-help-btn');
    if (helpBtnEl) {
        helpBtnEl.addEventListener('click', (e) => {
            e.stopPropagation();
            showHelp = !showHelp;
            renderGame();
        });
    }

    const dismissEl = gameContainer.querySelector('#g2048-dismiss');
    if (dismissEl) {
        dismissEl.addEventListener('click', (e) => {
            e.stopPropagation();
            showHelp = false;
            renderGame();
        });
    }
}

/**
 * Handle arrow key input
 */
function handleKeyDown(e) {
    const directions = {
        ArrowLeft: 'left',
        ArrowRight: 'right',
        ArrowUp: 'up',
        ArrowDown: 'down'
    };

    const dir = directions[e.key];
    if (!dir) return;

    e.preventDefault();

    if (hasWon && !keepPlaying) return;

    if (move(dir)) {
        renderGame();
    }
}

/**
 * Touch/swipe support
 */
function setupTouch(el) {
    if (!el) return;
    let startX = 0, startY = 0;

    el.addEventListener('touchstart', (e) => {
        startX = e.touches[0].clientX;
        startY = e.touches[0].clientY;
    }, { passive: true });

    el.addEventListener('touchend', (e) => {
        const dx = e.changedTouches[0].clientX - startX;
        const dy = e.changedTouches[0].clientY - startY;
        const absDx = Math.abs(dx);
        const absDy = Math.abs(dy);

        if (Math.max(absDx, absDy) < 30) return; // too short

        let dir;
        if (absDx > absDy) {
            dir = dx > 0 ? 'right' : 'left';
        } else {
            dir = dy > 0 ? 'down' : 'up';
        }

        if (hasWon && !keepPlaying) return;

        if (move(dir)) {
            renderGame();
        }
    }, { passive: true });
}
