/**
 * JottNote Editor Service
 * Handles editor functionality, auto-save, and content management
 */

import { saveNote, getNote } from './storage.js';

let saveTimeout = null;
const SAVE_DELAY = 100; // Debounce delay in ms

let currentNote = null;
let onContentChangeCallback = null;

/**
 * Initialize editor with a note
 * @param {Object} note - Note to load
 * @param {HTMLTextAreaElement} editor - Editor element
 */
export function initEditor(note, editor) {
    currentNote = note;
    editor.value = note.content || '';
    positionCursorAtEnd(editor);
}

/**
 * Position cursor at end of editor
 * @param {HTMLTextAreaElement} editor
 */
export function positionCursorAtEnd(editor) {
    editor.selectionStart = editor.selectionEnd = editor.value.length;
    editor.focus();
}

/**
 * Position cursor at start of editor
 * @param {HTMLTextAreaElement} editor
 */
export function positionCursorAtStart(editor) {
    editor.selectionStart = editor.selectionEnd = 0;
    editor.focus();
}

/**
 * Handle content change with debounced auto-save
 * @param {HTMLTextAreaElement} editor
 * @param {Function} onSave - Callback after save
 */
export function handleContentChange(editor, onSave) {
    if (!currentNote) return;

    // Update note content
    currentNote.content = editor.value;

    // Debounced save
    if (saveTimeout) {
        clearTimeout(saveTimeout);
    }

    saveTimeout = setTimeout(async () => {
        try {
            const saved = await saveNote(currentNote);
            currentNote = saved;
            if (onSave) onSave(saved);
        } catch (error) {
            console.error('Auto-save failed:', error);
        }
    }, SAVE_DELAY);
}

/**
 * Force immediate save
 * @returns {Promise<Object>} Saved note
 */
export async function forceSave(editor) {
    if (saveTimeout) {
        clearTimeout(saveTimeout);
        saveTimeout = null;
    }

    if (currentNote) {
        // Sync latest textarea value before saving
        if (editor && editor.value !== undefined) {
            currentNote.content = editor.value;
        }
        const saved = await saveNote(currentNote);
        currentNote = saved;
        return saved;
    }
    return null;
}

/**
 * Get current note content
 * @returns {string}
 */
export function getCurrentContent() {
    return currentNote?.content || '';
}

/**
 * Get current note
 * @returns {Object|null}
 */
export function getCurrentNote() {
    return currentNote;
}

/**
 * Set current note
 * @param {Object} note
 */
export function setCurrentNote(note) {
    currentNote = note;
}

/**
 * Get first line of content (for keyword detection)
 * @param {string} content
 * @returns {string}
 */
export function getFirstLine(content) {
    const lines = content.split('\n');
    return lines[0] || '';
}

/**
 * Check if note is empty
 * @returns {boolean}
 */
export function isNoteEmpty() {
    return !currentNote?.content?.trim();
}

/**
 * Copy content to clipboard (smart copy)
 * @param {HTMLTextAreaElement} editor
 * @param {Object} options
 * @returns {Promise<string>} Copied text
 */
export async function smartCopy(editor, options = {}) {
    const { skipKeywords = true, skipCheckTriggers = true, checkTrigger = '/x' } = options;

    let text = '';

    // Check if there's a selection
    if (editor.selectionStart !== editor.selectionEnd) {
        text = editor.value.substring(editor.selectionStart, editor.selectionEnd);
    } else {
        // Copy whole note
        text = editor.value;
    }

    // Process text based on options
    if (skipKeywords && text) {
        const lines = text.split('\n');
        const firstLine = lines[0].toLowerCase().trim();
        const keywords = ['list', 'math', 'sum', 'avg', 'count', 'code', 'timer'];

        // Check if first line is just a keyword
        for (const kw of keywords) {
            if (firstLine === kw || firstLine.startsWith(kw + ':')) {
                // Extract title if present
                const colonIndex = lines[0].indexOf(':');
                if (colonIndex > 0) {
                    lines[0] = lines[0].substring(colonIndex + 1).trim();
                    if (!lines[0]) {
                        lines.shift();
                    }
                } else {
                    lines.shift();
                }
                break;
            }
        }
        text = lines.join('\n');
    }

    if (skipCheckTriggers && text) {
        const trigger = checkTrigger.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
        const regex = new RegExp(trigger + '\\s*$', 'gm');
        text = text.replace(regex, '');
    }

    // Copy to clipboard
    try {
        await navigator.clipboard.writeText(text);
        return text;
    } catch (error) {
        console.error('Copy failed:', error);
        return '';
    }
}

/**
 * Smart paste (strip formatting)
 * @param {HTMLTextAreaElement} editor
 * @param {string} text
 * @param {Object} options
 */
export function smartPaste(editor, text, options = {}) {
    const {
        stripLeadingSpaces = true,
        stripListNumbers = true,
        stripBullets = true,
        stripMarkdown = false,
        stripEmptyLines = false
    } = options;

    let processed = text;

    // Strip leading/trailing whitespace per line
    if (stripLeadingSpaces) {
        processed = processed.split('\n').map(line => line.trim()).join('\n');
    }

    // Strip list numbers (1. 2. 3.)
    if (stripListNumbers) {
        processed = processed.replace(/^\s*\d+\.\s+/gm, '');
    }

    // Strip bullets (-, *, •)
    if (stripBullets) {
        processed = processed.replace(/^\s*[-*•]\s+/gm, '');
    }

    // Strip basic markdown
    if (stripMarkdown) {
        processed = processed
            .replace(/\*\*(.+?)\*\*/g, '$1')  // Bold
            .replace(/__(.+?)__/g, '$1')       // Underline  
            .replace(/\*(.+?)\*/g, '$1')       // Italic
            .replace(/_(.+?)_/g, '$1')         // Italic
            .replace(/~~(.+?)~~/g, '$1')       // Strikethrough
            .replace(/^#+\s+/gm, '');          // Headers
    }

    // Strip empty lines
    if (stripEmptyLines) {
        processed = processed.split('\n').filter(line => line.trim()).join('\n');
    }

    // Insert at cursor position
    const start = editor.selectionStart;
    const end = editor.selectionEnd;
    const before = editor.value.substring(0, start);
    const after = editor.value.substring(end);

    editor.value = before + processed + after;
    editor.selectionStart = editor.selectionEnd = start + processed.length;

    // Trigger change event
    editor.dispatchEvent(new Event('input', { bubbles: true }));
}
