/**
 * JottNote Dictionary & Thesaurus Mode
 * Fetches data via the Free Dictionary API (https://dictionaryapi.dev/)
 */

export async function fetchWordData(word) {
    if (!word || !word.trim()) {
        throw new Error("Please provide a word to look up.");
    }

    const trimmedWord = word.trim();
    const url = `https://api.dictionaryapi.dev/api/v2/entries/en/${encodeURIComponent(trimmedWord)}`;

    try {
        const res = await fetch(url);

        if (!res.ok) {
            if (res.status === 404) {
                throw new Error(`Word "${trimmedWord}" not found in the dictionary.`);
            }
            throw new Error(`Error looking up "${trimmedWord}": ${res.statusText}`);
        }

        const data = await res.json();
        if (!data || data.length === 0) {
            throw new Error(`No definitions found for "${trimmedWord}".`);
        }

        return parseDictionaryData(data[0]);

    } catch (error) {
        throw error; // Re-throw to be handled by the UI
    }
}

/**
 * Extracts the most relevant parts from the Free Dictionary API response
 */
function parseDictionaryData(entry) {
    const word = entry.word || '';

    // Find the first valid phonetic text
    let phonetic = '';
    if (entry.phonetic) {
        phonetic = entry.phonetic;
    } else if (entry.phonetics && entry.phonetics.length > 0) {
        const found = entry.phonetics.find(p => p.text);
        if (found) phonetic = found.text;
    }

    const meanings = [];
    const allSynonyms = new Set();
    const allAntonyms = new Set();

    if (entry.meanings && entry.meanings.length > 0) {
        entry.meanings.forEach(meaning => {
            const partOfSpeech = meaning.partOfSpeech || '';

            // Extract top 2 definitions for this part of speech
            const definitions = [];
            if (meaning.definitions && meaning.definitions.length > 0) {
                for (let i = 0; i < Math.min(2, meaning.definitions.length); i++) {
                    const def = meaning.definitions[i];
                    definitions.push({
                        definition: def.definition,
                        example: def.example || null
                    });
                }
            }

            // Collect synonyms for this part of speech
            if (meaning.synonyms && meaning.synonyms.length > 0) {
                meaning.synonyms.forEach(s => allSynonyms.add(s));
            }

            // Collect antonyms for this part of speech
            if (meaning.antonyms && meaning.antonyms.length > 0) {
                meaning.antonyms.forEach(a => allAntonyms.add(a));
            }

            meanings.push({
                partOfSpeech,
                definitions
            });
        });
    }

    return {
        word,
        phonetic,
        meanings,
        synonyms: Array.from(allSynonyms),
        antonyms: Array.from(allAntonyms)
    };
}
