/**
 * JottNote Date Mode
 * Handles date calculations, countdowns, and date insertion
 */

const MONTH_NAMES = [
    'January', 'February', 'March', 'April', 'May', 'June',
    'July', 'August', 'September', 'October', 'November', 'December'
];
const MONTH_SHORT = [
    'Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun',
    'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'
];
const DAY_NAMES = ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'];
const DAY_SHORT = ['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat'];

/**
 * Parse a flexible date string into a Date object
 * Supports: "today", "tomorrow", "yesterday", "Dec 25", "Dec 25 2025",
 *           "2025-12-25", "12/25/2025", "December 25, 2026"
 */
function parseFlexDate(str) {
    if (!str) return null;
    const s = str.trim().toLowerCase();

    const now = new Date();
    const today = new Date(now.getFullYear(), now.getMonth(), now.getDate());

    if (s === 'today') return today;
    if (s === 'tomorrow') return new Date(today.getTime() + 86400000);
    if (s === 'yesterday') return new Date(today.getTime() - 86400000);

    // ISO format: 2025-12-25
    const isoMatch = s.match(/^(\d{4})-(\d{1,2})-(\d{1,2})$/);
    if (isoMatch) {
        return new Date(+isoMatch[1], +isoMatch[2] - 1, +isoMatch[3]);
    }

    // US format: 12/25/2025 or 12/25
    const usMatch = s.match(/^(\d{1,2})\/(\d{1,2})(?:\/(\d{2,4}))?$/);
    if (usMatch) {
        const year = usMatch[3] ? (usMatch[3].length === 2 ? 2000 + +usMatch[3] : +usMatch[3]) : now.getFullYear();
        return new Date(year, +usMatch[1] - 1, +usMatch[2]);
    }

    // Month name formats: "Dec 25", "Dec 25 2025", "Dec 25, 2025", "December 25, 2026"
    const monthNameMatch = s.match(/^([a-z]+)\s+(\d{1,2})(?:\s*,?\s*(\d{4}))?$/);
    if (monthNameMatch) {
        const monthIdx = findMonthIndex(monthNameMatch[1]);
        if (monthIdx !== -1) {
            const year = monthNameMatch[3] ? +monthNameMatch[3] : now.getFullYear();
            return new Date(year, monthIdx, +monthNameMatch[2]);
        }
    }

    // "25 Dec 2025" or "25 Dec"
    const dayFirstMatch = s.match(/^(\d{1,2})\s+([a-z]+)(?:\s+(\d{4}))?$/);
    if (dayFirstMatch) {
        const monthIdx = findMonthIndex(dayFirstMatch[2]);
        if (monthIdx !== -1) {
            const year = dayFirstMatch[3] ? +dayFirstMatch[3] : now.getFullYear();
            return new Date(year, monthIdx, +dayFirstMatch[1]);
        }
    }

    return null;
}

function findMonthIndex(name) {
    const lower = name.toLowerCase();
    for (let i = 0; i < 12; i++) {
        if (MONTH_NAMES[i].toLowerCase().startsWith(lower) && lower.length >= 3) return i;
    }
    return -1;
}

/**
 * Format a Date object according to the format string
 * Supported tokens: YYYY, MM, DD, MMM, MMMM, D, ddd, dddd
 */
export function formatDate(date, format) {
    if (!date || isNaN(date.getTime())) return 'Invalid date';

    const y = date.getFullYear();
    const m = date.getMonth();
    const d = date.getDate();
    const dow = date.getDay();

    return format
        .replace('YYYY', String(y))
        .replace('MMMM', MONTH_NAMES[m])
        .replace('MMM', MONTH_SHORT[m])
        .replace('MM', String(m + 1).padStart(2, '0'))
        .replace('dddd', DAY_NAMES[dow])
        .replace('ddd', DAY_SHORT[dow])
        .replace('DD', String(d).padStart(2, '0'))
        .replace(/\bD\b/, String(d));
}

/**
 * Format a Date with time
 */
function formatDateTime(date, dateFormat) {
    const hours = date.getHours();
    const minutes = String(date.getMinutes()).padStart(2, '0');
    const ampm = hours >= 12 ? 'PM' : 'AM';
    const h12 = hours % 12 || 12;
    return `${formatDate(date, dateFormat)} ${h12}:${minutes} ${ampm}`;
}

/**
 * Calculate difference in days between two dates (date2 - date1)
 */
function daysBetween(date1, date2) {
    const ms = date2.getTime() - date1.getTime();
    return Math.round(ms / 86400000);
}

/**
 * Format a duration in days into a human-readable string
 */
function formatDuration(totalDays) {
    const absDays = Math.abs(totalDays);
    const parts = [];

    if (absDays >= 365) {
        const years = Math.floor(absDays / 365);
        parts.push(`${years} year${years !== 1 ? 's' : ''}`);
    }
    if (absDays >= 30) {
        const months = Math.floor((absDays % 365) / 30);
        if (months > 0) parts.push(`${months} month${months !== 1 ? 's' : ''}`);
    }
    const weeks = Math.floor((absDays % 30) / 7);
    if (weeks > 0) parts.push(`${weeks} week${weeks !== 1 ? 's' : ''}`);
    const days = absDays % 7;
    if (days > 0 || parts.length === 0) parts.push(`${days} day${days !== 1 ? 's' : ''}`);

    return parts.join(', ');
}

/**
 * Add a duration to a date
 * @param {Date} date
 * @param {number} amount
 * @param {string} unit - days, weeks, months, years
 * @returns {Date}
 */
function addToDate(date, amount, unit) {
    const result = new Date(date);
    switch (unit) {
        case 'day':
        case 'days':
            result.setDate(result.getDate() + amount);
            break;
        case 'week':
        case 'weeks':
            result.setDate(result.getDate() + amount * 7);
            break;
        case 'month':
        case 'months':
            result.setMonth(result.getMonth() + amount);
            break;
        case 'year':
        case 'years':
            result.setFullYear(result.getFullYear() + amount);
            break;
    }
    return result;
}

/**
 * Process a single date expression line
 * @param {string} line
 * @param {string} dateFormat
 * @returns {Object|null} result object or null
 */
function processDateLine(line, dateFormat) {
    const trimmed = line.trim().toLowerCase();
    if (!trimmed) return null;

    // "today" - show today's date
    if (trimmed === 'today') {
        const now = new Date();
        const today = new Date(now.getFullYear(), now.getMonth(), now.getDate());
        return {
            type: 'date-result',
            formatted: formatDate(today, dateFormat),
            raw: today.toISOString().split('T')[0]
        };
    }

    // "now" - show current date + time
    if (trimmed === 'now') {
        const now = new Date();
        return {
            type: 'date-result',
            formatted: formatDateTime(now, dateFormat),
            raw: now.toISOString()
        };
    }

    // "day" - day of week
    if (trimmed === 'day') {
        const now = new Date();
        return {
            type: 'date-result',
            formatted: DAY_NAMES[now.getDay()],
            raw: DAY_NAMES[now.getDay()]
        };
    }

    // "month" - current month name
    if (trimmed === 'month') {
        const now = new Date();
        return {
            type: 'date-result',
            formatted: MONTH_NAMES[now.getMonth()],
            raw: MONTH_NAMES[now.getMonth()]
        };
    }

    // "year" - current year
    if (trimmed === 'year') {
        const now = new Date();
        return {
            type: 'date-result',
            formatted: String(now.getFullYear()),
            raw: String(now.getFullYear())
        };
    }

    // Date arithmetic: "<date> + <n> <unit>" or "<date> - <n> <unit>"
    const arithMatch = trimmed.match(/^(.+?)\s*([+-])\s*(\d+)\s*(days?|weeks?|months?|years?)$/);
    if (arithMatch) {
        const baseDate = parseFlexDate(arithMatch[1]);
        if (baseDate) {
            const sign = arithMatch[2] === '+' ? 1 : -1;
            const amount = parseInt(arithMatch[3]) * sign;
            const unit = arithMatch[4];
            const result = addToDate(baseDate, amount, unit);
            return {
                type: 'date-result',
                formatted: formatDate(result, dateFormat),
                raw: result.toISOString().split('T')[0]
            };
        }
    }

    // Difference: "<date1> - <date2>" (where both are dates, not "date - N units")
    // or "<date1> to <date2>"
    const diffMatch = trimmed.match(/^(.+?)\s+(?:to|-)\s+(.+?)$/);
    if (diffMatch) {
        // Make sure second part isn't a number+unit (already handled above)
        const secondPart = diffMatch[2].trim();
        if (!/^\d+\s*(days?|weeks?|months?|years?)$/.test(secondPart)) {
            const date1 = parseFlexDate(diffMatch[1]);
            const date2 = parseFlexDate(diffMatch[2]);
            if (date1 && date2) {
                const days = daysBetween(date1, date2);
                const absDays = Math.abs(days);
                const duration = formatDuration(days);
                let label;
                if (days > 0) {
                    label = `${absDays} days (${duration})`;
                } else if (days < 0) {
                    label = `${absDays} days ago (${duration})`;
                } else {
                    label = 'today';
                }
                return {
                    type: 'date-result',
                    formatted: label,
                    raw: String(days)
                };
            }
        }
    }

    // Try to parse as a standalone date
    const parsed = parseFlexDate(trimmed);
    if (parsed) {
        return {
            type: 'date-result',
            formatted: formatDate(parsed, dateFormat),
            raw: parsed.toISOString().split('T')[0]
        };
    }

    return null;
}

/**
 * Process date mode content
 * @param {string} content - Full note content
 * @param {string} dateFormat - User's preferred date format
 * @returns {Array<Object>} Processed lines with results
 */
export function processDateContent(content, dateFormat = 'MMM D, YYYY') {
    const lines = content.split('\n');
    const results = [];

    // Skip first line (keyword)
    for (let i = 1; i < lines.length; i++) {
        const line = lines[i];
        const trimmed = line.trim();

        if (!trimmed) {
            results.push({ type: 'empty', lineIndex: i });
            continue;
        }

        // Comment lines
        if (trimmed.startsWith('//')) {
            results.push({ type: 'comment', lineIndex: i, text: trimmed.substring(2).trim() });
            continue;
        }

        const result = processDateLine(line, dateFormat);
        if (result) {
            results.push({
                ...result,
                lineIndex: i,
                expression: trimmed
            });
        } else {
            results.push({ type: 'text', lineIndex: i, text: trimmed });
        }
    }

    return results;
}

/**
 * Get today's date formatted according to user preference
 * @param {string} dateFormat
 * @returns {string}
 */
export function getFormattedToday(dateFormat = 'MMM D, YYYY') {
    const now = new Date();
    const today = new Date(now.getFullYear(), now.getMonth(), now.getDate());
    return formatDate(today, dateFormat);
}
