/**
 * JottNote Code Mode Module
 * Syntax highlighting for code snippets
 */

// Supported languages with aliases
const LANGUAGES = {
    javascript: ['js', 'javascript', 'jsx'],
    typescript: ['ts', 'typescript', 'tsx'],
    python: ['py', 'python'],
    html: ['html', 'htm'],
    css: ['css'],
    swift: ['swift'],
    json: ['json'],
    sql: ['sql'],
    ruby: ['rb', 'ruby'],
    go: ['go', 'golang'],
    rust: ['rs', 'rust'],
    cpp: ['cpp', 'c++', 'c'],
    java: ['java'],
    php: ['php'],
    shell: ['sh', 'bash', 'shell', 'zsh']
};

// Token types for highlighting
const TOKEN_TYPES = {
    keyword: 'code-keyword',
    string: 'code-string',
    number: 'code-number',
    comment: 'code-comment',
    function: 'code-function',
    property: 'code-property',
    tag: 'code-tag',
    attribute: 'code-attribute',
    value: 'code-value',
    selector: 'code-selector',
    operator: 'code-operator',
    punctuation: 'code-punctuation'
};

// Language-specific keywords
const KEYWORDS = {
    javascript: ['const', 'let', 'var', 'function', 'return', 'if', 'else', 'for', 'while', 'class', 'extends', 'import', 'export', 'from', 'default', 'async', 'await', 'try', 'catch', 'throw', 'new', 'this', 'super', 'typeof', 'instanceof', 'true', 'false', 'null', 'undefined'],
    typescript: ['const', 'let', 'var', 'function', 'return', 'if', 'else', 'for', 'while', 'class', 'extends', 'import', 'export', 'from', 'default', 'async', 'await', 'try', 'catch', 'throw', 'new', 'this', 'super', 'typeof', 'instanceof', 'true', 'false', 'null', 'undefined', 'interface', 'type', 'enum', 'implements', 'private', 'public', 'protected'],
    python: ['def', 'class', 'if', 'elif', 'else', 'for', 'while', 'return', 'import', 'from', 'as', 'try', 'except', 'finally', 'with', 'lambda', 'yield', 'True', 'False', 'None', 'and', 'or', 'not', 'in', 'is', 'pass', 'break', 'continue', 'raise', 'async', 'await'],
    swift: ['func', 'var', 'let', 'class', 'struct', 'enum', 'protocol', 'extension', 'if', 'else', 'guard', 'switch', 'case', 'for', 'while', 'return', 'import', 'true', 'false', 'nil', 'self', 'super', 'init', 'deinit', 'private', 'public', 'internal', 'fileprivate', 'static', 'override', 'mutating', 'throws', 'try', 'catch', 'async', 'await'],
    go: ['func', 'var', 'const', 'type', 'struct', 'interface', 'if', 'else', 'for', 'range', 'return', 'import', 'package', 'defer', 'go', 'chan', 'select', 'case', 'default', 'break', 'continue', 'fallthrough', 'true', 'false', 'nil', 'map', 'make', 'new', 'append', 'len', 'cap'],
    rust: ['fn', 'let', 'mut', 'const', 'struct', 'enum', 'impl', 'trait', 'if', 'else', 'match', 'for', 'while', 'loop', 'return', 'use', 'mod', 'pub', 'self', 'super', 'crate', 'true', 'false', 'Some', 'None', 'Ok', 'Err', 'async', 'await', 'move', 'ref', 'where'],
    java: ['class', 'interface', 'extends', 'implements', 'public', 'private', 'protected', 'static', 'final', 'void', 'int', 'boolean', 'String', 'if', 'else', 'for', 'while', 'return', 'new', 'this', 'super', 'try', 'catch', 'throw', 'throws', 'import', 'package', 'true', 'false', 'null'],
    cpp: ['int', 'float', 'double', 'char', 'void', 'bool', 'class', 'struct', 'enum', 'if', 'else', 'for', 'while', 'return', 'include', 'define', 'using', 'namespace', 'public', 'private', 'protected', 'virtual', 'override', 'const', 'static', 'new', 'delete', 'true', 'false', 'nullptr', 'this', 'template', 'typename'],
    ruby: ['def', 'class', 'module', 'if', 'elsif', 'else', 'unless', 'case', 'when', 'for', 'while', 'until', 'do', 'end', 'return', 'require', 'include', 'extend', 'attr_accessor', 'attr_reader', 'attr_writer', 'self', 'super', 'true', 'false', 'nil', 'yield', 'begin', 'rescue', 'ensure', 'raise'],
    php: ['function', 'class', 'interface', 'extends', 'implements', 'public', 'private', 'protected', 'static', 'if', 'else', 'elseif', 'for', 'foreach', 'while', 'return', 'echo', 'print', 'require', 'include', 'use', 'namespace', 'new', 'this', 'self', 'true', 'false', 'null', 'try', 'catch', 'throw'],
    sql: ['SELECT', 'FROM', 'WHERE', 'AND', 'OR', 'INSERT', 'INTO', 'VALUES', 'UPDATE', 'SET', 'DELETE', 'CREATE', 'TABLE', 'DROP', 'ALTER', 'INDEX', 'JOIN', 'LEFT', 'RIGHT', 'INNER', 'OUTER', 'ON', 'AS', 'ORDER', 'BY', 'GROUP', 'HAVING', 'LIMIT', 'OFFSET', 'NULL', 'NOT', 'IN', 'LIKE', 'BETWEEN'],
    shell: ['if', 'then', 'else', 'elif', 'fi', 'for', 'while', 'do', 'done', 'case', 'esac', 'function', 'return', 'exit', 'echo', 'export', 'source', 'alias', 'cd', 'pwd', 'ls', 'rm', 'cp', 'mv', 'mkdir', 'chmod', 'chown', 'grep', 'sed', 'awk', 'cat', 'head', 'tail', 'true', 'false']
};

/**
 * Parse code keyword from first line
 * @param {string} firstLine - First line of content
 * @returns {Object|null} { language, displayName } or null
 */
export function parseCodeKeyword(firstLine) {
    const trimmed = firstLine.trim().toLowerCase();

    // Check for "code" exactly (show help)
    if (trimmed === 'code') {
        return { showHelp: true, showLanguages: false, language: null };
    }

    // Check for "code:help" or "code: help" (show languages reference)
    if (trimmed.match(/^code:\s*help$/)) {
        return { showHelp: false, showLanguages: true, language: null };
    }

    // Check for "code: language" format
    const match = trimmed.match(/^code:\s*(\w+)$/);
    if (match) {
        const lang = match[1];
        const normalizedLang = normalizeLanguage(lang);
        if (normalizedLang) {
            return { showHelp: false, showLanguages: false, language: normalizedLang, displayName: lang };
        }
    }

    return null;
}

/**
 * Get all supported languages with their aliases
 * @returns {Array<{name: string, aliases: string[]}>}
 */
export function getSupportedLanguages() {
    return Object.entries(LANGUAGES).map(([name, aliases]) => ({
        name,
        aliases
    }));
}

/**
 * Normalize language alias to standard name
 * @param {string} alias
 * @returns {string|null}
 */
function normalizeLanguage(alias) {
    const lower = alias.toLowerCase();
    for (const [lang, aliases] of Object.entries(LANGUAGES)) {
        if (aliases.includes(lower)) {
            return lang;
        }
    }
    // If not found, still return as-is for basic highlighting
    return lower;
}

/**
 * Highlight code content
 * @param {string} code - Code to highlight
 * @param {string} language - Language identifier
 * @returns {string} HTML with syntax highlighting
 */
export function highlightCode(code, language) {
    if (language === 'html' || language === 'htm') {
        return highlightHTML(code);
    } else if (language === 'css') {
        return highlightCSS(code);
    } else {
        return highlightGeneric(code, language);
    }
}

/**
 * Highlight HTML code
 */
function highlightHTML(code) {
    let result = escapeHtml(code);

    // Comments <!-- -->
    result = result.replace(/(&lt;!--[\s\S]*?--&gt;)/g, `<span class="${TOKEN_TYPES.comment}">$1</span>`);

    // DOCTYPE
    result = result.replace(/(&lt;!DOCTYPE\s+\w+&gt;)/gi, `<span class="${TOKEN_TYPES.tag}">$1</span>`);

    // Tags with attributes
    result = result.replace(/(&lt;\/?)([\w-]+)((?:\s+[\w-]+(?:=(?:&quot;[^&]*&quot;|&#39;[^&]*&#39;|[^\s&gt;]+))?)*)\s*(\/?&gt;)/g,
        (match, open, tag, attrs, close) => {
            let highlighted = `<span class="${TOKEN_TYPES.punctuation}">${open}</span>`;
            highlighted += `<span class="${TOKEN_TYPES.tag}">${tag}</span>`;

            if (attrs) {
                // Highlight attributes
                attrs = attrs.replace(/([\w-]+)(=)(&quot;|&#39;)([^&]*)(&quot;|&#39;)/g,
                    `<span class="${TOKEN_TYPES.attribute}">$1</span><span class="${TOKEN_TYPES.punctuation}">$2$3</span><span class="${TOKEN_TYPES.string}">$4</span><span class="${TOKEN_TYPES.punctuation}">$5</span>`
                );
                attrs = attrs.replace(/([\w-]+)(=)([^\s&]+)/g,
                    `<span class="${TOKEN_TYPES.attribute}">$1</span><span class="${TOKEN_TYPES.punctuation}">$2</span><span class="${TOKEN_TYPES.value}">$3</span>`
                );
                highlighted += attrs;
            }

            highlighted += `<span class="${TOKEN_TYPES.punctuation}">${close}</span>`;
            return highlighted;
        }
    );

    return result;
}

/**
 * Highlight CSS code
 */
function highlightCSS(code) {
    let result = escapeHtml(code);

    // Comments /* */
    result = result.replace(/(\/\*[\s\S]*?\*\/)/g, `<span class="${TOKEN_TYPES.comment}">$1</span>`);

    // Selectors (before {)
    result = result.replace(/^([^{]+)(\{)/gm, (match, selector, brace) => {
        const highlightedSelector = selector.replace(/([.#]?[\w-]+)/g, `<span class="${TOKEN_TYPES.selector}">$1</span>`);
        return `${highlightedSelector}<span class="${TOKEN_TYPES.punctuation}">${brace}</span>`;
    });

    // Properties and values
    result = result.replace(/([\w-]+)(\s*:\s*)([^;{}]+)(;?)/g, (match, prop, colon, value, semi) => {
        return `<span class="${TOKEN_TYPES.property}">${prop}</span><span class="${TOKEN_TYPES.punctuation}">${colon}</span><span class="${TOKEN_TYPES.value}">${value}</span><span class="${TOKEN_TYPES.punctuation}">${semi}</span>`;
    });

    // Closing braces
    result = result.replace(/(\})/g, `<span class="${TOKEN_TYPES.punctuation}">$1</span>`);

    return result;
}

/**
 * Highlight generic code (JS, Python, etc.)
 */
function highlightGeneric(code, language) {
    let result = escapeHtml(code);
    const keywords = KEYWORDS[language] || KEYWORDS.javascript;

    // Comments
    // Single line //
    result = result.replace(/(\/\/[^\n]*)/g, `<span class="${TOKEN_TYPES.comment}">$1</span>`);
    // Single line #
    result = result.replace(/(^|\n)(#[^\n]*)/g, `$1<span class="${TOKEN_TYPES.comment}">$2</span>`);
    // Multi-line /* */
    result = result.replace(/(\/\*[\s\S]*?\*\/)/g, `<span class="${TOKEN_TYPES.comment}">$1</span>`);

    // Strings (double and single quotes)
    result = result.replace(/(&quot;[^&]*&quot;)/g, `<span class="${TOKEN_TYPES.string}">$1</span>`);
    result = result.replace(/(&#39;[^&]*&#39;)/g, `<span class="${TOKEN_TYPES.string}">$1</span>`);
    // Template literals
    result = result.replace(/(`[^`]*`)/g, `<span class="${TOKEN_TYPES.string}">$1</span>`);

    // Numbers
    result = result.replace(/\b(\d+\.?\d*)\b/g, `<span class="${TOKEN_TYPES.number}">$1</span>`);

    // Keywords
    for (const keyword of keywords) {
        const regex = new RegExp(`\\b(${keyword})\\b`, 'g');
        result = result.replace(regex, `<span class="${TOKEN_TYPES.keyword}">$1</span>`);
    }

    // Function calls
    result = result.replace(/\b([\w]+)(\s*\()/g, `<span class="${TOKEN_TYPES.function}">$1</span>$2`);

    return result;
}

/**
 * Escape HTML special characters
 */
function escapeHtml(text) {
    return text
        .replace(/&/g, '&amp;')
        .replace(/</g, '&lt;')
        .replace(/>/g, '&gt;')
        .replace(/"/g, '&quot;')
        .replace(/'/g, '&#39;');
}

/**
 * Get help content for code mode
 */
export function getCodeHelpContent() {
    return `Add a colon followed by a language name to apply syntax
highlighting to the text below.

Examples
code: swift
code: js
code: py

Disables formatting strip on paste.
Disables all link features.`;
}

/**
 * Check if code mode is active
 * @param {string} content - Note content
 * @returns {boolean}
 */
export function isCodeMode(content) {
    const firstLine = content.split('\n')[0].trim().toLowerCase();
    return firstLine === 'code' || firstLine.startsWith('code:');
}
