/**
 * JottNote Bookmarks Service
 * Handles bookmark storage and Chrome bookmarks import
 */

const BOOKMARKS_KEY = 'jottnote_bookmarks';

// Reserved keywords that cannot be used as tags
const RESERVED_TAG_KEYWORDS = ['add', 'search', 'edit', 'end', 'save', 'help', 'import'];

/**
 * Validate and filter tags to remove reserved keywords
 * @param {Array<string>} tags - Array of tag strings
 * @returns {Object} { validTags: Array, invalidTags: Array }
 */
export function validateTags(tags) {
  const validTags = [];
  const invalidTags = [];

  for (const tag of tags) {
    const lowerTag = tag.toLowerCase().trim();
    if (RESERVED_TAG_KEYWORDS.includes(lowerTag)) {
      invalidTags.push(tag);
    } else if (tag.trim()) {
      validTags.push(tag.trim());
    }
  }

  return { validTags, invalidTags };
}

/**
 * Bookmark schema:
 * {
 *   id: string (UUID),
 *   url: string,
 *   title: string,
 *   tags: array of strings,
 *   description: string,
 *   createdAt: string (ISO date),
 *   modifiedAt: string (ISO date),
 *   accessedAt: string (ISO date),
 *   chromeBookmarkId: string | null
 * }
 */

/**
 * Generate a UUID for new bookmarks
 */
function generateId() {
  return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, function(c) {
    const r = Math.random() * 16 | 0;
    const v = c === 'x' ? r : (r & 0x3 | 0x8);
    return v.toString(16);
  });
}

/**
 * Get all bookmarks from storage, optionally filtered by tag
 * @param {string} tag - Optional tag to filter by
 * @returns {Promise<Array>} Array of bookmarks sorted by accessedAt (most recent first)
 */
export async function getBookmarks(tag = null) {
  try {
    const result = await chrome.storage.local.get(BOOKMARKS_KEY);
    let bookmarks = result[BOOKMARKS_KEY] || [];

    // Filter by tag if provided
    if (tag) {
      bookmarks = bookmarks.filter(b =>
        b.tags && b.tags.some(t => t.toLowerCase() === tag.toLowerCase())
      );
    }

    // Sort by accessedAt descending (most recent first)
    return bookmarks.sort((a, b) => new Date(b.accessedAt) - new Date(a.accessedAt));
  } catch (error) {
    console.error('Error getting bookmarks:', error);
    return [];
  }
}

/**
 * Get a single bookmark by ID
 * @param {string} id - Bookmark ID
 * @returns {Promise<Object|null>} Bookmark object or null if not found
 */
export async function getBookmark(id) {
  try {
    const bookmarks = await getBookmarks();
    return bookmarks.find(b => b.id === id) || null;
  } catch (error) {
    console.error('Error getting bookmark:', error);
    return null;
  }
}

/**
 * Save a bookmark (create or update)
 * @param {Object} bookmark - Bookmark object
 * @returns {Promise<Object>} Saved bookmark
 */
export async function saveBookmark(bookmark) {
  try {
    // Get raw bookmarks from storage (not filtered/sorted)
    const result = await chrome.storage.local.get(BOOKMARKS_KEY);
    const bookmarks = result[BOOKMARKS_KEY] || [];

    // Find existing bookmark by ID
    const existingIndex = bookmarks.findIndex(b => b.id === bookmark.id);

    if (existingIndex >= 0) {
      // Update existing bookmark
      bookmark.modifiedAt = new Date().toISOString();
      bookmarks[existingIndex] = bookmark;
    } else {
      // Create new bookmark
      if (!bookmark.id) {
        bookmark.id = generateId();
      }
      if (!bookmark.createdAt) {
        bookmark.createdAt = new Date().toISOString();
      }
      bookmark.modifiedAt = new Date().toISOString();
      if (!bookmark.accessedAt) {
        bookmark.accessedAt = new Date().toISOString();
      }
      bookmarks.push(bookmark);
    }

    // Save to storage
    await chrome.storage.local.set({ [BOOKMARKS_KEY]: bookmarks });
    return bookmark;
  } catch (error) {
    console.error('Error saving bookmark:', error);
    throw error;
  }
}

/**
 * Delete a bookmark by ID
 * @param {string} id - Bookmark ID
 * @returns {Promise<boolean>} True if deleted, false otherwise
 */
export async function deleteBookmark(id) {
  try {
    const bookmarks = await getBookmarks();
    const filteredBookmarks = bookmarks.filter(b => b.id !== id);

    if (filteredBookmarks.length === bookmarks.length) {
      return false; // No bookmark found with this ID
    }

    await chrome.storage.local.set({ [BOOKMARKS_KEY]: filteredBookmarks });
    return true;
  } catch (error) {
    console.error('Error deleting bookmark:', error);
    return false;
  }
}

/**
 * Search bookmarks by query (searches title, URL, tags, description)
 * @param {string} query - Search query
 * @returns {Promise<Array>} Array of matching bookmarks
 */
export async function searchBookmarks(query) {
  try {
    const bookmarks = await getBookmarks();
    const lowerQuery = query.toLowerCase();

    return bookmarks.filter(b =>
      b.title.toLowerCase().includes(lowerQuery) ||
      b.url.toLowerCase().includes(lowerQuery) ||
      (b.description && b.description.toLowerCase().includes(lowerQuery)) ||
      (b.tags && b.tags.some(tag => tag.toLowerCase().includes(lowerQuery)))
    );
  } catch (error) {
    console.error('Error searching bookmarks:', error);
    return [];
  }
}

/**
 * Get all unique tags with counts
 * @returns {Promise<Array>} Array of {tag, count} objects sorted by count descending
 */
export async function getAllTags() {
  try {
    const bookmarks = await getBookmarks();
    const tagCounts = {};

    // Count tags
    for (const bookmark of bookmarks) {
      if (bookmark.tags) {
        for (const tag of bookmark.tags) {
          const lowerTag = tag.toLowerCase();
          tagCounts[lowerTag] = (tagCounts[lowerTag] || 0) + 1;
        }
      }
    }

    // Convert to array and sort by count descending
    return Object.entries(tagCounts)
      .map(([tag, count]) => ({ tag, count }))
      .sort((a, b) => b.count - a.count);
  } catch (error) {
    console.error('Error getting tags:', error);
    return [];
  }
}

/**
 * Update the accessedAt timestamp for a bookmark
 * @param {string} id - Bookmark ID
 * @returns {Promise<void>}
 */
export async function updateAccessTime(id) {
  try {
    const bookmark = await getBookmark(id);
    if (bookmark) {
      bookmark.accessedAt = new Date().toISOString();
      await saveBookmark(bookmark);
    }
  } catch (error) {
    console.error('Error updating access time:', error);
  }
}

/**
 * Import bookmarks from Chrome
 * @returns {Promise<number>} Number of bookmarks imported
 */
export async function importFromChrome() {
  try {
    // Request bookmarks permission at runtime (declared as optional_permissions)
    const granted = await chrome.permissions.request({ permissions: ['bookmarks'] });
    if (!granted) {
      throw new Error('Bookmarks permission was denied');
    }

    // Get all Chrome bookmarks
    const tree = await chrome.bookmarks.getTree();
    const flatBookmarks = flattenBookmarkTree(tree[0]);

    // Get existing bookmarks to avoid duplicates
    const existing = await getBookmarks();
    const existingUrls = new Set(existing.map(b => b.url));

    let imported = 0;
    for (const chromeBookmark of flatBookmarks) {
      // Skip if already exists
      if (existingUrls.has(chromeBookmark.url)) continue;

      const bookmark = {
        id: generateId(),
        url: chromeBookmark.url,
        title: chromeBookmark.title || extractTitleFromUrl(chromeBookmark.url),
        tags: [],
        description: '',
        createdAt: new Date(chromeBookmark.dateAdded || Date.now()).toISOString(),
        modifiedAt: new Date(chromeBookmark.dateAdded || Date.now()).toISOString(),
        accessedAt: new Date(chromeBookmark.dateAdded || Date.now()).toISOString(),
        chromeBookmarkId: chromeBookmark.id
      };

      await saveBookmark(bookmark);
      imported++;
    }

    return imported;
  } catch (error) {
    console.error('Error importing Chrome bookmarks:', error);
    throw error;
  }
}

/**
 * Flatten Chrome bookmark tree into array
 * @param {Object} node - Bookmark tree node
 * @param {Array} result - Accumulator array
 * @returns {Array} Flat array of bookmarks
 */
function flattenBookmarkTree(node, result = []) {
  if (node.url) {
    result.push(node);
  }
  if (node.children) {
    for (const child of node.children) {
      flattenBookmarkTree(child, result);
    }
  }
  return result;
}

/**
 * Extract a readable title from URL
 * @param {string} url - URL string
 * @returns {string} Extracted title
 */
/**
 * Export bookmarks to JSON format (for download)
 * @param {Array} bookmarks - Array of bookmark objects
 * @returns {string} JSON string
 */
export function exportBookmarksToJSON(bookmarks) {
    return JSON.stringify(bookmarks, null, 2);
}

/**
 * Import bookmarks from JSON data
 * @param {string} jsonData - JSON string of bookmarks array
 * @returns {Promise<Object>} { imported, skipped, errors }
 */
export async function importBookmarksFromJSON(jsonData) {
    try {
        const importedBookmarks = JSON.parse(jsonData);

        if (!Array.isArray(importedBookmarks)) {
            throw new Error('Invalid bookmarks data format');
        }

        const existing = await getBookmarks();
        const existingUrls = new Set(existing.map(b => b.url));
        const results = { imported: 0, skipped: 0, errors: [] };

        for (const bookmark of importedBookmarks) {
            try {
                if (!bookmark.url) {
                    results.errors.push(`Invalid bookmark: missing URL`);
                    results.skipped++;
                    continue;
                }

                // Skip duplicates (same URL)
                if (existingUrls.has(bookmark.url)) {
                    results.skipped++;
                    continue;
                }

                const newBookmark = {
                    ...bookmark,
                    id: generateId(),
                    title: bookmark.title || extractTitleFromUrl(bookmark.url),
                    tags: bookmark.tags || [],
                    description: bookmark.description || '',
                    createdAt: bookmark.createdAt || new Date().toISOString(),
                    modifiedAt: new Date().toISOString(),
                    accessedAt: bookmark.accessedAt || new Date().toISOString(),
                    chromeBookmarkId: bookmark.chromeBookmarkId || null
                };

                await saveBookmark(newBookmark);
                existingUrls.add(newBookmark.url);
                results.imported++;
            } catch (err) {
                results.errors.push(`Error importing bookmark: ${err.message}`);
                results.skipped++;
            }
        }

        return results;
    } catch (err) {
        throw new Error(`Failed to parse bookmarks data: ${err.message}`);
    }
}

export function extractTitleFromUrl(url) {
  try {
    const urlObj = new URL(url);
    let title = urlObj.hostname.replace('www.', '');

    // Try to get a better title from pathname
    const pathParts = urlObj.pathname.split('/').filter(p => p);
    if (pathParts.length > 0) {
      const lastPart = pathParts[pathParts.length - 1];
      // Use last path part if it looks meaningful
      if (lastPart.length > 2 && lastPart.length < 50) {
        title = lastPart.replace(/-|_/g, ' ');
      }
    }

    return title;
  } catch (e) {
    return url;
  }
}
