/**
 * JottNote Background Service Worker
 * Handles context menus and global commands
 */

importScripts('lib/ExtPay.js');

// Initialize ExtensionPay
const extpay = ExtPay('jottnote');
extpay.startBackground();

// Create context menus on install
chrome.runtime.onInstalled.addListener(async () => {
    // Check user preference for side panel mode
    const result = await chrome.storage.local.get('jottnote_settings');
    const settings = result.jottnote_settings || {};
    const sidePanelMode = settings.sidePanelMode || false;

    // Enable side panel to open when extension icon is clicked (if user prefers)
    if (sidePanelMode) {
        chrome.sidePanel.setPanelBehavior({ openPanelOnActionClick: true }).catch((error) => {
            console.error('Side panel setup failed:', error);
        });
    } else {
        // Disable side panel behavior (will use popup instead)
        chrome.sidePanel.setPanelBehavior({ openPanelOnActionClick: false }).catch((error) => {
            console.error('Side panel setup failed:', error);
        });
    }

    // Context menu for selected text
    chrome.contextMenus.create({
        id: 'save-to-jottnote',
        title: 'Save to JottNote',
        contexts: ['selection']
    });

    chrome.contextMenus.create({
        id: 'append-to-jottnote',
        title: 'Append to current JottNote',
        contexts: ['selection']
    });

    // Context menu for page (save URL)
    chrome.contextMenus.create({
        id: 'save-page-to-jottnote',
        title: 'Save page URL to JottNote',
        contexts: ['page']
    });
});

// Handle context menu clicks
chrome.contextMenus.onClicked.addListener(async (info, tab) => {
    const STORAGE_KEY = 'jottnote_notes';
    const CURRENT_NOTE_KEY = 'jottnote_current';

    try {
        switch (info.menuItemId) {
            case 'save-to-jottnote': {
                // Create new note with selected text
                const selectedText = info.selectionText || '';
                const note = createNewNote(selectedText);
                await saveNewNote(note);
                break;
            }

            case 'append-to-jottnote': {
                // Append to current note
                const selectedText = info.selectionText || '';
                await appendToCurrentNote(selectedText);
                break;
            }

            case 'save-page-to-jottnote': {
                // Save page title and URL
                const pageInfo = `${tab.title}\n${tab.url}`;
                const note = createNewNote(pageInfo);
                await saveNewNote(note);
                break;
            }
        }
    } catch (error) {
        console.error('Context menu action failed:', error);
    }
});

/**
 * Generate UUID
 */
function generateId() {
    return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, function (c) {
        const r = Math.random() * 16 | 0;
        const v = c === 'x' ? r : (r & 0x3 | 0x8);
        return v.toString(16);
    });
}

/**
 * Create a new note object
 */
function createNewNote(content) {
    const now = new Date().toISOString();
    return {
        id: generateId(),
        content,
        createdAt: now,
        modifiedAt: now
    };
}

/**
 * Save a new note to storage
 */
async function saveNewNote(note) {
    const STORAGE_KEY = 'jottnote_notes';
    const CURRENT_NOTE_KEY = 'jottnote_current';

    const result = await chrome.storage.local.get(STORAGE_KEY);
    const notes = result[STORAGE_KEY] || [];

    notes.unshift(note);

    await chrome.storage.local.set({
        [STORAGE_KEY]: notes,
        [CURRENT_NOTE_KEY]: note.id
    });
}

/**
 * Append text to current note
 */
async function appendToCurrentNote(text) {
    const STORAGE_KEY = 'jottnote_notes';
    const CURRENT_NOTE_KEY = 'jottnote_current';

    const result = await chrome.storage.local.get([STORAGE_KEY, CURRENT_NOTE_KEY]);
    const notes = result[STORAGE_KEY] || [];
    const currentId = result[CURRENT_NOTE_KEY];

    if (currentId) {
        const noteIndex = notes.findIndex(n => n.id === currentId);
        if (noteIndex >= 0) {
            notes[noteIndex].content += '\n' + text;
            notes[noteIndex].modifiedAt = new Date().toISOString();

            // Move to front
            const note = notes.splice(noteIndex, 1)[0];
            notes.unshift(note);

            await chrome.storage.local.set({ [STORAGE_KEY]: notes });
            return;
        }
    }

    // No current note, create new one
    const note = createNewNote(text);
    await saveNewNote(note);
}

// Handle keyboard commands
chrome.commands.onCommand.addListener((command) => {
    if (command === '_execute_action') {
        // This is handled automatically by Chrome to open the popup
    }
});

/**
 * Background Timer Management
 * Keeps timer running even when popup is closed
 */

const TIMER_STORAGE_KEY = 'jottnote_timer_state';
const TIMER_ALARM_NAME = 'jottnote_timer_tick';

// In-memory timer state
let timerState = {
    running: false,
    paused: false,
    type: null,
    startTime: null,
    pauseStartedAt: null,
    duration: 0,
    remaining: 0,
    title: null,
    workDuration: 0,
    breakDuration: 0,
    isBreak: false,
    completedRounds: 0
};

/**
 * Load timer state from storage on startup
 */
async function loadTimerState() {
    try {
        const result = await chrome.storage.local.get(TIMER_STORAGE_KEY);
        if (result[TIMER_STORAGE_KEY]) {
            timerState = result[TIMER_STORAGE_KEY];
            if (timerState.running && !timerState.paused) {
                // Restart alarm if timer was running
                startTimerAlarm();
            }
        }
    } catch (error) {
        console.error('Failed to load timer state:', error);
    }
}

/**
 * Save timer state to storage
 */
async function saveTimerState() {
    try {
        await chrome.storage.local.set({ [TIMER_STORAGE_KEY]: timerState });
    } catch (error) {
        console.error('Failed to save timer state:', error);
    }
}

/**
 * Start timer alarm
 * Chrome enforces a minimum period of 1 minute for MV3 alarms.
 * We use 1-minute ticks for background updates; the popup uses
 * its own setInterval for real-time display when open.
 */
function startTimerAlarm() {
    chrome.alarms.create(TIMER_ALARM_NAME, {
        delayInMinutes: 0.5,
        periodInMinutes: 1
    });
}

/**
 * Stop timer alarm
 */
function stopTimerAlarm() {
    chrome.alarms.clear(TIMER_ALARM_NAME);
}

/**
 * Handle timer tick
 */
function handleTimerTick() {
    if (!timerState.running || timerState.paused) return;

    if (timerState.type === 'stopwatch') {
        // Stopwatch just counts up
        const elapsed = Math.floor((Date.now() - timerState.startTime) / 1000);
        broadcastTimerUpdate({ seconds: elapsed });
    } else if (timerState.type === 'countdown' || timerState.type === 'pomodoro') {
        // Calculate remaining time
        const elapsed = Math.floor((Date.now() - timerState.startTime) / 1000);
        timerState.remaining = Math.max(0, timerState.duration - elapsed);

        broadcastTimerUpdate({ seconds: timerState.remaining });

        // Check if time is up
        if (timerState.remaining <= 0) {
            if (timerState.type === 'countdown') {
                // Stop countdown timer
                timerState.running = false;
                stopTimerAlarm();
                broadcastTimerComplete({ type: 'countdown' });
            } else if (timerState.type === 'pomodoro') {
                // Switch pomodoro phase
                const wasBreak = timerState.isBreak;
                timerState.isBreak = !timerState.isBreak;

                if (wasBreak && !timerState.isBreak) {
                    timerState.completedRounds++;
                }

                timerState.duration = timerState.isBreak ? timerState.breakDuration : timerState.workDuration;
                timerState.remaining = timerState.duration;
                timerState.startTime = Date.now();
                timerState.pauseStartedAt = null;

                broadcastTimerUpdate({ phaseChange: true });
            }
        }

        saveTimerState();
    }
}

/**
 * Broadcast timer update to popup if open
 */
function broadcastTimerUpdate(data) {
    chrome.runtime.sendMessage({
        type: 'TIMER_TICK',
        state: timerState,
        data
    }).catch(() => {
        // Popup not open, that's okay
    });
}

/**
 * Broadcast timer completion
 */
function broadcastTimerComplete(data) {
    chrome.runtime.sendMessage({
        type: 'TIMER_COMPLETE',
        data
    }).catch(() => {
        // Popup not open, that's okay
    });
    saveTimerState();
}

/**
 * Handle messages from popup
 */
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
    switch (message.type) {
        case 'TIMER_START': {
            const config = message.config || {};
            const duration = Math.max(0, Number(config.duration) || 0);
            const work = Math.max(0, Number(config.work) || 0);
            const brk = Math.max(0, Number(config.break) || 0);

            timerState = {
                running: true,
                paused: false,
                type: config.type || 'stopwatch',
                startTime: Date.now(),
                pauseStartedAt: null,
                duration: duration,
                remaining: duration,
                title: config.title || null,
                workDuration: work,
                breakDuration: brk,
                isBreak: false,
                completedRounds: 0
            };
            startTimerAlarm();
            saveTimerState();
            sendResponse({ success: true });
            break;
        }

        case 'TIMER_STOP':
            timerState.running = false;
            timerState.paused = false;
            stopTimerAlarm();
            saveTimerState();
            sendResponse({ success: true });
            break;

        case 'TIMER_PAUSE':
            if (timerState.running) {
                if (timerState.paused) {
                    // Resume — shift startTime forward by pause duration
                    const pauseDuration = Date.now() - (timerState.pauseStartedAt || Date.now());
                    timerState.startTime += pauseDuration;
                    timerState.paused = false;
                    timerState.pauseStartedAt = null;
                    startTimerAlarm();
                } else {
                    // Pause — record when pause began
                    timerState.paused = true;
                    timerState.pauseStartedAt = Date.now();
                    stopTimerAlarm();
                }
                saveTimerState();
                sendResponse({ success: true, paused: timerState.paused });
            } else {
                sendResponse({ success: false });
            }
            break;

        case 'TIMER_RESTART':
            if (timerState.running) {
                timerState.startTime = Date.now();
                timerState.pauseStartedAt = null;
                timerState.paused = false;
                timerState.remaining = timerState.duration;
                saveTimerState();
                sendResponse({ success: true });
            } else {
                sendResponse({ success: false });
            }
            break;

        case 'TIMER_GET_STATE':
            sendResponse({ state: timerState });
            break;

        case 'UPDATE_SIDE_PANEL_MODE':
            const mode = message.enabled;
            chrome.sidePanel.setPanelBehavior({ openPanelOnActionClick: mode })
                .then(() => {
                    sendResponse({ success: true });
                })
                .catch((error) => {
                    console.error('Side panel behavior update failed:', error);
                    sendResponse({ success: false, error: error.message });
                });
            return true;
    }
    return true; // Keep message channel open for async response
});

/**
 * Handle all alarm events (single consolidated listener)
 */
chrome.alarms.onAlarm.addListener(async (alarm) => {
    if (alarm.name === TIMER_ALARM_NAME) {
        handleTimerTick();
    } else if (alarm.name === BACKUP_ALARM_NAME) {
        const result = await chrome.storage.local.get('jottnote_settings');
        const settings = result.jottnote_settings || {};
        await createBackup(settings.backupCount || 12);
    }
});

// Load timer state on startup
loadTimerState();

/**
 * Listen for settings changes to update side panel behavior
 */
chrome.storage.onChanged.addListener((changes, areaName) => {
    if (areaName === 'local' && changes.jottnote_settings) {
        const newSettings = changes.jottnote_settings.newValue || {};
        const sidePanelMode = newSettings.sidePanelMode || false;

        // Update side panel behavior based on new setting
        chrome.sidePanel.setPanelBehavior({ openPanelOnActionClick: sidePanelMode }).catch((error) => {
            console.error('Side panel behavior update failed:', error);
        });

        // Re-schedule backup alarm if backup settings changed
        scheduleBackupAlarm(newSettings);
    }
});

/**
 * Backup Management
 * Periodically saves a snapshot of all notes based on backupInterval setting
 */
const BACKUP_ALARM_NAME = 'jottnote_backup';
const BACKUP_STORAGE_KEY = 'jottnote_backups';

/**
 * Schedule (or clear) the backup alarm based on settings
 * Slider values: 0=10min, 1=30min, 2=1hr, 3=3hr, 4=6hr, 5=12hr, 6=1day, 7=1week, 8=Never
 */
async function scheduleBackupAlarm(settings) {
    const interval = settings?.backupInterval ?? 8; // default: Never
    if (interval >= 8) {
        // Never — clear any existing alarm
        await chrome.alarms.clear(BACKUP_ALARM_NAME);
        return;
    }

    const minutes = [10, 30, 60, 180, 360, 720, 1440, 10080][interval];
    await chrome.alarms.create(BACKUP_ALARM_NAME, {
        delayInMinutes: minutes,
        periodInMinutes: minutes
    });
}

/**
 * Create a backup of all notes
 */
async function createBackup(maxBackups) {
    try {
        const notesResult = await chrome.storage.local.get('jottnote_notes');
        const notes = notesResult.jottnote_notes || [];

        if (notes.length === 0) return; // Nothing to back up

        const backupsResult = await chrome.storage.local.get(BACKUP_STORAGE_KEY);
        const backups = backupsResult[BACKUP_STORAGE_KEY] || [];

        // Create backup entry
        const backup = {
            id: Date.now().toString(36),
            timestamp: new Date().toISOString(),
            noteCount: notes.length,
            data: notes
        };

        backups.unshift(backup);

        // Trim to max backups
        const limit = maxBackups || 12;
        while (backups.length > limit) {
            backups.pop();
        }

        await chrome.storage.local.set({ [BACKUP_STORAGE_KEY]: backups });
    } catch (error) {
        console.error('Backup failed:', error);
    }
}

// Backup alarm handled by consolidated listener above

// Schedule backup alarm on startup
(async () => {
    const result = await chrome.storage.local.get('jottnote_settings');
    const settings = result.jottnote_settings || {};
    await scheduleBackupAlarm(settings);
})();
